/*
** Author:      James Painter
** Purpose:     Microcode interface header file for micro engine
**
** Copyright (c), 1988 GRAIL, University of Washington
**
** $Revision$
** $Date$
** $Locker$
** $Log$
*/


/* Exported types */
typedef enum { HiRes, LowRes } mode; /* Argument type for uEInitialize */

/* Exported procedures */
  /*  uEInitialize starts up the uEngine package */
extern int uEInitialize( /* mode */ );

  /* uEFinalize flushes any buffer commands and closes uEngine */
extern int uEFinalize  ( /* no args */ );

  /* uEFlush flushes all buffered commands */
extern int uEFlush     ();

  /* uESync waits for the microcode to catch up */
extern int uESync      ();


/* Exported macros */

  /*
    The graphics primitives are all macros (sorry)
   uEclear(color)              - clears the screen to color (int).
   uEClearAndSwap(color)       - clears the screen to color (int) and swaps
                                 buffers.  For use when double buffering.
   uEpoint(x,y,color)          - sets a single pixel at x,y to color.
   uEline (x1,y1,x2,y2,color)  - draws a line from (x1,y1) to (x2,y2) in color.
   uErect (x1,y1,x2,y2,color)  - fills a rect from (x1,y1) to (x2,y2) in color.
   uEshadeline(y,x1,x2,c1,c2)  - interpolates colors from c1 to c2 across a
                               - scanline at y from x1 to x2

  */

/* ------------------------------ PRIVATE ------------------------------ */
#define uEclear(color)    				    \
{							    \
  if (_uEArgs) {					    \
    if (_uEnext + 2 >= _uEend)				    \
      uEFlush();					    \
    *_uEnext = (int) CLEAR;				    \
    if (color != _uEcolor)  {				    \
      _uEcolor = color;					    \
      *_uEnext |= COLOR_BIT;				    \
      *++_uEnext = _uEcolor;			    \
    }							    \
    _uEnext++;                                              \
  }							    \
}

#define uEclearAndSwap(color)    				    \
{							    \
  if (_uEArgs) {					    \
    if (_uEnext + 2 >= _uEend)				    \
      uEFlush();					    \
    *_uEnext = (int) CLEAR_AND_SWAP;			    \
    if (color != _uEcolor)  {				    \
      _uEcolor = color;					    \
      *_uEnext |= COLOR_BIT;				    \
      *++_uEnext = _uEcolor;			    \
    }							    \
    _uEnext++;                                              \
  }							    \
}

#define uEpoint(x,y,color)				    \
{							    \
  if (_uEArgs) {					    \
    if (_uEnext + 3 >= _uEend)				    \
      uEFlush();					    \
    *_uEnext = (int) POINT;				    \
    if (color != _uEcolor) {				    \
      _uEcolor = color;					    \
      *_uEnext |= COLOR_BIT;				    \
      *++_uEnext = _uEcolor;			            \
    }							    \
    _uEnext++;     				            \
    *_uEnext++ = (y << 16) | x;				    \
  }							    \
}

#define uEline(x1,y1,x2,y2,color)                           \
{							    \
  if (_uEArgs) {					    \
    if (_uEnext + 6 >= _uEend)				    \
      uEFlush();					    \
    *_uEnext = (int) LINE;				    \
    if (color != _uEcolor) {				    \
      _uEcolor = color;					    \
      *_uEnext |= COLOR_BIT;				    \
      *++_uEnext = _uEcolor;			            \
    }							    \
    _uEnext++;                                              \
    *_uEnext++ = (y1 << 16) | x1;			    \
    *_uEnext++ = (y2 << 16) | x2;   		            \
  }							    \
}


#define uErect(x1,y1,x2,y2,color)                           \
{							    \
  if (_uEArgs) {					    \
    if (_uEnext + 6 >= _uEend)				    \
      uEFlush();					    \
    *_uEnext = (int) RECT;				    \
    if (color != _uEcolor) {				    \
      _uEcolor = color;					    \
      *_uEnext |= COLOR_BIT;				    \
      *++_uEnext = _uEcolor;        			    \
    }							    \
    _uEnext++;						    \
    *_uEnext++ = (y1 << 16) | x1;			    \
    *_uEnext++ = (y2 << 16) | x2;			    \
  }							    \
}


#define uEshadeline(y,x1,x2,color1, color2)             \
{							    \
  if (_uEArgs) {					    \
    if (_uEnext + 4 >= _uEend)				    \
      uEFlush();					    \
    *_uEnext++ = (y << 16) | (short) SHADELINE;		    \
    *_uEnext++ = (x2 << 16) | x1;			    \
    *_uEnext++ = color1;				    \
    *_uEnext++ = color2;				    \
  }							    \
}


/*
   This structure defines the microcode interface.
   the cmd field is actually BUFFER_SIZE long but is
   declared 1 to avoid increasing the size of the microcode
   load file.  On the host end, the allocated size should be:
   sizeof(Args) + sizeof(short)*(BUFFER_SIZE - 1)
*/
#define BUFFER_SIZE 4096

typedef struct {
  int running;			/* Syncronization flag */
  int count;			/* used size of cmd */
  int cmd[1];			/* Array of packed commands */
} Args;


#ifndef IKC
/* Private variables */
extern Args *_uEArgs;		/* Point to argument block */
extern int *_uEnext;		/* next pointer int _uEArgs->cmd */
extern int *_uEend;           /* end pointer for  _uEArgs->cmd */
extern int   _uEcolor;		/* Current color */
#endif


/* This is a path to the microcode support code */
#define MICROCODE_PATH "/usr/graphics/microcode/uEngine.abs"
#define HR_MICROCODE_PATH "/usr/graphics/microcode/HRuEngine.abs"
#define HR_FBC_INIT_PATH  "/usr/graphics/microcode/fbcregs.hr"

/* Possible values of the running field in the uArgs structure.
 *  These are used to syncronize with the microcode with the
 *  host processor.  RUNNING means the microcode is busy,
 *  WAITING means the microcode is waiting for data from the host.
 *  NOTRUNNING means the microcode has terminated or has not yet
 *  started.
 */
#define RUNNING		0xabcddcba
#define WAITING         0xaaaaaaaa
#define NOTRUNNING	0xffffffff

/* These are the possible op-codes */
typedef enum { CLEAR, LINE, RECT, POINT, SHADELINE,
	       CLEAR_AND_SWAP } OpCodeType;

/* Most commands use the "current color".  The current color can be
 * sent preceeding the other command arguments if the COLOR_BIT modifer
 * is set in the command op code.
 */
#define COLOR_BIT 0x8000

/* Commands are packed into an int array.  The following describe
 *  the command formats for each op code.  When the COLOR_BIT is set
 *  the color (as two shorts:  aaaarrrr ggggbbbb) preceeds the other
 *  command arguments.
 */

/*  op = CLEAR
        no arguments
    op = LINE
         (y1 << 16) | x1
         (y2 << 16) | y2
    op = RECT
         (y1 << 16) | x1
         (y2 << 16) | y2
    op = POINT
         (y1 << 16) | x1
    op = SHADELINE  | (y << 16)
         (x2 << 16) | x1
	 color1
         color2
*/



