/*
  File: ScanLine.c
  Author: K.R. Sloan
  Last Modified: 18 April 1990
  Purpose: hidden-surface rendering of triangles

           This is an alternative to "RenderTriangles".  The interface
           is virtually identical.

           ScanTriangles displays the entire collection of triangles,
           with hidden surfaces removed. 

           The user program should allocate an array of TriangleType3D,
           and fill in the geometry and color information for each vertex.

           The Triangles array is used for scratch storage, may be
           extended, and is meaningless after ScanTriangles returns.
          
           If the user program doesn't cull back-facing triangles, we
           do it here (and that ought to provide enough extra space).
           If not, the user program must allocate an array large enough
           to handle some expansion.  10-20% should be enough...

           Unless...the global TwoSided indicates that we should
           reverse all normals on back-facing triangles.  This may
           not do what you want - tough.    It's purely kludge-control.
           The tasteful user will not find it necessary.

           Similarly, the global LeftHanded indicates that triangles
           follow the left-hand rule for cross-product calculations.
           Adroit users won't need this feature.

           Yet another global, Facets, indicate that the normal to the
           plane containing the vertices should be used INSTEAD OF the
           vertex normals.

           Two more: Background and Transparency.  Background tells
           us the color of the Background.  Transparency indicates 
           the transparency OF EVERY
           SURFACE in the scene.  If Transparency == 0, then only the first
           surface is seen.  If Transparency == 1.0, then only the background
           is seen. 

           CAUTION:  Facets, TwoSided and LeftHanded
           are defined in "Triangles.c".
           If you build a program which uses Scanline, but NOT Triangles,
           then you need to define them, and initialize them, perhaps in your
           main program.

           On the other hand, Transparency and Background are defined
           here, so if you use Triangles, but not Scanline, you need to
           define them yourself.

           Actually, this version doesn't need any extra space - but it does
           change things.

           The global VERBOSE controls output

 */
#include <stdio.h>
#include <math.h>
#include "3D.h"
#include "Lighting.h"
#include "Triangles.h"
#include "Scanline.h"

extern int VERBOSE;
double fabs();

sgpColorType Background;
double Transparency;

#define EPSILON (0.0000000000001)

/*
   Is the (2D) point P inside triangle [A,B,C]?
   If so, fill in the barycentric coordinates and return -1
   If not, return 0
 */
static int BaryCentric(P,A,B,C,a,b,c)
 sgpPointType P,A,B,C;
 double *a, *b, *c;
 {
  double abx, aby, acx, acy, abCROSSac;
  double pax, pay, pbx, pby, pcx, pcy;
  double paCROSSpb, pbCROSSpc, pcCROSSpa; 

  /* check for tiny triangles - nothing is inside them... */
  abx = B.x-A.x; aby  = B.y-A.y;
  acx = C.x-A.x; acy  = C.y-A.y;
  abCROSSac = abx*acy - aby*acx;
  if ((-EPSILON < abCROSSac) && (abCROSSac < EPSILON)) return(0);

  pax = A.x-P.x; pay = A.y-P.y;
  pbx = B.x-P.x; pby = B.y-P.y;
  pcx = C.x-P.x; pcy = C.y-P.y;

  paCROSSpb = pax*pby - pay*pbx;
  pbCROSSpc = pbx*pcy - pby*pcx;
  pcCROSSpa = pcx*pay - pcy*pax;

  *a = pbCROSSpc / abCROSSac;
  *b = pcCROSSpa / abCROSSac;
  *c = 1.0 - *a - *b;

  return( (*a>=0.0) && (*b>=0.0) && (*c>=0.0) ); 
 }

/*
  Annotate a triangle with it's 3D bounding box
 */
static void BoundingBox(t)
 TriangleType3D *t;
 {
  PointType3D VB;

  /* calculate Bounding Box min/max(x,y,z) in StandardViewBox */

  WorldToViewBox3D(t->Vertex[0], &VB);
  t->MinX = VB.x;  t->MaxX = VB.x;
  t->MinY = VB.y;  t->MaxY = VB.y;
  t->MinZ = VB.z;  t->MaxZ = VB.z;

  WorldToViewBox3D(t->Vertex[1], &VB);
  if      (VB.x < t->MinX) t->MinX = VB.x;
  else if (VB.x > t->MaxX) t->MaxX = VB.x;
  if      (VB.y < t->MinY) t->MinY = VB.y;
  else if (VB.y > t->MaxY) t->MaxY = VB.y;
  if      (VB.z < t->MinZ) t->MinZ = VB.z;
  else if (VB.z > t->MaxZ) t->MaxZ = VB.z;

  WorldToViewBox3D(t->Vertex[2], &VB);
  if      (VB.x < t->MinX) t->MinX = VB.x;
  else if (VB.x > t->MaxX) t->MaxX = VB.x;
  if      (VB.y < t->MinY) t->MinY = VB.y;
  else if (VB.y > t->MaxY) t->MaxY = VB.y;
  if      (VB.z < t->MinZ) t->MinZ = VB.z;
  else if (VB.z > t->MaxZ) t->MaxZ = VB.z;

  /* and, might as well initialize this, too */
  t->Looping=0;
 }

/*
  Cull back facing triangles, calculate face normals, etc.
 */
static int PreprocessTriangles(n, T, EyePoint)
 int n;
 TriangleType3D T[];
 PointType3D EyePoint;
 {
  int s,t;
  VectorType3D A, B, C;
  int v;
  double back;

  /* t consumes triangles, s produces them */
  for(s=0,t=0;t<n;t++)
   {
    /*
      cull triangles which have any vertex on wrong side of hither 
      this ought to be fixed to clip the triangle against the hither
      plane - producing either 0, 1, or 2 triangles

      but then, we wouldn't be able to do this in place...
     */

    if (0 == ClipLineHither(&T[t].Vertex[0], &T[t].Vertex[0])) continue;
    if (0 == ClipLineHither(&T[t].Vertex[1], &T[t].Vertex[1])) continue;
    if (0 == ClipLineHither(&T[t].Vertex[2], &T[t].Vertex[2])) continue;

    /* calculate normal */
    
    A.dx = (T[t].Vertex[1].x) - (T[t].Vertex[0].x);
    A.dy = (T[t].Vertex[1].y) - (T[t].Vertex[0].y);
    A.dz = (T[t].Vertex[1].z) - (T[t].Vertex[0].z);
    Normalize3D(&A);

    B.dx = (T[t].Vertex[2].x) - (T[t].Vertex[0].x);
    B.dy = (T[t].Vertex[2].y) - (T[t].Vertex[0].y);
    B.dz = (T[t].Vertex[2].z) - (T[t].Vertex[0].z);
    Normalize3D(&B);

    if (LeftHanded) T[t].N = Cross3D(B,A); else T[t].N = Cross3D(A,B);
    Normalize3D(&T[t].N);

    /*
      cull back-facing (and on-edge) triangles 
      by comparing the face normal with the vector from the EyePoint
      to any vertex

      or, perhaps flip the normals

     */

    C.dx = T[t].Vertex[0].x - EyePoint.x;
    C.dy = T[t].Vertex[0].y - EyePoint.y;
    C.dz = T[t].Vertex[0].z - EyePoint.z;
    Normalize3D(&C);

/*
  Debugging - throw away soon 
    if (VERBOSE && (-EPSILON <= Dot3D(T[t].N, C)))
     {
      fprintf(stderr,"Backwards triangle:\n");
      fprintf(stderr,"[%f,%f,%f][%f,%f,%f],[%f,%f,%f]\n",
                     T[t].Vertex[0].x,T[t].Vertex[0].y,T[t].Vertex[0].z,
                     T[t].Vertex[1].x,T[t].Vertex[1].y,T[t].Vertex[1].z,
                     T[t].Vertex[2].x,T[t].Vertex[2].y,T[t].Vertex[2].z);
      fprintf(stderr,"N = [%f,%f,%f]\n",
                     T[t].N.dx,T[t].N.dy,T[t].N.dz);
      fprintf(stderr,"C = [%f,%f,%f]\n",
                      C.dx,C.dy,C.dz); 
      fprintf(stderr,"N dot C = %g\n",Dot3D(T[t].N,C));
     }
 */
    if (-EPSILON <= Dot3D(T[t].N, C))
     if (!TwoSided) continue;
     else
      {
       T[t].N.dx = -T[t].N.dx;
       T[t].N.dy = -T[t].N.dy;
       T[t].N.dz = -T[t].N.dz;
       for(v=0;v<3;v++)
        {
         T[t].Normal[v].dx = -T[t].Normal[v].dx;
         T[t].Normal[v].dy = -T[t].Normal[v].dy;
         T[t].Normal[v].dz = -T[t].Normal[v].dz;
        }
      }
    /* solve the rest of the plane equation */
    A.dx = T[t].Vertex[0].x; A.dy = T[t].Vertex[0].y; A.dz = T[t].Vertex[0].z;
    T[t].d = -Dot3D(A,T[t].N);

    /* does ANY normal face away from us? */
    /* if so, make it perpendicular to the line of sight to that vertex */
    /* Should we normalize these to unit vectors here?  maybe, but we don't */
    for(v=0;v<3;v++)
     {
      C.dx = T[t].Vertex[v].x - EyePoint.x;
      C.dy = T[t].Vertex[v].x - EyePoint.y;
      C.dz = T[t].Vertex[v].x - EyePoint.z;
      Normalize3D(&C);
      back = Dot3D(C,T[t].Normal[v]);
      if (back>0.0)
       {
        /* it does!, subtract out the away-facing component */
        T[t].Normal[v].dx -= back*C.dx;
        T[t].Normal[v].dy -= back*C.dy;
        T[t].Normal[v].dz -= back*C.dz;
       } 
     }

    /* If we want facets, replace the Vertex normals with the face normal */
    if (Facets) 
     for(v=0;v<3;v++)
      T[t].Normal[v] = T[t].N;

    BoundingBox(&T[t]);

    /* copy forward */
    T[s++] = T[t];
   } 
  return(s); /* number of surviving triangles */
 }

/*
  Compare two triangles by maximum Y coordinate  
 */
int ByMaxY(s,t)
 TriangleType3D *s, *t;
 {
  double dy;

  dy = (s->MaxY) - (t->MaxY);
  if      (dy<0.0) return(-1);
  else if (0.0<dy) return(1);
  return(0);
 }

/*
  list of hits for the ray through the center of a pixel
 */

typedef struct HT
         {
          int t;
          double u0, u1, u2;
          double z;
	 } HitType;

#define MaxHits (100)
static HitType HitList[102];


/*
  Render 'em
 */
void ScanTriangles(n, T, Tmax)
 int n;
 TriangleType3D T[];
 int Tmax;
 {
  PointType3D EyePoint;
  VectorType3D OpticalAxis, Up;
  sgpStateType sgpState;
  sgpColorType Diffuse, Specular, PixelColor, HitColor;
  double Specularity;
  double t1, t2; /* transparency */
  int m;
  int t, s;
  sgpPointType P, S, PStart, PStop, SStart, SStop, VStart, VStop;
  sgpPointType V0,V1,V2,V;
  double xlo, xhi, ylo, yhi;
  double Px, Py, Pdx, Pdy, width, height;
  double xMin, xMax;
  int yMin, yMax;
  int Hit,h;
  PointType3D VB0, VB1, VB2;
  double NewZ,z,b0,b1,b2,u0,u1,u2; 
  PointType3D W;
  VectorType3D N;
  int Pixels, InBoundingBox, InsideTriangle, Hits, Paint;

  Pixels = 0; InBoundingBox = 0; InsideTriangle = 0; Paint = 0;
  t1 = Transparency; t2 = 1.0-Transparency;

  GetCamera3D(&EyePoint, &OpticalAxis, &Up);

  /* we need a unit-length Optical Axis below, so normalize it here */
  Normalize3D(&OpticalAxis);

  m = PreprocessTriangles(n, T, EyePoint);

  if (m <= 0) return; /* nothing to see */

  /*
    find the number of physical pixels  Beware: dangerous turns ahead!
   */
  sgpInquire(&sgpState);

  /*
    start with the Window boundaries, in Viewing-plane coordinates
   */
  VStart.x = sgpState.Window.Left;  VStop.x = sgpState.Window.Right;
  VStart.y = sgpState.Window.Top;   VStop.y = sgpState.Window.Bottom;

  /*
    enforce proper order in World coordinates - see sort by MaxY below
   */
  if (VStart.x < VStop.x) {  xlo = VStart.x; xhi = VStop.x;}
  else                    {  xhi = VStart.x; xlo = VStop.x;}
  if (VStart.y < VStop.y) {  ylo = VStart.y; yhi = VStop.y;}
  else                    {  yhi = VStart.y; ylo = VStop.y;}

  /* 
    determine pixel boundaries
   */
  VStart.x = xlo; VStop.x = xhi;       /* lo to hi */
  VStart.y = yhi; VStop.y = ylo;       /* hi to lo */
  sgpWorldToScreen(VStart, &SStart);  sgpScreenToPhysical(SStart, &PStart);
  sgpWorldToScreen(VStop,  &SStop);   sgpScreenToPhysical(SStop,  &PStop);

  /*
    put the pixel boundaries in the middle of the pixels
    to avoid numerical problems.  Follow carefully, now:

    Suppose the underlying SGP truncates when moving from
    floating to fixed point Physical coordinates.  Then, small numerical
    errors can turn 123.0 into 122.9999999999, and we skip a pixel, or a 
    scanline.  So, we should make all the Physical coordinates be of the
    form 123.5 

    But, suppose that SGP rounds up(down) instead of truncating?  Then 123.5 
    might change to 123.4999999999(123.500000001).  ARRRRGH!

    So, we settle on Physical coordinates of the form 123.25 !?!

    All this to avoid the unclean alternative of adding sgpPixel(iX, iY)!
   */

  PStart.x = 0.25 + (int)PStart.x;  PStart.y = 0.25 + (int)PStart.y;
  PStop.x  = 0.25 + (int)PStop.x;   PStop.y  = 0.25 + (int)PStop.y;

  /*
    figure out which way we're going in Physical coordinates 
   */
  if (PStart.x < PStop.x) { Pdx =  1.0; width  = PStop.x  - PStart.x; } 
  else                    { Pdx = -1.0; width  = PStart.y - PStop.x;  }
  if (PStart.y < PStop.y) { Pdy =  1.0; height = PStop.y  - PStart.y; }
  else                    { Pdy = -1.0; height = PStart.y - PStop.y;  }

  /*
    and widen the ranges to make sure we cover ALL of the window
    We rely on 2D clipping to remove the potential extra pixels
   */

  PStart.x -= Pdx; PStop.x += Pdx; width  += 2.0;
  PStart.y -= Pdy; PStop.y += Pdy; height += 2.0;

  /*
    sort the triangles by MaxY to cut down the number considered
    on each scanline.
   */

  qsort(T, m, sizeof(TriangleType3D), ByMaxY);
  
  yMax = m-1;
  for(Py = 0.0, P.y = PStart.y; Py <= height; Py++, P.y += Pdy)
   {
    P.x = PStart.x; /* any reasonable value will do */
    sgpPhysicalToScreen(P, &S); sgpScreenToWorld(S, &V);

    /* discard triangles off the top */
    for(;(0 <= yMax) && (T[yMax].MinY > V.y);yMax--) ;
    if (yMax < 0) break; /* nobody home - all done */

    /*
      find a superset of the triangles which overlap this scanline,
      and find the extent of x-coordinates covered by this set
     */
    xMin = T[yMax].MinX; xMax = T[yMax].MaxX;
    for(yMin=yMax; yMin>=0;yMin--)
     {
      if (T[yMin].MaxY<V.y) break;
      if (T[yMin].MinX<xMin) xMin = T[yMin].MinX;
      if (T[yMin].MaxX>xMax) xMax = T[yMin].MaxX;
     }

    /*
       T[yMin+1..yMax] may intersect this scanline
       somewhere with x in [xMin, xMax]
     */

    if(1 > (yMax-yMin))                continue; /* none */
    if( (xMax < xlo) || (xhi < xMin) ) continue; /* not relevant */

    /*
      we COULD sort this set of triangles on MinX, and play the
      same game in x that we just played in y.

      But, that would be harder - we would have to allocate storage
      to keep the x-sorted list.  And, it's not clear how much it would
      save.  If you're ambitious, implement active-edge lists, etc.
     */

    for(Px = 0.0, P.x = PStart.x; Px <= width; Px++, P.x += Pdx)
     {
      sgpPhysicalToScreen(P, &S); sgpScreenToWorld(S, &V);

      if (V.x  < xMin) continue;  /* not there, yet */
      if (xMax < V.x ) break;     /* done with this scanline */
      Pixels++;
      Hit = 0;
      for(t=yMax;t>yMin;t--)
       {
        if(  (T[t].MinX <= V.x) && (V.x <= T[t].MaxX) 
           &&(T[t].MinY <= V.y) && (V.y <= T[t].MaxY) )
         {
          InBoundingBox++;  
          /* this triangle's bounding box surrounds this pixel */

          /* project the triangle to the screen */
          /*
            This is a good place to make changes.  We had to
            project all of the vertices when we calculted the
            bounding box.  If we augment the Triangles structure
            to include the Viewbox coordinates of each vertex,
            then we save these projections

            The reason we don't is that the storage is excessive,
            and this may not be the time sink - the BaryCentric
            calculation below is the probable time sink.  To save time,
            figure out how to speed THAT up.
           */ 

          /* first, to the standard viewbox */
          WorldToViewBox3D(T[t].Vertex[0], &VB0);
          WorldToViewBox3D(T[t].Vertex[1], &VB1);
          WorldToViewBox3D(T[t].Vertex[2], &VB2);

          /* and then, to the 2D world */
          V0.x = VB0.x; V0.y = VB0.y;
          V1.x = VB1.x; V1.y = VB1.y;
          V2.x = VB2.x; V2.y = VB2.y;

          /* 
            calculate the barycentric coordinates of this pixel
            with respect to this (projected) triangle

            if BaryCentric returns TRUE,
            this pixel is contained in this triangle

            Note that we do this in the 2D projection plane.  We
            could do it all in 3D (and remove the projections above).

            Exercise: find the barycentric coordinates (with respect 
            to this triangle) of the intersection point of the line
            which goes through the EyePoint and the center of this pixel
            and the plane of this triangle - all calculated in 3D World
            coordinates.  (Cheaply, please)
           */

          if(BaryCentric(V,V0,V1,V2,&b0,&b1,&b2))
           {
            InsideTriangle++;
            /*
              calculate the z-coordinate (in ViewBox coordinates)
              of the point on the 3D triangle 
              with these barycentric coordinates
             */
            NewZ  = b0*VB0.z + b1*VB1.z + b2*VB2.z;

            /*
              insert this hit into a short list - save the
              identity of the
              triangle, the barycentric coordinates, and the z value
             */
            
            for(h=Hit; h>0; h--)
             if (HitList[h-1].z > NewZ) HitList[h] = HitList[h-1]; 
             else break;
            HitList[h].t  =  t;
            HitList[h].u0 = b0;
            HitList[h].u1 = b1; 
            HitList[h].u2 = b2;
            HitList[h].z = NewZ;
            if (Hit < MaxHits) Hit++;
           }
	 }
       }

      /*
        If we have any hits, elaborate the positions, and normals,
        at this point.
        Evaluate the colors, combine, and plot the pixel.

        As long as we are renndering pixel-by-pixel, let's go all the way
        and evaluate the lighting model at every pixel.  And, as long as 
        we're doing that, let's interpolate normals.  The planes of the
        triangular patches dictate the VISIBILITY of each point, but we
        can model the lighting as if the surfaces were curved.  

        One minor hitch - it's possible that the normal will tilt AWAY
        from us, even though we can see the surface.  This does severe 
        violence to the Lighting calcultion.  We fixed it (in 
        PreprocessTriangles) by clamping each vertex normal so that
        AT WORST it is perpendicular to the line of sight.
        We don't Normalize there (maybe we should?) and the lighting model
        wants unit normals, so we do it here.  

        All of this lighting calculation is EXPENSIVE!  Again, the alternative
        is to store colors at the vertices (we could recruit "Diffuse", for
        example - so it wouldn't cost any more storage) and interpolate
        colors.  But then, the pictures would look identical to those 
        produced by "RenderTriangles" - why bother?

        Or, we could do real Phong shading (go read the original paper)

        Oh yeah, as long as we're interpolating, why not interpolate
        surface characteristics, too?  Just imagine the bizarre effects
        we can get!
       */

      if (!Hit) continue;  /* someone else painted the background */

      PixelColor = Background; /* the last surface */

      while(Hit--)
       {
        s  = HitList[Hit].t;
        u0 = HitList[Hit].u0;
        u1 = HitList[Hit].u1;
        u2 = HitList[Hit].u2;
        W.x  = u0*T[s].Vertex[0].x
              +u1*T[s].Vertex[1].x
              +u2*T[s].Vertex[2].x;
        W.y  = u0*T[s].Vertex[0].y
              +u1*T[s].Vertex[1].y
              +u2*T[s].Vertex[2].y;
        W.z  = u0*T[s].Vertex[0].z
              +u1*T[s].Vertex[1].z
              +u2*T[s].Vertex[2].z;
        N.dx = u0*T[s].Normal[0].dx
              +u1*T[s].Normal[1].dx
              +u2*T[s].Normal[2].dx;
        N.dy = u0*T[s].Normal[0].dy
              +u1*T[s].Normal[1].dy
              +u2*T[s].Normal[2].dy;
        N.dz = u0*T[s].Normal[0].dz
              +u1*T[s].Normal[1].dz
              +u2*T[s].Normal[2].dz;
        Diffuse.r = u0*T[s].Diffuse[0].r
                   +u1*T[s].Diffuse[1].r
                   +u2*T[s].Diffuse[2].r;
        Diffuse.g = u0*T[s].Diffuse[0].g
                   +u1*T[s].Diffuse[1].g
                   +u2*T[s].Diffuse[2].g;
        Diffuse.b = u0*T[s].Diffuse[0].b
                   +u1*T[s].Diffuse[1].b
                   +u2*T[s].Diffuse[2].b;
        Specular.r = u0*T[s].Specular[0].r
                    +u1*T[s].Specular[1].r
                    +u2*T[s].Specular[2].r;  
        Specular.g = u0*T[s].Specular[0].g
                    +u1*T[s].Specular[1].g
                    +u2*T[s].Specular[2].g;
        Specular.b = u0*T[s].Specular[0].b
                    +u1*T[s].Specular[1].b
                    +u2*T[s].Specular[2].b;
        Specularity =  u0*T[s].Specularity[0]
                      +u1*T[s].Specularity[1]
                      +u2*T[s].Specularity[2];
        SetSurfaceReflectance(Diffuse,Specular,Specularity);
        HitColor = ColorAtPoint(W,N);
        PixelColor.r=t1*PixelColor.r+t2*HitColor.r;
        PixelColor.g=t1*PixelColor.g+t2*HitColor.g;
        PixelColor.b=t1*PixelColor.b+t2*HitColor.b;
       }
      Paint++;
      sgpColor(PixelColor);
      sgpPoint(V);  /* all that, for one measly pixel!  let's do it again */
     }
   }
  /*
    report on our good deeds...
   */
  if (VERBOSE)
   {
    fprintf(stderr,"N              = %10d\n",n);
    fprintf(stderr,"M              = %10d\n",m-1);
    fprintf(stderr,"Pixels         = %10d\n",Pixels);
    fprintf(stderr,"InBoundingBox  = %10d\n",InBoundingBox);
    fprintf(stderr,"InsideTriangle = %10d\n",InsideTriangle);
    fprintf(stderr,"Paint          = %10d\n",Paint);
   }
 }
