/*
  File: Render.c
  Authors: K.R. Sloan   
  Last Modified:  4 December 1989
  Purpose: read a "sgpTriangles" file, and Render under standard lighting
           COMING SOON: the LightingSpec file!
           or...perhaps this should all be absorbed into sgpServer?  
           Note - all measurements are in meters.  

           as an option...render as a hidden-line drawing, with no lighting
           of color information. (-p)
 
           yet another option...no hidden lines! (-P)

           and, finally, high-Quality scan-line output
           -Q turns it on (DEFAULT)
           -G specifies Gouraud shading (which may actually be SLOWER!)

           one more...optionally normalize all vertex normals to unit
           vectors  (the anti-Tony hack!)
           -n [this is now a mandatory option]

           And now, the anti-Lounsbery hack: -2 causes back-facing
           triangles to have all of their normals flipped.  -1 negates
           this, and is the current default.

           And the LeftHanded hack: -l causes triangles to be treated
           according to the left-hand rule, as opposed to the obviously
           correct and superior right-hand rule.

           And the Tensor-product hack: -t causes Previewed triangles
               to have only two sides - the two shortest.  The hope is
               that you get quadrilateral patches.  Obviously, this is
               only useful for rectangular gridded data.

           And, the Flat Faces hack to make vertex normals the
               same as Face normals (-F)
             
           The -M option allows you to specify the Maximum number of
               triangles.  

           The -a option allows you to specify the Ambient

           The -b option allows you to specify the Background
              [default Background = Ambient]

           -w asks for a wireframe display with NO hidden surface
              removal, but WITH backface culling - this is almost
              hidden surface removal, for almost convex objects

           -T t causes surfaces to be transparent
 */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "3D.h"
#include "Lighting.h"
#include "Triangles.h"
#include "Render.h"
#include "Scanline.h"
 
double atof();

int VERBOSE = 0;    /* exported for the real renderers to see */
static int DEBUG = 0;
static int Preview = PREVIEW;
static int Hidden = 1;
static int Quality = 1;
static int Normalize = 1;  /* a mandatory option */
static int TensorProduct = 0;
static sgpColorType InteriorColor = { 1.0, 1.0, 1.0 };
static sgpColorType OutlineColor  = { 0.0, 0.0, 0.0 };

static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s\n", RoutineName);
  fprintf(stderr,"\t\t[-e x y z]    -    (e)yepoint position\n");
  fprintf(stderr,"\t\t[-O x y z]    -    (O)bject position\n");
  fprintf(stderr,"\t\t[-o dx dy dz] -    (o)ptical axis\n");
  fprintf(stderr,"\t\t[-u dx dy dz] -    (u)p vector\n");
  fprintf(stderr,"\t\t[-f]          -    (f)ocal length (in meters)\n");
  fprintf(stderr,"\t\t[-x]          -    e(x)posure\n");
  fprintf(stderr,"\t\t[-a r g b]    -    (a)mbient light\n");
  fprintf(stderr,"\t\t[-b r g b]    -    (b)ackground color\n");
  fprintf(stderr,"\t\t[-1][-2]      -    (1) or (2) sided triangles\n");
  fprintf(stderr,"\t\t[-T t]        -    (T)ransparency\n");
  fprintf(stderr,"\t\t[-l]          -    (l)eft handed triangles\n");
  fprintf(stderr,"\t\t[-Q][-G]      -    (Q)uality or (G)ouraud shading\n");
  fprintf(stderr,"\t\t[-F]          -    (F)lat shading\n");
  fprintf(stderr,"\t\t[-p][-P]      -    (p)review, or REALLY (P)REVIEW\n");
  fprintf(stderr,"\t\t[-w]          -    (w)ireframe\n");
  fprintf(stderr,"\t\t[-t]          -    (t)ensor product surface\n");
  fprintf(stderr,"\t\t[-n]          -    (n)ormalize to unit normals\n");
  fprintf(stderr,"\t\t[-v]          -    (v)erbose\n");
  fprintf(stderr,"\t\t[-M m]        -    (M)aximum number of triangles\n");
  fprintf(stderr,"\t\t[-D]          -    (D)ebug display\n");
 }

static int t=0;
static TriangleType3D *T;
static MaxTriangles = MAXTRIANGLES;

/*
  this program is a (possibly unnecessary) specialization of sgpServer
  the following code was ripped off from there...
 */

static int ServerVERBOSE = 0;    /* don't chatter on so much */
static int StackWARNING = 1;     /* worry about stack */
static int ErrorWARNING = -1;    /* do notice errors       */ 

#define STACKSIZE (45)          
static int StackTop = 0;
static double Stack[STACKSIZE+1]; /* watch those fenceposts! */

/*
    General purpose utilities
 */

static void Push(Value)
 double Value;
 {
  if (StackTop < STACKSIZE)
   Stack[StackTop++] = Value;
  else
   {
    int i;
    (void) fprintf(stderr, "\n%s ERROR: stack overflow\n", RoutineName);
    for(i=StackTop;i>=0;i--)
     {
      fprintf(stderr,"%f ",Stack[i]);
      if (7 == (i%8)) fprintf(stderr,"\n");
     }
    exit(-1);
   }
 }

static double Pop()
 {
  if (StackTop > 0)
   return(Stack[--StackTop]);
  else  
   {
    (void)fprintf(stderr, "\n%s ERROR: attempt made to pop from empty stack",
                           RoutineName);
    return(0.0);
   }
 }

static double GetNumber(InputFile)
 FILE *InputFile;
 {
  double NumberRead;

  (void) fscanf(InputFile, "%lf", &NumberRead); /* errors?  n a a h! */
  return(NumberRead);
 }

static void GobbleComment(InputFile)
 FILE *InputFile;
 {
  char ch;

  do
   ch = getc(InputFile);
  while(ch != '\n');
 }

/*
  supported routines:
    sgpTriangle
 */

static int WARNED = 0;
static void CallsgpTriangle()
 {
  int i;

  if (t >= MaxTriangles) 
   {
    t = MaxTriangles-1;  /* you lied to me... */
    if (!WARNED)
     fprintf(stderr,"More than %d triangles! Use -M, Michael\n",
                       MaxTriangles);
    WARNED = 1;
   }

  for (i = 2; i >=0; i--)
   {
    T[t].Specularity[i] = Pop();
    T[t].Specular[i].b = Pop();   
    T[t].Specular[i].g = Pop(); 
    T[t].Specular[i].r = Pop();
    T[t].Diffuse[i].b = Pop();   
    T[t].Diffuse[i].g = Pop(); 
    T[t].Diffuse[i].r = Pop();
   }

  for (i = 2; i >=0; i--)
   {
    T[t].Normal[i].dz = Pop();   
    T[t].Normal[i].dy = Pop(); 
    T[t].Normal[i].dx = Pop();
    if (Normalize) Normalize3D(&T[t].Normal[i]);
   }

  for (i = 2; i >=0; i--)
   {
    T[t].Vertex[i].z = Pop();   
    T[t].Vertex[i].y = Pop(); 
    T[t].Vertex[i].x = Pop();
   }

  if (Hidden) t++;
  else
   {
    sgpColor(OutlineColor);
    if (TensorProduct)
     {
      double dx,dy,dz,d1,d2,d3;

      dx = (T[t].Vertex[0].x)-(T[t].Vertex[1].x);
      dy = (T[t].Vertex[0].y)-(T[t].Vertex[1].y);
      dz = (T[t].Vertex[0].z)-(T[t].Vertex[1].z);
      d1 = dx*dx + dy*dy + dz*dz;

      dx = (T[t].Vertex[1].x)-(T[t].Vertex[2].x);
      dy = (T[t].Vertex[1].y)-(T[t].Vertex[2].y);
      dz = (T[t].Vertex[1].z)-(T[t].Vertex[2].z);
      d2 = dx*dx + dy*dy + dz*dz;

      dx = (T[t].Vertex[2].x)-(T[t].Vertex[0].x);
      dy = (T[t].Vertex[2].y)-(T[t].Vertex[0].y);
      dz = (T[t].Vertex[2].z)-(T[t].Vertex[0].z);
      d3 = dx*dx + dy*dy + dz*dz;

      if ((d1 <= d2) || (d1 <= d3)) Line3D(T[t].Vertex[0], T[t].Vertex[1]);
      if ((d2 <= d3) || (d2 <= d1)) Line3D(T[t].Vertex[1], T[t].Vertex[2]);
      if ((d3 <= d1) || (d3 <= d2)) Line3D(T[t].Vertex[2], T[t].Vertex[0]);
     }
    else
     {
      Line3D(T[t].Vertex[0], T[t].Vertex[1]);    
      Line3D(T[t].Vertex[1], T[t].Vertex[2]);    
      Line3D(T[t].Vertex[2], T[t].Vertex[0]);    
     }
   }
 }


/*
  In this version of DoFunctionCall, we only support a very few functions
  If you feel the urge to add more, see sgpServer.c for the necessary
  boilerplate
 */
static void DoFunctionCall(FunctionString)
 char *FunctionString;
  {
   if(!strcmp(FunctionString,"sgpTriangle")) CallsgpTriangle();

   else if (ErrorWARNING)
         (void) fprintf(stderr, "\n%s ERROR: unrecognized function name %s\n",
                                RoutineName, FunctionString);

   if (StackWARNING && (StackTop > 0))
    (void) fprintf(stderr, 
            "\n%s WARNING: %d arguments remain on stack after Call to %s\n",
                        RoutineName, StackTop, FunctionString);

  }

void ReadTriangles(InputFile)
 FILE *InputFile;
 {
  char ch;
  char FunctionString[81];

  do
   {
    ch = getc(InputFile);
    switch (ch)
     {
      case '[': break;                            /* ignore brackets */
      case ']': break;
      case '%': GobbleComment(InputFile); break;  /* comments to eol */
      case ' ': break;                            /* ignore white space */
      case '\t': break;
      case '\n': break;
      default:
       if (isalpha(ch))
        {
         (void) ungetc(ch, InputFile);
         (void) fscanf(InputFile, "%80s", FunctionString);
         DoFunctionCall(FunctionString);
        }
       else if (isdigit(ch) || ('.' == ch) || ('-' == ch) || ('+' == ch))
        {
         (void) ungetc(ch, InputFile); /* put the character back */
         Push(GetNumber(InputFile)); /* we push the number on the stack */
        }
       else
        {
         if (!(feof(InputFile)))
          (void) fprintf(stderr, "Unrecognized input character\n");
        }
       break;
     } 
   }
  while (!(feof(InputFile)));

 }

int main(argc, argv)
 int argc;
 char *argv[];
 {
  static int ArgsParsed = 0;
  static PointType3D  EyePoint    = { 10.0, 10.0, 10.0 };
  static VectorType3D OpticalAxis = { -1.0, -1.0, -1.0 };
  static PointType3D Origin       = {  0.0,  0.0,  0.0 };
  static VectorType3D Up          = {  0.0,  1.0,  0.0 };

  static double FocalLength = 0.040;
  static sgpRectangleType Window   = {-0.015, -0.012, 0.015, 0.012};
  static sgpRectangleType Viewport = {-0.125,  0.000, 1.125, 1.000};  
  static double Exposure = 1.0;
  static sgpColorType Ambient = {0.2, 0.2, 0.3};
  static int STRANGEBACKGROUND = 0; /* unless YOU set it with -b */
  PointType3D LightPosition;
  VectorType3D Right;
  sgpColorType Color;

  RoutineName = argv[ArgsParsed++];

  LeftHanded = 0; TwoSided = 0; Facets = 0;  /* but, of course! */
  Transparency = 0.0;
  while (ArgsParsed < argc)
   {
    if ('-' == argv[ArgsParsed][0])
     {
      switch (argv[ArgsParsed++][1])
       {
        case 'e':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         EyePoint.x = atof(argv[ArgsParsed++]);
         EyePoint.y = atof(argv[ArgsParsed++]);
         EyePoint.z = atof(argv[ArgsParsed++]);
         OpticalAxis.dx = Origin.x - EyePoint.x;
         OpticalAxis.dy = Origin.y - EyePoint.y;
         OpticalAxis.dz = Origin.z - EyePoint.z;
         break;
        case 'o':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         OpticalAxis.dx = atof(argv[ArgsParsed++]);
         OpticalAxis.dy = atof(argv[ArgsParsed++]);
         OpticalAxis.dz = atof(argv[ArgsParsed++]);
         break;
        case 'O':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         Origin.x = atof(argv[ArgsParsed++]);
         Origin.y = atof(argv[ArgsParsed++]);
         Origin.z = atof(argv[ArgsParsed++]);
         OpticalAxis.dx = Origin.x - EyePoint.x;
         OpticalAxis.dy = Origin.y - EyePoint.y;
         OpticalAxis.dz = Origin.z - EyePoint.z;
         break;
        case 'u':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         Up.dx = atof(argv[ArgsParsed++]);
         Up.dy = atof(argv[ArgsParsed++]);
         Up.dz = atof(argv[ArgsParsed++]);
         break;
        case 'f':
         if ((argc-ArgsParsed)<1) { usage(); exit (-1); }
         FocalLength = atof(argv[ArgsParsed++]);
         break;
        case 'F':
         Facets = 1;
         break; 
        case 'x':
         if ((argc-ArgsParsed)<1) { usage(); exit (-1);}
         Exposure = atof(argv[ArgsParsed++]);
         break;
        case 'p':
         Hidden = 1;
         Preview = 1;
         break;
        case 'P':
         Hidden = 0;
         Preview = 1;
         break;
        case 'w':
         Hidden = 1;
         Preview = 2;  /* Steve asked for it - wireframe, with culling */
         break;
        case 'Q':
         Quality = 1;
         Hidden = 1;
         Preview = 0;
         break;
        case 'G':
         Quality = 0;
         Hidden = 1;
         Preview = 0;
         break;
        case 'n':
         Normalize = 1;
         break;
        case 'D':
         DEBUG = 1;
         break;
        case '1':
         TwoSided = 0;
         break;
        case '2':
         TwoSided = 1;
         break;
        case 'l':
         LeftHanded = 1;
         break;
        case 't':
         TensorProduct = 1;
         break;
        case 'a':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1);}
         Ambient.r = atof(argv[ArgsParsed++]); 
         Ambient.g = atof(argv[ArgsParsed++]); 
         Ambient.b = atof(argv[ArgsParsed++]); 
	 break;
        case 'b':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1);}
         Background.r = atof(argv[ArgsParsed++]); 
         Background.g = atof(argv[ArgsParsed++]); 
         Background.b = atof(argv[ArgsParsed++]); 
         STRANGEBACKGROUND = 1; 
	 break;
        case 'v':
         VERBOSE = 1;
         break;
        case 'M':
         if ((argc-ArgsParsed)<1) { usage(); exit (-1);}
         MaxTriangles = atoi(argv[ArgsParsed++]);
	 break;
        case 'T':
         if ((argc-ArgsParsed)<1) { usage(); exit (-1); }
         Transparency = atof(argv[ArgsParsed++]);
         break;
        case 'h': 
        default:
         usage(); exit(-1);
       }
     }
    else { usage(); exit (-1); }
   }   

  if (!STRANGEBACKGROUND) Background = Ambient;

  T = (TriangleType3D *) calloc(MaxTriangles, sizeof(TriangleType3D));
  if (T == (TriangleType3D *) 0)
   {
    fprintf(stderr,"%s: Not enough memory for %d triangles\n", 
            RoutineName, MaxTriangles);
    exit(-1);
   }

  if (DEBUG) sgpInit(DebugDisplay);
  else       sgpInit(ColorDisplay);

  sgpSetWindow(Window);
  sgpSetViewport(Viewport);

  Camera3D(EyePoint, OpticalAxis, Up);
  Lens3D(FocalLength);  SetHither(FocalLength);

  GetCamera3D(&EyePoint, &OpticalAxis, &Up);

  if (VERBOSE)
   {
    fprintf(stderr,"%s: EyePoint = { %f, %f, %f }\n", RoutineName,
                        EyePoint.x,
                        EyePoint.y,
                        EyePoint.z);
    fprintf(stderr,"%s: OpticalAxis = { %f, %f, %f }\n", RoutineName,
                        OpticalAxis.dx,
                        OpticalAxis.dy,
                        OpticalAxis.dz);
    fprintf(stderr,"%s: Up = { %f, %f, %f }\n", RoutineName,
                        Up.dx,
                        Up.dy,
                        Up.dz);
    fprintf(stderr,"%s: FocalLength = %f\n", RoutineName, FocalLength);
   }

  if (Preview)
   {
    if (VERBOSE)
     {
      fprintf(stderr,"%s: Previewing\n", RoutineName);
      fprintf(stderr,"%s: InteriorColor = { %f, %f, %f }\n", RoutineName,
                          InteriorColor.r, InteriorColor.g, InteriorColor.b);
      fprintf(stderr,"%s: OutlineColor = { %f, %f, %f }\n", RoutineName,
                          OutlineColor.r, OutlineColor.g, OutlineColor.b);
     }
    RenderMode(Preview, TensorProduct, InteriorColor, OutlineColor);
    sgpColor(InteriorColor); sgpClearScreen();
   }
  else
   {
    Right = Cross3D(OpticalAxis, Up);
    Normalize3D(&Right); Normalize3D(&Up);

    SetAmbient(Ambient);
    sgpColor(Background); sgpClearScreen();

    LightPosition.x = EyePoint.x + Up.dx + Right.dx;
    LightPosition.y = EyePoint.y + Up.dy + Right.dy;
    LightPosition.z = EyePoint.z + Up.dz + Right.dz;
    Color.r = 0.75; Color.g = 0.75; Color.b = 0.75;
    AddLightSource(LightPosition, Color);

    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    SetExposure(Exposure);

   if (VERBOSE)
    {
     fprintf(stderr,"%s: Exposure = %f\n", RoutineName, Exposure);
     fprintf(stderr,"%s: Ambient = { %f, %f, %f }\n",RoutineName,
                     Ambient.r, Ambient.g, Ambient.b);
     fprintf(stderr,"%s: Background = { %f, %f, %f }\n",RoutineName,
                     Background.r, Background.g, Background.b);
     fprintf(stderr,"%s: Transparency = %f\n",RoutineName,Transparency);
    }
   }

  if (VERBOSE) fprintf(stderr,"%s: Reading triangles\n",RoutineName);
  t = 0; 
  ReadTriangles(stdin);

  if (t >= MaxTriangles)
   fprintf(stderr,"%s: WARNING - MaxTriangles limit (%d) exceeded\n",
                   RoutineName,MaxTriangles);

  if (Hidden)
   if (Quality && (!Preview))
    {
     if (VERBOSE)
      fprintf(stderr,"%s: calling ScanTriangles(%d, T, %d)\n",
                    RoutineName,t,MAXTRIANGLES);
     ScanTriangles(t,T,MAXTRIANGLES);

     if (VERBOSE) fprintf(stderr,"%s: ScanTriangles returns\n",RoutineName);
   }
   else
    {
     if (VERBOSE)
      fprintf(stderr,"%s: calling RenderTriangles(%d, T, %d)\n",
                    RoutineName,t,MAXTRIANGLES);
     RenderTriangles(t,T,MAXTRIANGLES);

     if (VERBOSE) fprintf(stderr,"%s: RenderTriangles returns\n",RoutineName);
    }
  sgpQuit();
  if (VERBOSE) fprintf(stderr,"-30-\n");
  exit(0);
 }
