/*
  File: Aggregates.c
  Author: David Meyers
  Last Modified: 15 August 1990
  Purpose: Functions for handling aggregate contours
 */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <TypeDefinitions.h>
#include <Contours.h>
#include <Tiler.h>

extern void exit();
extern void free();
extern char *malloc();
static ContourDescriptor *MergeContours2();

/**************************************************************************/

/*
  FindContour finds the contour with name ContourNamed in section TheSection
  if it is there. If found, TRUE is returned, and the value of ContourIndex
  is set to the index of the desired contour in TheSection's array of contours.
  If not found, FALSE is returned.
 */
extern int FindContour(InSection, ContourNamed, ContourIndex, UseAlias)
 Section *InSection;
 char *ContourNamed;
 int *ContourIndex;
 int UseAlias;
{
 int i, j;

 for(i=0; i<InSection->NContours; i++)
  {
   if (!strcmp(ContourNamed, InSection->TheContours[i].Name))
    {
     *ContourIndex = i;
     return(TRUE);
    }
   else if(UseAlias)
    {
     for (j=0; j < InSection->TheContours[i].NAliases; j++)
      if (!strcmp(ContourNamed, &(InSection->TheContours[i].Aliases[j*80])))
       {
	*ContourIndex = i;
	return(TRUE);
       }       
    }
  }
 return(FALSE);
}

/**************************************************************************/

extern void FreeDescriptor(TheDescriptor)
 ContourDescriptor *TheDescriptor;
{
 ContourDescriptor *Current;

 while(TheDescriptor != NULL)
  {
   Current = TheDescriptor->next;
   free((char *) TheDescriptor);
   TheDescriptor = Current;
  }
}

/**************************************************************************/

extern PointsType *MakeAggregateContour(TheSection, TheDescriptor)
 Section *TheSection;
 ContourDescriptor *TheDescriptor;
{
 int Npts = 0, ContourIndex;
 int i, j;
 PointsType *AggregateContour;
 ContourDescriptor *Current;

 Current = TheDescriptor;
 while (NULL != Current)
  {
   Npts = Npts + Current->Npts;
   Current = Current->next;
  }
 AggregateContour = NewPoints(Npts);
 Current = TheDescriptor;
 i = 0;
 while (NULL != Current)
  {
   if(FindContour(TheSection, Current->Name, &ContourIndex, TRUE))
    {
     int TPN;
     PointsType *ThePoints;

     ThePoints = TheSection->TheContours[ContourIndex].ContourPoints;
     TPN = ThePoints->n;

     for(j=0; j<Current->Npts; j++)
      {
       int SegmentIndex;

       SegmentIndex = Current->FirstPoint;
       AggregateContour->P[i] = ThePoints->P[(j+SegmentIndex) % TPN];
       i++;
      }
    }
   else
    {
     (void) fprintf(stderr,"MakeAggregateContour: ContourName %s not found\n",
	     Current->Name);
     exit(-1);
    }
   Current = Current->next;
  }
 return(AggregateContour);
}

/**************************************************************************/

int Find(TheSection, TheContour, ThePoint, TheName)
 Section *TheSection;
 int TheContour;
 int ThePoint;
 char *TheName;
{
 int i;

 for(i=0; i<TheSection->Last; i++)
  {
   if(((TheContour == TheSection->TheIndex[i].FirstContour) &&
       (ThePoint == TheSection->TheIndex[i].FCPoint)) ||
      ((TheContour == TheSection->TheIndex[i].SecondContour) &&
       (ThePoint == TheSection->TheIndex[i].SCPoint)))
    {
     (void) strcpy(TheName, TheSection->TheIndex[i].Name);
     return(TRUE);
    }
  }
 return(FALSE);
}

/**************************************************************************/

extern char *PointName(TheSection, PIndex, TheDescriptor)
 Section *TheSection;
 int PIndex;
 ContourDescriptor *TheDescriptor;
{
 ContourDescriptor *Current;
 char NameString[80];
 char IndexString[20];
 int NPtsInContour;

 Current = TheDescriptor;
 while (NULL != Current)
  {
   NPtsInContour = TheSection->TheContours[Current->Index].ContourPoints->n;
   if (Current->Npts <= PIndex)
    {
     PIndex = PIndex - Current->Npts;
     Current = Current->next;
    }
   else if(Find(TheSection, Current->Index,
		(Current->FirstPoint + PIndex) % NPtsInContour, NameString))
    {
     return(NameString);
    }
   else
    {
     (void) strcpy (NameString, Current->SectionName);
     (void) strcat (NameString, Current->Name);
     (void) sprintf(IndexString, "%d",
		    (Current->FirstPoint + PIndex) % NPtsInContour);
     (void) strcat (NameString, IndexString);
     return(NameString);
    }
  }
 (void) fprintf(stderr,"PointName: PIndex too large\n");
 return("\0");
}

/**************************************************************************/

extern void DumpAggregateContour(TheSection, TheAggregate, TheDescriptor)
 Section *TheSection;
 PointsType *TheAggregate;
 ContourDescriptor *TheDescriptor;
{
 int i;
 char NameString[80];

 (void) fprintf(stderr,"Aggregate Contour in Section: %s\n\n",
		TheSection->Name);

 for (i=0; i<TheAggregate->n; i++)
  {
   (void) strcpy(NameString, PointName(TheSection, i, TheDescriptor));
   (void) fprintf(stderr,"i=%d, PointName=%s ", i, NameString);
   (void) fprintf(stderr,"%f, %f, %f\n",
	   TheAggregate->P[i].x,
	   TheAggregate->P[i].y,
	   TheAggregate->P[i].z);
  }
 (void) fprintf(stderr,"\nTotal points in Aggregate: %d", i);
 (void) fprintf(stderr,"\n\n");
 (void) fflush(stderr);
}

/**************************************************************************/

/*
  MergeContours creates a ContourDescriptor list describing the resultant
  merged contour. The actual memory allocation for the merged contour and its
  construction are handled by MakeAggregateContour. It should not be necessary
  to modify MakeAggregateContour to change the way in which contours are
  merged, since MergeContours handles the policy decisions. MergeContours does
  a recursive traversal of the adjacency list of a contour and any contours
  reachable from them, creating a list of ContourDescriptors.
 */

/*
  Top level routine, which calls a recursive lower level routine. The top level
  routine knows about any differences in the top level call.
 */
extern ContourDescriptor *MergeContours (TheSection,
					 ContourName,
					 Above,
					 Done,
					 TriangleNumber)
 Section *TheSection;
 char *ContourName;
 int Above;
 int Done;
 int *TriangleNumber;
{
 ContourDescriptor *Root, *Current;
 Adjacency *Neighbor;
 Contour *TheContour;
 int ContourIndex, StartPoint, LocationInContour;
 char NameString[80];
 int SpecialAdjacency = FALSE;

 if(FindContour(TheSection, ContourName, &ContourIndex, TRUE))
  {
   TheContour = &(TheSection->TheContours[ContourIndex]);
   TheContour->Visited = Done;
   if(Above)
    Neighbor = TheContour->AdjContPrev;
   else
    Neighbor = TheContour->AdjContNext;

   if(NULL == Neighbor)
    /* Handle the case of no adjacent contours */
    {
     Current = (ContourDescriptor *) malloc(sizeof(ContourDescriptor));
     (void) strcpy(Current->SectionName, TheSection->Name);
     (void) strcpy(Current->Name, TheContour->Name);
     Current->Index = ContourIndex;
     Current->FirstPoint = 0;
     Current->LastPoint = TheContour->ContourPoints->n - 1;
     Current->Npts = TheContour->ContourPoints->n;
     Current->next = NULL;
     return(Current);
    }
   else
    {
     /*
       There are adjacent contours. We are at a top level call, since we are in
       MergeContours, not MergeContours2. We set StartPoint to the LocalIndex
       of the first adjacency.
      */
     StartPoint = Neighbor->LocalIndex;
     LocationInContour = StartPoint;

     Root = (ContourDescriptor *) malloc(sizeof(ContourDescriptor));
     Current = Root;

     do /* while(Neighbor->LocalIndex != StartPoint) */
      {
       /*
	 put the information for the current segment of the composite contour
	 into the record pointed to by Current.
	*/
       (void) strcpy(Current->SectionName, TheSection->Name);
       (void) strcpy(Current->Name, TheContour->Name);
       Current->Index = ContourIndex;
       Current->FirstPoint = LocationInContour;
       if(!Find(TheSection, ContourIndex, Neighbor->LocalIndex, NameString))
	{
	 Current->LastPoint = Neighbor->LocalIndex;
	 SpecialAdjacency = FALSE;
	}
       else
	{
	 SpecialAdjacency = TRUE;
	 Current->LastPoint = Neighbor->LocalIndex;
	}
       /*
	 The following test is similar to one which occurs outside the
	 current loop. This one MUST be a >= test, the other one MUST be a
	 > test. We are handling wraparound with these tests and the situation
	 is different depending on whether we are about to return from or
	 about to make a call to MergeContours.
	*/
       if (Current->LastPoint >= Current->FirstPoint)
	Current->Npts = Current->LastPoint + 1 - Current->FirstPoint;
       else /* handle wraparound */
	{
	 Current->Npts = TheContour->ContourPoints->n + 1 + Current->LastPoint
	                   - Current->FirstPoint;
	}
       if(TRUE == SpecialAdjacency)
	{
	 /* 
	   If this ends up with 0 or less, the MakeAggregateContour routine
	   will just skip over this record in the list of descriptors, so
	   there should be no problem.
	  */
	 Current->Npts = Current->Npts - 1;
	 Current->LastPoint = Current->LastPoint - 1;
	 if (Current->LastPoint < 0)
	  Current->LastPoint = TheContour->ContourPoints->n - 1;
	}
       /*
	 recursively build a list of segments of contours
	*/
       Current->next = MergeContours2(TheSection, Neighbor->Name,
				     TheContour->Name,
				     Neighbor->AdjacentIndex,
				     Above,
				     Done, 
				     &LocationInContour,
				     TriangleNumber);
       /* Now move the Current pointer to the end of the added stuff. */
       while(NULL != Current->next)
	Current = Current->next;
/*
  At this point we need to do the tiling of the border between the two
  adjacent contours.
  Needed information on the region to be tiled is in the records
  pointed to by Neighbor and Current, and in LocationInContour.
  At this point, Neighbor->LocalIndex is the first point on the caller's
  adjacency region, LocationInContour is the last point.
  Neighbor->AdjacentIndex is the last point in the adjacent contour's
  adjacency   region and Current->LastPoint is the first point in the
  neighbor's adjacency region.
  The index of the adjacent contour is Current->Index, the index of the
  caller is ContourIndex.
 */
       if(((LocationInContour != Neighbor->LocalIndex) ||
	   (Current->LastPoint != Neighbor->AdjacentIndex)) &&
	  (!SpecialAdjacency))
	TileSaddle(TheSection,
		   ContourIndex, Neighbor->LocalIndex, LocationInContour,
		   Current->Index, Current->LastPoint, Neighbor->AdjacentIndex,
		   Above, TriangleNumber);

       /* now add a new cell to the descriptor */
       Current->next = (ContourDescriptor *) malloc(sizeof(ContourDescriptor));
       Current = Current->next;

       /* advance along the list of adjacencies */
       Neighbor = Neighbor->next;
      }
     while(Neighbor->LocalIndex != StartPoint);

       /* Handle end of top level call. */
     (void) strcpy(Current->SectionName, TheSection->Name);
     (void) strcpy(Current->Name, TheContour->Name);
     Current->Index = ContourIndex;
     Current->FirstPoint = LocationInContour;
     if (StartPoint > 0)
      Current->LastPoint = StartPoint - 1;
     else
      Current->LastPoint = TheContour->ContourPoints->n - 1;

     if (Current->LastPoint >= Current->FirstPoint)
      Current->Npts = Current->LastPoint + 1 - Current->FirstPoint;
     else /* handle wraparound */
      {
       Current->Npts = TheContour->ContourPoints->n + 1 + Current->LastPoint
	               - Current->FirstPoint;
      }
     Current->next = NULL;
    }
  }
 else
  {
   (void) fprintf(stderr,
		  "MergeContours: contour name %s not found\n", ContourName);
   exit(-1);
  }
 return (Root);
}

static ContourDescriptor *MergeContours2 (TheSection,
					  ContourName,
					  CallerName,
					  StartPoint,
					  Above,
					  Done,
					  NewLocation,
					  TriangleNumber)
 Section *TheSection;
 char *ContourName;
 char *CallerName;
 int StartPoint;
 int Above;
 int Done;
 int *NewLocation; /* update LocationInContour, call with NULL at top level */
 int *TriangleNumber;
{
 ContourDescriptor *Root, *Current;
 Adjacency *Neighbor;
 Contour *TheContour;
 int ContourIndex, LocationInContour = StartPoint;
 char NameString[80];
 int SpecialAdjacency = FALSE;

 Root = (ContourDescriptor *) malloc(sizeof(ContourDescriptor));
 Current = Root;
 if(FindContour(TheSection, ContourName, &ContourIndex, FALSE))
  {
   TheContour = &(TheSection->TheContours[ContourIndex]);
   TheContour->Visited = Done;
   if(Above)
    Neighbor = TheContour->AdjContPrev;
   else
    Neighbor = TheContour->AdjContNext;

   if(NULL == Neighbor)
    /* Handle the case of no adjacent contours */
    {
     (void) strcpy(Current->SectionName, TheSection->Name);
     (void) strcpy(Current->Name, TheContour->Name);
     Current->Index = ContourIndex;
     Current->FirstPoint = 0;
     Current->LastPoint = TheContour->ContourPoints->n - 1;
     Current->Npts = TheContour->ContourPoints->n;
     Current->next = NULL;
     Root = Current;
     return(Root);
    }
   else
    {
     /*
       We need to enter the circular list of adjacent contours in the
       appropriate  place. We then need to continue until we get back to from
       where we started, then we return.
      */

     /* Find the right place in the circular list */
     if(StartPoint > Neighbor->prev->LocalIndex);
     /*
       Start is larger than all so we can use the first adjacency in the list.
      */
     else
      /*
	Advance through the list of adjacencies until we come to one which
	is at a point with index higher than the start point.
       */
      while(Neighbor->LocalIndex < StartPoint)
       Neighbor = Neighbor->next;

     /* 
       Neighbor->LocalIndex >= StartPoint here, OR StartPoint is > all of the
       Local indices of the elements of the adjacency list.
       Now march around the list creating a ContourDescriptor for each of the
       required pieces of each contour.
      */
     if(Neighbor->LocalIndex == StartPoint)
      Neighbor = Neighbor->next;
     while(strcmp(Neighbor->Name,CallerName))
      {
       /*
	 put the information for the current segment of the composite contour
	 into the record pointed to by Current.
	*/
       (void) strcpy(Current->SectionName, TheSection->Name);
       (void) strcpy(Current->Name, TheContour->Name);
       Current->Index = ContourIndex;
       Current->FirstPoint = LocationInContour;
       if(!Find(TheSection, ContourIndex, Neighbor->LocalIndex, NameString))
	{
	 Current->LastPoint = Neighbor->LocalIndex;
	 SpecialAdjacency = FALSE;
	}
       else
	/*
	  The adjacency point is in the Name Index, we don't want to
	  use it twice since it is really only one point.
	 */
	{
	 SpecialAdjacency = TRUE;
	 if (0 < Neighbor->LocalIndex)
	  Current->LastPoint = Neighbor->LocalIndex - 1;
	 else
	  Current->LastPoint = TheContour->ContourPoints->n - 1;
	}
       /*
	 The following test is similar to one which occurs outside the
	 current loop. This one MUST be a >= test, the other one MUST be a
	 > test. We are handling wraparound with these tests and the situation
	 is different depending on whether we are about to return from or
	 about to make a call to MergeContours. THIS MAY STILL BE A BUG.
	*/
       if (Current->LastPoint >= Current->FirstPoint)
	Current->Npts = Current->LastPoint + 1 - Current->FirstPoint;
       else /* handle wraparound */
	{
	 Current->Npts = TheContour->ContourPoints->n + 1 + Current->LastPoint
	                   - Current->FirstPoint;
	}
       /*
	 recursively build a list of segments of contours
	*/
       Current->next = MergeContours2(TheSection, Neighbor->Name,
				      TheContour->Name,
				      Neighbor->AdjacentIndex,
				      Above,
				      Done, 
				      &LocationInContour,
				      TriangleNumber);
       /* Now move the Current pointer to the end of the added stuff. */
       while(NULL != Current->next)
	Current = Current->next;
/*
  At this point we need to do the tiling of the border between the two
  adjacent contours.
  Needed information on the region to be tiled is in the records
  pointed to by Neighbor and Current, and in LocationInContour.
  At this point, Neighbor->LocalIndex is the first point on the caller's
  adjacency region, LocationInContour is the last point.
  Neighbor->AdjacentIndex is the last point in the adjacent contour's
  adjacency   region and Current->LastPoint is the first point in the
  neighbor's adjacency region.
  The index of the adjacent contour is Current->Index, the index of the
  caller is ContourIndex.
 */
       if(((LocationInContour != Neighbor->LocalIndex) ||
	   (Current->LastPoint != Neighbor->AdjacentIndex)) &&
	  (!SpecialAdjacency))
	TileSaddle(TheSection,
		   ContourIndex, Neighbor->LocalIndex, LocationInContour,
		   Current->Index, Current->LastPoint, Neighbor->AdjacentIndex,
		   Above, TriangleNumber);

       /* now add a new cell to the descriptor */
       Current->next = (ContourDescriptor *) malloc(sizeof(ContourDescriptor));
       Current = Current->next;

       /* advance along the list of adjacencies */
       Neighbor = Neighbor->next;
      }
     /*
       We have reached an adjacency which points back to the caller, and need
       to fill in the descriptor for the rest of this contour and return.
      */

     /* set the new value of LocationInContour */
     *NewLocation = Neighbor->AdjacentIndex;
     (void) strcpy(Current->SectionName, TheSection->Name);
     (void) strcpy(Current->Name, TheContour->Name);
     Current->Index = ContourIndex;
     Current->FirstPoint = LocationInContour;
     if(!Find(TheSection, ContourIndex, Neighbor->LocalIndex, NameString))
      Current->LastPoint = Neighbor->LocalIndex;
     else
      {
       if (0 < Neighbor->LocalIndex)
	Current->LastPoint = Neighbor->LocalIndex - 1;
       else
	Current->LastPoint = TheContour->ContourPoints->n - 1;
      }
     if (Current->LastPoint > Current->FirstPoint)
      Current->Npts = Current->LastPoint - Current->FirstPoint + 1;
     else /* handle wraparound */
      {
       Current->Npts = TheContour->ContourPoints->n + 1 + Current->LastPoint
	                                                - Current->FirstPoint;
      }
     Current->next = NULL;
    }
  }
 else
  {
   (void) fprintf(stderr,
		  "MergeContours: contour name %s not found\n", ContourName);
   exit(-1);
  }
 return (Root);
}
/**************************************************************************/

double Dist(TheSection, C1, C1P, C2, C2P)
 Section *TheSection;
 int C1, C1P, C2, C2P;
{
 double dx, dy, dz, d;

 dx = TheSection->TheContours[C1].ContourPoints->P[C1P].x -
      TheSection->TheContours[C2].ContourPoints->P[C2P].x;
 dy = TheSection->TheContours[C1].ContourPoints->P[C1P].y -
      TheSection->TheContours[C2].ContourPoints->P[C2P].y;
 dz = TheSection->TheContours[C1].ContourPoints->P[C1P].z -
      TheSection->TheContours[C2].ContourPoints->P[C2P].z;
 d = (dx * dx) + (dy * dy) + (dz * dz);
 d = sqrt(d);
 return(d);
}

/**************************************************************************/

/*
  BuildNameIndex handles adjacencies in which the adjacent contours touch,
  or nearly touch.
  If a point on one contour is very close to its adjacent
  point on the other contour, the points are merged and given a unique name,
  which is entered into this Index. If points are farther
  than Tolerance apart, then this is not done. The Index created is needed
  so that the same point does not get multiple names, thereby confusing the
  mesh programs which see the output of the ContourFitter program.
 */
void BuildNameIndex(TheSection, Tolerance)
 Section *TheSection;
 double Tolerance;
{
 int i, NAdjacencies = 0;
 Adjacency *Adj, *C2Adj;
 AdjIndex *IndexList = NULL, *Temp;
 double x1, x2, y1, y2, z1, z2, distance;
 int C1, C2, PC1, PC2;

 /* Step through the contours one at a time */
 for (i=0; i<TheSection->NContours; i++)
  {
   /* First check the list of previous section fusing adjacencies */
   Adj = TheSection->TheContours[i].AdjContPrev;
   if(NULL != Adj)
    /* we have some adjacencies on this contour */
    {
     Adj->prev->next = NULL; /*simplifies finding the end of the list*/
     while(NULL != Adj)
      /* step through the list of adjacencies for this contour */
      {
       C1 = i;
       PC1 = Adj->LocalIndex;
       if(!FindContour(TheSection, Adj->Name, &C2, TRUE))
	{
	 (void) fprintf(stderr,"Contour %s not found: BuildNameIndex\n",
			Adj->Name);
	 exit(-1);
	}
       PC2 = Adj->AdjacentIndex;
       distance = Dist(TheSection, C1, PC1, C2, PC2);
       if((distance < Tolerance)&&(C2 > C1))
	/*
	  If the points are close together, and the contour index of the 
	  adjacent contour is greater than the  contour index of the current
	  contour, we check to see that the adjacency is of the a->b, b->a
	  type by looking through the list of adjacencies of the adjacent 
	  contour for the "reverse arrow"
	 */
	{
	 C2Adj = TheSection->TheContours[C2].AdjContPrev;
	 if(NULL != C2Adj)
	  {
	   C2Adj->prev->next = NULL; /*simplifies finding the end of the list*/
	   while (NULL != C2Adj)
	    {
	     if((C2Adj->LocalIndex == PC2)&&(C2Adj->AdjacentIndex == PC1))
	      /*
		the adjacency meets the criteria, so add it to the list and
		set the coordinates of the two versions of the point to their
		mean.
	       */
	      {
	       NAdjacencies += 1;
	       Temp = (AdjIndex *) malloc(sizeof(AdjIndex));
	       Temp->FirstContour = C1;
	       Temp->FCPoint = PC1;
	       Temp->SecondContour = C2;
	       Temp->SCPoint = PC2;
	       (void) strcpy(Temp->Name, TheSection->Name);
	       (void) strcat(Temp->Name, Adj->Name);
	       (void) strcat(Temp->Name, C2Adj->Name);
	       Temp->next = IndexList;
	       IndexList = Temp;

	       x1 = TheSection->TheContours[C1].ContourPoints->P[PC1].x;
	       y1 = TheSection->TheContours[C1].ContourPoints->P[PC1].y;
	       z1 = TheSection->TheContours[C1].ContourPoints->P[PC1].z;
	       x2 = TheSection->TheContours[C2].ContourPoints->P[PC2].x;
	       y2 = TheSection->TheContours[C2].ContourPoints->P[PC2].y;
	       z2 = TheSection->TheContours[C2].ContourPoints->P[PC2].z;

	       TheSection->TheContours[C1].ContourPoints->P[PC1].x =
		(x1+x2)/2.0;
	       TheSection->TheContours[C1].ContourPoints->P[PC1].y =
		(y1+y2)/2.0;
	       TheSection->TheContours[C1].ContourPoints->P[PC1].z =
		(z1+z2)/2.0;
	       TheSection->TheContours[C2].ContourPoints->P[PC2].x =
		(x1+x2)/2.0;
	       TheSection->TheContours[C2].ContourPoints->P[PC2].y =
		(y1+y2)/2.0;
	       TheSection->TheContours[C2].ContourPoints->P[PC2].z =
		(z1+z2)/2.0;

	       break;
	      }
	     else
	      C2Adj = C2Adj->next;
	    }
	   TheSection->TheContours[C2].AdjContPrev->prev->next = 
	    TheSection->TheContours[C2].AdjContPrev; /* fix up the list */
	  }
	}
       Adj = Adj->next;
      }
     TheSection->TheContours[i].AdjContPrev->prev->next = 
      TheSection->TheContours[i].AdjContPrev; /* fix up the list */
    }
   Adj = TheSection->TheContours[i].AdjContNext;
   /* now do all of the above stuff for the other adjacency list */
   if(NULL != Adj)
    {
     Adj->prev->next = NULL; /*simplifies finding the end of the list*/
     while(NULL != Adj)
      {
       C1 = i;
       PC1 = Adj->LocalIndex;
       if(!FindContour(TheSection, Adj->Name, &C2, TRUE))
	{
	 (void) fprintf(stderr,"Contour not found: BuildNameIndex\n");
	 exit(-1);
	}
       PC2 = Adj->AdjacentIndex;
       distance = Dist(TheSection, C1, PC1, C2, PC2);
       if((distance < Tolerance)&&(C2 > C1))
	/*
	  We don't need to add one if C2 < C1 since it was added from the 
	  other end.
	 */
	{
	 C2Adj = TheSection->TheContours[C2].AdjContNext;
	 if(NULL != C2Adj)
	  {
	   C2Adj->prev->next = NULL; /*simplifies finding the end of the list*/
	   while (NULL != C2Adj)
	    {
	     if((C2Adj->LocalIndex == PC2)&&(C2Adj->AdjacentIndex == PC1))
	      {
	       NAdjacencies += 1;
	       Temp = (AdjIndex *) malloc(sizeof(AdjIndex));
	       Temp->FirstContour = C1;
	       Temp->FCPoint = PC1;
	       Temp->SecondContour = C2;
	       Temp->SCPoint = PC2;
	       (void) strcpy(Temp->Name, TheSection->Name);
	       (void) strcat(Temp->Name, Adj->Name);
	       (void) strcat(Temp->Name, C2Adj->Name);
	       Temp->next = IndexList;
	       IndexList = Temp;

	       x1 = TheSection->TheContours[C1].ContourPoints->P[PC1].x;
	       y1 = TheSection->TheContours[C1].ContourPoints->P[PC1].y;
	       z1 = TheSection->TheContours[C1].ContourPoints->P[PC1].z;
	       x2 = TheSection->TheContours[C2].ContourPoints->P[PC2].x;
	       y2 = TheSection->TheContours[C2].ContourPoints->P[PC2].y;
	       z2 = TheSection->TheContours[C2].ContourPoints->P[PC2].z;

	       TheSection->TheContours[C1].ContourPoints->P[PC1].x =
		(x1+x2)/2.0;
	       TheSection->TheContours[C1].ContourPoints->P[PC1].y =
		(y1+y2)/2.0;
	       TheSection->TheContours[C1].ContourPoints->P[PC1].z =
		(z1+z2)/2.0;
	       TheSection->TheContours[C2].ContourPoints->P[PC2].x =
		(x1+x2)/2.0;
	       TheSection->TheContours[C2].ContourPoints->P[PC2].y =
		(y1+y2)/2.0;
	       TheSection->TheContours[C2].ContourPoints->P[PC2].z =
		(z1+z2)/2.0;

	       break;
	      }
	     else
	      C2Adj = C2Adj->next;
	    }
	   TheSection->TheContours[C2].AdjContNext->prev->next = 
	    TheSection->TheContours[C2].AdjContNext; /* fix up the list */
	  }
	}
       Adj = Adj->next;
      }
     TheSection->TheContours[i].AdjContNext->prev->next = 
      TheSection->TheContours[i].AdjContNext; /* fix up the list */
    }
  }
 
 if(0 == NAdjacencies) return;

/*
  We have some entries for a Name Index, so copy them into an array attached
  to TheSection, after allocating space.
 */

 TheSection->TheIndex = (AdjIndex *) malloc((unsigned)NAdjacencies *
					    sizeof(AdjIndex));

 TheSection->Last = 0;
 Temp = IndexList;
 while(NULL != Temp)
  {
   IndexList = Temp->next;
   Temp->next = NULL;
   TheSection->TheIndex[TheSection->Last] = *Temp;
   free((char *) Temp); /* don't let that memory leak out there! */
   Temp = IndexList;
   TheSection->Last += 1;
  }
}

/**************************************************************************/
