/*****************************************************************************
 * $Id: tape_buf.c,v 1.2 1992/01/10 13:27:29 ak Exp $
 *****************************************************************************
 * $Log: tape_buf.c,v $
 * Revision 1.2  1992/01/10  13:27:29  ak
 * [tar/tape -> ] buffer -> compress [ -> tar/tape] (-Z)
 * DOS & OS/2 file attributes (-p)
 * Don't recurse (-Y)
 *
 * Revision 1.1.1.1  1992/01/06  20:41:33  ak
 * -Y = don't recurse: new.
 * -X = exclude list: for extract.
 * Use BUFFER for OS/2 tape compression.
 * No own tape buffering for OS/2.
 * Support for SYSTEM and HIDDEN files.
 *
 * Revision 1.1  1992/01/06  20:41:32  ak
 * Initial revision
 *
 *****************************************************************************/

static char *rcsid = "$Id: tape_buf.c,v 1.2 1992/01/10 13:27:29 ak Exp $";

/*
 *	tape_buf.c
 *
 * Tape interface for GNU tar.
 * Use "Remote Tape Interface".
 */

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#ifdef __ZTC__
# include <dos.h>
#else
# include <malloc.h>
#endif
#include <sys/types.h>
#ifdef OS2
# define INCL_DOSFILEMGR
# define INCL_DOSSEMAPHORES
# define INCL_DOSERRORS
# include <os2.h>
#else
# include <dos.h>
#endif

#include "tar.h"
#include "port.h"
#include "rmt.h"

#include "scsi.h"
#include "tape.h"

extern FILE *msg_file;
char *__rmt_path;

static void far *	buffer;
static unsigned		length;
static void far *	bufp;
static long		nblocks;
static unsigned		nbytes;
static int		active;
static long		actual;
static int		pending;

#ifdef OS2
static HFILE		handle;
static ULONG		sema = 0;
static USHORT		asy_written;
static USHORT		asy_status;
#endif

	/* position: */
#define Known		1
#define EndOfFile	2

static long	offset;
static long	blockno;
static int	position = 0;
static int	errcode;
static int	wrflag;
static int	tape_no = 0;


static int
error(int code)
{
	errcode = code;
	errno = EIO;
	return -1;
}

#undef perror

void
my_perror(const char _FAR_ *msg)
{
	if (errno >= 2000) {
		if (msg)
			fprintf(msg_file, "%s: ", msg);
		switch (errno) {
		case EINVOP:
			fprintf(msg_file, "invalid operation\n");
			break;
		case ENOSPC:
			fprintf(msg_file, "reached end of tape\n");
			break;
		case EIO:
			fprintf(msg_file, "tape error\n");
			if (errcode)
				tape_print_sense(msg_file, errcode);
			break;
		}
	} else
		perror(msg);
}

int
__rmt_open(char *name, int mode, int prot, int bias)
{
	int r, unit;
	char *cp;
	extern int map_read;

#ifdef OS2
	USHORT	disk = 0;
	USHORT	action;

	if (disk = _isdisk(name)) {
		if ((mode & 3) != O_RDONLY) {
			fprintf(stderr, "Cannot write to disk\n");
			exit(EX_SYSTEM);
		}
	} else
		while (*name == '+')
			++name;
	r = DosOpen(name, &handle, &action, 0L, 0, FILE_OPEN,
		OPEN_SHARE_DENYNONE
		+ OPEN_FLAGS_FAIL_ON_ERROR
		+ (disk ? OPEN_FLAGS_DASD : 0)
		+ (mode & 3), 0L);
	if (r) {
		errno = r;
		return -1;
	}
	if (disk)
		return handle;
	bias += handle;

	DosSemClear((HSEM)&sema);
#endif

	unit = -1;
	for (cp = name; *cp; ++cp)
		if (isdigit(*cp))
			unit = *cp - '0';

	if (buffer == NULL) {
#ifdef __ZTC__
		buffer = farmalloc(blocksize);
#else
		buffer = _fmalloc(blocksize);
#endif
		if (buffer == NULL) {
			fprintf(stderr, "Out of memory\n");
			exit(EX_SYSTEM);
		}
	}

	tape_init();
	if (unit >= 0)
		tape_target(unit);
	tape_ready();		/* skip "Cartridge Changed" */
	r = tape_ready();
	if (r != 0) {
		fprintf(msg_file, "Tape not ready\n");
		tape_print_sense(msg_file, r);
		exit(EX_SYSTEM);
	}

	position = 0;
	blockno = 0;

	if (++tape_no == 1)
		switch (mode & 3) {
		case O_RDONLY:
			blockno = 0;
			break;
		case O_WRONLY:
			wrflag = 1;
			if (tape_space(SpaceLogEndOfMedia, 0L, NULL) != 0)
				tape_rewind(0);
			break;
		case O_RDWR:
			errno = EINVOP;
			return -1;
		}
	else
		tape_rewind(0);

	offset = tape_tell();
	if (offset == 0)
		offset = 1;
	else if (offset < 0)
		offset = 0;
	if (offset > 0) {
		extern FILE *map_file;
		position = Known;
		if (f_map_file)
			fprintf(map_file, "vol %10ld\n", offset);
	}

	pending = 0;
	
	return bias;
}

int
__rmt_read(int fd, void *p, unsigned len)
{
	int	r;
	long	actual;

	r = tape_read(p, len, len / 512, &actual);
	if (position & Known)
		blockno += actual;
	if (actual == 0) {
		if (r == SenseKey+MediaOverflow) {
			errcode = r;
			errno = ENOSPC;
			return -1;
		}
		if (r < 0 && r != EndOfTape)
			return error(r);
	}
	return actual * 512;
}

#ifdef OS2

static int
flush_buffer(void)
{
	int r, count;
	USHORT rc, nwritten;

	r = DosSemWait((HSEM)&sema, 30000L);
	if (r) {
		fprintf(stderr, "Fatal tape semaphore timeout\n");
		exit(EX_SYSTEM);
	}
	if (asy_status) {
		errno = asy_status==ERROR_OUT_OF_PAPER ? ENOSPC : EIO;
		r = -1;
	} else
		r = 0;
	pending = 0;

	/* OS/2 fault: If device driver returns an error,
	   DosWrite always returns "0 bytes written", no
	   matter how many bytes are written according to
	   the driver.

	   All I can do is hopefully assume that the data
	   is written successfully if the driver returns
	   OUT_OF_PAPER.
	 */
	if (asy_status != ERROR_OUT_OF_PAPER) {
		nwritten = asy_written;
		while (nwritten < nbytes) {
			bufp = (char far *)bufp + nwritten;
			nbytes -= nwritten;
			rc = DosWrite(handle, bufp, nbytes, &nwritten);
			if (rc && nwritten == 0) {
				fprintf(stderr, "Fatal tape write error\n");
				exit(EX_SYSTEM);
			}
		}
	}
	return r;
}

#else

static int
flush_buffer(void)
{
	int r, count;

	r = tape_buffered_wait(&actual);
	pending = 0;
	if (r == NoCommand)
		return 0;
	while (actual < nblocks) {
		bufp = (char far *)bufp + actual * 512;
		nblocks -= actual;
		r = tape_write(bufp, nblocks * 512, nblocks, &actual);
		if (r && actual == 0) {
			fprintf(stderr, "Fatal tape write error\n");
			tape_print_sense(stderr, r);
			exit(EX_SYSTEM);
		}
	}
	if (r == SenseKey+MediaOverflow) {
		errcode = r;
		errno = ENOSPC;
		return -1;
	}
	return (r < 0) ? error(r) : 0;
}

#endif

int
__rmt_write(int fd, void *p, unsigned len)
{
	int r;
	void far *q = p;

	if (pending && flush_buffer())
		return -1;
# ifdef __ZTC__
	movedata(FP_SEG(q), FP_OFF(q), FP_SEG(buffer), FP_OFF(buffer), len);
# else
	_fmemcpy(buffer, q, len);
# endif
	bufp = buffer;
	nbytes = len;
	nblocks = len / 512;
	blockno += nblocks;
# ifdef OS2
	DosSemSet((HSEM)&sema);
	r = DosWriteAsync(handle, (HSEM)&sema, &asy_status, 
		buffer, len, &asy_written);
	if (r) {
		errno = r;
		return -1;
	}
	pending = 1;
	return len;
# else
	r = tape_buffered_write(buffer, nbytes, nblocks, &actual);
	if (r == ComeAgain) {
		pending = 1;
		return len;
	}
	return (r < 0) ? error(r) : len;
# endif
}

long
__rmt_lseek(int fd, long offs, int mode)
{
	/* used:	0L, 1	-> tell
			nn, 0	-> seek
			0L, 0	-> backspace to filemark
			nn, 3	-> physical block
	*/
	long r;

	if (wrflag || offs & 0x1FF) {
		errno = EINVOP;
		return -1;
	}
	offs >>= 9;

	switch (mode) {
	case 3:
		/* physical block, direct seek required */
		if (!offset) {
			errno = EINVOP;
			return -1;
		}
		r = tape_seek(0, offs);
		if (r < 0) {
			position = 0;
			blockno = 0;
			return error(r);
		}
		blockno = offs - offset;
		return 0;
		break;
	case 2:
		/* seek to end of file */
		if (!(position & EndOfFile)) {
			r = tape_space(SpaceFileMarks, 1L, NULL);
			if (r != 0 && r != EndOfTape)
				return error(r);
			position = EndOfFile;
		}
		r = tape_tell();
		if (r == 0)
			r = 1;
		if (r > 0) {
			position |= Known;
			return blockno = r - offset;
		}
		return 1;
	case 1:
		if (offs == 0) {
			/* just tell current position */
			if (offset) {
				r = tape_tell();
				if (r == 0)
					r = 1;
				if (r > 0) {
					position |= Known;
					return blockno = r - offset;
				}
				return error(r);
			} else if (!(position & Known)) {
				errno = EINVOP;
				return -1;
			} else
				return blockno;
			break;
		}
		/* relative seek */
		if (!(position & Known)) {
			errno = EINVOP;
			return -1;
		}
		offs += blockno;
	}
	if (offset) {
		/* direct seek possible */
		r = tape_seek(0, offset + offs);
		if (r < 0) {
			position = 0;
			blockno = 0;
			return error(r);
		}
	} else if (position & Known) {
		/* known position within file, sequential seek */
		r = tape_space(SpaceBlocks, offs - blockno, NULL);
		if (r < 0) {
			blockno = 0;
			if (r==EndOfData || r==EndOfTape || r==FileMark)
				position = (offs < blockno) ? Known : EndOfFile;
			return error(r);
		}
	} else {
		/* unkown position, rewind to filemark, sequential seek */
		r = tape_space(SpaceFileMarks, -1L, NULL);
		if (r != 0 && r != FileMark && r != EndOfTape)
			return error(r);
		r = tape_space(SpaceFileMarks,  1L, NULL);
		if (r != 0 && r != FileMark && r != EndOfTape)
			return error(r);
		if (r == EndOfTape || r == FileMark)
			position |= EndOfFile;
		position |= Known;
		if (offs) {
			r = tape_space(SpaceBlocks, offs, &blockno);
			if (r) {
				if (r==EndOfData || r==EndOfTape || r==FileMark)
					position = Known | EndOfTape;
				return error(r);
			}
		}
	}
	return blockno = offs;
}

int
__rmt_close(int fd)
{
	int r;

	r = flush_buffer();
	if (wrflag) {
		r = tape_filemark(0, 1L);
		if (r < 0)
			r = error(r);
	}
	tape_term();
	wrflag = 0;
	return r;
}
