/*****************************************************************************
 * $Id: tctl.c,v 1.1.1.1 1992/01/06 20:27:09 ak Exp $
 *****************************************************************************
 * $Log: tctl.c,v $
 * Revision 1.1.1.1  1992/01/06  20:27:09  ak
 * Interface now based on ST01 and ASPI.
 * AHA_DRVR no longer supported.
 * Files reorganized.
 *
 * Revision 1.1  1992/01/06  20:27:08  ak
 * Initial revision
 *
 *****************************************************************************/

static char *rcsid = "$Id: tctl.c,v 1.1.1.1 1992/01/06 20:27:09 ak Exp $";

/*
 *	tctl.c
 *
 * Tape Control.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <io.h>
#include <limits.h>
#include <conio.h>
#include "scsi.h"
#include "tape.h"
#include "scsitape.h"

#define NBLOCKS	 60

#define INT16(x) ((x)[0] << 8 | (x)[1])
#define INT24(x) ((unsigned long)INT16(x) << 8 | (x)[2])
#define INT32(x) (INT24(x) << 8 | (x)[3])

static ExtSenseData	sense;
static unsigned char	cdb [10];
static unsigned char	inq [255];
static unsigned char	mode [255];
static unsigned char	buffer [NBLOCKS][512];
static int		trace = 0;

typedef int	(*fcnptr)(void _far *, long, long, long *);

static void
print_inquiry(void)
{
	int k;

	printf("Device type %02X.%02X, %sremovable, ISO=%d, ECMA=%d, ANSI=%d\n",
		inq[0], inq[1] & 0x7F, (inq[1] & 0x80) ? "" : "not ",
		inq[2] >> 6 & 3, inq[2] >> 3 & 7, inq[2] >> 0 & 3);
	for (k = 0; k < inq[4]; ++k)
		printf("%c", inq[5+k]);
	printf("\n");
}

static void
dump(unsigned char *p, int len)
{
	int k;
	for (k = 0; k < len; ++k) {
		if (k && k % 16 == 0)
			printf("\n");
		if (k == 0)
			printf("\t");
		printf("%02X ", p[k]);
	}
	printf("\n");
}

static void
print_mode(int page, int skip)
{
	int e, b, k, df;
	static char * pageval[] = {
		"current",
		"changeable",
		"default",
		"saved"
	};
	static char * ifcid[] = {
		"SCSI (X3.131)",
		"SMI (X3.91M-1987",
		"ESDI (X3.170)",
		"IPI-2 (X3.130-1986; X3T9.3(87-002",
		"IPI-3 (X3.132-1987; X3.147-1988",
	};

	if (skip)
		b = 4 + mode[3];
	else {
		printf("Header:\tmedia type %02X", mode[1]);
		if (mode[2] & 0x80)
			printf(", write protected");
		if (mode[2] & 0x70)
			printf(", buffer mode %d", (mode[2] & 0x70) >> 4);
		else
			printf(", unbuffered");
		if (mode[2] & 0x0F)
			printf(", speed code %d\n", mode[2] & 0xF);
		printf("\n");
		if (trace)
			dump(mode, 4);

		for (b = 4; b < 4 + mode[3]; b += 8) {
			printf("Block:\tdensity code %02X, %ld blocks of %ld bytes\n",
				mode[b+0], INT24(mode+b+1), INT24(mode+b+5));
			if (trace)
				dump(mode+4+b, 8);
		}
	}
	
	printf("Page %02X (%s values, %ssavable): ",
		page & 0x3F,
		pageval[(page & 0xC0) >> 6],
		(mode[b+0] & 0x80) ? "" : "not ");

	switch(k = mode[b+0] & 0x3F) {

	case 0x0A:
		printf("Control Mode\n");
		break;
	case 0x02:
		printf("Disconnect-Reconnect\n");
		printf("\tbuffer full ratio: %u\n", mode[b+2]);
		printf("\tbuffer empty ratio: %u\n", mode[b+3]);
		printf("\tbus inactivity limit: %lu\n", INT16(mode+b+4));
		printf("\tdisconnect time limit: %lu\n", INT16(mode+b+6));
		printf("\tconnect time limit: %lu\n", INT16(mode+b+8));
		printf("\tmaximum burst size: %lu\n", INT16(mode+b+10));
		if (mode[b+12] & 3)
			printf("\tdon't disconnect within transfer\n");
		return;
	case 0x09:
		printf("Periperal Device\n");
		printf("\tinterface: ");
		if ((k = INT16(mode+b+2)) < 5)
			printf("%s\n", ifcid[k]);
		else
			printf("%04X\n", k);
		break;

	case 0x10:
		printf("Device Configuration\n");
		break;
	case 0x11:
		printf("Medium Partition (1)\n");
		break;
	case 0x12:
		printf("Medium Partition (2)\n");
		break;
	case 0x13:
		printf("Medium Partition (3)\n");
		break;
	case 0x14:
		printf("Medium Partition (4)\n");
		break;
	case 0x01:
		printf("Read-Write Error Recovery\n");
		break;
	case 0x00:
		printf("Vendor specific\n");
		dump(mode+b, 12);
		return;
	default:
		if (k < 0x15)
			printf("Reserved\n");
		else if (k == 0x3F)
			printf("All pages\n");
		else
			printf("Vendor specific\n");
		break;
	}
	dump(mode+b, mode[b+1]);
}

int
eq(const char *p, const char *q, int len)
{
	int n = strlen(p);
	if (n < len)
		return 0;
	return strncmp(p, q, n) == 0;
}

void
help(void)
{
	static char *text[] = {
		"tape <option|command>+",
		"options:",
#if 0
		"	-Unit [<n>]	SCSI target ID <n> (default=4)",
#endif
		"	-Wait		always wait for completion (default)",
		"	-Nowait		don't wait for completion",
		"	-Abort		abort on errors (default)",
		"	-Ignore		ignore errors",
		"	-Trace		trace mode",
		"commands:",
		"	Load		Load, rewind",
		"	RETension	Load, retension",
		"	Unload		Unload, rewind",
		"	UNLOADEnd	Unload, position to end of tape",
		"	REWind		Rewind",
		"	STatus		Print status",
		"	Inquiry		Print inquiry data",
		"	MOde [<n>]	Print mode page <n> (default=all)",
		"	File [<n>]	Seek files forward/backward (default=1)",
		"	Block [<n>]	Seek blocks forward/backward (default=1)",
		"	TEll		Current block number (TDC specific)",
		"	SEek <n>	Seek block <n> (TDC specific)",
		"	ENd		Position to logical end of media",
		"	REad <file>	Read file",
		"	WRITE <file>	Write file",
		"	MARK		Write filemark",
		"	ERASE		Erase tape",
		"	Verify [<n>]	Verify <n> files (default=1)",
		"	RESET		Device reset",
		"	BUSRESET	SCSI bus reset",
		"	CDB byte+	CDB command",
		"	RCDB byte+ len	CDB command, read <len> bytes",
		NULL
	};
	char **p;
	for (p = text; *p; ++p)
		printf("%s\n", *p);
	exit(1);
}

int
transfer_in(fcnptr fcn, char *name)
{
	int file;
	int r;

	file = open(name, O_CREAT|O_TRUNC|O_WRONLY, 0600);
	if (file < 0) {
		perror(name);
		exit(1);
	}
	do {
		long n;

		r = fcn(buffer, NBLOCKS * 512, NBLOCKS, &n);
		if (write(file, buffer, (unsigned)n * 512) != n * 512) {
			printf("%s: disk full\n", name);
			exit(3);
		}
	} while (r == 0 || r == SenseKey+RecoveredData);
	close(file);
	return r;
}

int
transfer_out(fcnptr fcn, char *name)
{
	int file;
	int r;

	file = open(name, O_RDONLY);
	if (file < 0) {
		perror(name);
		exit(1);
	}
	do {
		long n, b;

		memset(buffer, 0, NBLOCKS * 512);
		n = read(file, buffer, NBLOCKS * 512);
		if (n < 0) {
			perror(name);
			exit(3);
		}
		if (n == 0)
			break;
		b = (n + 511) / 512;
		r = fcn(buffer, b * 512, b, &b);
	} while (r == 0 || r == SenseKey+RecoveredData);
	close(file);
	return r;
}

main(int argc, char **argv)
{
	int immed = 0, ignore = 0;
	int unit;
	char *p;

	if (argc <= 1)
		help();

	for (++argv; argc >= 2; --argc, ++argv) {
		char *cmd = *argv;
		if (eq(cmd, "-wait", 2))
			immed = 0;
		else if (eq(cmd, "-nowait", 2))
			immed = 1;
		else if (eq(cmd, "-ignore", 2))
			ignore = 1;
		else if (eq(cmd, "-abort", 2))
			ignore = 0;
		else if (eq(cmd, "-trace", 2))
			tape_trace(++trace);
		else
			break;
	}

	tape_init();

	unit = -1;
	if ((p = getenv("TAPE")) != NULL)
		for (; *p; ++p)
			if (isdigit(*p))
				unit = *p - '0';
	if (unit >= 0)
		tape_target(unit);

	tape_sense(&sense, sizeof sense);

	while (argc-- >= 2) {
		char *cmd = *argv++;
		int r = 0;
		if (eq(cmd, "load", 1)) {
			printf("Load\n"),
			r = tape_load(immed, 0);
		} else if (eq(cmd, "retension", 3)) {
			printf("Retension\n");
			r = tape_load(immed, 1);
		} else if (eq(cmd, "unload", 1)) {
			printf("Unload\n"),
			r = tape_unload(immed, UnloadRewind);
		} else if (eq(cmd, "unloadend", 7)) {
			printf("Unload end\n"),
			r = tape_unload(immed, UnloadEndOfTape);
		} else if (eq(cmd, "rewind", 3)) {
			printf("Rewind\n");
			r = tape_rewind(immed);
		} else if (eq(cmd, "status", 2)) {
			tape_print_sense(stdout, tape_ready());
		} else if (eq(cmd, "inquiry", 1)) {
			if ((r = tape_inquiry(inq, sizeof inq)) == 0)
				print_inquiry();
		} else if (eq(cmd, "tell", 2)) {
			long n;
			printf("Current block number\n");
			n = tape_tell();
			if (n >= 0)
				printf("	%ld\n", n);
			else
				r = n;
		} else if (eq(cmd, "end", 2)) {
			printf("Logical end of media\n");
			r = tape_space(SpaceLogEndOfMedia, 0L, NULL);
		} else if (eq(cmd, "erase", 5)) {
			printf("Erase\n");
			r = tape_erase();
		} else if (eq(cmd, "mark", 4)) {
			printf("Write filemark\n");
			r = tape_filemark(immed, 1L);
		} else if (eq(cmd, "read", 2)) {
			if (argc-- < 2) {
				printf("Missing file\n");
				exit(1);
			}
			printf("Read %s\n", *argv);
			transfer_in(tape_read, *argv++);
		} else if (eq(cmd, "compare", 2)) {
			if (argc-- < 2) {
				printf("Missing file\n");
				exit(1);
			}
			printf("Compare %s\n", *argv);
			transfer_out(tape_compare, *argv++);
		} else if (eq(cmd, "write", 5)) {
			if (argc-- < 2) {
				printf("Missing file\n");
				exit(1);
			}
			printf("Write %s\n", *argv);
			transfer_out(tape_write, *argv++);
		} else if (eq(cmd, "reset", 5)) {
			printf("SCSI reset\n");
			r = tape_reset(0);
		} else if (eq(cmd, "busreset", 8)) {
			printf("SCSI bus reset\n");
			r = tape_reset(1);
		} else if (eq(cmd, "CDB", 3)) {
			int i;
			for (i = 0; argc-- >= 2 && isdigit(**argv); )
				mode[i++] = strtoul(*argv++, NULL, 16);
			r = tape_cmd(mode, i);
		} else if (eq(cmd, "RCDB", 4)) {
			int i, n;
 			for (i = 0; argc-- >= 2 && isdigit(**argv); )
				cdb[i++] = strtoul(*argv++, NULL, 16);
			n = strtoul(*(argv-1), NULL, 10);
			r = tape_read_cmd(cdb, i-1, *buffer, n);
			if (r == 0)
				dump(*buffer, n);
		} else {
			int present = 0;
			long n = 1;
			if (argc >= 2 && strchr("0123456789+-", **argv)) {
				--argc;
				n = strtol(*argv++, NULL, 0);
				present = 1;
			}
			if (eq(cmd, "-unit", 2)) {
				tape_target((int)n);
			} else if (eq(cmd, "mode", 2)) {
				if (present) {
					if ((r = tape_mode_sense((int)n, mode, sizeof mode)) == 0)
						print_mode((int)n, 0);
				} else {
					for (n = 0; n < 0x3F; ++n)
						if ((r = tape_mode_sense((int)n, mode, sizeof mode)) == 0)
							print_mode((int)n, present++);
					r = 0;
				}
			} else if (eq(cmd, "file", 1)) {
				printf("Space over %ld filemark%s\n",
					n, n==1 ? "" : "s");
				r = tape_space(SpaceFileMarks, n, &n);
				printf("Spaced over %ld filemark%s\n",
					n, n==1 ? "" : "s");
			} else if (eq(cmd, "block", 1)) {
				printf("Space over %ld block%s\n",
					n, n==1 ? "" : "s");
				r = tape_space(SpaceBlocks, n, &n);
				printf("Spaced over %ld block%s\n",
					n, n==1 ? "" : "s");
			} else if (eq(cmd, "seek", 2)) {
				printf("Seek to block %ld\n", n);
				r = tape_seek(immed, n);
			} else if (eq(cmd, "verify", 3)) {
				long files = 0;
				printf("Verify %ld file%s\n",
					n, n==1 ? "" : "s");
				while (--n >= 0) {
					long blocks = 0, k;
					do {
						r = tape_verify(200L, &k);
						blocks += k;
					} while (r == 0);
					printf("Verified %ld block%s\n",
						blocks, blocks==1 ? "" : "s");
					if (r == SenseKey+BlankCheck
					 || r == EndOfData
					 || r == EndOfTape
					 || r != FileMark && r != 0 && !ignore)
						break;
					++files;
				}
				printf("Verified %ld file%s\n",
					files, files==1 ? "" : "s");
			} else {
				printf("Unkown command: %s\n\n", cmd);
				help();
			}
		}
		if (r != 0) {
			tape_print_sense(stdout, r);
			if (!ignore)
				exit(2);
		} else if (trace && cmd[0] != '-')
			tape_print_sense(stdout, r);
	}
	tape_term();
	exit(0);
}
