/*****************************************************************************
 * $Id: scsitape.c,v 1.1.1.1 1992/01/06 20:27:14 ak Exp $
 *****************************************************************************
 * $Log: scsitape.c,v $
 * Revision 1.1.1.1  1992/01/06  20:27:14  ak
 * Interface now based on ST01 and ASPI.
 * AHA_DRVR no longer supported.
 * Files reorganized.
 *
 * Revision 1.1  1992/01/06  20:27:12  ak
 * Initial revision
 *
 *****************************************************************************/

static char *rcsid = "$Id: scsitape.c,v 1.1.1.1 1992/01/06 20:27:14 ak Exp $";

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#define INCL_DOSFILEMGR
#define INCL_DOSDEVICES
#define INCL_DOSPROCESS
#define INCL_DOSSEMAPHORES
#define INCL_ERRORS
#include <os2.h>

#include "scsi.h"
#include "aspi.h"
#include "tapedrvr.h"

#define CDB_LEN		12
#define SENSE_LEN	40

typedef struct SCB {
	void *		cdb;
	void _far *	data;
	long		data_len;
	BYTE		target;
	BYTE		lun;
	BYTE		cdb_len;
	BYTE		sense_len;
	BYTE		readflag;
	BYTE		sense [SENSE_LEN];
	unsigned	error;
} SCB;

static HFILE	hdev;
static BYTE	sense_cmd[6] = { CmdRequestSense, 0, 0, 0, 0, 0 };
static TID	thread_id;
static BYTE	thread_stack[4096];
static ULONG	sema;
static SCB	*thread_scb;
static BYTE	trace = 0;
static BYTE	drvLevel = 0;
enum SenseMode	senseMode = Sensekey;

static void
fatal(char *msg, ...)
{
	va_list ap;
	fprintf(stderr, "Fatal SCSI error: ");
	va_start(ap, msg);
	vfprintf(stderr, msg, ap);
	va_end (ap);
	fprintf(stderr, "\n");
	exit (2);
}

command(BYTE *cdb, int len)
{
	int i;
	printf("SCSI op:");
	for (i = 0; i < len; ++i)
		printf(" %02X", cdb[i]);
	printf("\n");
}

status(BYTE *sense, int len)
{
	int i;
	printf("SCSI status:");
	for (i = 0; i < len; ++i) {
		if (i && (i % 16) == 0)
			printf("\n            ");
		printf(" %02X", sense[i]);
	}
	printf("\n");
}

void
scsi_init(void)
{
	USHORT rc, action;
	BYTE data[2];
	char *name;

	name = getenv("TAPE");
	if (name == NULL)
		fatal("Missing environment name TAPE\n");
	while (*name == '+')
		++name;

	rc = DosOpen((PSZ)name, &hdev, &action, 0L, 0,
		FILE_OPEN, OPEN_ACCESS_READWRITE+OPEN_SHARE_DENYNONE+OPEN_FLAGS_FAIL_ON_ERROR, 0L);
	if (rc)
		fatal("Cannot access device %s, return code %u", name, rc);

	DosDevIOCtl2(data, 2, 0, 0, IOCtlLevel, IOCtlCategory, hdev);
	drvLevel = data[0];
	senseMode = data[1];
	if (trace)
		printf("Device %s driver level: %d, sense mode: %d\n",
			name, drvLevel, senseMode);
}

void
scsi_term(void)
{
	DosClose(hdev);
}

void
scsi_trace(int level)
{
	trace = level;
}

void *
scsi_alloc(void)
{
	void *p = malloc(sizeof(SCB));
	if (p == NULL)
		fatal("No memory for SCSI Control Block\n");
}

void
scsi_free(void *p)
{
	free(p);
}

static unsigned
mapDriverError(unsigned rc)
{
	if (rc < 0xFF00)
		return SystemError + ErrorCode(rc);
	if (!drvLevel)
		return DriverError + ErrST01Driver + (rc & ErrMask);
	switch (rc & ErrSource) {
	case ErrTargetStatus:
		return StatusError + (rc & ErrMask);
	case ErrST01Driver:
	case ErrASPIDriver1:
	case ErrASPIDriver2:
		return DriverError + (rc & 0xFF);
	case ErrHostAdapter:
		return HostError + (rc & ErrMask);
	default:
		return UnknownError + (rc & 0xFF);
	}
}

int
scsi_cmd(int target, int lun,
	void *cdb, int cdb_len,	void *sense, int sense_len,
	void _far *data, long data_len,
	int readflag)
{
	BYTE	cmd, *sptr;
	USHORT	fcn, rc, len;

	if (cdb_len > CDB_LEN || sense_len > SENSE_LEN)
		return SenseKey+IllegalRequest;

	cmd = *(BYTE *)cdb;
	fcn = (cmd == SeqRead || cmd == SeqWrite) ? IOCtlFast : IOCtlSlow;
	if (drvLevel)
		fcn += readflag ? IOCtlRead : IOCtlWrite;
	if (data_len)
		rc = DosDevIOCtl2(data, (USHORT)data_len, cdb, cdb_len,
				fcn, IOCtlCategory, hdev);
	else
		rc = DosDevIOCtl2(0, 0, cdb, cdb_len,
				fcn, IOCtlCategory, hdev);

	if (drvLevel && rc == 0)
		return NoError;
	if (trace >= 2)
		printf("IOCtl return %X\n", rc);
	if (cmd == CmdRequestSense)
		return rc ? mapDriverError(rc) : 0; 
	if (rc) {
		if (drvLevel) {
			if (rc != 0xFF00+ErrTargetStatus+CheckStatus)
				return mapDriverError(rc); 
		} else
			if (rc != 0xFF00+CheckStatus)
				return mapDriverError(rc); 
	}

	rc = DosDevIOCtl2(sense, (USHORT)sense_len, sense_cmd, sizeof sense_cmd,
			drvLevel ? IOCtlSense : IOCtlSlow, IOCtlCategory, hdev);
	if (rc)
		return mapDriverError(rc);

	if (trace == 1)
		command(cdb, cdb_len);
	if (trace)
		status(sense, sense_len);

	sptr = (BYTE *)sense;
	if ((sptr[0] & 0x7E) != 0x70)
		return ExtendedError + (sptr[0] & 0x7F);
	if (sense_len <= 2)
		return UnknownError;
	return SenseKey + (sptr[2] & 0x0F);
}

#pragma check_stack(off)

static void _far
thread_fcn(void)
{
	for (;;) {
		DosSuspendThread(thread_id);
		if (thread_scb) {
			thread_scb->error = scsi_cmd(thread_scb->target,
				thread_scb->lun,
				thread_scb->cdb, thread_scb->cdb_len,
				thread_scb->sense, thread_scb->sense_len,
				thread_scb->data, thread_scb->data_len,
				thread_scb->readflag);
			DosSemClear((HSEM)&sema);
		}
	}
}

#pragma check_stack()

int
scsi_start(void *dcb, 
	int target, int lun,
	void *cdb, int cdb_len, int sense_len,
	void _far *data, long data_len,
	int readflag)
{
	SCB *scb = (SCB *)dcb;

	if (sense_len > SENSE_LEN)
		return SenseKey+IllegalRequest;

	if (thread_id == 0) {
		USHORT rc;
		rc = DosCreateThread(thread_fcn, &thread_id, thread_stack+4096);
		if (rc)
			fatal("Cannot create thread, return code %u", rc);
		DosSemClear((HSEM)&sema);
	}

	if (thread_scb)
		return SystemError+ErrorCode(ERROR_DEVICE_IN_USE);

	DosSemWait((HSEM)&sema, -1);

	scb->target    = target;
	scb->lun       = lun;
	scb->cdb       = cdb;
	scb->cdb_len   = cdb_len;
	scb->data      = data;
	scb->data_len  = data_len;
	scb->sense_len = sense_len;
	scb->readflag  = readflag;
	thread_scb = scb;

	DosSemSet((HSEM)&sema);
	DosResumeThread(thread_id);

	return ComeAgain;
}

int
scsi_wait(void *dcb, void *sense, int wait)
{
	SCB *scb = (SCB *)dcb;

	if (scb != thread_scb)
		return SystemError+ErrorCode(ERROR_INVALID_HANDLE);
	if (DosSemWait((HSEM)&sema, wait ? -1 : 0))
		return ComeAgain;
	thread_scb = 0;
	memcpy(sense, scb->sense, scb->sense_len);
	return scb->error;
}

int
scsi_reset(int target, int lun, int bus)
{
	USHORT rc = DosDevIOCtl(0, 0,
		bus ? IOCtlBusReset : IOCtlDevReset, IOCtlCategory, hdev);
	if (rc)
		return mapDriverError(rc);
	return NoError;
}

char *
scsi_error(int code)
{
	static char text[80];
	static ErrorTable driverTab[] = {
			/* ST01 driver error codes */
		ErrST01Driver+0x02,	"Device not ready",
		ErrST01Driver+0x0C,	"Bus sequence error",
		ErrST01Driver+0x13,	"Parameter error",

			/* ASPI status codes */
		ErrASPIDriver1+0x00,	"Busy",
		ErrASPIDriver1+0x01,	"Done",
		ErrASPIDriver1+0x02,	"Aborted",
		ErrASPIDriver1+0x03,	"Bad aborted",
		ErrASPIDriver1+0x04,	"Error",
		ErrASPIDriver1+0x10,	"Busy POST",
		ErrASPIDriver2+0x00,	"Invalid ASPI request",
		ErrASPIDriver2+0x01,	"Invalid host adapter",
		ErrASPIDriver2+0x02,	"Device not installed",

			/* ASPITAPE driver error codes */
		TapeInvalidFcn,		"Invalid ioctl cat/fcn code",
		TapeInvalidParm,	"Invalid parm pointer/length",
		TapeInvalidData,	"Invalid data pointer/length",

		-1
	};
	static ErrorTable hostTab[] = {
			/* Adaptec 154x host adapter status */
		SRB_NoError,		"No error",
		SRB_Timeout,		"Selection timeout",
		SRB_DataLength,		"Data length error",
		SRB_BusFree,		"Unexpected bus free",
		SRB_BusSequence,	"Target bus sequence failure",
		-1
	};

	if (code == 0)
		return "No error";
	if (code == ComeAgain)
		return "Busy";
	switch (ErrorClass(code)) {
	case SenseKey:
		return senseTab[code & 0x0F];
	case ExtendedError:
		switch (senseMode) {
		case TDC3600:
			sprintf(text, "Error code: %s",
				find_error(tdc3600ercd, ErrorCode(code)));
			break;
		default:
			sprintf(text, "Additional sense code: %02X",
				ErrorCode(code));
		}
		break;
	case StatusError:
		sprintf(text, "Target status: %s",
			find_error(targetStatusTab, ErrorCode(code)));
		break;
	case DriverError:
		sprintf(text, "Driver error: %s",
			find_error(driverTab, ErrorCode(code)));
		break;
	case SystemError:
		sprintf(text, "System error: %u", ErrorCode(code));
		break;
	case HostError:
		sprintf(text, "Host adapter error: %s",
			find_error(hostTab, ErrorCode(code)));
		break;
	default:
		sprintf(text, "Other error: %04X", code);
		break;
	}
	return text;
}
