/*****************************************************************************
 * $Id: extract.c,v 1.2 1992/01/10 13:27:37 ak Exp $
 *****************************************************************************
 * $Log: extract.c,v $
 * Revision 1.2  1992/01/10  13:27:37  ak
 * [tar/tape -> ] buffer -> compress [ -> tar/tape] (-Z)
 * DOS & OS/2 file attributes (-p)
 * Don't recurse (-Y)
 *
 * Revision 1.1.1.1  1992/01/06  20:40:56  ak
 * -Y = don't recurse: new.
 * -X = exclude list: for extract.
 * Use BUFFER for OS/2 tape compression.
 * No own tape buffering for OS/2.
 * Support for SYSTEM and HIDDEN files.
 *
 * Revision 1.1  1992/01/06  20:40:54  ak
 * Initial revision
 *
 *****************************************************************************/

static char *rcsid = "$Id: extract.c,v 1.2 1992/01/10 13:27:37 ak Exp $";

/* Extract files from a tar archive.
   Copyright (C) 1988 Free Software Foundation

This file is part of GNU Tar.

GNU Tar is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GNU Tar is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Tar; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/*
 * Extract files from a tar archive.
 *
 * Written 19 Nov 1985 by John Gilmore, ihnp4!hoptoad!gnu.
 *
 * @(#) extract.c 1.32 87/11/11 - gnu
 */

#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef BSD42
#include <sys/file.h>
#endif

#ifdef USG
#include <fcntl.h>
#endif

#ifdef	MSDOS
#include <fcntl.h>
#endif	/* MSDOS */

/*
 * Some people don't have a #define for these.
 */
#ifndef	O_BINARY
#define	O_BINARY	0
#endif
#ifndef O_NDELAY
#define	O_NDELAY	0
#endif

#ifdef NO_OPEN3
/* We need the #define's even though we don't use them. */
#include "open3.h"
#endif

#ifdef EMUL_OPEN3
/* Simulated 3-argument open for systems that don't have it */
#include "open3.h"
#endif

#include "tar.h"
#include "port.h"
#include "rmt.h"

#ifndef __ZTC__
#ifndef MICROSOFT
extern int errno;			/* From libc.a */
#endif
#endif
extern time_t time();			/* From libc.a */
extern char *index();			/* From libc.a or port.c */

extern FILE *msg_file;

extern union record *head;		/* Points to current tape header */
extern struct stat hstat;		/* Stat struct corresponding */
extern int head_standard;		/* Tape header is in ANSI format */

extern char *save_name;
extern long save_totsize;
extern long save_sizeleft;

extern void print_header();
extern void skip_file();
extern void pr_mkdir();

int make_dirs();			/* Makes required directories */

static time_t now = 0;			/* Current time */
static we_are_root = 0;			/* True if our effective uid == 0 */
static int notumask = ~0;		/* Masks out bits user doesn't want */

scan_for(do_something, name, rec)
	void (*do_something)();
	char *name;
	long rec;
{
	extern long baserec;
	extern long from_oct();

	while (read_header() == 1) {
		/* We should decode next field (mode) first... */
		/* Ensure incoming names are null terminated. */
		head->header.name[NAMSIZ-1] = '\0';

#ifdef MSDOS
		if (stricmp(head->header.name, name) == 0) {
#else
		if (strcmp(head->header.name, name) == 0) {
#endif
			(*do_something)();
			return 1;
		} else {
			/* Skip past it in the archive */
			userec(head);
			/* Skip to the next header on the archive */
			if(head->header.linkflag!='5')
				skip_file((long)hstat.st_size);
		}
	}
	return 0;
}

/*
 * Random archive access.
 */
void
seek_exec(do_something, name, vol, voloff, rec)
	void (*do_something)();
	char *name;
	int vol;
	long voloff, rec;
{
	long r;
	int offs;
	static int error_flag;
	extern int volno;
	extern long baserec;

	if (f_multivol) {
		if (vol != volno) {
			/* next volume */
			close_archive(1);
			sprintf(ar_file + strlen(ar_file) - 3, "%03d", vol);
			open_archive(1);
			ar_last = ar_record = ar_block;
			baserec = f_split ? voloff : 0;
			volno = vol;
		}

		if (f_split) {
			if (!scan_for(do_something, name, rec)) {
				msg("%s not found at vol %d rec %ld", name, vol, rec);
				exit(EX_BADFILE);
			}
			return;
		}

		/* align seek to blocking boundary */
		offs = rec % blocking;
		rec -= offs;
		r = rmtlseek(archive, (voloff + rec) * RECORDSIZE, 3);
		ar_last = ar_record = ar_block;

		/* skip alignment blocks */
		if (offs)
			userec(findrec() + offs-1);
	} else {
		r = rmtlseek(archive, (voloff + rec) * RECORDSIZE, 3);
		ar_last = ar_record = ar_block;
		baserec = rec;
	}
	if (r < 0) {
		if (vol)
			msg_perror("Seek error on %s, volume %d record %ld",
				 ar_file, vol, rec);
		else
			msg_perror("Seek error on %s, record %ld",
				 ar_file, rec);
		if (++error_flag > 10) {
			msg("Too many errors, quitting.");
			exit(EX_BADARCH);
		}
		return;
	}
	if (read_header() == 1) {
		/* We should decode next field (mode) first... */
		/* Ensure incoming names are null terminated. */
		head->header.name[NAMSIZ-1] = '\0';

		/* Handle the archive member */
		(*do_something)();
	} else {
		if (vol)
			msg("File %s volume %d record %ld has invalid header, wrong map file?",
				name, vol, rec);
		else
			msg("File %s record %ld has invalid header, wrong map file?",
				name, rec);
		if (++error_flag > 10) {
			msg("Too many errors, quitting.");
			exit(EX_BADARCH);
		}
	}
}

/*
 * Main loop for random access.
 */
void
seek_and(do_something)
	void (*do_something)();
{
	extern FILE *map_file;
	struct name *nlp;
	char	line[200];
	long	voloff, offset;
	char	*p;
	int	len, volume;
	struct Volmap {
		long		base;	/* physical base record no */
		short		ctrl;	/* no of inserted control records */
		short		num;	/* volume no */
		struct Volmap	*prev;	/* previous volume */
	} *volmap, *q;

	name_gather();			/* Gather all the names */
	open_archive(1);		/* Open for reading */

	if (f_split) {
		volmap = NULL;
		rewind(map_file);
		for (len = 0; fgets(line, 200, map_file); )
			if (line[0] == 'V'
			 && sscanf(line+1, "%d %ld", &volume, &voloff) == 2) {
				q = (struct Volmap *)malloc(sizeof(struct Volmap));
				if (q == NULL) {
					msg("Out of memory");
					exit(EX_SYSTEM);
				}
				q->num = volume;
				q->base = voloff;
				q->ctrl = ++len;
				q->prev = volmap;
				volmap = q;
			} else if (strncmp(line, "rec ", 4) == 0) {
				p = strchr(line, ':');
				if (volmap == NULL) {
					msg("Need -V records for multivolume -F");
					exit(EX_BADARCH);
				} else if (p && *(p+2) == 'M')
					volmap->ctrl = ++len;
			}
	}

	volume = voloff = 0;
	rewind(map_file);
	while (fgets(line, 200, map_file)) {
		char *name = line + strlen(line);
		char *key  = strtok(line, " \t");
		char *arg  = key ? strtok(NULL, ": \t\r\n") : NULL;
		char *mode = arg ? strtok(NULL, " \t") : NULL;
		offset = arg ? strtoul(arg, NULL, 0) : 0;
		if (strncmp(key, "vol", 3) == 0) {
			/* base address of tape file */
			voloff = offset;
			continue;
		}
		if (key[0] == 'V') {
			/* tape number */
			volume = atoi(key+1);
			continue;
		}
		if (strncmp(key, "rec", 3) != 0)
			continue;
		if (mode[0] == 'M')
			continue;

		/* get file name from record line -- skip "(..)" descr */
		while (isspace(*--name))
			;
		if (mode[0] == 'A' && *name == ')')
			while (*--name != '(')
				;
		while (isspace(*(name-1)))
			--name;
		*(name+1) = '\0';
		while (!isspace(*(name-1)))
			--name;
		len = strlen(name);

		/* look if name is mentioned in command line */
		for (nlp = namelist; nlp; nlp = nlp->next) {
			/* fast check for first character */
			if (nlp->firstch && nlp->name[0] != name[0])
				continue;

			/* regular expression? */
			if (nlp->regexp) {
				if (wildmat(name, nlp->name))
					goto match;
				continue;
			}

			/* plain name */
			if (nlp->length <= len
			 && (name[nlp->length] == '\0'
			  || name[nlp->length] == '/')
#ifdef MSDOS
			 && strnicmp(name, nlp->name, nlp->length) == 0)
#else
			 && strncmp(name, nlp->name, nlp->length) == 0)
#endif
			 	goto match;
			continue;

		match:	--offset;
			if (f_split) {
				/* get volume offset */
				for (q = volmap; q; q = q->prev)
					if (q->base <= offset)
						goto found;
				continue;
			found:	seek_exec(do_something, name, q->num, q->base, q->ctrl-1);
			} else
				seek_exec(do_something, name, volume, voloff, offset);
			if (mode[0] != 'A')
				nlp->found = 1;
		}
	}

	close_archive(1);
	names_notfound();		/* Print names not found */
}

/*
 * Set up to extract files.
 */
extr_init()
{
	int ourmask;

	now = time((time_t *)0);
	if (geteuid() == 0)
		we_are_root = 1;

	/*
	 * We need to know our umask.  But if f_use_protection is set,
	 * leave our kernel umask at 0, and our "notumask" at ~0.
	 */
#ifdef MSDOS
	notumask = ~0;
#else
	ourmask = umask(0);		/* Read it */
	if (!f_use_protection) {
		(void) umask (ourmask);	/* Set it back how it was */
		notumask = ~ourmask;	/* Make umask override permissions */
	}
#endif
}


/*
 * Extract a file from the archive.
 */
void
extract_archive()
{
	register char *data;
	int fd, check, namelen, written, openflag;
	long size;
	time_t acc_upd_times[2];
	register int skipcrud;

	saverec(&head);			/* Make sure it sticks around */
	userec(head);			/* And go past it in the archive */
	decode_header(head, &hstat, &head_standard, 1);	/* Snarf fields */

	if(f_exclude && check_exclude(head->header.name)) {
		skip_file((long)hstat.st_size);
		saverec((union record **)0);
		return;
	}
	if(f_confirm && !confirm("extract",head->header.name)) {
		skip_file((long)hstat.st_size);
		saverec((union record **)0);
		return;
	}

	if (f_mapnames)
		mapname(head->header.name);

	/* Print the record from 'head' and 'hstat' */
	if (f_verbose)
		print_header();

	/*
	 * Check for fully specified pathnames and other atrocities.
	 *
	 * Note, we can't just make a pointer to the new file name,
	 * since saverec() might move the header and adjust "head".
	 * We have to start from "head" every time we want to touch
	 * the header record.
	 */
	skipcrud = 0;
	while ('/' == head->header.name[skipcrud]) {
		static int warned_once = 0;

		skipcrud++;	/* Force relative path */
		if (!warned_once++) {
			msg("Removing leading / from absolute path names in the archive.");
		}
	}

	switch (head->header.linkflag) {

	default:
		msg("Unknown file type '%c' for %s, extracted as normal file",
			head->header.linkflag, skipcrud+head->header.name);
		/* FALL THRU */

	case LF_OLDNORMAL:
	case LF_NORMAL:
	case LF_CONTIG:
		/*
		 * Appears to be a file.
		 * See if it's really a directory.
		 */
		namelen = strlen(skipcrud+head->header.name)-1;
		if (head->header.name[skipcrud+namelen] == '/')
			goto really_dir;

		/* FIXME, deal with protection issues */
	again_file:
		openflag = f_keep?
#if 1
			O_BINARY|O_NDELAY|O_WRONLY|O_CREAT|O_EXCL:
			O_BINARY|O_NDELAY|O_WRONLY|O_CREAT|O_TRUNC;
#else	/* Was soll denn der Unsinn? O_TRUNC/EXCL + O_APPEND? */
			O_BINARY|O_NDELAY|O_WRONLY|O_APPEND|O_CREAT|O_EXCL:
			O_BINARY|O_NDELAY|O_WRONLY|O_APPEND|O_CREAT|O_TRUNC;
#endif
		if(f_exstdout) {
			fd = 1;
			goto extract_file;
		}
#ifdef O_CTG
		/*
		 * Contiguous files (on the Masscomp) have to specify
		 * the size in the open call that creates them.
		 */
		if (head->header.lnkflag == LF_CONTIG)
			fd = open(skipcrud+head->header.name, openflag | O_CTG,
				hstat.st_mode, hstat.st_size);
		else
#endif
		{
#ifdef NO_OPEN3
			/*
			 * On raw V7 we won't let them specify -k (f_keep), but
			 * we just bull ahead and create the files.
			 */
			fd = creat(skipcrud+head->header.name, 
				hstat.st_mode);
#else
			/*
			 * With 3-arg open(), we can do this up right.
			 */
			fd = open(skipcrud+head->header.name, openflag,
# ifdef MSDOS
				/* some libs cannot create readonly files */
				0666);
# else
				hstat.st_mode);
# endif
#endif
		}

		if (fd < 0) {
			if (make_dirs(skipcrud+head->header.name))
				goto again_file;
			msg_perror("Could not create file %s",skipcrud+head->header.name);
			skip_file((long)hstat.st_size);
			goto quit;
		}

	extract_file:
		for (size = hstat.st_size;
		     size > 0;
		     size -= written) {

			if(f_multivol) {
				save_name=head->header.name;
				save_totsize=hstat.st_size;
				save_sizeleft=size;
			}

			/*
			 * Locate data, determine max length
			 * writeable, write it, record that
			 * we have used the data, then check
			 * if the write worked.
			 */
			data = findrec()->charptr;
			if (data == NULL) {	/* Check it... */
				msg("Unexpected EOF on archive file");
				break;
			}
			written = endofrecs()->charptr - data;
			if (written > size)
				written = size;
			errno = 0;
			check = write(fd, data, written);
			/*
			 * The following is in violation of strict
			 * typing, since the arg to userec
			 * should be a struct rec *.  FIXME.
			 */
			userec((union record *)(data + written - 1));
			if (check == written) continue;
			/*
			 * Error in writing to file.
			 * Print it, skip to next file in archive.
			 */
			if(check<0)
				msg_perror("couldn't write to file %s",skipcrud+head->header.name);
			else
				msg("could only write %d of %d bytes to file %s",written,check,skipcrud+head->header.name);
			skip_file((long)(size - written));
			break;	/* Still do the close, mod time, chmod, etc */
		}

		if(f_multivol)
			save_name = 0;

			/* If writing to stdout, don't try to do anything
			   to the filename; it doesn't exist, or we don't
			   want to touch it anyway */
		if(f_exstdout)
			break;

		check = close(fd);
		if (check < 0) {
			msg_perror("Error while closing %s",skipcrud+head->header.name);
		}
		
	set_filestat:
#ifdef OS2
		if (eabuf && strcmp(eaname, head->header.name) == 0)
			if (ea_store(eabuf, skipcrud+head->header.name) == -1)
				msg_perror("Error while setting extended attributes of %s",skipcrud+head->header.name);
#endif

		/*
		 * Set the modified time of the file.
		 * 
		 * Note that we set the accessed time to "now", which
		 * is really "the time we started extracting files".
		 * unless f_gnudump is used, in which case .st_atime is used
		 */
		if (!f_modified) {
			/* fixme if f_gnudump should set ctime too, but how? */
			if(f_gnudump) acc_upd_times[0]=hstat.st_atime;
			else acc_upd_times[0] = now;	         /* Accessed now */
			acc_upd_times[1] = hstat.st_mtime; /* Mod'd */
			if (utime(skipcrud+head->header.name, acc_upd_times) < 0) {
#ifndef MSDOS
				msg_perror("couldn't change access and modification times of %s",skipcrud+head->header.name);
#endif
			}
		}

		/*
		 * If we are root, set the owner and group of the extracted
		 * file.  This does what is wanted both on real Unix and on
		 * System V.  If we are running as a user, we extract as that
		 * user; if running as root, we extract as the original owner.
		 */
		if (we_are_root) {
			if (chown(skipcrud+head->header.name, hstat.st_uid,
				  hstat.st_gid) < 0) {
				msg_perror("cannot chown file %s to uid %d gid %d",skipcrud+head->header.name,hstat.st_uid,hstat.st_gid);
			}
		}

		/*
		 * If '-k' is not set, open() or creat() could have saved
		 * the permission bits from a previously created file,
		 * ignoring the ones we specified.
		 * Even if -k is set, if the file has abnormal
		 * mode bits, we must chmod since writing or chown() has
		 * probably reset them.
		 *
		 * If -k is set, we know *we* created this file, so the mode
		 * bits were set by our open().   If the file is "normal", we
		 * skip the chmod.  This works because we did umask(0) if -p
		 * is set, so umask will have left the specified mode alone.
		 */
		if (!f_keep
#ifndef MSDOS
		    || (hstat.st_mode & (S_ISUID|S_ISGID|S_ISVTX))
#endif
		) {
			if (chmod(skipcrud+head->header.name,
				  notumask & (int)hstat.st_mode) < 0) {
				msg_perror("cannot change mode of file %s to %ld",skipcrud+head->header.name,notumask & (int)hstat.st_mode);
			}
		}

#ifdef MSDOS
		if (f_use_protection)
			put_fileattr(skipcrud+head->header.name, &hstat);
#endif
	quit:
		break;

	case LF_LINK:
	again_link:
		check = link (head->header.linkname,
			      skipcrud+head->header.name);
		if (check == 0)
			break;
		if (make_dirs(skipcrud+head->header.name))
			goto again_link;
		if(f_gnudump && errno==EEXIST)
			break;
		msg_perror("Could not link %s to %s",
			skipcrud+head->header.name,head->header.linkname);
		break;

#ifdef S_IFLNK
	case LF_SYMLINK:
	again_symlink:
		check = symlink(head->header.linkname,
			        skipcrud+head->header.name);
		/* FIXME, don't worry uid, gid, etc... */
		if (check == 0)
			break;
		if (make_dirs(skipcrud+head->header.name))
			goto again_symlink;
		msg_perror("Could not create symlink to %s",head->header.linkname);
		break;
#endif

#ifdef S_IFCHR
	case LF_CHR:
		hstat.st_mode |= S_IFCHR;
		goto make_node;
#endif

#ifdef S_IFBLK
	case LF_BLK:
		hstat.st_mode |= S_IFBLK;
		goto make_node;
#endif

#ifdef S_IFIFO
	/* If local system doesn't support FIFOs, use default case */
	case LF_FIFO:
		hstat.st_mode |= S_IFIFO;
		hstat.st_rdev = 0;		/* FIXME, do we need this? */
		goto make_node;
#endif

	make_node:
		check = mknod(skipcrud+head->header.name,
			      (int) hstat.st_mode, (int) hstat.st_rdev);
		if (check != 0) {
			if (make_dirs(skipcrud+head->header.name))
				goto make_node;
			msg_perror("Could not make %s",skipcrud+head->header.name);
			break;
		};
		goto set_filestat;

	case LF_DIR:
	case LF_DUMPDIR:
		namelen = strlen(skipcrud+head->header.name)-1;
	really_dir:
		/* Check for trailing /, and zap as many as we find. */
		while (namelen && head->header.name[skipcrud+namelen] == '/')
			head->header.name[skipcrud+namelen--] = '\0';
		if(f_gnudump) {		/* Read the entry and delete files
					   that aren't listed in the archive */
			gnu_restore(skipcrud+head->header.name);
		
		} else if(head->header.linkflag==LF_DUMPDIR)
			skip_file((long)(hstat.st_size));

	
	again_dir:
		check = mkdir(skipcrud+head->header.name,
			      0300 | (int)hstat.st_mode);
		if (check != 0) {
			if (make_dirs(skipcrud+head->header.name))
				goto again_dir;
			/* If we're trying to create '.', let it be. */
			if (head->header.name[skipcrud+namelen] == '.' && 
			    (namelen==0 ||
			     head->header.name[skipcrud+namelen-1]=='/'))
				goto check_perms;
			if(f_gnudump && errno==EEXIST)
				break;
			msg_perror("Could not make directory %s",skipcrud+head->header.name);
			break;
		}
		
	check_perms:
#ifndef MSDOS
		if (0300 != (0300 & (int) hstat.st_mode)) {
			hstat.st_mode |= 0300;
			msg("Added write and execute permission to directory %s",
			  skipcrud+head->header.name);
		}
#endif

#ifdef OS2
		if (eabuf && strcmp(eaname, head->header.name) == 0)
			if (ea_store(eabuf, skipcrud+head->header.name) == -1)
				msg_perror("Error while setting extended attributes of %s",skipcrud+head->header.name);
		break;
#else
		goto set_filestat;
#endif
		/* FIXME, Remember timestamps for after files created? */
		/* FIXME, change mode after files created (if was R/O dir) */
	case LF_VOLHDR:
		if(f_verbose) {
			printf("Reading %s\n",head->header.name);
		}
	case LF_MULTIVOL:
		msg("Can't extract '%s'--file is continued from another volume\n",head->header.name);
		skip_file((long)hstat.st_size);
		break;

	case LF_EATTR:
#ifdef OS2
	{	char _far *eaptr;

		ea_free(eabuf);
		ealen = hstat.st_size;
		eabuf = ea_alloc(ealen);
		eaptr = (char _far *)eabuf;
		strcpy(eaname, head->header.name);

		if (eabuf == (PEAList)0)
			msg("Not enough memory for extended attributes (%ld bytes)", ealen);

		for (size = hstat.st_size;
		     size > 0;
		     size -= written) {
			/*
			 * Locate data, determine max length
			 * writeable, write it, record that
			 * we have used the data, then check
			 * if the write worked.
			 */
			data = findrec()->charptr;
			if (data == NULL) {	/* Check it... */
				msg("Unexpected EOF on archive file");
				break;
			}
			written = endofrecs()->charptr - data;
			if (written > size)
				written = size;
			if (eabuf)
				_fmemcpy(eaptr, data, written);
			eaptr += written;
			/*
			 * The following is in violation of strict
			 * typing, since the arg to userec
			 * should be a struct rec *.  FIXME.
			 */
			userec((union record *)(data + written - 1));
		}
	}
#else
		msg("OS/2 extended attributes for '%s' ignored", head->header.name);
		for (size = hstat.st_size;
		     size > 0;
		     size -= written) {
			data = findrec()->charptr;
			if (data == NULL) {	/* Check it... */
				msg("Unexpected EOF on archive file");
				break;
			}
			written = endofrecs()->charptr - data;
			if (written > size)
				written = size;
			/*
			 * The following is in violation of strict
			 * typing, since the arg to userec
			 * should be a struct rec *.  FIXME.
			 */
			userec((union record *)(data + written - 1));
		}
#endif
		break;

	}

	/* We don't need to save it any longer. */
	saverec((union record **) 0);	/* Unsave it */
}

/*
 * After a file/link/symlink/dir creation has failed, see if
 * it's because some required directory was not present, and if
 * so, create all required dirs.
 */
int
make_dirs(pathname)
	char *pathname;
{
	char *p;			/* Points into path */
	int madeone = 0;		/* Did we do anything yet? */
	int save_errno = errno;		/* Remember caller's errno */
	int check;
	switch (errno) {
#ifdef __ZTC__
	case ENOTDIR:	/* sometimes... */
	case -1:	/* Zortech bug */
#endif
	case ENOENT:
		break;
	default:
		return 0;		/* Not our problem */
	}
	for (p = index(pathname, '/'); p != NULL; p = index(p+1, '/')) {
		/* Avoid mkdir of empty string, if leading or double '/' */
		if (p == pathname || p[-1] == '/')
			continue;
		/* Avoid mkdir where last part of path is '.' */
		if (p[-1] == '.' && (p == pathname+1 || p[-2] == '/'))
			continue;
		*p = 0;				/* Truncate the path there */
		check = mkdir (pathname, 0777);	/* Try to create it as a dir */
		if (check == 0) {
			/* Fix ownership */
			if (we_are_root) {
				if (chown(pathname, hstat.st_uid,
					  hstat.st_gid) < 0) {
					msg_perror("cannot change owner of %s to uid %d gid %d",pathname,hstat.st_uid,hstat.st_gid);
				}
			}
			pr_mkdir(pathname, p-pathname, notumask&0777);
			madeone++;		/* Remember if we made one */
			*p = '/';
			continue;
		}
		*p = '/';
#ifdef MSDOS
		if (errno == EACCES)		/* Directory already exists */
			continue;
#else
		if (errno == EEXIST)		/* Directory already exists */
			continue;
#endif
		/*
		 * Some other error in the mkdir.  We return to the caller.
		 */
		break;
	}

	errno = save_errno;		/* Restore caller's errno */
	return madeone;			/* Tell them to retry if we made one */
}

mapname(name)
	char *name;
{
	char *p, *q;
	int dot = 0;

	p = q = head->header.name;
 	for (q += strlen(p); p < q; ) {
 		if (*q == '.' && ++dot > 1)
				*q = '_';
		else if (!isalnum(*p) && !strchr("-_.", *p))
	  		*p = '_';
	  	if (*p == '/')
	  		dot = 0;
	}
}


