/***************************************
  $Revision: 1.5 $

  Error reporting (er) er_macro.c - simple macro processor

  Status: NOT REVUED, PARTLY TESTED

  Design and implementation by: Marek Bukowy

  ******************/ /******************
  Copyright (c) 1999,2000                             RIPE NCC
 
  All Rights Reserved
  
  Permission to use, copy, modify, and distribute this software and its
  documentation for any purpose and without fee is hereby granted,
  provided that the above copyright notice appear in all copies and that
  both that copyright notice and this permission notice appear in
  supporting documentation, and that the name of the author not be
  used in advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.
  
  THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
  AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
  DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
  ***************************************/

#include <string.h>
#include <glib.h>
#include "stubs.h"
#include "sk.h"
#include "er_macro.h"
#include "er_paths.h"
#include "er_yacc_helper.h" 
#include "memwrap.h"

#include "ca_configFns.h"
#include "ca_dictSyms.h"
#include "ca_macros.h"

GHashTable *er_macro_hash = NULL;

/* process a macro call, i.e. execute one of the predefined macros 
   selected by the 0th argument, using other arguments.
   
   Uses the er_macro_array[] to find the macro definition.
   
   Allocates the result string and stores the pointer to it in **output.

   returns 0 on success, non-0 on failure.
*/

int
ER_process_split(int argc, char **argv, char **output)
{
  char *pattern, *ch;
  GString *result = g_string_new("");

  dieif( argc == 0 ); /* may not be called without the macro name */

  /* find macro, error if not found */
  if( (pattern = g_hash_table_lookup(er_macro_hash, argv[0])) == NULL ) {
    return -1;
  }

  /* copy the macro definition by portions, substituting the $([0-9]) 
     entries with arguments. Error if not enough arguments.
  */
  do {

    if( (ch = strstr( pattern, "$(" )) == NULL ) {
      /* no more entries. copy the rest */
      g_string_append ( result, pattern );
      break;
    }
    else {
      /* pass the string between here and ch */
      while( pattern != ch ) {
	g_string_append_c ( result, *pattern );
	pattern++;
      }
      /* check the next 3 characters exist, break the look if not */
      if( *(ch+2) == '\0' ||  *(ch+3) == '\0') {
	break;
      }

      /* look for the digit and ")", pass the $( through if not present */
      if( ! isdigit(*(ch+2)) || *(ch+3) != ')' ) {
	/* not need to do anything to make it pass through */
	;
      }
      else {
	/* substitute the $(?) with the appropriate argument.
	   error if not enough arguments or $(0) is used.*/
	int a = *(ch+2) - '0';
	
	if( argc < a || a==0) {
	  return -1;
	}
	g_string_append( result, argv[a]);
	/* advance the pattern pointer */
	pattern += strlen("$(1)");
      }
    }
  } while(1);

  /* copy the pointer, free the orig structure, keep the text */

  *output = (result->str); 
  
  g_string_free( result, FALSE );

  return 0;
}


/* wrapper around the above that splits the string into argv 
   and calls the ER_parse. 

   sets the errbuf to the ER_parse_spec result

   returns 0 on success, non-0 on failure.
*/
int
ER_macro_spec(char *input, char **errbuf)
{
  char **argv = g_strsplit(input, " ", 0);
  int argc = 0, ret;
  char *fullspec;

  while( argv[argc] != NULL ) {
    argc++;
  }
  

  if( ER_process_split(argc, argv, &fullspec) != 0 ) {
    /* macro unknown. That's OK, just parse that text now */

    fullspec = strdup(input);
  }

  ret = ER_parse_spec(fullspec, errbuf);

  free(fullspec);
  g_strfreev(argv);
  return ret;
  
}


void
ER_make_macro(char *name, char *def)
{
  char *cp_name = wr_string(name);
  char *cp_def  = wr_string(def);

  void *oldkey, *oldval;
  
  /* cleanup on redefinition */
  if( g_hash_table_lookup_extended(er_macro_hash, name, 
				   &oldkey, &oldval) == TRUE ) {
    g_hash_table_remove(er_macro_hash, name);
    wr_free(oldkey);
    wr_free(oldval);
  }
  
  g_hash_table_insert(er_macro_hash, cp_name, cp_def);
}


/* predefine some macros */
void
ER_macro_predef(void)
{
  /* create the hash with hashing and equality testing functions
     specific for strings 
  */
  er_macro_hash = g_hash_table_new(g_str_hash, g_str_equal);
  
#define DBUPDLOG_FORMAT "  FORMAT SEVCHAR|FACSYMB|TEXTLONG|DATETIME|PIDFULL|PROGNAME|MNEMONIC  "
#define RIPLOG_FORMAT   "  FORMAT SEVCHAR|FACSYMB|TEXTLONG|DATETIME|PIDFULL|PROGNAME|THR_ID|MNEMONIC  "

  /* catch-all for dbupdate */
  ER_make_macro("DBUPERR", "CREATE dbuperr {"
		DBUPDLOG_FORMAT "NAME $(1) DATE}"
		" ( FAC MM|UP SEV W- )");

  /* catch-all for rip */
  ER_make_macro("ALLRIPERR", "CREATE allriperr { " 
		RIPLOG_FORMAT "NAME $(1) DATE}" 
		" (FAC ALL SEV W- )");

  /* selected: errors in ripupdate */
  ER_make_macro("RIPUPERR", "CREATE ripuperr {" 
		RIPLOG_FORMAT "NAME $(1) DATE}" 
		" (FAC UD SEV W- )");

  /* querylog: logs all rip queries */
  ER_make_macro("QRYLOG", "CREATE qrylog {" 
		RIPLOG_FORMAT "NAME $(1) DATE}" 
		" (FAC PW ASP PW_I_QRYLOG SEV I )");

  /* audit: any security related messages from RIP */
  ER_make_macro("RIPAUDIT", "CREATE ripaudit {"
		RIPLOG_FORMAT "NAME $(1) DATE}"
		"( FAC PW ASP PW_I_PASSUN SEV i )" 
		" (  FAC AC ASP AC_I_PERMBAN SEV I )");

  /* ripupdlog: logs all update transactions */
  ER_make_macro("RIPUPDLOG", "CREATE ripupdlog_$(2) {" 
		RIPLOG_FORMAT "NAME $(1)_$(2) DATE}" 
		" ( FAC UD ASP 0xffffffff SEV I THR self)");

  /* ripmirlog */
  ER_make_macro("RIPMIRLOG", "CREATE ripmirlog {"  
		RIPLOG_FORMAT "NAME $(1) DATE }"
		"( FAC PM ASP 0xffffffff SEV I )");

  /* server log: all administration by SV (startup, shutdown, etc) and errors */
  ER_make_macro("RIPSVRLOG", "CREATE ripsvrlog {" 
		RIPLOG_FORMAT "NAME $(1) DATE}" 
		" ( FAC SV ASP 0xffffffff SEV I-F )");											      
  /* dbase log: all errors of SQ */
  ER_make_macro("SQLOG", " CREATE sqlog {" 
		RIPLOG_FORMAT "NAME $(1) DATE}" 
		" ( FAC SQ SEV W- )");
  
}

static
void er_macro_list_hook (void* key, void * value, void *condat)
{
  SK_cd_printf(condat, "%s: %s\n", (char *) key, (char *) value);
}
     
void 
ER_macro_list(sk_conn_st *condat)
{
  g_hash_table_foreach(er_macro_hash, er_macro_list_hook, condat );
}

/* override macros with the definitions from the config file */
void 
ER_proc_ca_macro(void)
{
  char *alldef = ca_get_er_macro ;
  char *this_line = alldef;
  char *defname, *defbody, *end_line;
  
  /* alldef is a copy of the configured value. so we can modify it
     if it helps us to do it line by line */
  
  /* ER_MACRO may not be present in the configuration, in which case 
     ca_get_er_macro returns NULL */
  
  if( alldef != NULL ) {
    
    while( *this_line != '\0' ) {
      /* separate the line */
      end_line = strchr(this_line, '\n');
      *end_line = '\0';
      
      /* advance to non-whitespace */
      while( isspace(*this_line) ) {
	this_line++;
      }
      
      /* find the name and body of the definition */
      defname = strsep(&this_line, " \t");
      defbody = this_line;
      
      /* fire */
      dieif( defname == NULL || defbody == NULL );
      ER_make_macro( defname, defbody );
      
      this_line = end_line + 1;
    }
    
    free(alldef);
  }
}


/* process the error definitions from the config file */
void 
ER_proc_ca_err(void)
{
  char *alldef = ca_get_er_def ;
  char *this_line = alldef;
  char *defname, *defbody, *end_line;
  char *erret = NULL;
  int res;
 
    /* alldef is a copy of the configured value. so we can modify it
       if it helps us to do it line by line */

  /* ER_DEF may not be present in the configuration, in which case 
     ca_get_er_def returns NULL */
  if( alldef != NULL ) {
    
    while( *this_line != '\0' ) {
      /* separate the line */
      end_line = strchr(this_line, '\n');
      *end_line = '\0';
      
      /* fire */      
      if( (res = ER_macro_spec(this_line, &erret)) != 0 ) {
	fputs(erret, stderr);
	die;
      }
      
      free(erret); 
      
      this_line = end_line + 1;
    }
    
    free(alldef);
  }
}
