/*
 * Copyright (C) 1992, Board of Trustees of the University of Illinois.
 *
 * Permission is granted to copy and distribute source with out fee.
 * Commercialization of this product requires prior licensing
 * from the National Center for Supercomputing Applications of the
 * University of Illinois.  Commercialization includes the integration of this 
 * code in part or whole into a product for resale.  Free distribution of 
 * unmodified source and use of NCSA software is not considered 
 * commercialization.
 *
 */


#include <stdio.h>
#include <Xm/TextP.h>


#define CARET_WIDTH 9
#define CARET_HEIGHT 5


extern void TextExposed();
extern void AddBuff();
extern void FlushMinMax();


extern Boolean EricBuff;



typedef struct {
   XmTextWidget tw;
} TextGCDataRec, *TextGCData;

static XContext _XmTextGCContext = NULL;


static XmTextPosition ULbuf = -1;
static XmTextPosition LRbuf = -1;
static int buf_x, buf_y, buf_width, buf_height;



static void GetRect(tw, rect)
XmTextWidget tw;
XRectangle *rect;
{
  Dimension margin_width = tw->text.margin_width +
                           tw->primitive.shadow_thickness +
                           tw->primitive.highlight_thickness;
  Dimension margin_height = tw->text.margin_height +
                           tw->primitive.shadow_thickness +
                           tw->primitive.highlight_thickness;

  if (margin_width < tw->core.width)
     rect->x = margin_width;
  else
     rect->x = tw->core.width;

  if (margin_height < tw->core.height)
     rect->y = margin_height;
  else
     rect->y = tw->core.height;

  if ((2 * margin_width) < tw->core.width)
     rect->width = (int) tw->core.width - (2 * margin_width);
  else
     rect->width = 0;

  if ((2 * margin_height) < tw->core.height)
     rect->height = (int) tw->core.height - (2 * margin_height);
  else
     rect->height = 0;
}


static TextGCData GetTextGCData(w)
Widget w;
{
   TextGCData gc_data;

   if (_XmTextGCContext == NULL)
      _XmTextGCContext = XUniqueContext();

   if (XFindContext(XtDisplay(w), (Window) 0, _XmTextGCContext, 
		    (caddr_t*)&gc_data)) {
       gc_data = (TextGCData)XtCalloc(sizeof(TextGCDataRec), 1);

       XSaveContext(XtDisplay(w), (Window) 0, _XmTextGCContext, (char *)gc_data);
       gc_data->tw = (XmTextWidget) w;
   }

   if (gc_data->tw == NULL) gc_data->tw = (XmTextWidget) w;

   return gc_data;

}


static int FindWidth(data, x, block, from, to)
OutputData data;
int x;                          /* Starting position (needed for tabs) */
XmTextBlock block;
int from;                       /* How many positions in to start measuring */
int to;                         /* How many positions in to stop measuring */
{
    XFontStruct *font = data->font;
    char *ptr;
    unsigned char c;
    int i, result;

    if (block->format == FMT16BIT) {
        int direction, ascent, descent;
        XCharStruct overall;
        (void) XTextExtents16(font, (XChar2b *)(block->ptr + from * 2),
                              to - from, &direction, &ascent, &descent,
                              &overall);
        return overall.width;
    }
    result = 0;

    for (i=from, ptr = block->ptr + from; i<to ; i++, ptr++) {
        c = (unsigned char) *ptr;
        if (c == '\t')
            result += (data->tabwidth -
                       ((x + result - data->leftmargin) % data->tabwidth));
        /* %%% Do something for non-printing? */
        else {
            if (font->per_char &&
                (c >= font->min_char_or_byte2 && c <= font->max_char_or_byte2))
                result += font->per_char[c - font->min_char_or_byte2].width;
            else
        result += font->min_bounds.width;
        }
    }
    return result;
}


static void XmSetNormGC (tw, gc)
XmTextWidget tw;
GC gc;
{
    unsigned long valueMask = (GCForeground | GCBackground | GCFunction |
                               GCGraphicsExposures);
    XGCValues values;

    values.foreground = tw->primitive.foreground;
    values.background = tw->core.background_pixel;
    values.function = GXcopy;
    values.graphics_exposures = (Bool) TRUE;

    XChangeGC(XtDisplay(tw), gc, valueMask, &values);
}

static void XmSetInvGC (tw, gc)
XmTextWidget tw;
GC gc;
{
    unsigned long valueMask = (GCForeground | GCBackground | GCFunction |
                               GCGraphicsExposures);
    XGCValues values;

    values.foreground = tw->core.background_pixel;
    values.background = tw->primitive.foreground;
    values.function = GXcopy;
    values.graphics_exposures = (Bool) TRUE;

    XChangeGC(XtDisplay(tw), gc, valueMask, &values);
}


static void XmSetFullGC(tw, gc)
XmTextWidget tw;
GC gc;
{
    OutputData data = tw->text.output->data;
    XRectangle ClipRect;

    ClipRect.width = tw->core.width - (2 *(tw->primitive.highlight_thickness +
                                             tw->primitive.shadow_thickness));
    ClipRect.height = tw->core.height - (2 *(tw->primitive.highlight_thickness +
                                             tw->primitive.shadow_thickness));
    ClipRect.x = tw->primitive.highlight_thickness +
                 tw->primitive.shadow_thickness;
    ClipRect.y = tw->primitive.highlight_thickness +
                 tw->primitive.shadow_thickness;

    XSetClipRectangles(XtDisplay(tw), gc, 0, 0, 
                        &ClipRect, 1, Unsorted);

   if (!data->has_rect) {
     OutputData old_data;
     TextGCData gc_data = GetTextGCData((Widget)tw);
     old_data = gc_data->tw->text.output->data; 
     old_data->has_rect = False;
     gc_data->tw = tw;
     data->has_rect = True;
   }
}


static void XmSetMarginGC (tw, gc)
XmTextWidget tw;
GC gc;
{
  OutputData data = tw->text.output->data;
  XRectangle ClipRect;

  GetRect(tw, &ClipRect);
  XSetClipRectangles(XtDisplay(tw), gc, 0, 0, &ClipRect, 1,
                     Unsorted);
 /*
  * Make sure the cached GC has the clipping rectangle
  * set to the current widget.
  */
   if (!data->has_rect) {
     OutputData old_data;
     TextGCData gc_data = GetTextGCData((Widget)tw);
     old_data = gc_data->tw->text.output->data; 
     old_data->has_rect = False;
     gc_data->tw = tw;
     data->has_rect = True;
   }
}


static void MyDraw(widget, line, start, end, highlight)
XmTextWidget widget;
LineNum line;
XmTextPosition start, end;
XmHighlightMode highlight;
{
    OutputData data = widget->text.output->data;
    XFontStruct *font = data->font;
    XmTextPosition linestart, nextlinestart;
    LineTableExtra extra;
    XmTextBlockRec block;
    int x, y, length, newx, i;
    int text_border;
    int rightedge = ((int)widget->text.inner_widget->core.width)+ data->hoffset;
    int MyX, MyY, MyW, MyH;

    int width, height;
    Boolean cleartoend, cleartobottom;

    if (!XtIsRealized((Widget) widget)) return;
    _XmTextLineInfo(widget, line+1, &nextlinestart, &extra);
    _XmTextLineInfo(widget, line, &linestart, &extra);

    if (!data->has_rect) _XmTextAdjustGC(widget);

    if (linestart == PASTENDPOS) {
        start = end = nextlinestart = PASTENDPOS;
        cleartoend = cleartobottom = TRUE;
    } else if (nextlinestart == PASTENDPOS) {
        nextlinestart = (*widget->text.source->Scan)(widget->text.source, 0,
                                        XmSELECT_ALL, XmsdRight, 1, FALSE);
        cleartoend = cleartobottom = (end >= nextlinestart);
        if (start >= nextlinestart) highlight = XmHIGHLIGHT_NORMAL;
    } else {
        cleartobottom = FALSE;
        cleartoend = (end >= nextlinestart);
        if (cleartoend && (!extra || !extra->wrappedbychar))
            end = (*widget->text.source->Scan)(widget->text.source,
                         nextlinestart, XmSELECT_POSITION, XmsdLeft, 1, TRUE);
    }
    y = data->topmargin + line * data->lineheight + font->ascent;
    x = data->leftmargin;
    while (linestart < start && x <= rightedge) {
        linestart = (*widget->text.source->ReadSource)(widget->text.source,
                                                      linestart, start, &block);
        x += FindWidth(data, x, &block, 0, block.length);
    }
MyX = x - data->hoffset;
MyY = y - font->ascent;

    while (start < end && x <= rightedge) {
        start = (*widget->text.source->ReadSource)(widget->text.source, start,
                                                                 end, &block);
        while (block.length > 0) {
            if (block.format == FMT16BIT) length = block.length / 2;
            else {
                while (block.ptr[0] == '\t') {
                    newx = x;
                    while (block.length > 0 &&
			   newx - data->hoffset < data->leftmargin) {
                        width = FindWidth(data, newx, &block, 0, 1);
                        newx += width;
    
                        if (newx - data->hoffset < data->leftmargin) {
                           block.length--;
                           block.ptr++;
                           x = newx;
                        }
                    }
                    if (block.length <= 0 || block.ptr[0] != '\t') break;

                    width = FindWidth(data, x, &block, 0, 1);
   
                    if (highlight == XmHIGHLIGHT_SELECTED)
		       XmSetNormGC(widget, data->gc);
                    else
                       XmSetInvGC(widget, data->gc);
                    XmSetFullGC(widget, data->gc);
                    XFillRectangle(XtDisplay(widget),
                                   XtWindow(widget->text.inner_widget),
                                   data->gc, x - data->hoffset,
				   y - font->ascent,
                                   width, data->lineheight);
                    XmSetMarginGC(widget, data->gc);
                    if (highlight == XmHIGHLIGHT_SECONDARY_SELECTED) {
                        if (highlight == XmHIGHLIGHT_SELECTED)
			    XmSetInvGC(widget, data->gc);
                        else
                            XmSetNormGC(widget, data->gc);
                        XDrawLine(XtDisplay(widget),
                                  XtWindow(widget->text.inner_widget),
                                  data->gc, x - data->hoffset, y,
				  ((x - data->hoffset) + width) - 1, y);
                    }
                    x += width;
                    block.ptr++;
                    if (--block.length <= 0) break;
                }
                for ( length = 0; length < block.length; length++ ) {
                    if (block.ptr[length] == '\t') break;
                }
            }
            if (length <= 0) break;
            if (block.format == FMT16BIT) {
                if (highlight == XmHIGHLIGHT_SELECTED)
		   XmSetInvGC(widget, data->gc);
                else
		   XmSetNormGC(widget, data->gc);
                XDrawImageString16(XtDisplay(widget),
                                  XtWindow(widget->text.inner_widget), data->gc,                                  x, y, (XChar2b *) block.ptr, length);
                x += FindWidth(data, x, &block, 0, length);
                block.length -= length * 2;
                block.ptr += length * 2;
            } else {
                newx = x;
                while (length > 0 && newx - data->hoffset < data->leftmargin) {
                    newx += FindWidth(data, newx, &block, 0, 1);

                    if (newx - data->hoffset < data->leftmargin) {
                       length--;
                       block.length--;
                       block.ptr++;
                       x = newx;
                    }
                }
                if (length == 0) continue;
                    newx = x + FindWidth(data, x, &block, 0, length);
                if (newx > rightedge) {
                    int lastx;
                    newx = x;
                    for (i=0 ; i<length && newx <= rightedge ; i++) {
                        lastx = newx;
                        newx += FindWidth(data, newx, &block, i, i+1);
                    }
                    length = i;
                    newx = lastx;
                    start = end; /* Force a break out of the outer loop. */
                    block.length = length; /* ... and out of the inner loop. */
                }
                if (highlight == XmHIGHLIGHT_SELECTED)
		   XmSetInvGC(widget, data->gc);
                else
		   XmSetNormGC(widget, data->gc);
                XDrawImageString(XtDisplay(widget),
                                 XtWindow(widget->text.inner_widget), data->gc,
                                 x - data->hoffset, y, block.ptr, length);
                if (highlight == XmHIGHLIGHT_SECONDARY_SELECTED)
                    XDrawLine(XtDisplay(widget),
                              XtWindow(widget->text.inner_widget), data->gc,
                              x - data->hoffset, y,
			      (newx - data->hoffset) - 1, y);
                x = newx;
                block.length -= length;
                block.ptr += length;
            }
        }
    }
MyW = x - MyX + 1;
MyH = data->lineheight;

    /* clear left margin */
    text_border = widget->primitive.shadow_thickness +
                  widget->primitive.highlight_thickness;
    if (data->leftmargin - text_border > 0 && y + font->descent > 0)
           XClearArea(XtDisplay(widget), XtWindow(widget->text.inner_widget),
                      text_border, text_border, data->leftmargin - text_border,
                      y + font->descent - text_border, FALSE);

    if (cleartoend) {
        x -= data->hoffset;
        if (x > ((int)widget->text.inner_widget->core.width)- data->rightmargin)
            x = ((int)widget->text.inner_widget->core.width)- data->rightmargin;
        if (x < data->leftmargin)
            x = data->leftmargin;
        width = ((int)widget->text.inner_widget->core.width) - x -
                (widget->primitive.shadow_thickness +
                 widget->primitive.highlight_thickness);
        if (width > 0 && data->lineheight > 0) {
          if (highlight == XmHIGHLIGHT_SELECTED) XmSetNormGC(widget, data->gc);
          else XmSetInvGC(widget, data->gc);
          XmSetFullGC(widget, data->gc);
          XFillRectangle(XtDisplay(widget), XtWindow(widget->text.inner_widget),
                         data->gc, x, y - font->ascent, width,
                         data->lineheight);
          XmSetMarginGC(widget, data->gc);
if (((x + width) - MyX) > MyW) MyW = (x + width) - MyX;
        }
    }
    if (cleartobottom) {
        x =  widget->primitive.shadow_thickness +
             widget->primitive.highlight_thickness;
        width = widget->text.inner_widget->core.width -
                2 * (widget->primitive.shadow_thickness +
                widget->primitive.highlight_thickness);
        height = widget->text.inner_widget->core.height -
                 ((y + font->descent) +
                 widget->primitive.shadow_thickness +
                 widget->primitive.highlight_thickness);
        if (width > 0 && height > 0)
	{
            XClearArea(XtDisplay(widget), XtWindow(widget->text.inner_widget),
                       x, y + font->descent, width, height, FALSE);
MyH = MyH + height;
if (((x + width) - MyX) > MyW) MyW = (x + width) - MyX;
	}
    }
/*
fprintf(stderr, "MyDraw(%d, %d, %d, %d)\n", MyX, MyY, MyW, MyH);
*/
TextExposed(widget->primitive.user_data, MyX, MyY, MyW, MyH);
}


/*
static Boolean MyMoveLines(widget, fromline, toline, destline)
XmTextWidget widget;
LineNum fromline, toline, destline;
{
    OutputData data = widget->text.output->data;
    if (!XtIsRealized((Widget) widget)) return FALSE;
    if (!data->has_rect) _XmTextAdjustGC(widget);
    XmSetNormGC(widget, data->gc);
    XmSetFullGC(widget, data->gc);
    XCopyArea(XtDisplay(widget), XtWindow(widget->text.inner_widget),
              XtWindow(widget->text.inner_widget), data->gc,
              widget->primitive.shadow_thickness +
              widget->primitive.highlight_thickness,
              (Position) data->lineheight * fromline + data->topmargin,
              widget->text.inner_widget->core.width -
              2 * (widget->primitive.shadow_thickness +
              widget->primitive.highlight_thickness),
              data->lineheight * (toline - fromline + 1),
              widget->primitive.shadow_thickness +
              widget->primitive.highlight_thickness,
              (Position) data->lineheight * destline + data->topmargin);
    XmSetMarginGC(widget, data->gc);
    data->exposevscroll++;
    return TRUE;
}
*/


static XmTextPosition XYToPos(widget, x, y)
XmTextWidget widget;
Position x, y;
{
    OutputData data = widget->text.output->data;
    LineTableExtra extra;
    int i, width, lastwidth, length;
    LineNum line;
    XmTextPosition start, end, laststart;
    XmTextBlockRec block;
    XmTextPosition position;
    int delta = 0;

    x += data->hoffset;
    y -= data->topmargin;
   /* take care of negative y case */
    if (y < 0) {
       delta = ((int)(y + 1)/ (int) data->lineheight) - 1;
       y = 0;
    }
    line = y / data->lineheight;
    if (line > _XmTextNumLines(widget)) line = _XmTextNumLines(widget);
    _XmTextLineInfo(widget, line, &start, &extra);
    if (start == PASTENDPOS)
        return (*widget->text.source->Scan)(widget->text.source, 0, XmSELECT_ALL,
                                      XmsdRight, 1, FALSE);
    _XmTextLineInfo(widget, line+1, &end, &extra);
    end = (*widget->text.source->Scan)(widget->text.source, end, XmSELECT_POSITION,
                               XmsdLeft, 1, TRUE);
    width = lastwidth = data->leftmargin;
    if (start >= end) return start;
    do {
        laststart = start;
        start = (*widget->text.source->ReadSource)(widget->text.source, start,
						   end, &block);
        length = block.format == FMT16BIT ? block.length / 2 : block.length;
        for (i=0 ; i<length && width < x; i++) {
            lastwidth = width;
            width += FindWidth(data, width, &block, i, i+1);
        }
    } while (width < x && start < end);
    if (abs(lastwidth - x) < abs(width - x)) i--;
   /* if original y was negative, we need to find new laststart */
    if (delta && laststart > 0)
       laststart = _XmTextFindScroll(widget, laststart, delta);
    return (*widget->text.source->Scan)(widget->text.source, laststart,
				        XmSELECT_POSITION, (i < 0) ?
					XmsdLeft : XmsdRight, abs(i), TRUE);
}


static void _XmTextDrawShadow(tw)
XmTextWidget tw;
{
   if (XtIsRealized(tw)) {
      if (tw->primitive.shadow_thickness > 0)
         _XmDrawShadow (XtDisplay (tw), XtWindow (tw),
                        tw->primitive.bottom_shadow_GC,
                        tw->primitive.top_shadow_GC,
                        tw->primitive.shadow_thickness,
                        tw->primitive.highlight_thickness,
                        tw->primitive.highlight_thickness,
                        tw->core.width - 2 * tw->primitive.highlight_thickness,
                        tw->core.height-2 * tw->primitive.highlight_thickness);


      if (tw -> primitive.highlighted)
         _XmHighlightBorder ((Widget)tw);
      else if (_XmDifferentBackground ((Widget)tw, XtParent (tw)))
         _XmUnhighlightBorder ((Widget)tw);
   }
}


static void RedrawRegion(widget, x, y, width, height)
XmTextWidget widget;
int x, y, width, height;
{
    OutputData data = widget->text.output->data;
    int i;
    XmTextPosition first, last;
    for (i = y ; i < y + height + data->lineheight ; i += data->lineheight) {
       first = XYToPos(widget, x, i);
       last = XYToPos(widget, x + width, i);
       first = (*widget->text.source->Scan)(widget->text.source, first, XmSELECT_POSITION,
                             XmsdLeft, 1, TRUE);
       last = (*widget->text.source->Scan)(widget->text.source, last, XmSELECT_POSITION,
                              XmsdRight, 1, TRUE);
/*
fprintf(stderr, "MarkRedraw(%d, %d)\n", first, last);
*/
       _XmTextMarkRedraw(widget, first, last);
    }
} 



/*
 * Clear the rectangle containing the cursor.
 */

static void ClearCursor(widget)
XmTextWidget widget;
{
    OutputData data = widget->text.output->data;
    Position x, y;
    XFontStruct *font = data->font;

    if (!XtIsRealized((Widget) widget)) return;

    if (!data->has_rect) _XmTextAdjustGC(widget);

    x = data->insertx - (data->cursorwidth >> 1) - 1;
    y = data->inserty + font->descent - data->cursorheight;

/*EJB*/
    {
	XmSetInvGC(widget, data->gc);
	XmSetFullGC(widget, data->gc);
	XFillRectangle(XtDisplay(widget),
		      XtWindow(widget->text.inner_widget), data->gc,
		      x, y, data->cursorwidth, data->cursorheight);
	XmSetMarginGC(widget, data->gc);
    }
/*EJB
    XClearArea(XtDisplay(widget), XtWindow(widget->text.inner_widget),
               x, y, data->cursorwidth, data->cursorheight, FALSE);
EJB*/
    RedrawRegion(widget, x, y, data->cursorwidth, data->cursorheight);
}



static void OutputExpose(widget, event)
XmTextWidget widget;
XEvent *event;
{
    XExposeEvent *xe = (XExposeEvent *) event;
    OutputData data = widget->text.output->data;
    Boolean erased_cursor = False;
    int old_number_lines = data->number_lines;
    Boolean changed_visible = False;
    Arg args[1];

    if (event->xany.type != Expose)
        return;
    
    if (data->dest_visible) {
       _XmTextDestinationVisible((Widget) widget, False);
       changed_visible = True;
    }

    if (XtSensitive(widget) && data->hasfocus)
          _XmTextChangeBlinkBehavior(widget, False);

    if (widget->text.input->data->has_destination) {
/*EJB*/
    {
	XmSetInvGC(widget, data->gc);
	XmSetFullGC(widget, data->gc);
	XFillRectangle(XtDisplay(widget),
			XtWindow(widget->text.inner_widget), data->gc,
			xe->x - CARET_WIDTH, xe->y - CARET_HEIGHT,
			xe->width + (2 * CARET_WIDTH),
			xe->height + (2 * CARET_HEIGHT));
	XmSetMarginGC(widget, data->gc);
    }
/*EJB
       XClearArea(XtDisplay(widget), XtWindow(widget->text.inner_widget),
                  xe->x - CARET_WIDTH, xe->y - CARET_HEIGHT,
                  xe->width + (2 * CARET_WIDTH),
                  xe->height + (2 * CARET_HEIGHT), FALSE);
EJB*/
       data->dest_on = False;
       erased_cursor = True;
    }

    data->number_lines = widget->text.inner_widget->core.height -
                     data->topmargin - data->bottommargin;
    if (data->number_lines < (int) data->lineheight) data->number_lines = 1;
    else data->number_lines /= data->lineheight;

    if (data->vbar && old_number_lines != data->number_lines)
    {
        if (data->number_lines > 1)
           XtSetArg(args[0], XmNpageIncrement, data->number_lines - 1);
        else
           XtSetArg(args[0], XmNpageIncrement, 1);
        XtSetValues(data->vbar, args, 1);
    }

    if (!data->handlingexposures) {
        _XmTextDisableRedisplay(widget, FALSE);
        data->handlingexposures = TRUE;
        ClearCursor(widget);    /* %%% needs reviewing */
        data->putback = NULL;
    }
    if (data->exposehscroll != 0) {
        xe->x = 0;
        xe->width = widget->core.width;
    }
    if (data->exposevscroll != 0) {
        xe->y = 0;
        xe->height = widget->core.height;
    }
    if (xe->x == 0 && xe->y == 0 && xe->width == widget->core.width &&
          xe->height == widget->core.height)
        _XmTextMarkRedraw(widget, (XmTextPosition)0, 9999999);
    else {
        if (!erased_cursor)
           RedrawRegion(widget, xe->x, xe->y, xe->width, xe->height);
        else
           RedrawRegion(widget, xe->x - CARET_WIDTH, xe->y - CARET_HEIGHT,
			xe->width + (2 * CARET_WIDTH),
			xe->height + (2 * CARET_HEIGHT));
    }

    _XmTextInvalidate(widget, (XmTextPosition) widget->text.top_character,
		      (XmTextPosition) widget->text.cursor_position, NODELTA);

    _XmTextEnableRedisplay(widget);

    if (widget->text.input->data->has_destination)
       if (erased_cursor && (!data->hasfocus ||
           widget->text.dest_position != widget->text.cursor_position))
          _XmTextDrawDestination(widget);

    data->handlingexposures = FALSE;

    _XmTextDrawShadow(widget);

    if (changed_visible)
       _XmTextDestinationVisible((Widget) widget, True);
    if (XtSensitive(widget) && data->hasfocus)
       _XmTextChangeBlinkBehavior(widget, True);
}



void
MyOutputCreate(widget, args, num_args)
	XmTextWidget widget;
	ArgList args;
	Cardinal num_args;
{
	_XmTextOutputCreate(widget, args, num_args);
	widget->text.output->Draw = MyDraw;
/*
	widget->text.output->expose = OutputExpose;
	widget->text.output->MoveLines = MyMoveLines;
*/
}


void
TextRedraw(w, first, last)
	XmTextWidget w;
	XmTextPosition first, last;
{
       _XmTextMarkRedraw(w, first, last);
}


static void
TextRegion(widget, x, y, width, height)
	XmTextWidget widget;
	int x, y, width, height;
{
	OutputData data = widget->text.output->data;
	int i;
	XmTextPosition first, last;
	XmTextPosition min, max;

	for (i = y ; i < y + height + data->lineheight ; i += data->lineheight)
	{
		if (i > (y + height))
		{
			first = XYToPos(widget, x, (y + height));
			last = XYToPos(widget, (x + width), (y + height));
		}
		else
		{
			first = XYToPos(widget, x, i);
			last = XYToPos(widget, (x + width), i);
		}
		first = (*widget->text.source->Scan)(widget->text.source,
			first, XmSELECT_POSITION, XmsdLeft, 1, TRUE);
		last = (*widget->text.source->Scan)(widget->text.source,
			last, XmSELECT_POSITION, XmsdRight, 1, TRUE);
		AddBuff(widget, first, last);
	}
	min = XYToPos(widget, 0, y);
	max = XYToPos(widget, 0, y + height + data->lineheight);
	FlushMinMax(widget, min, max);
} 


void
TextClearFlush(widget)
	XmTextWidget widget;
{
/*
fprintf(stderr, "TextClearFlush(%d, %d)\n", ULbuf, LRbuf);
*/
if (EricBuff)
{
	TextRegion(widget, buf_x, buf_y, buf_width, buf_height);
	FlushMinMax(widget, 0, 0);
}
else
{
	RedrawRegion(widget, buf_x, buf_y, buf_width, buf_height);
}
	ULbuf = -1;
	LRbuf = -1;
}


void
TextClear(widget, x, y, width, height)
	XmTextWidget widget;
	int x, y, width, height;
{
	OutputData data = widget->text.output->data;
	XmTextPosition UL, LR;

	UL = XYToPos(widget, x, y);
	LR = XYToPos(widget, (x + width), (y + height + data->lineheight - 1));

	if ((ULbuf == -1)&&(LRbuf == -1))
	{
		ULbuf = UL;
		LRbuf = LR;
		buf_x = x;
		buf_y = y;
		buf_width = width;
		buf_height = height;
	}
	else if ((UL != ULbuf)||(LR != LRbuf))
	{
/*
fprintf(stderr, "TextClear(%d, %d)\n", ULbuf, LRbuf);
*/
if (EricBuff)
{
		TextRegion(widget, buf_x, buf_y, buf_width, buf_height);
}
else
{
		RedrawRegion(widget, buf_x, buf_y, buf_width, buf_height);
}
		ULbuf = UL;
		LRbuf = LR;
		buf_x = x;
		buf_y = y;
		buf_width = width;
		buf_height = height;
	}
}


void
GetMargins(widget, top, bottom, left, right, border)
	XmTextWidget widget;
	int *top, *bottom, *left, *right, *border;
{
	OutputData data = widget->text.output->data;

	*top = data->topmargin;
	*bottom = data->bottommargin;
	*left = data->leftmargin;
	*right = data->rightmargin;
	*border = widget->primitive.shadow_thickness +
		widget->primitive.highlight_thickness;

}


int
CheckLines(widget)
	XmTextWidget widget;
{
	return(widget->text.total_lines);
}

