#include <stdio.h>
#include <string.h>
#include "wutils.h"
#include "wdefs.h"
#include "wwrap.h"


/*  
 *  All `lhs' and `rhs' pointers are assumed to point to malloc()ed storage,
 *  since wFreeList() will try to free them.
 */    


static WItem *wAppendItem_ proto_((WItem *lst, Val fmt, const char *lhs, const char *rhs, const char *caller));


/*  
 *
 *
 *                             Internal routines.
 *
 *
 */    

/*  
 *  Append an item to the end of a list.
 *  
 *  `lst' should point to the beginning of the list.
 *  
 *  `fmt' determines how `lhs' and `rhs' are to be interpreted.  The range of
 *  values it may take on depend on the type of list to which it is being
 *  appended (i.e. search, command, reply or constraint).
 *  
 *  Return a pointer to the item appended if successful, NULL otherwise.
 */    
static WItem *wAppendItem_(lst, fmt, lhs, rhs, caller)
  WItem *lst;
  Val fmt;
  const char *lhs;
  const char *rhs;
  const char *caller;
{
  WItem *l;
  WItem *w = (WItem *)0;
  
  ptr_check(lst, "lst", "wAppendItem_", (WItem *)0);
  ptr_check(lhs, "lhs", "wAppendItem_", (WItem *)0);
  ptr_check(rhs, "rhs", "wAppendItem_", (WItem *)0);
  ptr_check(caller, "caller", "wAppendItem_", (WItem *)0);

  if (lst->type == T_SYSTEM)
  {
    d2fprintf(stderr, "wAppendItem_ (for %s): you may not specify multiple commands.\n", caller);
  }
  else
  {
    /*  
     *  bug: Should do consistency checking. (I.e. whether `fmt' matches list
     *  type).
     */      
    for (l = lst; l->next; l = l->next);
    if ( ! (w = wNewWItem()))
    {
      d2fprintf(stderr, "wAppendItem_ (for %s): wNewWItem() failed.\n", caller);
    }
    else
    {
      w->type = lst->type;
      w->format = fmt;
      w->lhs = strdup(lhs);
      w->rhs = strdup(rhs);
      if (w->lhs && w->rhs)
      {
        l->next = w;
      }
      else
      {
        d2fprintf(stderr, "wAppendItem_ (for %s): strdup() failed.\n", caller);
        wFreeList(w);
        w = (WItem *)0;
      }
    }
  }

  return w;
}


/*  
 *
 *
 *                             External routines.
 *
 *
 */    


WItem *wAppendTerm(lst, fmt, lhs, rhs)
  WItem *lst;
  Val fmt;
  const char *lhs;
  const char *rhs;
{
  return wAppendItem_(lst, fmt, lhs, rhs, "wAppendTerm");
}


WItem *wAppendConstraint(lst, lhs, rhs)
  WItem *lst;
  const char *lhs;
  const char *rhs;
{
  ptr_check(lst, "lst", "wAppendConstraint", (WItem *)0);

  if ( ! lst->cons)
  {
    WItem *w = wNewWItem();

    if ( ! w)
    {
      return (WItem *)0;
    }
    else
    {
      w->type = T_CONS;
      w->format = F_HEAD;
      lst->cons = w;
    }
  }

  return wAppendItem_(lst->cons, NO_VAL, lhs, rhs, "wAppendConstraint");
}


WItem *wCreateSearch()
{
  WItem *w = (WItem *)0;
  
  w = wNewWItem();
  if ( ! w)
  {
    d2fprintf(stderr, "wCreateSearch: wNewWItem() failed.\n");
  }
  else
  {
    w->type = T_SEARCH;
    w->format = F_HEAD;
  }
  return w;
}


WItem *wCreateCommand(cmd, arg)
  const char *cmd;
  const char *arg;
{
  WItem *h = (WItem *)0;
  WItem *w;

  ptr_check(cmd, "cmd", "wCreateCommand", (WItem *)0);

  h = wNewWItem();
  if (h)
  {
    h->type = T_SYSTEM;
    h->format = F_HEAD;

    w = wNewWItem();
    if ( ! w)
    {
      goto failed;
    }
    else
    {
      w->type = h->type;
      w->lhs = strdup(cmd);
      if ( ! w->lhs) goto failed;
      if (arg && ! (w->rhs = strdup(arg))) goto failed;
      h->next = w;
    }
  }

  return h;

failed:
  if (h) wFreeList(h);
  return (WItem *)0;
}


#if 0
WItem *wCreateConstraint(WItem *lst)
{
  WItem *w = (WItem *)0;

  ptr_check(lst, "lst", "wCreateConstraint", (WItem *)0);

  if (lst->type != T_SEARCH && lst->type != T_SYSTEM)
  {
    d2fprintf(stderr, "wCreateConstraint: argument type is neither a search nor a command.\n");
  }
  else
  {
    w = wNewWItem();
    if (w)
    {
      w->type = T_CONS;
      w->format = F_HEAD;
    }
  }
  return w;
}
#endif


/*  
 *  If the number of hits is not yet known, pass a negative number.
 */  
WItem *wCreateReply(fmt, hits)
  Val fmt;
  int hits;
{
  WItem *w;

  /*  
   *  Check if `fmt' is a valid value for replies.
   */  

  w = wNewWItem();
  if (w)
  {
    static StrVal rfmts[] = /* bug: things like this should be made globally available */
    {
      { "abridged", R_ABRIDGED },
      { "full",     R_FULL   },
      { "handle",   R_HANDLE },
      { "mime",     R_MIME   },
      { "pointer",  R_POINT  },
      { "summary",  R_SUM    },

      { (const char *)0, NO_VAL }
    };
    const char *f;

    w->type = T_REPLY;
    w->format = F_HEAD;

    f = getStr(rfmts, fmt);
    if ( ! f)
    {
      d2fprintf(stderr, "wCreateReply: unknown reply format `%ld'.\n", (long)fmt);
      wFreeList(w);
      w = (WItem *)0;
    }
    else
    {
      char *s = strdup(f);

      if ( ! s)
      {
        d2fprintf(stderr, "wCreateReply: strdup() failed on format string `%s'.\n", f);
        wFreeList(w);
        w = (WItem *)0;
      }
      else
      {
        char *n;
        char num[32];

        w->lhs = s;
        sprintf(num, "%d", hits);
        if ((n = strdup(num)))
        {
          w->rhs = n;
        }
        else
        {
          d2fprintf(stderr, "wCreateReply: strdup() failed on number `%s'.\n", num);
          wFreeList(w);
          w = (WItem *)0;
        }
      }
    }
  }

  return w;
}


WItem *wFirstConstraint(lst)
  WItem *lst;
{
  ptr_check(lst, "lst", "wFirstConstraint", (WItem *)0);
  
  if (lst->type == T_REPLY || lst->type == T_CONS)
  {
    d2fprintf(stderr, "wFirstConstraint: argument points to wrong list type.\n");
    return (WItem *)0;
  }
  else
  {
    if (lst->cons)
    {
/*      return lst->cons->next; */
        return lst->cons;
    }
    else
    {
      return (WItem *)0;
    }
  }
}


Val wGetFormat(w)
  const WItem *w;
{
  ptr_check(w, "w", "wGetFormat", NO_VAL);

  return w->format;
}


char *wGetLHS(w)
  WItem *w;
{
  ptr_check(w, "w", "wGetLHS", (char *)0);

  return w->lhs;
}


WItem *wSetLHS(lst, lhs)
   WItem *lst;
   char *lhs;
{
  ptr_check(lst, "lst", "wSetLHS", (WItem *)0);

  if(lst -> lhs)
     free(lst -> lhs);

  lst -> lhs = strdup(lhs);

  return lst;
}


char *wGetRHS(w)
  WItem *w;
{
  ptr_check(w, "w", "wGetRHS", (char *)0);

  return w->rhs;
}


WItem *wSetRHS(lst, rhs)
   WItem *lst;
   char *rhs;
{
  ptr_check(lst, "lst", "wSetLHS", (WItem *)0);

  if(lst -> rhs)
     free(lst -> rhs);

  lst -> rhs = strdup(rhs);

  return lst;
}


Val wGetType(w)
  const WItem *w;
{
  ptr_check(w, "w", "wGetType", NO_VAL);

  return w->type;
}


WItem *wNextItem(lst)
  WItem *lst;
{
  ptr_check(lst, "lst", "wNextItem", (WItem *)0);

  return lst->next;
}

WItem *wAppendSystemMessage(lst, msg)
  WItem *lst;
  char *msg;
{
  WItem *w, *l;

  ptr_check(lst, "lst", "AppendSystemMessage", (WItem *)0);
  ptr_check(msg, "msg", "AppendSystemMessage", (WItem *)0);

  if(!( w = wNewWItem())){
    d2fprintf(stderr, "wAppendSystemMessage (for %s): wNewWItem() failed.\n", msg);
    return (WItem *) 0;
  }

  w -> type = T_SYSTEM;
  w -> format = NO_VAL;
  w -> lhs = strdup(msg);
  w -> rhs = strdup("");

  for (l = lst; l->next; l = l->next);

  l -> next = w;

  return w;
}
  
