// IconWell.h
//
// Free software created 1 Feb 1992
// by Paul Burchard <burchard@math.utah.edu>.
//
// This class creates a well for drag-and-drop of icons.  It conforms to the 
// target-action paradigm of Controls and so may be configured easily in IB.
// In communicating with other objects it works via its stringValue, a 
// TAB-separated list of the complete path names of the file(s) represented by 
// the icon.  Double-clicks open the files; either via the IconWell's delegate,
// if present, or via WorkSpace otherwise.  (Actually, incomplete path names
// work, as long as the icon doesn't depend on the absolute path name, and
// except that you can't double-click to open such an icon.)  The delegate also 
// receives notice when an icon is about to be dragged out in case the file
// needs to be prepared in any way.
//
// To avoid circularity, if an action message is received from the target, our 
// action is not sent back to it.
//
// While IconWells may be dropped into an app in IB like any other Control, two 
// things must be present in the NIB file to support their function: (1) the 
// opaque white 48 x 48 image "Blank.tiff"; (2) one IconWellControl object 
// for each Window in which IconWells appear, created with
//	[[IconWellControl alloc] initWindow:theWindow];
// (you don't have to do anything else with it).
//
// The appearance of the IconWell may be set with the setBezeled: and
// setBordered: methods.  The setHoldOnDrag: method determines whether dragging
// out of the well is a "copy" or "move" operation; i.e., whether or not the
// icon should remain in the well after it has been successfully dragged out.
// Defaults: bezeled, no border, hold on drag.  Dragging and dropping can be
// selectively disabled with the setDraggable:droppable: method.
//
// To implement a different file-opening procedure, create an object with an
//	- (int)openFile:(const char *)fullPath ok:(int *)flag;
// method (see Speaker/Listener class documentation for details), and connect
// it up as the IconWell's delegate in IB.  Similarly, to intercept drag-outs
// and prep files as needed, implement a method like this:
//	- (int)prepFile:(const char *)fullPath ok:(int *)flag;
//
//
// NOTES ON IMPLEMENTATION:
//
// The design of this class is a bit convoluted because drag-and-drop is not 
// implemented in a truly robust or object-oriented way in NeXTstep 2.0.  You 
// have to arrange things just so.
// 
// One problem is that WorkSpace handles drag-and-drop messages window by 
// window; there is no view-level interface yet in NeXTstep.  A Listener which 
// has registered to get all the remote msgs for a window sends them to a 
// single delegate.  It is this delegate, then (an IconWellControl object in 
// this scheme), which must delegate these messages to the individual accepting 
// views.
//
// The scheme used here is enabled just by initializing an IconWellControl for
// each Window; all organizing between IconWells and, IconWellControls, and 
// Listeners is then done automatically.  Presumably, the features of the 
// IconWellControl class could be built into the Window class itself in a 
// future version of NeXTstep.  
//
// Another problem is that WorkSpace is tied up while the drop operation is 
// being completed.  This means that WorkSpace services (such as getting the 
// correct icon for a path) are not available for the IconWell methods that 
// handle drag-and-drop messages.  Thus, low-level PostScript calls are needed 
// for some operations.
//
// TO DO:
//
// Use custom ActionCell subclass so background gray can be set.
//

#import <appkit/Control.h>

@interface IconWell : Control
{
    id delegate;
    id iconPath;
    char iconName[512];
    BOOL isDropping;
    BOOL isDragging;
    BOOL isDraggable;
    BOOL isDroppable;
    BOOL isHoldOnDrag;
    NXEvent dragFromEvent;
}

+ initialize;
+ wellListFor:aWindow;

- initFrame:(const NXRect *)frameRect;
- free;
- windowChanged:newWindow;

- setBezeled:(BOOL)flag;
- (BOOL)isBezeled;
- setBordered:(BOOL)flag;
- (BOOL)isBordered;
- setHoldOnDrag:(BOOL)flag;
- (BOOL)isHoldOnDrag;
- setDraggable:(BOOL)dragFlag droppable:(BOOL)dropFlag;
- (BOOL)isDraggable;
- (BOOL)isDroppable;

- (const char *)stringValue;
- setStringValue:(const char *)aString;
- takeStringValueFrom:sender;
- clear:sender;

- (BOOL)acceptsFirstMouse;
- mouseDown:(NXEvent *)theEvent;
- mouseDragged:(NXEvent *)theEvent;
- mouseUp:(NXEvent *)theEvent;
- (int)iconEntered:(int)windowNum at:(double)x :(double)y
    iconWindow:(int)iconWindowNum iconX:(double)iconX iconY:(double)iconY
    iconWidth:(double)iconWidth iconHeight:(double)iconHeight
    pathList:(char *)pathList;
- (int)iconMovedTo:(double)x :(double)y;
- (int)iconExitedAt:(double)x :(double)y;
- (int)iconReleasedAt:(double)x :(double)y ok:(int *)flag;

- (int)openFile:(const char *)fullPath ok:(int *)flag;
- (int)prepFile:(const char *)fullPath ok:(int *)flag;
- setDelegate:anObject;
- delegate;

@end
