/* WIDE AREA INFORMATION SERVER SOFTWARE:
   No guarantees or restrictions.  See the readme file for the full standard
   disclaimer. */

/* Include file for the irhash.c file.
   Implements the building functions in irext.h */

#ifndef IREXT_H
#define IREXT_H


/* An interface for adding new server types into the WAIS system.
 * The idea is to use the parsing and bookkeeping operatios of the serial 
 * indexer, while allowing different invered file and signiture systems 
 * to be added as back ends.
 *
 * - Tracy Shen and Brewster 3/91
 */

/*
 * Tracy changes:
 *  - in function "add_word", add two more parameters, source and date
 *  - add a new function "set_query_parameter"
 * proposed changes by brewster:
 *  replace date_type with time_t: accepted
 *  take out all "unsigned" type modifiers (tracey will concider this)
 *  replace short with long: accepted
 *  replace int with long (we port to 16 bit machines still): accepted
 *  added source to delete_doc_id parameters: accepted
 * Proposed changes by brewster and tracy:
 *  if routines are successful return 0, otherwise an error code: accepted
 * Proposed changes by harry:
 *  make the dictionary value be any size.  This can be done by 
 *   passing in a size arg or by passing in read and write routines.
 *  have a function that says we will not be calling best_hit anymore.
 * proposed changes by brewster:
 *  took out hash_pos from add word.
 *  change source to a database* db.
 *  added finished_best_hit, finished_search_word(db), finished_add_word
 *
 */

#include "cdialect.h"
#include "irfiles.h" /* for database */

#ifdef __cplusplus
/* declare these as C style functions */
extern "C"
	{
#endif /* def __cplusplus */



/* ============================
 * ===  Building Functions  ===
 * ============================*/

long init_add_word _AP ((database *db, long parameter1, long parameter2));

/*
 *  add_word: add this word to the database
 *   return values: 0 if successful, non-0 if error
 *       defined error conditions:
 *
 */
long add_word _AP((
		   char *word,	/* the word to be indexed, this could be a
				   word pair. If NULL there are no more words
				   to be indexed */
		   long char_pos, /* the position of the start of the
				     word */
		   long line_pos, /* this is passed for the best
				     section calculation */
		   long weight,	/* how important the word looks
				   syntactically (such as is it bold)
				   NOT used by signature system */
		   long doc_id,	/* current document, this will never be 0 */
		   time_t date, /* display day of this document, 0 if not known */
		   database* db /* database to insert the document */
		   ));

/*
 *  finished_add_word: states that there are no more words to add
 *   to this database.
 *
 *   return values: 0 if successful, non-0 if error
 *       defined error conditions:
 *
 */

long finished_add_word _AP((database *db));

/* ===============================
 * ===  Maintenance Functions  ===
 * ===============================*/

/*
 *  delete_doc_id : delete a document
 *   return values:  0, successfull
 *                  -1, document not found
 *
 */
long delete_doc_id _AP((long doc_id, database *db));

/* =============================
 * ===  Searching Functions  ===
 * =============================*/


/*
 *  set_query_parameter : set query parameter
 *      set search attributes such as date factor, document source ids,
 *      and maximum number of documents returned in a search ( the last
 *      one is an important performance factor to signature  type system)
 *      The search artributes applies to all comming queries until
 *      they are re-set by next set_query_parameter call.
 *
 *   return values:  none
 *
 */
#define SET_MAX_RETRIEVED_MASK 1
#define SET_DATE_FACTOR_MASK   2
#define SET_SELECT_SOURCE      4

/* enum literals for date_factor */
#define DF_INDEPENDENT          1
#define DF_LATER                2
#define DF_EARLIER              3

typedef struct {
  long max_hit_retrieved;
  /* max number of hits can be returned by
   * the search engine. For a signature
   * type system, the default value is 20
   */
  long date_factor;		/* default is DF_INDEPENDENT */
  long num_db;			/* value of zero indicating select all,
				 * default is selecting all
				 */
  database **db_list;		/* list of databases to be searched */
}  query_parameter_type;

/*
 *  set_query_parameter: set a mode variable for the search engine
 *   return values: 0 if successful, non-0 if error
 *       defined error conditions:
 *
 */

long set_query_parameter _AP ((
			 long mask,
			 query_parameter_type *parameters
			 /* fields in the query parameter structure are only
			  * interpreted when the corresponding mask bit 
			  * is set in the mask argument.
			  */
			 ));


/*
 *  search_word: searches for a word in the index.  it side effects 
 *               internal state so that best_hit will return the correct 
 *               results.
 *   return values: 0 if successful, non-0 if error
 *       defined error conditions:
 *
 */

long search_word 
  _AP ((char *word, /* the word to be searched for */
	long char_pos,		/* the position of the start of the word */
	long line_pos,		/* is this needed? not for signature system */
	long weight,		/* how important the word looks syntactically,
				   such as is it bold */
	long doc_id,		/* current document, seed words is 0,
				   then it increments into the relevant 
				   document */
	long dictionary_value,	/* this is from the disk dictionary,
				   a signature system would use weight,
				   inverted file systems would put
				   position information */
	database *db
	));


/*
 *  finished_search_word: states that there are no more words that will
 *   be searched for before best_hit will be called.
 *
 *   return values: 0 if successful, non-0 if error
 *       defined error conditions:
 *
 */

long finished_search_word _AP((database *db));

/*
 *  best-hit :
 *
 *   return values:  0, successfull
 *                  -1, no more documents to return
 *		    Other values returned to signal future signals.
 *
 */
long best_hit _AP ((long *doc_id, long *score));

/*
 *  finished_best_hit: states that there are no more best_hits will be called
 *   before the next set of search_words or add_words.
 *
 *   return values: 0 if successful, non-0 if error
 *       defined error conditions:
 *
 */

long finished_best_hit _AP((void));


#ifdef __cplusplus
	}
#endif /* def __cplusplus */

#endif /* ndef IREXT_H */
