/*
 * nntpgimme
 *
 * Connects to the specified nntp server, and gets all articles whose ids
 * appear on the standard input.
 *
 * This is useful for collecting articles whose IHAVEs leaked through nntpd
 * to rnews, which rejects them as "corrupt header" articles.
 *
 * Jacob Gore, Northwestern U., May 1988
 *
 * Derived from:
 *	nntpxfer
 *	Brian Kantor, UCSD 1986
 *	(some bug fixes by ambar@athena.mit.edu)
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <net/if.h>
#include <netinet/in.h>

#include <stdio.h>
#include <errno.h>
#include <netdb.h>

#ifndef RNEWS
#define RNEWS	"/usr/notes/lib/rnews"
#endif  RNEWS

char	*strcpy();
char	*strcat();
u_long	inet_addr();

extern int errno;

static int server;		/* stream socket to the nntp server */
static int misart = 0;
static int verbose = 0;

main(argc, argv)
    int argc;
    char *argv[];
{
    char buf[BUFSIZ];
    char artrequest[BUFSIZ];
    char *article;
    int connected = 0;		/* 1 = connected */
    struct hostent *hp;
    struct servent *sp;
    struct sockaddr_in sin;
    char *server_host;

    if (argc < 2 || argc > 3) {
	(void) printf(
	 "Usage: %s [-v] host\n   (Standard input: message-ids, 1 per line)\n",
		      argv[0]);
	exit(1);
    }

    if (argc == 2) {
	server_host = argv[1];
    } else {
	if (strcmp(argv[1], "-v") == 0) {
	    ++verbose;
	    server_host = argv[2];
	} else if (strcmp(argv[2], "-v") == 0) {
	    ++verbose;
	    server_host = argv[2];
	} else {
	    (void) printf(
	 "Usage: %s [-v] host\n   (Standard input: message-ids, 1 per line)\n",
			  argv[0]);
	    exit(1);
	}
    }

    sin.sin_addr.s_addr = inet_addr(server_host);
    if (sin.sin_addr.s_addr != -1) {
	sin.sin_family = AF_INET;
    } else {
	hp = gethostbyname(server_host);
	if (hp == NULL) {
	    (void) printf("%s: unknown host\n", server_host);
	    exit(1);
	}

	sin.sin_family = hp->h_addrtype;
#ifdef	BSD43
	bcopy(hp->h_addr_list[0], (caddr_t)&sin.sin_addr, hp->h_length);
#else	BSD43
	bcopy(hp->h_addr        , (caddr_t)&sin.sin_addr, hp->h_length);
#endif	BSD43
    }
	
    sp = getservbyname("nntp", "tcp");
    if (sp == NULL) {
	perror("nntp/tcp");
	exit(1);
    }

    sin.sin_port = sp->s_port;

    do {
	server = socket(AF_INET, SOCK_STREAM, 0);
	if (server < 0) {
	    perror("nntpgimme: socket");
	    exit(1);
	}

	if (connect(server, (struct sockaddr *)&sin, sizeof (sin)) < 0) {
#ifdef	BSD43
	    if (hp && hp->h_addr_list[1]) {
		hp->h_addr_list++;
		bcopy(hp->h_addr_list[0],(caddr_t)&sin.sin_addr, hp->h_length);
		(void) close(server);
		continue;
	    }
#endif	BSD43
	    perror("nntpgimme: connect");
	    exit(1);
	}
	connected++;
    }
    while (connected == 0);

    if (verbose) {
	(void) printf("connected to nntp server at %s\n", server_host);
	fflush(stdout);
    }
    /*
     * ok, at this point we're connected to the nntp daemon 
     * at the distant host.
     */

    /* get the greeting herald */
    (void) sockread(buf);
    if (verbose) {
	(void) printf("%s\n", buf);
	fflush(stdout);
    }
    if (buf[0] != '2') {	/* uh-oh, something's wrong! */
	(void) printf("protocol error: got '%s'\n", buf);
	(void) close(server);
	exit(1);
    }


    /* first, tell them we're a slave process to get priority */
    sockwrite("SLAVE");
    (void) sockread(buf);
    if (verbose) {
	(void) printf("%s\n", buf);
	fflush(stdout);
    }
    if (buf[0] != '2') {	/* uh-oh, something's wrong! */
	(void) printf("protocol error: got '%s'\n", buf);
	(void) close(server);
	exit(1);
    }

    /* prime the "ARTICLE ..." command */
    strcpy(artrequest, "ARTICLE ");
    article=artrequest+strlen(buf);   /* article points to where
					 message-id belongs */

    /* now, ask for each new article */
    for (;;) {
	if (gets(article) == NULL) {
	    break;
	}
	(void) artfetch(article);
    }

    if (verbose) {
	(void) printf("%d missing articles\n", misart);
	fflush(stdout);
    }

    /* we're all done, so tell them goodbye */
    sockwrite("QUIT");
    (void) sockread(buf);
    if (verbose) {
	(void) printf("%s\n", buf);
	fflush(stdout);
    }
    if (buf[0] != '2') {	/* uh-oh, something's wrong! */
	(void) printf("error: got '%s'\n", buf);
	(void) close(server);
	exit(1);
    }
    (void) close(server);

    exit(0);
}

artfetch(articleid)
    char *articleid;
{
    int lines = 0;
    char buf[BUFSIZ];
    FILE *rnews;
    FILE *popen();

    /* now, ask for the article */
    (void) sprintf(buf,"ARTICLE %s", articleid);
    sockwrite(buf);
    (void) sockread(buf);
    if (verbose) {
	(void) printf("%s\n", buf);
	fflush(stdout);
    }
    if (buf[0] == '4') {	/* missing article, just skipit */
	misart++;
	return(0);
    }

    if (buf[0] != '2') {	/* uh-oh, something's wrong! */
	(void) printf("protocol error: got '%s'\n", buf);
	(void) close(server);
	exit(1);
    }
    if (verbose) {
	(void) printf("command: %s\n", RNEWS);
	fflush(stdout);
    }
    if ( (rnews = popen(RNEWS, "w")) == NULL) {
	perror(RNEWS);
	exit(1);
    }

    /* and here comes the article, terminated with a "." */
    if (verbose) {
	(void) printf("data\n");
	fflush(stdout);
    }
    while (1) {
	(void) sockread(buf);
	if (buf[0] == '.' && buf[1] == '\0') {
	    break;
	}
	lines++;
	if (verbose &&
	    strncmp(buf, "Newsgroups", 9) == 0) {
	    (void)puts(buf);
	    fflush(stdout);
	}
	(void) strcat(buf,"\n");
	(void) fputs(((buf[0] == '.') ? buf + 1 : buf), rnews);
    }
    if (verbose) {
	(void) printf(".\n%d lines\n", lines);
	fflush(stdout);
    }
    (void) fflush(rnews);
    (void) pclose(rnews);
    return(0);
}

int
sockread(buf)
    char *buf;
{
    char c;
    int j = 0;
#ifdef BSD43
    fd_set rf;
#else BSD43
    int rf;
#endif BSD43
    struct timeval tv;
    int r;
    char *p = buf;

    while ( 1 ) {
	tv.tv_sec = 180;	/* 3 minutes */
	tv.tv_usec = 0L;
#ifdef BSD43
	FD_ZERO(&rf);
	FD_SET(server, &rf);
#else BSD43
	rf = 1 << server;
#endif BSD43
	r = select(20, (fd_set *)&rf, (fd_set *)0, (fd_set *)&rf, &tv);

	if (r < 0) {
	    if (errno == EINTR) {
		continue;
	    }
	    perror("getsock select");
	    exit(1);
	}
	if (r == 0) {
	    printf("read timed out.\n");
	    exit(1);
	}

	if (read(server, &c, 1) <= 0) {
	    break;
	}

	/* mask off any chance parity bits */
	*p = c & 0x7f;

	/* look for end of line (== LF) */
	if (c == 0x0a) {
	    if (j > 0 && *(p-1) == 0x0d) {
		*(p-1) = '\0';
	    } else {
		*p = '\0';
	    }
	    return(strlen(buf));
	}
	j++; p++;
    }
    perror("sockread");
    (void) close(server);
    exit(1);
    /* NOTREACHED */
}

sockwrite(buf)
    char *buf;
{
    register int sz;
    char buf2[BUFSIZ];
    if (verbose) {
	(void) printf(">>> %s\n", buf);
	fflush(stdout);
    }
    (void) strcpy(buf2,buf);
    (void) strcat(buf2,"\r\n");
    sz = strlen(buf2);
    if (write(server,buf2,sz) != sz) {
	(void) printf("write error on server socket\n");
	(void) close(server);
	exit(1);
    }
}
