

#include "nc.h"

/*
 * Newsclip compiler symbol table routines.
 *
 * This code handles general symbol manipulation for the user program
 */

 /*
  * Newsclip(TM) Compiler Source Code.
  * Copyright 1989 Looking Glass Software Limited.  All Rights Reserved.
  * Unless otherwise licenced, the only authorized use of this source
  * code is compilation into a binary of the newsclip compiler for the
  * use of licenced Newsclip customers.  Minor source code modifications
  * are allowed before compiling.
  * A short time evaluation of this product is also permitted.  See the file
  * 'Licence' in the library source directory for details.
  */
 /*
  * There are 3 levels of symbol table.  There's the table of special
  * globals, the global symbol table and the local symbol table for
  * each routine.  We do not support symbol tables inside compound
  * statments.
  */

#define MAX_ST_INDEX 2

dbptr sym_stack[MAX_ST_INDEX+1];	/* stack of 3 symbol tables */
int cur_st_index = 0;
dbptr cur_symtab;		/* current symbol table */

dbptr outer_symtab;		/* the table of special symbols */
dbptr global_symtab;		/* the gloal symtab for the user */

struct sym_entry Unknown = {
"Unknown", 0, 0, 0, 0, 0, 0 };


symtab_init()
{
	int i;
	symptr thesym;
	extern struct outsym predefs[];

	outer_symtab = init_db( 40, sizeof( struct sym_entry ) );

	for( i = 0; predefs[i].name; i++ ) {
		thesym = (symptr)add_rec( outer_symtab, predefs[i].name,
						AR_CREATE | AR_NOALLOC );
		thesym->decl_type = predefs[i].odecl_type;
		thesym->type = predefs[i].otype;
		/* do something with flags */
		thesym->sflags = predefs[i].flags;
		thesym->argtypes = predefs[i].atlist;
		}
	/* Link up the arg lists for predefined routines */

	global_symtab = init_db( 80, sizeof( struct sym_entry ) );
	cur_symtab = global_symtab;
	sym_stack[0] = outer_symtab;
	sym_stack[1] = global_symtab;
	cur_st_index = 1;
}

nodep
extern_var( varname, type )
char *varname;		/* name of variable */
dtype type;		/* type of variable */
{
	symptr thesym;

	if( thesym = extern_decl(varname,ST_VAR,type,(listp)0) ) {
		if( thesym->sflags & OSF_CONST ) 
			return NIL;
		 else
			return tree( N_EXTERN, declid(thesym) );
		}
	 else
		return tree( N_EXTERN, NIL );
	
}

/* Do a general external declaration.
 * If the symbol exists at this level, complain about a
 * redeclaration. 
 * If the symbol exists at a higher level, check that things
 * match, and if so, create the symbol at this level.
 */

symptr
extern_decl( name, sytype, type, argtlist )
char *name;			/* name of external symbol */
int sytype;			/* type of symbol */
dtype type;			/* user type */
struct typelist * argtlist;	/* arglist if needed */
{
	symptr thesym;
	extern bool no_externals;	/* forbid undefined externals */

	thesym = (symptr)add_rec( cur_symtab, name, AR_NEWONLY );
	if( thesym ) {
		symptr globname;
		globname = (symptr)get_rec( outer_symtab, name );
		/* should be an option to disable true externals */
		if( globname ) {
			int flags;
			if( globname->decl_type != sytype ||
					globname->type != type ) {
				parerror( "External '%s' is of invalid type.",
							name );
				}
			else if( (sytype == ST_FUNC || sytype == ST_PROC) &&
				!arglists_match(argtlist, globname->argtypes )){
					parerror("Invalid argument list for '%s'",
							name );
				}
			 else {
				handle_outer( globname, TRUE );
				thesym->sflags = globname->sflags;
				}
			}
		 else {
			if( no_externals )
				parerror( "Undefined external references disallowed" );
			thesym->sflags = 0;
			}
		thesym->decl_type = sytype;
		thesym->type = type;
		thesym->argtypes = argtlist;
		return thesym;
		}
	 else {
		parerror( "Symbol '%s' redeclared", name );
		return (symptr)0;
		}
}


nodep
extern_func( funcname, type, arglist, is_external )
char *funcname;		/* name of proc or func */
dtype type;		/* return type or 0 for procedure */
listp arglist;		/* list of arguments */
bool is_external;	/* is this an external or a forward declaration */
{
	symptr funcsym;		/* the symbol created for the function */
	struct typelist *atlist;/* the argument type list */
	int fstype;		/* the type of subroutine */

	fstype = type ? ST_FUNC : ST_PROC;

	atlist = buildargs( arglist );

	if( is_external ) {
		funcsym = extern_decl( funcname, fstype, type, atlist );
		if( !funcsym )
			return NIL;
		}
	 else {
		if( cur_st_index != 1 ) {
			parerror( "Forward declaration of '%s' must be a global declaration", funcname );
			return NIL;
			}
		if( funcsym = declare_local( funcname, fstype, type ) ) {
			funcsym->sflags |= SF_FORWARD;
			funcsym->argtypes = atlist;
			}
		 else
			return NIL;
		}
	if( funcsym->sflags & OSF_CONST )
		return NIL;
	 else
		return tree( N_EXT_FUNC, declid(funcsym), arglist );
}

symptr
declare_local( name, sytype, type )
char *name;		/* symbol name */
int sytype;
dtype type;
{
	symptr thesym;

	thesym = (symptr)add_rec( cur_symtab, name, AR_NEWONLY );
	if( thesym ) {
		thesym->decl_type = sytype;
		thesym->type = type;
		thesym->sflags |= SF_LOCAL;
		}
	 else
		parerror( "Symbol '%s' redeclared", name );
	return thesym;
}

nodep
declare_var( varname, type )
char *varname;		/* name of variable */
dtype type;		/* type of variable */
{
	symptr sym;

	if( sym = declare_local( varname, ST_VAR, type ) )
		return tree( N_DECL_VAR, declid(sym) );
	 else
		return NIL;
}

nodep
gen_declare( name )
char *name;		/* name of user routine */
{
	symptr sym;
	sym = (symptr)add_rec( cur_symtab, name, AR_CREATE );
	/* is nil possible? */
	return sym ? declid(sym) : NIL;
}

nodep
declare_arg( name, type )
char *name;		/* name of the argument */
dtype type;		/* type for the argument */
{
	/* for our purposes, these are just like variables */
	return declare_var( name, type );
}

nodep
declare_lab( name )
char *name;		/* name of the label */
{
	symptr sym;

	sym = (symptr)add_rec( cur_symtab, name, AR_CREATE );
	if( sym->decl_type != 0 && sym->decl_type != ST_LABREF ) {
		parerror( "Label '%s' redeclared", name );
		return declid( &Unknown );
		}
	 else {
		sym->decl_type = ST_LABEL;
		return declid(sym);
		}


}

/* special globals to use while checking a routine */

int in_routine = 0;
int routine_type = 0;

/* General procedure to check and output a subroutine */

routine_decl( rid, rargs, rcode, type, symtype )
nodep rid;		/* identifier for routine */
listp rargs;		/* the argument list */
nodep rcode;		/* the code block for the routine */
dtype type;		/* type to give the routine */
int symtype;		/* type of symbol -- proc or function */
{
	symptr thesym;
	struct typelist *atlist;/* the argument type list */
	extern int got_error;	/* parsing error status */

	if( !rid )
		return;

	thesym = (symptr)kid0(rid);

	atlist = buildargs( rargs );

	/* set up the symbol */

	if( thesym->decl_type == 0 )  {
		thesym->type = type;
		thesym->sflags |= SF_LOCAL;
		thesym->decl_type = symtype;
		thesym->argtypes = atlist;
		}
	 else {
		if( thesym->sflags & SF_FORWARD ) {
			if( thesym->decl_type != symtype ||
					thesym->type != type ||
					!arglists_match( thesym->argtypes,
					atlist ) )
				terror( rargs, "Subroutine declaration does not match forward declaration" );
			/* turn off forward for future */
			thesym->sflags &= ~SF_FORWARD;
			}
		 else {
			parerror( "Symbol '%s' redeclared", thesym->name );
			return;
			}
		}

	/* set special externals */	

	if( got_error < SYNTAX_ERROR ) {
		in_routine = symtype;
		routine_type = type;

		check( rcode );

		outsubr( thesym, rargs, rcode );

		/* clear special externals */

		in_routine = 0;
		routine_type = 0;
		}
	/* free up the code and arglist.  Symbol and type list stay in
	   the symbol table */
	treefree( rcode );
	treefree( rargs );
}


/*
 * Look up a label.  Unusual in that it is possible the label has
 * not been declared yet.  We will create it if so and wait for it
 * to get declared.
 */

nodep
goto_lookup( name )
char *name;
{
	symptr sym;

	sym = (symptr)get_rec( cur_symtab, name );
	if( sym ) {
		if( sym->decl_type == ST_LABEL || sym->decl_type == ST_LABREF )
			return declid(sym);
		 else {
			parerror( "Symbol '%s' is not a label", name );
			return declid( &Unknown );
			}
		}
	 else {
		/* the symbol was not found.  That's not an error yet */
		sym = (symptr)add_rec( cur_symtab, name, AR_CREATE );
		sym->decl_type = ST_LABREF;
		return declid(sym);
		}
}

/* 
 * General symbol lookup.
 */

nodep
symlookup( symname, t1, t2 )
char *symname;
dtype t1,t2;			/* possible types for symbol */
{
	int i;
	symptr sym;

	for( i = cur_st_index; i >= 0; i-- ) {
		sym = (symptr)get_rec( sym_stack[i], symname );
		if( sym ) {
			if( i == 0 && !(sym->sflags & OSF_PREDEF) ) {
				parerror( "Symbol '%s' must be declared external before it can be referenced", symname );
				return declid( &Unknown );
				}
			if( sym->decl_type != t1 && sym->decl_type != t2 )
				parerror( "Incorrect kind of identifier: '%s'",
						symname );
			return declid(sym);
			}
		}

	/* never found it */
	parerror( "Undeclared symbol: '%s'", symname );
	return declid( &Unknown );
}

nodep
declid(sym)
symptr sym;
{
	return tree( N_ID, (nodep)sym );
}

push_table()
{
	if( cur_st_index < MAX_ST_INDEX ) {
		/* create a new table */
		cur_symtab = init_db( 20, sizeof( struct sym_entry ) );
		sym_stack[++cur_st_index] = cur_symtab;
		}
}

pop_table()
{
	/* free the old table */
	free_db( cur_symtab );
	cur_symtab = sym_stack[--cur_st_index];
}

struct typelist *
buildargs( alist )
listp alist;		/* argument declaration list */
{
	dtype tempdt[255];		/* build arglist here */
	struct typelist *ret;		/* final return pointer */
	int anum;
	int i;

	for( anum = 0; alist; anum++,alist = alist->next ) {
		nodep decvar, decid;
		symptr arsym;

		if( decvar = alist->kid ) {
			if( decvar->ntype == N_INT )
				tempdt[anum] = (int)kid0(decvar);
			 else {
				if( (decid = kid0(decvar)) &&
						(arsym = (symptr)kid0(decid)) )
					tempdt[anum] = arsym->type;
				 else
					anum--;	/* nil arg */
				}
			}
		 else
			anum--;		/* nil argument */
		}

	/* allocate room for a real arglist */

	ret = (struct typelist *) checkalloc( sizeof(struct typelist) );
	if( anum )
		ret->args = (dtype *) checkalloc( anum * sizeof(dtype) );
	 else
		ret->args = (dtype *)0;

	ret->argmin = ret->argmax = ret->argcount = anum;
	/* copy over the found arguments */
	for( i = 0; i < anum; i++ )
		ret->args[i] = tempdt[i];

	return ret;
}

/* handle references to predefined symbols */

handle_outer( globname, complain )
symptr globname;		/* global symbol */
int complain;			/* complain about local header refs */
{

	int flags;
	extern bool needs_stat;
	extern bool wants_dist;

	flags = globname->sflags;

	/* if we have already seen this symbol, ignore it */

	if( flags & OSF_REFERENCED )
		return;
	globname->sflags |= OSF_REFERENCED;

	switch( flags & OSF_SPECIAL_MASK ) {
		case SPC_HEADER:
			if( complain )
				insist_global(globname);
			hcreate( globname->name, globname->type );
			break;
		case SPC_STAT:
			needs_stat = TRUE;
			break;
		case SPC_FROM:
			if( complain )
				insist_global(globname);
			makeref( globname->name + 1 );
			makeref( "from" );
			break;
		case SPC_NEWSGROUPS:
			if( complain )
				insist_global(globname);
			makeref( globname->name + 1 );
			/* newsgroups already pre-referenced */
			break;
		case SPC_REF:
			makeref( "references" );
			break;
		case SPC_DIST:
			makeref( "distribution" );
			wants_dist = TRUE;
			break;
		}
}

insist_global(sym)
symptr sym;
{
	if( cur_st_index > 1 ) 
		parerror( "Header variable declaration for '%s' must be global, not local", sym->name );
}

/* Make as though an external reference has been made to a given name */

makeref( name )
char *name;
{
	symptr globname;
	globname = (symptr)get_rec( outer_symtab, name );
	if( globname )
		handle_outer( globname, FALSE );
}
