/*  This module contains the code for the user interface for eep.  */

#include	"eep.h"
#include	<stdio.h>
#include	<string.h>
#include	<curses.h>
#include	<ctype.h>
#include	<signal.h>


extern char     *malloc();
extern char     *getenv();
extern int	qcompare();	/* sort alphabetically */

extern char	buffer[], /* general purpose line buffer */
		tmp[];

extern struct  actif *act[];	/* main data structure */
extern struct  actif *topact[]; /* top level names only */

extern int	i_active,	/* index into arrays */
		c_active,	/* number of lines in act array */
		t_active,	/* number of lines in act array */
		eepoint,	/* pointer switch */
		noshell,	/* noshell switch */
		really_active;

extern FILE	*fnewsrc;	/* .newsrc.new */

/* The levels array contains the Head pointers which control the
linked lists of newsgroups grouped by their "depth" in the
hierarchy, and sorted alphabetically.  E.g. alt, sci, soc are 
level 0, alt.pagan, sci.physics are level 1 and so on. */

extern struct actif *levels[];  /* keep track of levels with this array */
extern struct actif *aptr;	/* temporary pointer */
extern int	i_levels;	/* index into array */

/* Global to this module */

int	scrnpos = 0,	/* position of highlight on screen */
	current = 0,	/* matching highlight in data structure */
	top = 0,	/* data element on top of screen */
	eeplines = 0,	/* number of lines on screen */
	eepcolumns = 0;	/* columns on screen */

WINDOW	*over;	/* used for on-screen help window etc. */

/* pad() -- this function will take a pointer to a string and a 
numeric argument, and will pad the string with spaces to reach
the desired length.  If the string is too long, it will be 
truncated to fit.  The function will return a pointer to a
buffer declared in the global scope that will contain the padded
string (or original if it's long enough already). 
If str is a NULL pointer, we'll return spaces only. */

char	*pad(str,len)
char	*str;
int	len;
{
	buffer[0] = '\0';
	if (len >= BUFSIZE) return(buffer); /* safety */
	/* In case we are passed a NULL pointer... */
	if (str == (char *) NULL) {
		buffer[0] = '\0';
		while (strlen(buffer) < len) strcat(buffer," ");
		return(buffer);
	}
	if (strlen(str) >= len) {
		strncpy(buffer,str,len);
		return(buffer);
	}
	strcpy(buffer,str);
	while (strlen(buffer) < len)
		strcat(buffer," ");
	return(buffer);
}


/* newsmain()  -- this provides the mainline user interface. */

newsmain()
{

char	*ptr;
char	search[BUFSIZE];	/* search string buffer */
int	ch,		/* input character */
	found = FALSE,	/* flag to say we've found something */
	quit = FALSE,	/* flag used when ready to quit */
	index;		/* used when searching */

	/* Work out how many lines and columns we have.
	Use the environment variables first -- if they don't
	exist, we'll try terminfo.  Yes, I know terminfo
	should do this, but don't bet on it for all versions.
	If that doesn't work, default to the ones in eep.h. */
	if ((ptr = getenv("LINES")) != NULL)
		eeplines = atoi(ptr);
	if ((ptr = getenv("COLUMNS")) != NULL)
		eepcolumns = atoi(ptr);
	if (eeplines <= 0)
		eeplines = tigetnum("lines");
	if (eepcolumns <= 0)
		eepcolumns = tigetnum("columns");
	if (eeplines <= 0)
		eeplines = EEPLINES;
	if (eepcolumns <= 0)
		eepcolumns = EEPCOLUMNS;

	initscr();	/* set up for curses */
	raw();
	noecho();
	nonl();
	keypad(stdscr,TRUE);
	idlok(stdscr,TRUE);
	showlist(0);
	while (quit == FALSE) {
	ch = getch();
	switch(ch) {
		case 'q':	/* quit */
		case 'Q':
		case '\033':
			move(eeplines-1,0);
			deleteln();
			addstr("Do you want to quit without saving your changes? [n]: ");
			refresh();
			if (tolower(getch()) == (int) 'y') quit = TRUE;
			deleteln();
			move(scrnpos,0);
			refresh();
			break;
		case '?':	/* on-line help */
		case 'h':
		case 'H':
		case KEY_F(1):
			over = newwin(18,60,3,5);
			if (!eepoint) wstandout(over);
			box(over,'\0','\0');
			if (!eepoint) wstandend(over);
			wmove(over,2,5);
			waddstr(over,"eep v1.1: experimental .newsrc edit program");
			wmove(over,3,5);
			waddstr(over,"by P. Gillingwater, paul@actrix.gen.nz");
			wmove(over,5,5);
			waddstr(over,"/   Search          n   Search for next");
			wmove(over,6,5);
			waddstr(over,"^D  Page down       ^U  Page up");
			wmove(over,7,5);
			waddstr(over,"j   Line down       k   Line up");
			wmove(over,8,5);
			waddstr(over,"p   Pointer change  c   Catch up");
			wmove(over,9,5);
			waddstr(over,"x   Save and exit   q   Exit without saving");
			wmove(over,10,5);
			waddstr(over,"s   Subscribe       u   Unsubscribe");
			wmove(over,11,5);
			waddstr(over,"^L  Redraw screen   ?   This Help");
			wmove(over,12,5);
			waddstr(over,"a   Alphabetise");
			wmove(over,13,5);
			waddstr(over,"ENTER or RETURN to show details");
			wmove(over,15,5);
			waddstr(over,"Press SPACE BAR to exit from help");
			wrefresh(over);
			ch = wgetch(over);
			delwin(over);
			touchwin(stdscr);
			refresh();
			break;
		case '!':	/* shell out */
			if (noshell != 0) break;
			/* code here later for shelling out */
			break;
		case 'r':	/* redraw */
		case 'R':	/* redraw */
		case '\014':	/* form feed ^L */
			touchwin(stdscr);
			clearok(stdscr,TRUE);
			refresh();
			break;
		case '\r':
		case '\n':	/* new line to select a group */
			if ((aptr = act[current]) == NULL) break;
			over = newwin(18,65,4,5);
			if (!eepoint) wstandout(over);
			box(over,'\0','\0');
			if (!eepoint) wstandend(over);
			wmove(over,2,5);
			wprintw(over,"Name: %.50s",pad(aptr->name,50));
			wmove(over,4,5);
			wprintw(over,"Desc: %.50s",pad(aptr->desc,50));
			wmove(over,6,5);
			wprintw(over,"This news group is ");
			switch(aptr->flag) {
			case 'y':
				wprintw(over,"Active.");
				break;
			case 'n':
				wprintw(over,"Not Active.");
				break;
			case 'm':
				wprintw(over,"Moderated.");
				break;
			default:
				wprintw(over,"Unknown.");
			}
			wmove(over,7,5);
			wprintw(over,"This news group is ");
			switch(aptr->status) {
			case ':':
				wprintw(over,"Subscribed to.");
				break;
			case '!':
				wprintw(over,"Not Subscribed to.");
				break;
			default:
				wprintw(over,"Not valid.");
				break;
			}
			if (aptr->hi > 0) {
				wmove(over,9,5);
				wprintw(over,"Active high message is %d",
					aptr->hi);
				wmove(over,11,5);
				wprintw(over,"Messages already read by you include:");
				wmove(over,12,10);
				wprintw(over,"%s",pad(aptr->hilo,30));
			}
			wmove(over,16,5);
			waddstr(over,"Press SPACE BAR to continue");
			ch = wgetch(over);
			delwin(over);
			touchwin(stdscr);
			refresh();
			break;
		case '/':	/* search */
			move(eeplines - 1,0);
			deleteln();
			addch('/');
			refresh();
			getbuf(search);
			if (strlen(search) == 0) {
				move(eeplines - 1,0);
				deleteln();
				refresh();
				break;
			}
			found = FALSE;
			/* make it lower case for searching */
			strcpy(search,shift_lower(search));
			index = current;	/* track progress */
			while (++index != current) {
				if (index >= c_active) index = 0;
				aptr = act[index];
				if (in_string(shift_lower(aptr->name),search) ||
				    in_string(shift_lower(aptr->desc),search)) {
					/* found match */
					found = TRUE;
					current = index;
					showlist(current);
					break;
				}
			}
			break;
		case 'n':	/* look for next occurence */
		case 'N':
			if (!found) 	break;
			index = current;	/* track progress */
			while (++index != current) {
				if (index >= c_active) index = 0;
				aptr = act[index];
				if (in_string(shift_lower(aptr->name),search) ||
				    in_string(shift_lower(aptr->desc),search)) {
					/* found match */
					found = TRUE;
					current = index;
					showlist(current);
					break;
				}
			}
			break;
		case 'a':	/* change sorting order */
		case 'A':
/* Later we will extend this command to have a sub-menu that can
choose other sorting methods, e.g. sort by original order found in
.newsrc */
			move(eeplines-1,0);
			deleteln();
			addstr("Do you wish to sort alphabetically? [n]: ");
			refresh();
			if (tolower(getch()) != (int) 'y') {
				deleteln();
				refresh();
				break;
			}
			deleteln();
			move(scrnpos,0);
			qsort( act, (unsigned) c_active, 
				(unsigned) sizeof(act[0]), qcompare);
			showlist(current);
			break;
		case '+':	/* subscribe to this newsgroup */
		case 's':
		case 'S':
			aptr = act[current];
			/* check if group is valid */
			if ((aptr->flag == 'y') ||
			    (aptr->flag == 'm')) {
				aptr->status = ':';
				move(scrnpos,2);
				standout();
				addch('+');
				standend();
				move(scrnpos,0);
			} else {
				move(eeplines - 1,0);
				printw("Not a newsgroup -- cannot subscribe");
				refresh();
				sleep(2);
				deleteln();
				move(scrnpos,0);
			}
			refresh();
			goto	scroll_down;
			break;
		case '-':	/* unsubscribe to this newsgroup */
		case 'u':
		case 'U':
			aptr = act[current];
			/* check if group is valid */
			if ((aptr->flag == 'y') ||
			    (aptr->flag == 'm')) {
				aptr->status = '!';
				move(scrnpos,2);
				standout();
				addch(' ');
				standend();
				move(scrnpos,0);
			} else {
				move(eeplines - 1,0);
				printw("Not a newsgroup -- cannot unsubscribe");
				refresh();
				sleep(2);
				deleteln();
				move(scrnpos,0);
			}
			refresh();
			goto	scroll_down;
			break;
		case ' ':
		case 'j':	/* for vi users */
		case 'J':	/* for vi users */
		case '\016':	/* for emacs users */
		case KEY_DOWN:
scroll_down:
			aptr = act[current];
			move(scrnpos,0);
			if (!eepoint) standend();
			switch(aptr->status) {
			case ':':	printw("  +"); /* subscribed */
					break;
			case '!':	printw("   "); /* unsubscribed */
					break;
			default:	printw("   "); /* mystery! */
			}
			if (!eepoint) {
				printw("%.28s ", pad(aptr->name,28));
				printw("%.46s ", pad(aptr->desc,46));
			}
			refresh();
			if (++current == c_active) current = 0;
			if (++scrnpos == eeplines - 1) { /* scroll up */
				move(0,0);
				deleteln();
				move(eeplines - 2,0);
				insertln();
				refresh();
				scrnpos = eeplines - 2;
				if (++top == c_active) top = 0;
			};
			/* Now paint our new position */
			move(scrnpos,0);
			aptr = act[current];
			if (!eepoint) standout();
			switch(aptr->status) {
			case ':':	printw("->+"); /* subscribed */
					break;
			case '!':	printw("-> "); /* unsubscribed */
					break;
			default:	printw("-> "); /* mystery! */
			}
			printw("%.28s ", pad(aptr->name,28));
			printw("%.46s ", pad(aptr->desc,46));
			if (!eepoint) standend();
			move(scrnpos,0);
			refresh();
			break;
		case '\010':	/* backspace */
		case 'k':	/* for vi users */
		case 'K':	/* for vi users */
		case '\020':	/* for emacs users */
		case KEY_UP:
			move(scrnpos,0);
			if (!eepoint) standend();
			aptr = act[current];
			switch(aptr->status) {
			case ':':	printw("  +"); /* subscribed */
					break;
			case '!':	printw("   "); /* unsubscribed */
					break;
			default:	printw("   "); /* mystery! */
			}
			if (!eepoint) {
				printw("%.28s ", pad(aptr->name,28));
				printw("%.46s ", pad(aptr->desc,46));
			}
			move(scrnpos,0);
			refresh();
			if (--current == -1) current = c_active - 1;
			if (--scrnpos == -1) {
				move(eeplines - 2,0);
				deleteln();
				move(0,0);
				insertln();
				refresh();
				move(0,0);
				scrnpos = 0;
				if (--top == 0) top = c_active - 1;
			}; /* cause scroll */
			move(scrnpos,0);
			aptr = act[current];
			if (!eepoint) standout();
			switch(aptr->status) {
			case ':':	printw("->+"); /* subscribed */
					break;
			case '!':	printw("-> "); /* unsubscribed */
					break;
			default:	printw("-> "); /* mystery! */
			}
			printw("%.28s ", pad(aptr->name,28));
			printw("%.46s ", pad(aptr->desc,46));
			if (!eepoint) standend();
			move(scrnpos,0);
			refresh();
			break;
		case KEY_NPAGE: /* next page */
		case '\004':	/* ^D */
		case '\006':	/* ^F */
			if ((current += EEPPAGE) >= c_active)
				current -= c_active;
			showlist(current);
			break;
		case KEY_PPAGE: /* previous page */
		case '\025':	/* ^U */
		case '\002':	/* ^B */
			if ((current -= EEPPAGE) < 0)
				current += c_active;
			showlist(current);
			break;
		case 'p':	/* change type of pointer */
		case 'P':
			if (eepoint) eepoint = FALSE;
			else eepoint = TRUE;
			showlist(current);
			break;
#ifdef NEVER
		case 'm':	/* move groups around */
		case 'M':
			movem();
			break;
#endif /* NEVER */
		case 'c':	/* catch up */
		case 'C':
			aptr = act[current];
			/* only if it's a real news group */
			if (aptr->hi <= 0)	goto scroll_down;
			if ((aptr->flag == 'y') ||
			    (aptr->flag == 'm') ||
			    (aptr->flag == 'n')) {
				move(eeplines-1,0);
				deleteln();
				addstr("Catch up this news group? [n]: ");
				refresh();
				if (tolower(getch()) == 'y') {
					sprintf(buffer,"1-%d",
						aptr->hi);
					aptr->hilo = malloc(strlen(buffer)+1);
					if (aptr->hilo == NULL) {
						fprintf(stderr,
						"Malloc error.");
						exit(2);
					}
					strcpy(aptr->hilo,buffer);
					deleteln();
					move(eeplines-1,0);
					printw("High message now %d",
						aptr->hi);
				} else deleteln();
				move(scrnpos,0);
				refresh();
			}
			goto scroll_down;
			break;
		case 'x':	/* write the local .newsrc and exit */
		case 'X':
			move(eeplines-1,0);
			deleteln();
			if (fnewsrc == (FILE *) NULL) {
				addstr("Can't open .newsrc -- write failed.");
				refresh();
				break;
			}
			addstr("Do you want to save and exit? [n]: ");
			refresh();
			if (tolower(getch()) != (int) 'y') {
				deleteln();
				move(scrnpos,0);
				refresh();
				break;
			}
			deleteln();
			index = 0;
			while (index < c_active) {
				aptr = act[index];
				if ((aptr->status != ':') ||
				    (aptr->flag == '\0') ||
				    (aptr->name == NULL)) {
					index++;
					continue;
				}
				fprintf(fnewsrc,"%s: ",
					aptr->name);
				if (aptr->hilo != NULL) {
					fprintf(fnewsrc,"%s",
					aptr->hilo);
				}
				fprintf(fnewsrc,"\n");
				index++;
				move(eeplines-1,0);
				printw("%d",index);
				refresh();
			}
			fclose(fnewsrc);
			fnewsrc = (FILE *) NULL;

			if ((ptr = getenv("HOME")) != NULL) {
				sprintf(tmp, "%s/.newsrc", ptr);
				sprintf(buffer, "%s/.newsrc.old", ptr);
			} else {
				sprintf(tmp, ".newsrc");   
				sprintf(buffer, ".newsrc.old");   
			}
			unlink(buffer);	/* don't care about errors */
			if (link(tmp,buffer) < 0) {
				fprintf(stderr,"[1] .newsrc not replaced.");
				break;
			}
			if (unlink(tmp) < 0) {
				fprintf(stderr,"[2] .newsrc not replaced.");
				break;
			}
			if ((ptr = getenv("HOME")) != NULL)
				sprintf(buffer, "%s/.newsrc.new", ptr);
			else
				sprintf(buffer, ".newsrc.new");

			if (link(buffer,tmp) < 0) {
				fprintf(stderr,"[3] .newsrc not replaced.");
				break;
			}
			if (unlink(buffer) < 0) {
				fprintf(stderr,"[4] .newsrc not replaced.");
				break;
			}
			noraw();
			endwin();	/* terminate */
			return(0);
			break;
		}
	}
	move(eeplines - 1,0);
	clrtoeol();
	refresh();
	noraw();
	endwin();	/* terminate */
}

/* This routine will show the window into the current list
of newsgroup lines.  Current is the one highlighted.  The
current line will always be at the top of the screen.  (This
simplifies much coding, although it's ugly in some ways.) */
 
showlist(current)
int	current;
{
int	index, counter;

	counter = 0;
	index = current;
	while ((index < c_active) && (counter < eeplines - 1)) {
		move(counter,0);
		if (current == index) {
			if (!eepoint) standout();
			printw("->");
			scrnpos = counter;
		} else	printw("  ");
		aptr = act[index];
		/* Show whether subscribed or not */
		switch(aptr->status) {
		case ':':	printw("+"); /* subscribed */
				break;
		case '!':	printw(" "); /* unsubscribed */
				break;
		default:	printw(" "); /* mystery! */
		}
			
		printw("%.28s ", pad(aptr->name,28));
		printw("%.46s ", pad(aptr->desc,46));
		move(counter,0);
		if (current == index) 
			if (!eepoint) standend();
		index++;
		counter++;
		if (index == c_active)	index = 0;
	}
	move(eeplines - 1,0);
	printw("There are %d active newsgroups",really_active);
	move(eeplines - 1,60);
	printw("Press ? for Help");
	move(scrnpos,0);
	refresh();
}


/* getbuf() -- read input into the buffer */

getbuf(buf)
char	*buf;
{
int	count = 0,
	c;	/* character */

	while (count < BUFSIZE - 1) { /* -1 to allow for nul */
		c = getch();
		switch(c) {
		case KEY_BREAK:
		case '\033':
		case '\003':
		case KEY_EXIT:
			deleteln();
			refresh();
			buf[0] = '\0';
			return;
			break;	/* just in case */
		case KEY_BACKSPACE:
		case '\010':
		case '\177':
		case KEY_LEFT:
			if (count == 0)	continue;
			addstr("\010 \010");
			refresh();
			count--;
			continue;
		case KEY_ENTER:
		case '\n':
		case '\r':
			buf[count] = '\0';
			return;
			break;
		}
		/* yes, this is very ASCII and doesn't take into
		account non-English alphabets.  This is because
		I'm not planning to fix curses in free software. */
		if ((c < '\020') || /* some other control code */
		    (c > '\177'))
			continue;	/* ignore it */
		buf[count] = c;
		echochar(buf[count]);
		count++;
	}
	buf[count] = '\0';	/* nul terminate the string */
}
