/****************************************************************
 *								*
 *	LSV-Input						*
 *								*
 *	Read in and unfold continued header lines, no limit	*
 *	on line length or amount of header lines.		*
 *								*
 *	Pick up reply address.					*
#
#  Copyright 1990-1993   Matti.Aarnio @ FUNET.FI
#  This software is free under similar rules as BSD copyrights.
#  (Definitely this is NOT Public Domain.  "Just" FREELY AVAILABLE.
#   Don't clain you did this..)
#  You can use this, but you shall not held us liable for anything.
#  You must not use our name in marketing, in case you decide to
#  use this.  We do appreciate bug-reports  -> mailserver-owner@nic.funet.fi
#  for improving this piece of software.
 *								*
 ****************************************************************/


#include <stdio.h>
#include <sys/types.h>
/* #include <sys/param.h> */
#include <sys/wait.h>
#include <signal.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/resource.h>
#ifndef BSD
#include "malloc.h"
extern char *malloc();
#endif

#include <ctype.h>
#include <netdb.h>
extern int h_errno;

#include "config.h"
#include "input.h"

#ifdef SYSV
# define COMMAND_PROGRAM	"input-program"
#else
# define COMMAND_PROGRAM	"/usr/local/lib/mailserver/input-program"
#endif
#define REQUESTARCHIVEDIR "requests" /* Under TRICKLE_HOME */
/* Invoked with args:
	0: `MAIL', 1: Fromaddr, 2: Replyaddr, 3: MessageId, 4: Subject 5: UFrom */

struct headerstruct {
  char *From;
  char *To;
  char *UFrom;
  char *Reply;
  char *Subject;
  char *MessageId;
  char *ReplyMsgId;
};


char myhostname[80];


extern char *malloc(), *strchr(), *realloc();
extern int errno;
extern char *sys_errlist[];

/* **************************************************************
 *
 *  int mail_header_input( FILE *fd, char ***headers )
 *
 *  Read in header lines from designated input stream, and stop at
 *  EOF or after last header line (white space/ '\n'-only).
 *  Unfold continued header lines, zap '\n' from end of line.
 *
 * *************************************************************** */

int
mail_header_input(fd,headers)
FILE *fd;
char ***headers;
{
#define INLINELEN 512
	char *InpLine = malloc(INLINELEN);
	char *inpline = NULL;
	int   inplinesize = INLINELEN;
	int   inplinelen  = 0;
	char *prevline = malloc(INLINELEN);
	int   prevlinesize = INLINELEN;
	int   prevlinelen  = 0;

	char **Hdrs = NULL;
	int    HdrsCnt = 0;
	
	char *rcc;

	/* Grab in headers, allocate - out of blue - header lines, etc. */

	*prevline = 0;
	prevlinelen = 0;
	while( !ferror(fd) && !feof(fd) ) {
	  inpline = InpLine;
	  *inpline = 0;
	  inplinelen = 0;
	  if (fgets( inpline+inplinelen,inplinesize-inplinelen,fd ) == NULL)
	    break; /* EOF */
	  if (*inpline == 0 ||
	      *inpline == '\n') break;  /* EOF or last entry of header! */
	  inplinelen += strlen( inpline+inplinelen );
	  rcc = strchr( inpline,'\n' );
	  if (rcc == NULL) {
	    ; /* TODO:  Must extend input, and get in more! */
	  } else
	    *rcc = 0; /* Zap that trailing `\n' */
	  --inplinelen;

	  /* Convert TABs to spaces */
	  rcc = strchr( inpline,'\t' );
	  while (rcc != NULL) {
	    *rcc = ' ';
	    rcc = strchr( rcc,'\t' );
	  }

	  if (*inpline == ' ') {
	    /* Append to prevline... */
	    if (inplinelen+prevlinelen >= prevlinesize) {
	      prevline = realloc( prevline,inplinelen+prevlinelen+8 );
	      prevlinesize = inplinelen+prevlinelen+8;
	    }
	    while (*inpline == ' ' && inplinelen >0) {
	      ++inpline; --inplinelen;
	    }
	    memcpy( prevline+prevlinelen,inpline,inplinelen+1 );
	    prevlinelen += inplinelen;
	  } else {
	    /* A new input line, start a new.. */
	    if (*prevline != 0) {
	      /* Insert into headers... */
	      if (Hdrs == NULL) {
		HdrsCnt = 0;
		Hdrs = (char**) xcalloc(sizeof(char*)*2);
	      }
	      Hdrs[HdrsCnt] = malloc( prevlinelen+1 );
	      memcpy( Hdrs[HdrsCnt],prevline,prevlinelen+1 );
	      ++HdrsCnt;
	      Hdrs = (char**) realloc( Hdrs,(HdrsCnt+2)*sizeof(char*) );
	      Hdrs[HdrsCnt] = NULL;
	    }
	    if (prevlinesize <= inplinelen ) {
	      prevline = realloc( prevline,inplinelen+8 );
	      prevlinesize = inplinelen+8;
	    }
	    while (*inpline == ' ' && inplinelen >0) {
	      ++inpline; --inplinelen;
	    }
	    memcpy( prevline,inpline,inplinelen+1 );
	    prevlinelen = inplinelen;
	  }
	} /* while() */
	if (*prevline) {
	  /* Insert into headers... */
	  if (Hdrs == NULL) {
	    HdrsCnt = 0;
	    Hdrs = (char**) xcalloc(2 * sizeof(char*));
	  }
	  Hdrs[HdrsCnt] = malloc( prevlinelen+1 );
	  memcpy( Hdrs[HdrsCnt],prevline,prevlinelen+1 );
	  ++HdrsCnt;
	  Hdrs = (char **) realloc( Hdrs,(HdrsCnt+2)*sizeof(char*) );
	  Hdrs[HdrsCnt] = NULL;
	}
	  
	*headers = Hdrs;
	free(prevline);
	free(inpline);

	return HdrsCnt;
}


char *FindHdrData( headers,searchtag )
char **headers;
char *searchtag;
{
	char **hdr = headers;
	int taglen = strlen(searchtag);

	while ( *hdr )  {
	  if (strncasecmp( *hdr,searchtag,taglen )==0 &&
	      (*hdr)[taglen] == ' ')
	    return *hdr;
	  ++hdr;
	}
	return NULL;
}


/* **************************************************************** *
 *
 *  char **search_addresses( char **headers );
 *
 *  Returns a set of pointers to lines which have relevant entries.
 *  If entry contents is invalid ... :-(
 *
 * **************************************************************** */

/* responces:
   0: from
   1: reply to
   2: ufrom (unix from)
   3: subject
   4: messageid
*/

struct headerstruct *
search_addresses(headers)
char **headers;
{
	struct headerstruct *R = (struct headerstruct *) malloc(sizeof (struct headerstruct));

	memset(R,0,sizeof(struct headerstruct));

	R->To = FindHdrData(headers,"resent-to:");
	if (R->To == NULL)
	  R->To = FindHdrData(headers,"to:");

	R->From = FindHdrData(headers,"resent-from:");
	if (R->From == NULL)
	  R->From = FindHdrData(headers,"from:");

	R->Reply = FindHdrData(headers,"resent-reply-to:");
	if (R->Reply == NULL)
	  R->Reply= FindHdrData(headers,"reply-to:");

	R->UFrom = FindHdrData(headers,"from"); /* UNIX from */

	R->Subject = FindHdrData(headers,"resent-subject:");
	if (R->Subject == NULL)
	  R->Subject = FindHdrData(headers,"subject:");
	if (R->Subject == NULL)
	  R->Subject = "(no subject given)";

	R->MessageId = FindHdrData(headers,"message-id:");

	if (R->Reply == NULL)
	  R->Reply = R->From;
	return R;
}

/* ****************************************************************
 *
 * tag_skip()  Skip over first non-space word, until encounter first
 *             space/tab, skip over that too to begin of 2nd word.
 *
 * **************************************************************** */

char *
tag_skip(str)
register char *str;
{
	if (!str) return str;
	while (*str && *str != ' ' && *str != '\t') ++str;
	while (*str && (*str == ' ' || *str == '\t')) ++str;
	return str;
}

/* ****************************************************************
 *
 *  process_commands(infile,hdrfile,outfile,Hdr)
 *
 *  Do actual command processing.  Fork a program to do it :-)
 *  stdin   is infile,
 *  stdout  is outfile,
 *  Hdr data are given in as arguments with prefix token `MAIL'
 *
 * **************************************************************** */

void
process_commands(infile,hdrfile,outfile,Hdr)
FILE *infile, *hdrfile, *outfile;
struct headerstruct *Hdr;
{
	int pid;
	int rc;
	int status;
	int i;

	pid = fork();
	if (pid < 0) exit(98);

	/* Infile, and Outfile  and unbuffered! */

	if (pid == 0) { /* CHILD! */
	  dup2(fileno(infile),0);
	  dup2(fileno(outfile),1);
	  dup2(fileno(hdrfile),3);

	  if( fcntl(0,F_GETFL,&i) < 0 )
	    fprintf(outfile,"\nError on STDIN, errno=%d (%s)\n",
		    errno,sys_errlist[errno]);
	  if( fcntl(1,F_GETFL,&i) < 0 )
	    fprintf(outfile,"\nError on STDOUT, errno=%d (%s)\n",
		    errno,sys_errlist[errno]);

	  if (Hdr->MessageId == NULL)
	    Hdr->MessageId = "";
	  if (Hdr->Subject == NULL)
	    Hdr->Subject = "";

	  execl (COMMAND_PROGRAM,
		 "MAIL",Hdr->From,Hdr->Reply,Hdr->MessageId,
		 Hdr->Subject,Hdr->UFrom,Hdr->ReplyMsgId, NULL );
	  _exit(97);
	}
	/* Parent! */
    retry_wait:
	do {
	  rc = wait(&status);
	} while(rc == -1 && errno == EINTR);
	if (WIFEXITED(status)) {
	  if (WEXITSTATUS(status) != 0)
	    fprintf(outfile,
		    "command processing program exited with ERROR status: %d\n",
		    WEXITSTATUS(status));
	} else if (WIFSIGNALED(status))
	  fprintf(outfile,
		  "command processing program terminated ABNORMALLY on signal: %d (%s)\n",
		  WTERMSIG(status),strsignal(WTERMSIG(status)));
#ifdef	BSD
	else if (WIFSTOPPED(status)) {
	  fprintf(outfile,
		  "What -- Child STOPPED on signal %d (%s) What ?  Will issue SIGCONT\n",
		  WSTOPSIG(status),strsignal(WSTOPSIG(status)));
	  kill(rc,SIGCONT);
	  goto retry_wait;
	}
#endif
	else {
	  fprintf(outfile,
		  "What ?  Child process exited without good reason ?  status=0x%X\n",
		  status);
	}
}


/* ****************************************************************
 *
 *  usage()  -- the usual :-)
 *
 * **************************************************************** */

void
usage()
{
	fprintf(stderr,"\
LSVINPUT: Mail based command input parser program.\n\
          Args:  [-S] [-H] [-d debugrecipient] inputfilename\n\
          -S     Output to STDOUT instead of sendmailed to sender.\n\
          -H     Header Debug - print out header processing.\n\
          -d XX  Send a Cc: to debug recipient XX\n\
          Filename may be `-' for stdin.\n\
          This in turn invokes `%s'\n\
with arguments:\n\
   0: `MAIL', 1: Fromaddr, 2: Replyaddr, 3: MessageId, 4: Subject\n",
		COMMAND_PROGRAM);

}


/* ****************************************************************
 *
 *  collect_input()   -- Collect input to current directory into
 *		         file `reqXXXXXXXXX.YYY' where X's are time of
 *			 request to the seconds + millisecs (Y's)
 *
 * **************************************************************** */

static FILE *
collect_input(infile,reqname,chdirrc)
FILE *infile;
char **reqname;
int chdirrc;
{
	FILE *ofile;
	static char filename[280];
	struct stat stats;
	char buf[1024];
	time_t curtim = time(NULL);
	struct tm *tm;
	int len;

	sprintf (filename,"req%8.8lX",(unsigned long)curtim);

	if (chdirrc == 0)
	  ofile = fopen (filename,"w+");
	else {
	  ofile = tmpfile();
	  strcpy(filename,"/TMPFILE/");
	}
	if (!ofile) {
	  /* Catastrophe!  How ?? */
	  *reqname = NULL;
	  return infile;
	}
	if (*filename == 'r' && stat(".",&stats) == 0) {
	  /* Able to get owner of current directory... */
	  chown(filename,stats.st_uid,stats.st_gid);
	  /* Now reqXXX files are owned by directory owner... */
	}

	setbuf (ofile,NULL);

	/* Copy file over */
	while (!feof(infile) && !ferror(infile)) {
	  int rdcnt = fread(buf,1,sizeof buf,infile);
	  if (rdcnt == 0) break;
	  fwrite(buf,1,rdcnt,ofile);
	}

	/* Return ptr to our static data.. */
	*reqname = filename;
	/* .. and append that data with unique (?) timestamp */
	len = strlen(filename);
	tm = gmtime(&curtim);
	if (0 == strftime(filename+len,sizeof(filename)-len,
			  "%%%a-%Y%m%d%H%M%SZ",tm))
	  strcpy(filename+len,":time_unconvertible");

	/* Replace infile with ofile... */
	rewind(ofile);
#ifndef	HOLD_PARENT_PROCESS
	fclose(infile);
#endif
	/* Return info about our file.. */
	return ofile;
}


/* ****************************************************************
 *
 * main()  - main body of LSVINPUT program
 *
 *
 * **************************************************************** */

int
main(argc,argv)
int argc;
char **argv;
{
	char **headers, *str;
	FILE *infile, *hdrfile, *outfile;
	struct headerstruct *H, Hdr;
	char address[512],fullname[512];
	char inpline[1024];
	char MailIdentity[512];
	char debugline[256];	/* Should be enough - For putenv() */
	int  cracked, rc;
#ifdef	BSD
	struct rusage Rusage;
#endif
	char *reqname, *s;
	char *DebugTarget = NULL;
	int StdOutDump = 0;
	int HeaderDebug = 0;

	struct hostent *hostent; /* For finding -- via BIND -- my real host name */
	extern int crackfrom();

	if (argc < 2) {
	  usage();
	  exit(2);
	}

#ifdef SYSV
#if 0
    /* set up malloc debugging */
    mallopt(MALLOC_WARN, 0);
    mallopt(MALLOC_FATAL, 129);
    mallopt(MALLOC_CKCHAIN, 1);
/*    mallopt(MALLOC_ERRFILE, "malloc.err"); */
#endif
#endif
	/* Figure out our domain */
	gethostname( myhostname,sizeof(myhostname) );
	hostent = gethostbyname( myhostname );
	if (hostent == NULL && h_errno == TRY_AGAIN)
	  /* Well, we try again -- once */
	  hostent = gethostbyname( myhostname );
	if (hostent != NULL)
	  strcpy( myhostname, hostent->h_name );

	strcpy(MailIdentity,MAIL_IDENTITY);
	strcpy(debugline,"DEBUG=");
	putenv(debugline);
	putenv("FTPD_LS=-Z");
#ifndef	PATHENV
# define PATHENV "PATH=/bin:/usr/local/bin:/usr/ucb:/etc:/usr/etc:/usr/local/etc"
#endif
	putenv(PATHENV);
	putenv(LOCAL_TZ); /* Set TZ environment */

	++argv; --argc;
	while (*argv) {
	  if (strcmp(*argv,"-d")==0) {
	    ++argv; --argc;
	    if (*argv == NULL) {
	      printf("-d requires parameter!\n");
	      exit(8);
	    }
	    DebugTarget = *argv;
	    ++argv; --argc;
	    sprintf(debugline,"DEBUG=%s",DebugTarget);
	    /* putenv(debugline); -- debugline[] modify implies putenv().. */
	    continue;
	  }
	  if (strcmp(*argv,"-S")==0) {
	    ++argv; --argc;
	    StdOutDump = 1;
	    continue;
	  }
	  if (strcmp(*argv,"-H")==0) {
	    ++argv; --argc;
	    HeaderDebug = 1;
	    continue;
	  }
	  break;
	}

	if (*argv == NULL) {
	  printf("A required source file name (or `-' for stdin) is missing!\n");
	  exit(8);
	}

	if (strcmp(*argv,"-")==0)
	  infile = stdin;
	else
	  infile = fopen( *argv,"r" );
	if (infile == NULL) exit(3);
	setbuf( infile, NULL );
	outfile = tmpfile();
	setbuf( outfile,NULL );
	hdrfile = tmpfile();	/* Trying to make sure this file handle is over 3.. */
	setbuf( hdrfile,NULL );

	chdir(TRICKLE_HOME);
	rc = chdir(REQUESTARCHIVEDIR);

	if (StdOutDump == 0)
	  infile = collect_input(infile,&reqname,rc);

	mail_header_input(infile,&headers);

	if (!headers) exit(98); /* If input is completely void.. */
	H = search_addresses(headers);
	if (!H) exit(99); /* Impossible ? */
	Hdr = *H;

	*address = 0; *fullname = 0;
	cracked = crackfrom( address,fullname,tag_skip(H->To));
	str = malloc( strlen(address)+1); strcpy( str,address );
	Hdr.To = str;

	s = strchr(Hdr.To,'@');
	if (s) {
	  register char *t = strchr(s+1,'@'); /* Another ?? */
	  if (t)
	    *t = 0;
	  t = s+1;
	  while (*t) {
	    if (!isalnum(*t) &&
		(*t != '-') && (*t != '.') && (*t != '_')) {
	      s = NULL;
	      break;
	    }
	    ++t;
	  }
	}

	/* if (s)
	   strcat(MailIdentity,s);
	 */

	if (HeaderDebug) {
	  fprintf(stderr,"Found header entries:\n");
	  fprintf(stderr,"From: `%s'\n",H->From);
	}

	*address = 0; *fullname = 0;
	cracked = crackfrom( address,fullname,tag_skip(H->From));
	str = malloc( strlen(address)+1); strcpy( str,address );
	Hdr.From = str;
	if (HeaderDebug) {
	  fprintf(stderr," -> `%s'\n",address);

	  printf("Reply: `%s'\n",H->Reply);
	}
	*address = 0; *fullname = 0;
	cracked = crackfrom( address,fullname,tag_skip(H->Reply));
/* fprintf(hdrfile,"X-Orig-Reply: `%s' -> `%s'\n",tag_skip(H->Reply),address); */
	str = malloc( strlen(address)+1); strcpy( str,address );
	Hdr.Reply = str;
	if (HeaderDebug) {
	  fprintf(stderr," -> `%s'\n",address);

	  fprintf(stderr,"UFrom: `%s'\n",H->UFrom);
	}
	*address = 0; *fullname = 0;
	cracked = crackfrom( address,fullname,tag_skip(H->UFrom));
	if (HeaderDebug) {
	  fprintf(stderr," -> `%s'\n",address);

	  printf("Subject: `%s'\n",H->Subject);
	}
	Hdr.Subject = tag_skip(H->Subject);

	if (HeaderDebug)
	  fprintf(stderr," -> `%s'\n",Hdr.Subject);

	if (Hdr.Subject != NULL) {
	  if (strncasecmp(Hdr.Subject,"re ",3)==0)
	    Hdr.Subject += 3;
	  if (strncasecmp(Hdr.Subject,"re: ",4)==0)
	    Hdr.Subject += 3;
	}

	if (HeaderDebug)
	  fprintf(stderr,"MessageId: `%s'\n",H->MessageId);

	*address = 0; *fullname = 0;
	cracked = crackfrom( address,fullname,tag_skip(H->MessageId));
	str = malloc( strlen(address)+1); strcpy( str,address );
	Hdr.MessageId = str;

	if (HeaderDebug)
	  fprintf(stderr," -> `%s'\n",str);

	fprintf(hdrfile,"From:       %s\n",MailIdentity );
	if (Hdr.Reply != NULL) {
	  if (DebugTarget != NULL) {
	    fprintf(hdrfile,"Cc:         %s\n",DebugTarget);
	    fprintf(hdrfile,"X-Warning:  NEVER use REPLY on the answers that arrive to you from\n            this server, the reply 'comes' from an error-trap address.\n");
	  }
	  fprintf( hdrfile,"Errors-To:   mailserver-owner\n" );
	  fprintf( hdrfile,"Subject:     Re: %s\n",Hdr.Subject );
	  Hdr.ReplyMsgId = reqname ? reqname : "bogus-reply-no-original-request";
	  fprintf( hdrfile,"Message-Id:  <%s@%s>\n", Hdr.ReplyMsgId,
		  myhostname);
	  fprintf( hdrfile,"In-Reply-To: <%s>\n",
		  Hdr.MessageId );
	  /* To: is filled in by routine behind process_commands! */

	  process_commands( infile,hdrfile,outfile,&Hdr );

#ifdef	BSD
	  /* Print resource usage summary! */
	  fputs ("Resource usages:\n",outfile);
	  rc = getrusage( RUSAGE_SELF,&Rusage );
	  fprintf (outfile,
		   "  Self:     Utime:  %2d.%3.3d,  Stime:  %2d.%3.3d\n",
		   Rusage.ru_utime.tv_sec,Rusage.ru_utime.tv_usec/1000,
		   Rusage.ru_stime.tv_sec,Rusage.ru_stime.tv_usec/1000);
	  rc = getrusage( RUSAGE_CHILDREN,&Rusage );
	  fprintf (outfile,
		   "  Children: Utime:  %2d.%3.3d,  Stime:  %2d.%3.3d\n",
		   Rusage.ru_utime.tv_sec,Rusage.ru_utime.tv_usec/1000,
		   Rusage.ru_stime.tv_sec,Rusage.ru_stime.tv_usec/1000);
#endif
	} else {
	  /* Hmm... Illegal reply address - none found! */
	  fprintf( hdrfile,"To:          %s\n",MailIdentity );
	  fprintf( hdrfile,"Subject:     Re: %s\n",Hdr.Subject );
	  fprintf( hdrfile,"             Illegal reply address - none found!\n");
	  fprintf( hdrfile,"In-Reply-To: <%s>\n",Hdr.MessageId );

	  while(!feof(infile) && !ferror(infile)) {
	    *inpline = 0;
	    if (fgets(inpline,sizeof inpline,infile) == NULL) break;
	    if (*inpline == 0) break;
	    fputs("*>",outfile); fputs(inpline,outfile);
	  }
	}

	fprintf( hdrfile,"\n" ); /* One blank line in between.. */

	fseek( outfile,0,0 ); /* Into beginning... */
	while (!feof(outfile) && !ferror(hdrfile)) {
	  int rdcnt = fread(inpline,1,sizeof inpline,outfile);
	  if (rdcnt == 0) break;
	  fwrite(inpline,1,rdcnt,hdrfile);
	}
	fclose(outfile);

	fseek( hdrfile,0,0 );

	if (!StdOutDump) {
	  /* Pipe it to sendmail... */
	  dup2(fileno(hdrfile),0);
	  execl(_SENDMAIL_PATH,"sendmail","-f",MAIL_IDENTITY,NULL);
	} else {
	  /* Display what we did.. */
	  while (!feof(hdrfile) && !ferror(hdrfile)) {
	    int rdcnt = fread(inpline,1,sizeof inpline,hdrfile);
	    if (rdcnt == 0) break;
	    fwrite(inpline,1,rdcnt,stdout);
	  }
	}	
	return 0;
}
