/************************************************************************
 *									*
 *	MailServiceSerializer						*
 *									*
 *	Takes input from  stdin feed by mailer in program run config:	*
 *		mailservice: "|/.../mailserializer -args args"		*
 *	Stores incoming mail at first into spool file, keeps tab on it	*
 *	and runs them first come first served fashion never hanging for *
 *	long time to wait for a finish of service and thus forcing	*
 *	mailer to wait.							*
 *									*
 *	This is to be run as ROOT.					*
 *									*
 *  By Matti Aarnio <mea@nic.funet.fi>  (C) FUNET			*
 *	Freely distributable, and usable, but please contact me if	*
 *	you want to exploit this commercially.  (That is, to sell it..)	*
 *									*
 ************************************************************************/

/*  Possible portability problems:
    - S_IFMT & S_IFREG vs. S_ISREG of POSIX style.
    - strchr()/strrchr() vs. index()/rindex() of BSD.
    - kill(pid,0) functionality on probing of process existance.
    - rename(2) system call used in one particular place.
 */

#ifndef	__
# ifdef	__STDC__
#  define __(x) x
# else
#  define __(x) ()
# endif
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <malloc.h>
#include <pwd.h>
#include <grp.h>
#include <errno.h>
#include "sysexits.h"	/* Very Berkeley thing, but d.. VERY usefull! */

#ifndef	S_ISREG
# define S_ISREG(x) ((S_IFMT & (x)) == S_IFREG)
#endif


#define DAEMON_UID 1 	/* If set, allows suid-root:ing this, and will
			   let DAEMON (usually uid 1) processes to run this
			   with root priviledges..  For BSD sendmail alias
			   defined usages. */
#define	SENDMAIL	"/usr/lib/sendmail"
#define	HAS_SETEUID	/* If you don't have it, you don't need it.
			   If you have it, do use it just in case.. */
#define	CAN_FTRUNCATE	/* If ftruncate(2) is known, or something similar
			   can be deviced -- like trunc() (?) in XENIX.
			   Any such alternatives are left to the user for
			   excercise */
#define USE_MMAP	/* If mmap() can be used to read the queue-file,
			   use it to improve the system performance.. */
#ifdef USE_MMAP
# include <sys/mman.h>
#endif

#ifdef	CAN_FTRUNCATE
extern int ftruncate __((int fd, off_t length));
#endif
extern char *strchr  __((char *s, int c));
extern char *strrchr __((char *s, int c));
/* extern void *malloc  __((unsigned size)); */
extern int   strlen  __((const char *s));
extern int   errno;
extern FILE *tmpfile __((void));

FILE *logfile = NULL;
static char *progname;

void
usage()
{
	fprintf(stderr,"\
Usage:  %s /spool/file/path uid gid err-addr program [args]\n\
       Uses also files:  .../path.pid,  .../path.lock\n\
       uid or username, gid or groupname\n\
       err-addr  is email address where errors are sent to.\n\
       program with explicite path, and necessary arguments.\n\
\n\
This is to be run by root, and program below this is then run with\n\
whatever is defined in the startup arguments.\n",progname);
#ifdef DAEMON_UID
	fprintf(stderr,"\
DAEMON-user (usually a result from running this from BSD sendmail's\n\
alias) can utilize  setuid-root, but no other can.\n");
#endif
	exit(EX_USAGE);
}


void
ERR(str,errcode)
char *str;
int errcode;
{
	fprintf(stderr,"%s: Error: (%d) %s\n",progname,errno,str);
	if (logfile)
	  fprintf(logfile,"%s: Error: (%d) %s\n",progname,errno,str);
/*	fprintf(stderr,"    Fatal! Exit!\n");	*/
	exit(errcode);
}

uid_t
parse_numname(arg)
char *arg;
{
	struct passwd *pwd = getpwnam(arg);
	int uid;

#ifdef	DAEMON_UID
	/* This allows setuid:ing this to root, and let  DAEMON (uid=1)
	   to run it..  Root won't get kicked because it has (e)uid=0..  */
	if ( ((uid = getuid()) != DAEMON_UID) && (uid = geteuid()))
	  return (uid_t)uid;
#else
	/* Non-root will run as him/her/it-self */
	if (uid = getuid()) return uid;
#endif
	if (pwd) return pwd->pw_uid;
	if (sscanf(arg,"%d",&uid) == 1) return (uid_t)uid;
	ERR("Bad username/uid argument!",EX_USAGE);
}

gid_t
parse_numgrp(arg)
char *arg;
{
	struct group *grp = getgrnam(arg);
	int gid;

#ifdef	DAEMON_UID
	/* This allows setuid:ing this to root, and let  DAEMON (uid=1)
	   to run it..  Root won't get kicked because it has (e)uid=0..  */
	if ( (getuid() != DAEMON_UID) && geteuid() )
	  return getgid();
#else
	/* Non-root will run as him/her/it-self */
	if (getuid())
	  return getgid();
#endif

	if (grp) return grp->gr_gid;
	if (sscanf(arg,"%d",&gid) == 1) return (gid_t)gid;
	ERR("Bad groupname/gid argument!",EX_USAGE);
}


void
getalock(queuename,lockname, pidname)
char *queuename, *lockname, *pidname;
{
	int lockloop = 20;
	int lockdelay = 5;
	int rc = 1;
	FILE *queuefile = fopen(queuename,"a");

	/* Probe for process with PID from  pidname. */
	int there_is_process = find_proc_by_pid(pidname);


	if (queuefile) fclose(queuefile);
	else	ERR("Couldn't open for append/create queue file ??",EX_USAGE);
	/* Created it if it didn't exist.. */

	while (lockloop-- > 0) {
	  if ((rc = link(queuename,lockname)) == 0) return; /* Made it */
	  if (errno == EEXIST) {
	    if (!there_is_process)
	      unlink(lockname);
	    sleep(lockdelay);
	  } else ERR("Can't make lock with link() method!",EX_USAGE);
	  there_is_process = find_proc_by_pid(pidname);
	}
	if (rc != 0)
	  ERR("Timeout on lock making... 100 seconds should be enough :-(",
	      EX_TEMPFAIL);
}

void
releasealock(lockname)
char *lockname;
{
	unlink(lockname);
}

void
copyintoqueue(queuename,uid,gid)
char *queuename;
uid_t uid;
gid_t gid;
{
	/* .. store incoming message with prefix of its length --
	   32bit int in hex written AFTER length is known for sure.
	   (Need to seek around..) */
	FILE *mailqueue = NULL, *mailqueue2 = NULL;
	char buf[4096];
	int rdlen;
	long wrtsize = 0;
	int err = 0;

	if (!(mailqueue = fopen(queuename,"a+"))) {
	  ERR("Queue open failed in `a+' mode -- mystic ?",EX_CANTCREAT);
	}
	if (!(mailqueue2 = fopen(queuename,"r+"))) {
	  ERR("Queue open failed in `r+' mode -- mystic ?",EX_CANTCREAT);
	}
	chown(queuename,uid,gid);
	chmod(queuename,0600);

	fseek(mailqueue2,0,2);	/* To the end.. */
	fprintf(mailqueue,"%08x?\n",0); /* where we write this, and then.. */
	while(!feof(stdin)) {
	  rdlen = fread(buf,1,sizeof(buf),stdin);
	  if (rdlen == 0) break;
	  /* Never mind why, input terminated, and thats all we care */
	  wrtsize += rdlen;
	  if (fwrite(buf,1,rdlen,mailqueue) != rdlen) {
	    /* Something dead wrong!  Can't write it all to the disk! */
	    err = 1;
	    /* XX: Should trunc back and exit with some nasty error code ? */
	    /*  Problem:  File truncation isn't omnipresent.. */
#ifdef	CAN_FTRUNCATE
	    fclose(mailqueue);
	    ftruncate(fileno(mailqueue2),ftell(mailqueue2));
	    fclose(mailqueue2);
	    exit(EX_TEMPFAIL);	/* Am I too hopefull ? */
#endif
	    break;
	  }
	}
	fflush(mailqueue);
	fprintf(mailqueue2,"%08x%s",wrtsize, err ? "-":"+");
	fflush(mailqueue2);
	fclose(mailqueue);
	fclose(mailqueue2);

	if (err) exit(EX_CANTCREAT);
}

/* Probe for process with PID from  pidname. */
int	/* rc != 0: there is process */
find_proc_by_pid(pidname)
char *pidname;
{
	int pid, rc;
	FILE *pidfile = fopen(pidname,"r");

	if (!pidfile) return 0;		/* No pid file.. */
	rc = fscanf(pidfile,"%d",&pid);
	fclose(pidfile);
	if (rc != 1) {
	  unlink(pidname);
	  return 0;
	}

	if (kill(pid,0) == 0) return 1;	/* There is a process! */
	unlink(pidname);
	return 0;
}

void
store_pid(pidname)	/* This is called AFTER all handles are closed.. */
char *pidname;
{
	FILE *pidfile = fopen(pidname,"w");
	if (!pidfile) exit(EX_UNAVAILABLE);	/* Waste of time ? */
	fprintf(pidfile,"%d\n",getpid());
	fclose(pidfile);
}


/* Fork up a child and feed to it the  first message
   (via pipe?) Run child with r&e uids and gids set
   to what was requested */
/* If child return code was not 0, send message to
   error_address in proper envelope. */
void
run_child(queuename,uid,gid,erraddr,cmd)
char *queuename;
uid_t uid;
gid_t gid;
char *erraddr;
char **cmd;
{
	/* These are run when all handles are closed.. */
	FILE *procinp = tmpfile();	/* Handle 0: stdin  */
	FILE *errfile = tmpfile();	/* Handle 1: stdout */
	FILE *queuefile;
	int  pid, rc, blksize;
	long inpsize,copysize = 0;
	char buf[4096];

	if (!procinp) ERR("run_child() failed to get handle 0 file",EX_UNAVAILABLE);
	if (!errfile) ERR("run_child() failed to get handle 1 file",EX_UNAVAILABLE);

	dup(1);	/* Magic knowledge which can be wrong ? Created handle 2: stderr */
	fprintf(errfile,"From:\tqueue-runner-errors (Queue: %s)\n",queuename);
	fprintf(errfile,"To:\tQueue error recipient <%s>\n",erraddr);
	fprintf(errfile,"Subject:\tMailServiceSerializer queue cmd output\n\n");
	fprintf(errfile,"Filenos: errfile=%d, procinp=%d\n",
			fileno(errfile),fileno(procinp));
	fflush(errfile);
	/* Right, if rc = 0, we just fclose() and thats that, else we seek to
	   the beginning, and give some arguments to SENDMAIL with this as
	   stdin.. */

	queuefile = fopen(queuename,"r");
	*buf = 0;
	if (!queuefile || fgets(buf,sizeof(buf),queuefile) == NULL
	    || sscanf(buf,"%8x",&inpsize) != 1) {
	  int err = errno;
	  fclose(queuefile);
	  fclose(procinp);
	  fprintf(errfile,
		  "** Couldn't open queue file, or bad first line. errno = %d\n%s",
		  err,buf);
	  goto err_out;
	}
	if (buf[8] != '+') {
	  /* Ah well, discard it! Let queue purger to handle its removal */
	  fclose(queuefile);
	  fclose(procinp);
	  fclose(errfile);
	  return;
	}
	while(!feof(queuefile) && (inpsize > 0) && !ferror(procinp)) {
	  /* Read a block at the time, write out partial if need be */
	  blksize = fread(buf,1,sizeof(buf),queuefile);
	  if (blksize == 0) break;
	  if (blksize > inpsize)
	    blksize = inpsize;
	  fwrite(buf,blksize,1,procinp);	/* Will write all, or fail.. */
	  if (ferror(procinp)) {		/* ..and we see failure here.*/
	    int err = errno;
	    fclose(queuefile);
	    fclose(procinp);
	    fprintf(errfile,
		    "Error while writing to procinp temporary file, errno=%d\n",
		    err);
	    goto err_out;
	  }
	  inpsize  -= blksize;
	  copysize += blksize;
	}
	fflush(procinp);
	fseek(procinp,0,0);	/* To the beginning.. */
	fclose(queuefile);

	/* What have we ?  handle 0: procinp (stdin), 1: errfile (stdout), 2: errfile (stdout)
	   Time to fork() and wait() */

	if ((pid = fork()) < 0) {
	  /* Argh.. Can't fork :-(  Exit all the way! */
	  exit(EX_UNAVAILABLE); /* XX: Some other better ? */
	}
	if (pid == 0) {
	  gid_t groups[2];
	  
	  /* CHILD */
	  dup2(dup(fileno(procinp)),0);
	  dup2(dup(fileno(errfile)),1);
	  dup2(1,2);

	  groups[0] = gid;
	  setgroups(1,groups);

	  setgid(gid);
#ifdef	HAS_SETEUID
	  setegid(gid);
	  seteuid(uid);
#endif
	  setuid(uid);

	  execv(*cmd,cmd);
	  /* If THAT fails, what can we do ?  Nothing :-( */
	  ERR("Running child failed",90);
	}
	/* Parent.. */

	pid = wait(&rc);
	if (rc == 0) {		/* exit(0) gives 0 into rc. Others are errors*/
	  fclose(procinp);
	  fclose(errfile);
	  close(0); close(1); close(2);
	  return;
	}
	fprintf(errfile,
		"\nProgram %s terminated because of:\n ",
		*cmd);
	if ((rc & 0377) == 0177) fprintf(errfile,"Got stopped by signal %d",rc>>8);
	else if (rc & 0377) fprintf(errfile,"Terminated by signal %d%s",rc&0177,rc&0200 ?" (with core)":"");
	else fprintf(errfile,"Exited with status code %d",rc >> 8);
	fprintf(errfile,"\nInput was %d bytes:\n",copysize);
err_out:
	/* Ahh.. An error.  Lets try to send it with sendmail.. */
	fseek(procinp,0,0);
	for(;;) {
	  blksize = fread(buf,1,sizeof(buf),procinp);
	  if (blksize == 0) break;
	  fwrite(buf,blksize,1,errfile);
	}
	fflush(errfile);
	fseek(errfile,0,0);	/* Errfile into beginning */
	dup2(fileno(errfile),0);
	/* Handle 0 is the error file, close all else */
	close(1); close(2);
	dup2(fileno(procinp = fopen("/dev/null","w+")),2);
	if ((pid = fork()) < 0) {
	  close(0); exit(EX_UNAVAILABLE);	/* Can't fork, give up :-( */
	}
	if (!pid) { /* CHILD */
	  execl(SENDMAIL,SENDMAIL,erraddr,NULL);
	  exit(EX_UNAVAILABLE);
	}
	fclose(errfile);
	fclose(procinp);
	close(0); close(1); close(2);
	pid = wait(&rc);
	if (rc != 0) exit(EX_UNAVAILABLE);
}

/* Remove first entry from queue file.. */
void
remove_first(queuename,uid,gid)
char *queuename;
uid_t uid;
gid_t gid;
{
	char *queue2name = malloc(strlen(queuename)+5);
	long inpsize;
	int  blksize;
#ifdef	USE_MMAP
	char *buf;
#else
	char buf[4096];
#endif

#ifndef	USE_MMAP
	FILE *queuefile, *queue2file;

	sprintf(queue2name,"%s.cpy",queuename);
	/* If malloc fails, let this drop core.. */

	queuefile = fopen(queuename,"r");
	*buf = 0;
	if (!queuefile || fgets(buf,sizeof(buf),queuefile) == NULL
	    || sscanf(buf,"%8x",&inpsize) != 1) {
	  ERR("Block size magic read wrong in remove_first()",EX_USAGE);
	}
	if (buf[8] == '?') {
	  ERR("Bad block size magic read in remove_first()",EX_USAGE);
	}

	fseek(queuefile,inpsize,1);	/* Skip forward */

	queue2file = fopen(queue2name,"w");
	chown(queue2name,uid,gid);
	chmod(queue2name,0600);
	if (!queue2file) {
	  ERR("Opening error in remove_first()",EX_UNAVAILABLE);
	}
	while(!feof(queuefile) && !ferror(queue2file)) {
	  blksize = fread(buf,1,sizeof(buf),queuefile);
	  if (blksize == 0) break;
	  fwrite(buf,1,blksize,queue2file);
	}
	if (ferror(queue2file)) {
	  int err = errno;
	  fclose(queue2file);
	  unlink(queue2name);
	  ERR("Copying error in remove_first()",EX_UNAVAILABLE);
	}
	/* Finally copied output to a new name */
	fclose(queuefile);
	rename(queue2name,queuename);
	fclose(queue2file);
#else /* USE_MMAP */
	long filesize;
	int queuefd = open(queuename,O_RDONLY,0);
	int queue2fd;
	int i;
	struct stat stats;

	if (queuefd < 0)
	  ERR("Queue-file didn't open R/O ??",EX_USAGE);

	fstat(queuefd,&stats);
	filesize = stats.st_size;

	buf = (char*)mmap(NULL, filesize, PROT_READ, MAP_SHARED, queuefd, 0);
	if ((int)buf == -1)
	  ERR("mmap()ing queue-file r/o failed!",EX_UNAVAILABLE);
	close(queuefd);

	sprintf(queue2name,"%s.cpy",queuename);
	/* If malloc fails, let this drop core.. */

	if (sscanf(buf,"%8x",&inpsize) != 1) {
	  ERR("Block size magic read wrong in remove_first()",EX_USAGE);
	}
	if (buf[8] == '?') {
	  ERR("Bad block size magic read in remove_first()",EX_USAGE);
	}

	for (i=0; i < filesize; ++i)
	  if (buf[i] == '\n')
	    break;
	if (i < filesize) ++i;

	inpsize += i;	/* Skip-size is the header + first message size */

	queue2fd = open(queue2name,O_WRONLY|O_TRUNC|O_CREAT,0600);
	if (queue2fd < 0)
	  ERR("Opening error of queue2name in remove_first()",EX_UNAVAILABLE);

	chown(queue2name,uid,gid);
	chmod(queue2name,0600);

	blksize = write(queue2fd, buf+inpsize, filesize - inpsize);
	if (blksize != (filesize - inpsize) || blksize == -1) {
	  int err = errno;
	  munmap(buf,filesize);
	  close(queue2fd);
	  unlink(queue2name);
	  ERR("Copying error to queue2name in remove_first()",EX_UNAVAILABLE);
	}
	munmap(buf,filesize);
	/* Finally copied output to a new name */
	rename(queue2name,queuename);
	close(queue2fd);
#endif
	free(queue2name);
}


int main(argc,argv)
int argc;
char *argv[];
{
	struct stat stats;
	char *queuename = argv[1];
	char *lockname = NULL;
	char *logname = NULL;
	char *pidname = NULL;
	char *erraddr = argv[4];
	char *s;
	uid_t uid;
	gid_t gid;
	time_t now;
	char **cmd = NULL;
	int there_is_process = 0;
	int has_lock = 0;
	int processcnt = 0;

	progname = argv[0];
	if (s = strrchr(progname,'/'))
	    progname = ++s;

#ifndef	DAEMON_UID
	if (getuid() != geteuid() || getgid() != getegid()) {
	    fprintf(stderr,"%s: RUN AS SUID/SGID PROGRAM NOT CONFIGURED! SECURITY BREACH! ABORT!\n",progname);
	    exit(EX_USAGE);
	}
#else
	/* This allows setuid:ing this to root, and let  DAEMON (uid=1)
	   to run it..  Root won't get kicked because it has (e)uid=0..  */
	uid = getuid();
	if (uid != DAEMON_UID && uid != 0 && geteuid() != uid) {
	  /* Ok, not root, not DAEMON, but is suid-something.. */
	  setegid(getgid());
#ifdef	HAS_SETEUID
	  seteuid(uid);
#else
	  setuid(uid); /* Earlier SYSV's with saved-uid-bit.. */
#endif
	}
#endif

	if (argc < 5) usage();

	/* argv[1]  has been stored */
	lockname = malloc(strlen(queuename)+8);
	sprintf(lockname,"%s.lock",queuename);
	logname = malloc(strlen(queuename)+8);
	sprintf(logname,"%s.log",queuename);
	logfile = fopen(logname,"a");
	pidname = malloc(strlen(queuename)+8);
	sprintf(pidname,"%s.pid",queuename);
	/* If malloc fails in here, we study core file.. */

	uid = parse_numname(argv[2]);
	gid = parse_numgrp(argv[3]);

	/* erraddr = argv[4]; */ /* Just reminder */

	cmd = &argv[5];
	if (**cmd != '/') {
	  fprintf(stderr,
		  "%s requires ABSOLUTE path for the program  to be run\n",
		  progname);
	  usage();
	}
	if ((stat(*cmd,&stats) != 0)
	    || (stats.st_mode & 0111) == 0
	    || !S_ISREG(stats.st_mode)) {
	  fprintf(stderr,
		  " Command argument must point to regular file with some execute permissions set.\n");
	  usage();
	}

	errno = 0;
	if (((stat(queuename,&stats) != 0)
	     && errno != ENOENT && errno != 0)
	    || !S_ISREG(stats.st_mode)) {
	  ERR("Queue file exists, and isn't regular file, or path to it doesn't exist, or some other error",EX_USAGE);
	}


	/* Get a lock..  link(queuename,lockname) -method. */
	getalock(queuename,lockname,pidname);
	/* Now we have the lock, or we are dead.. */
	has_lock = 1;

	/* Probe for process with PID from  pidname. */
	there_is_process = find_proc_by_pid(pidname);

	/* .. store incoming message with prefix of its length --
	   32bit int in hex written AFTER length is known for sure.
	   (Need to seek around..) */
	copyintoqueue(queuename,uid,gid);

	/* Was there a process with known pid already handling this
	   queue ?  If so, we release lock, and exit now! */
	if (there_is_process) {
	  releasealock(lockname);
	  exit(EX_OK);
	}

	/* -- we are about to form that process to handle things -- */

	/* Close all file handles, fork, and parent exits!
	   Store child PID into  pidname-file. */
	close(0);close(1);close(2);
	daemon(0,0);	/* Detach, close handles 0,1,2 */
	store_pid(pidname);

	/* -- Ready to run child programs -- */
	do {	/* While queue is not empty */

	  /* If locked, release lock */
	  if (has_lock) releasealock(lockname);
	  has_lock = 0;
	       
	  /* Fork up a child and feed to it the  first message
	     (via pipe?) Run child with r&e uids and gids set
	     to what was requested */
	  /* If child return code was not 0, send message to
	     error_address in proper envelope. */
	  run_child(queuename,uid,gid,erraddr,cmd);

	  /* Get a lock */
	  getalock(queuename,lockname,pidname);
	  has_lock = 1;
	  /* Remove 1st entry - unspecified method.

	     When queue becomes empty, its size is 0 */
	  remove_first(queuename,uid,gid);
	  ++processcnt;
	  /* end while loop */
	  stat(queuename,&stats);	/* Error test ? */
	} while(stats.st_size > 0);
	time(&now);
	if (logfile)
	  fprintf(logfile,"queue `%s' done %d items on %s",
		  queuename, processcnt, ctime(&now));

	/* Remove pidname, remove lock, exit. */
	unlink(pidname);
	unlink(lockname);
	exit(EX_OK);
}
