
/*
 * refile.c -- move or link message(s) from a source folder
 *          -- into one or more destination folders
 *
 * $Id$
 */

#include <h/mh.h>
#include <fcntl.h>
#include <errno.h>

static struct swit switches[] = {
#define	DRAFTSW          0
    { "draft", 0 },
#define	LINKSW           1
    { "link", 0 },
#define	NLINKSW          2
    { "nolink", 0 },
#define	PRESSW           3
    { "preserve", 0 },
#define	NPRESSW          4
    { "nopreserve", 0 },
#define	SRCSW            5
    { "src +folder", 0 },
#define	FILESW           6
    { "file file", 0 },
#define	RPROCSW          7
    { "rmmproc program", 0 },
#define	NRPRCSW          8
    { "normmproc", 0 },
#define VERSIONSW        9
    { "version", 0 },
#define	HELPSW          10
    { "help", 4 },
    { NULL, 0 }
};

extern int errno;

static char maildir[BUFSIZ];

struct st_fold {
    char *f_name;
    struct msgs *f_mp;
};

/*
 * static prototypes
 */
static void opnfolds (struct st_fold *, int);
static void clsfolds (struct st_fold *, int);
static void remove_files (int, char **);
static int m_file (char *, struct st_fold *, int, int);
static int addmsg (struct msgs **, char *, int);


int
main (int argc, char **argv)
{
    int	linkf = 0, preserve = 0, filep = 0;
    int foldp = 0, msgp = 0, isdf = 0;
    int i, msgnum;
    char *cp, *folder = NULL, buf[100];
    char **ap, **argp, *arguments[MAXARGS];
    char *filevec[NFOLDERS + 2];
    char **files = &filevec[1];	   /* leave room for remove_files:vec[0] */
    char *msgs[MAXARGS];
    struct st_fold folders[NFOLDERS + 1];
    struct msgs *mp;

#ifdef LOCALE
    setlocale(LC_ALL, "");
#endif
    invo_name = r1bindex (argv[0], '/');
    if ((cp = m_find (invo_name)) != NULL) {
	ap = brkstring (cp = getcpy (cp), " ", "\n");
	ap = copyip (ap, arguments);
    } else {
	ap = arguments;
    }
    copyip (argv + 1, ap);
    argp = arguments;

    while ((cp = *argp++)) {
	if (*cp == '-')
	    switch (smatch (++cp, switches)) {
		case AMBIGSW: 
		    ambigsw (cp, switches);
		    done (1);
		case UNKWNSW: 
		    adios (NULL, "-%s unknown\n", cp);

		case HELPSW: 
		    sprintf (buf, "%s [msgs] [switches] +folder ...", invo_name);
		    print_help (buf, switches);
		    done (1);
		case VERSIONSW:
		    print_version(invo_name);
		    done (1);

		case LINKSW: 
		    linkf++;
		    continue;
		case NLINKSW: 
		    linkf = 0;
		    continue;

		case PRESSW: 
		    preserve++;
		    continue;
		case NPRESSW: 
		    preserve = 0;
		    continue;

		case SRCSW: 
		    if (folder)
			adios (NULL, "only one source folder at a time!");
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    folder = path (*cp == '+' || *cp == '@' ? cp + 1 : cp,
				   *cp != '@' ? TFOLDER : TSUBCWF);
		    continue;
		case DRAFTSW:
		    if (filep > NFOLDERS)
			adios (NULL, "only %d files allowed!", NFOLDERS);
		    isdf = 0;
		    files[filep++] = getcpy (m_draft (NULL, NULL, 1, &isdf));
		    continue;
		case FILESW: 
		    if (filep > NFOLDERS)
			adios (NULL, "only %d files allowed!", NFOLDERS);
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    files[filep++] = path (cp, TFILE);
		    continue;

		case RPROCSW: 
		    if (!(rmmproc = *argp++) || *rmmproc == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    continue;
		case NRPRCSW: 
		    rmmproc = (char *)0;
		    continue;
	    }
	if (*cp == '+' || *cp == '@') {
	    if (foldp > NFOLDERS)
		adios (NULL, "only %d folders allowed!", NFOLDERS);
	    folders[foldp++].f_name =
		    path (cp + 1, *cp == '+' ? TFOLDER : TSUBCWF);
	} else {
	    msgs[msgp++] = cp;
	}
    }

    if (!m_find ("path"))
	free (path ("./", TFOLDER));
    if (foldp == 0)
	adios (NULL, "no folder specified");

#ifdef WHATNOW
    if (!msgp && !foldp && !filep && (cp = getenv ("mhdraft")) && *cp)
	files[filep++] = cp;
#endif /* WHATNOW */

    if (filep > 0) {
	if (folder || msgp)
	    adios (NULL, "use -file or some messages, not both");
	opnfolds (folders, foldp);
	for (i = 0; i < filep; i++)
	    if (m_file (files[i], folders, foldp, preserve))
		done (1);
	/* If -nolink, then "remove" files */
	if (!linkf)
	    remove_files (filep, filevec);
	done (0);
    }

    if (!msgp)
	msgs[msgp++] = "cur";
    if (!folder)
	folder = m_getfolder ();
    strcpy (maildir, m_maildir (folder));

    if (chdir (maildir) == NOTOK)
	adios (maildir, "unable to change directory to");

    /* read source folder */
    if (!(mp = m_readfolder (folder)))
	adios (NULL, "unable to read folder %s", folder);
    if (mp->hghmsg == 0)
	adios (NULL, "no messages in %s", folder);

    /* parse the message range/sequence/name and set SELECTED */
    for (msgnum = 0; msgnum < msgp; msgnum++)
	if (!m_convert (mp, msgs[msgnum]))
	    done (1);
    seq_setprev (mp);	/* set the previous-sequence */

    /* create folder structures for each destination folder */
    opnfolds (folders, foldp);

    /* Link all the selected messages into destination folders */
    for (msgnum = mp->lowsel; msgnum <= mp->hghsel; msgnum++) {
	if (is_selected (mp, msgnum)) {
	    cp = getcpy (m_name (msgnum));
	    if (m_file (cp, folders, foldp, preserve))
		done (1);
	    free (cp);
	}
    }

    /* If -nolink, then "remove" messages from source folder */
    if (!linkf) {
	folder_delmsgs (mp, 0);
    }

    clsfolds (folders, foldp);

    if (mp->hghsel != mp->curmsg
	    && (mp->numsel != mp->nummsg || linkf))
	seq_setcur (mp, mp->hghsel);
    seq_save (mp);	/* synchronize message sequences */

    m_replace (pfolder, folder);	/* update current folder   */
    m_update ();			/* update the context file */
    done (0);
}


/*
 * Read all the destination folders and
 * create folder structures for all of them.
 */

static void
opnfolds (struct st_fold *folders, int nfolders)
{
    register char *cp;
    char nmaildir[BUFSIZ];
    register struct st_fold *fp, *ep;
    register struct msgs *mp;
    struct stat st;

    for (fp = folders, ep = folders + nfolders; fp < ep; fp++) {
	chdir (m_maildir (""));
	strcpy (nmaildir, m_maildir (fp->f_name));

	if (stat (nmaildir, &st) == NOTOK) {
	    if (errno != ENOENT)
		adios (nmaildir, "error on folder");
	    cp = concat ("Create folder \"", nmaildir, "\"? ", NULL);
	    if (!getanswer (cp))
		done (1);
	    free (cp);
	    if (!makedir (nmaildir))
		adios (NULL, "unable to create folder %s", nmaildir);
	}

	if (chdir (nmaildir) == NOTOK)
	    adios (nmaildir, "unable to change directory to");
	if (!(mp = m_readfolder (fp->f_name)))
	    adios (NULL, "unable to read folder %s", fp->f_name);
	mp->curmsg = 0;

	fp->f_mp = mp;

	chdir (maildir);
    }
}


/*
 * Set the Previous-Sequence and then sychronize the
 * sequence file, for each destination folder.
 */

static void
clsfolds (struct st_fold *folders, int nfolders)
{
    register struct st_fold *fp, *ep;
    register struct msgs *mp;

    for (fp = folders, ep = folders + nfolders; fp < ep; fp++) {
	mp = fp->f_mp;
	seq_setprev (mp);
	seq_save (mp);
    }
}


/*
 * If you have a "rmmproc" defined, we called that
 * to remove all the specified files.  If "rmmproc"
 * is not defined, then just unlink the files.
 */

static void
remove_files (int filep, char **files)
{
    int i;
    char **vec;

    /* If rmmproc is defined, we use that */
    if (rmmproc) {
	vec = files++;		/* vec[0] = filevec[0] */
	files[filep] = NULL;	/* NULL terminate list */

	fflush (stdout);
	vec[0] = r1bindex (rmmproc, '/');
	execvp (rmmproc, vec);
	adios (rmmproc, "unable to exec");
    }

    /* Else just unlink the files */
    files++;	/* advance past filevec[0] */
    for (i = 0; i < filep; i++) {
	if (unlink (files[i]) == NOTOK)
	    admonish (files[i], "unable to unlink");
    }
}


/*
 * Link (or copy) the message into each of
 * the destination folders.
 */

static int
m_file (char *msgfile, struct st_fold *folders, int nfolders, int preserve)
{
    int msgnum;
    struct st_fold *fp, *ep;

    for (fp = folders, ep = folders + nfolders; fp < ep; fp++) {
	if ((msgnum = addmsg (&fp->f_mp, msgfile, preserve)) == -1)
	    return 1;
    }
    return 0;
}


/*
 * Link message into a folder.  Return the new number
 * of the message.  If an error occurs, return -1.
 */

static int
addmsg (struct msgs **mpp, char *msgfile, int preserve)
{
    int in, out, linkerr, msgnum;
    char *nmsg, newmsg[BUFSIZ];
    struct msgs *mp;
    struct stat st1, st2;

    mp = *mpp;

    /* Preserve the numbering of the message */
    if (preserve && (msgnum = m_atoi (msgfile)) > 0) {
	nmsg = msgfile;
	if (msgnum >= mp->hghoff) {
	    if ((mp = m_remsg (mp, 0, msgnum + MAXFOLDER)))
		*mpp = mp;
	    else
		adios (NULL, "unable to allocate folder storage");
	}
	if (!(mp->msgstats[msgnum] & EXISTS)) {
	    mp->msgstats[msgnum] |= EXISTS;
#if 0
	    mp->msgstats[msgnum] &= ~DELETED;
#endif
	    mp->nummsg++;
	}
	set_selected (mp, msgnum);
	if (msgnum > mp->hghmsg)
	    mp->hghmsg = msgnum;
    } else {
	/* Else assign it next highest number available */
	if (mp->hghmsg >= mp->hghoff) {
	    if ((mp = m_remsg (mp, 0, mp->hghoff + MAXFOLDER)))
		*mpp = mp;
	    else
		adios (NULL, "unable to allocate folder storage");
	}

	msgnum = ++mp->hghmsg;	/* get next number available */
	nmsg = m_name (msgnum);
	mp->nummsg++;
	mp->msgstats[msgnum] |= EXISTS;
	set_selected (mp, msgnum);
    }

    if (mp->lowmsg == 0)
	mp->lowmsg = msgnum;
    if (mp->lowsel == 0 || msgnum < mp->lowsel)
	mp->lowsel = msgnum;
    if (msgnum > mp->hghsel)
	mp->hghsel = msgnum;

    sprintf (newmsg, "%s/%s", mp->foldpath, nmsg);

    /*
     * Now link message into new folder
     */
    if (link (msgfile, newmsg) == NOTOK) {
	linkerr = errno;

#ifdef EISREMOTE
	if (linkerr == EISREMOTE)
	    linkerr = EXDEV;
#endif /* EISREMOTE */

	/*
	 * Check if the file in our desired location is the
	 * same as the source file.  If so, then just continue,
	 * else print error message and return.
	 */
	if (linkerr == EEXIST) {
	    if (stat (msgfile, &st2) == OK && stat (newmsg, &st1) == OK
		&& st2.st_ino == st1.st_ino) {
		return msgnum;
	    } else {
		advise (NULL, "message %s already exists", newmsg);
		return -1;
	    }
	}

	/*
	 * If link failed because we are trying to link
	 * across devices, then check if there is a message
	 * already in the desired location.  If so, then return
	 * error, else just copy the message.
	 */
	if (linkerr == EXDEV) {
	    if (stat (newmsg, &st1) == OK) {
		advise (NULL, "message %s:%s already exists", newmsg);
		return -1;
	    } else {
		if ((in = open (msgfile, O_RDONLY)) == NOTOK) {
		    advise (msgfile, "unable to open message %s");
		    return -1;
		}
		fstat (in, &st1);
		if ((out = creat (newmsg, (int) st1.st_mode & 0777)) == NOTOK) {
		    advise (newmsg, "unable to create");
		    close (in);
		    return -1;
		}
		cpydata (in, out, msgfile, newmsg);
		close (in);
		close (out);
		return msgnum;
	    }
	}

	/*
	 * Else, some other type of link error,
	 * so just return error.
	 */
	advise (newmsg, "error linking %s to", msgfile);
	return -1;
    }

    return msgnum;
}
