
/*
 * mark.c -- add message(s) to sequences in given folder
 *        -- delete messages (s) from sequences in given folder
 *        -- list sequences in given folder
 *
 * $Id$
 */

#include <h/mh.h>

static struct swit switches[] = {
#define	ADDSW               0
    { "add", 0 },
#define	DELSW               1
    { "delete", 0 },
#define	LSTSW               2
    { "list", 0 },
#define	SEQSW               3
    { "sequence name", 0 },
#define	PUBLSW              4
    { "public", 0 },
#define	NPUBLSW             5
    { "nopublic", 0 },
#define	ZEROSW              6
    { "zero", 0 },
#define	NZEROSW             7
    { "nozero", 0 },
#define VERSIONSW           8
    { "version", 0 },
#define	HELPSW              9
    { "help", 4 },
#define	DEBUGSW            10
    { "debug", -5 },
    { NULL, 0 }
};

/*
 * static prototypes
 */
static void seq_printall (struct msgs *);
static void seq_print (struct msgs *, char *);
static void print_debug (struct msgs *);
static void seq_printdebug (struct msgs *);


main (int argc, char **argv)
{
    int addsw = 0, deletesw = 0, debugsw = 0;
    int listsw = 0, publicsw = -1, zerosw = 0;
    int seqp = 0, msgp = 0, msgnum;
    char *cp, *maildir, *folder = NULL, buf[100];
    char **ap, **argp, *arguments[MAXARGS];
    char *seqs[NUMATTRS + 1], *msgs[MAXARGS];
    struct msgs *mp;

#ifdef LOCALE
    setlocale(LC_ALL, "");
#endif
    invo_name = r1bindex (argv[0], '/');
    if ((cp = m_find (invo_name)) != NULL) {
	ap = brkstring (cp = getcpy (cp), " ", "\n");
	ap = copyip (ap, arguments);
    } else {
	ap = arguments;
    }
    copyip (argv + 1, ap);
    argp = arguments;

    while ((cp = *argp++)) {
	if (*cp == '-')
	    switch (smatch (++cp, switches)) {
		case AMBIGSW: 
		    ambigsw (cp, switches);
		    done (1);
		case UNKWNSW: 
		    adios (NULL, "-%s unknown\n", cp);

		case HELPSW: 
		    sprintf (buf, "%s [+folder] [msgs] [switches]", invo_name);
		    print_help (buf, switches);
		    done (1);
		case VERSIONSW:
		    print_version(invo_name);
		    done (1);

		case ADDSW: 
		    addsw++;
		    deletesw = listsw = 0;
		    continue;
		case DELSW: 
		    deletesw++;
		    addsw = listsw = 0;
		    continue;
		case LSTSW: 
		    listsw++;
		    addsw = deletesw = 0;
		    continue;

		case SEQSW: 
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    if (seqp < NUMATTRS)
			seqs[seqp++] = cp;
		    else
			adios (NULL, "only %d sequences allowed!", NUMATTRS);
		    continue;

		case PUBLSW: 
		    publicsw = 1;
		    continue;
		case NPUBLSW: 
		    publicsw = 0;
		    continue;

		case DEBUGSW: 
		    debugsw++;
		    continue;

		case ZEROSW: 
		    zerosw++;
		    continue;
		case NZEROSW: 
		    zerosw = 0;
		    continue;
	    }
	if (*cp == '+' || *cp == '@') {
	    if (folder)
		adios (NULL, "only one folder at a time!");
	    else
		folder = path (cp + 1, *cp == '+' ? TFOLDER : TSUBCWF);
	}
	else
	    msgs[msgp++] = cp;
    }

    /*
     * If we haven't specified -add, -delete, or -list,
     * then use -add if a sequence was specified, else
     * use -list.
     */
    if (!addsw && !deletesw && !listsw) {
	if (seqp)
	    addsw++;
	else
	    listsw++;
    }

    if (!m_find ("path"))
	free (path ("./", TFOLDER));
    if (!msgp)
	msgs[msgp++] = listsw ? "all" :"cur";
    if (!folder)
	folder = m_getfolder ();
    maildir = m_maildir (folder);

    if (chdir (maildir) == NOTOK)
	adios (maildir, "unable to change directory to");

    /* read folder and create message structure */
    if (!(mp = m_readfolder (folder)))
	adios (NULL, "unable to read folder %s", folder);

    if (mp->hghmsg == 0)
	adios (NULL, "no messages in %s", folder);

    /* parse all the message ranges/sequences and set SELECTED */
    for (msgnum = 0; msgnum < msgp; msgnum++)
	if (!m_convert (mp, msgs[msgnum]))
	    done (1);

    if (publicsw == -1)
	publicsw = (mp->msgflags & READONLY) ? 0 : 1;
    if (publicsw && (mp->msgflags & READONLY))
	adios (NULL, "folder %s is read-only, so -public not allowed",
		folder);

    /* print some general debugging info */
    if (debugsw) {
	print_debug(mp);
    }

    /*
     * Make sure at least one sequence has been
     * specified if we are adding or deleting.
     */
    if (seqp == 0 && (addsw || deletesw))
	adios (NULL, "-%s requires at least one -sequence argument",
		addsw ? "add" : "delete");
    seqs[seqp] = NULL;

    /* Adding messages to sequences */
    if (addsw) {
	for (seqp = 0; seqs[seqp]; seqp++)
	    if (!seq_addsel (mp, seqs[seqp], publicsw, zerosw))
		done (1);
    }

    /* Deleting messages from sequences */
    if (deletesw) {
	for (seqp = 0; seqs[seqp]; seqp++)
	    if (!seq_delsel (mp, seqs[seqp], publicsw, zerosw))
		done (1);
    }

    /* Listing messages in sequences */
    if (listsw) {
	if (!seqp) {
	    seq_printall (mp);
	} else {
	    for (seqp = 0; seqs[seqp]; seqp++)
		seq_print (mp, seqs[seqp]);
	}

	/* print debugging info about SELECTED messages */
	if (debugsw) {
	    seq_printdebug (mp);
	}
    }

    m_replace (pfolder, folder);	/* update current folder         */
    seq_save (mp);			/* synchronize message sequences */
    m_freefolder (mp);			/* free folder/message structure */
    m_update ();			/* update the context file       */
    done (0);
}


#define empty(s) ((s) ? (s) : "")

/*
 * Print all the sequences in a folder
 */
static void
seq_printall (struct msgs *mp)
{
    int i;

    for (i = 0; mp->msgattrs[i]; i++)
	printf ("%s%s: %s\n", mp->msgattrs[i],
	    is_seq_private (mp, i) ? " (private)" : "",
	    empty(seq_list(mp, mp->msgattrs[i])));
}


/*
 * Print a particular sequence in a folder
 */
static void
seq_print (struct msgs *mp, char *seqname)
{
    printf ("%s: %s\n", seqname, empty(seq_list(mp, seqname)));
}


/*
 * Print general debugging info
 */
static void
print_debug (struct msgs *mp)
{
    char buf[100];

    printf ("invo_name     = %s\n", invo_name);
    printf ("mypath        = %s\n", mypath);
    printf ("defpath       = %s\n", defpath);
    printf ("ctxpath       = %s\n", ctxpath);
    printf ("context flags = %s\n", sprintb (buf, (unsigned) ctxflags, DBITS));
    printf ("foldpath      = %s\n", mp->foldpath);
    printf ("folder flags  = %s\n\n", sprintb(buf, (unsigned) mp->msgflags, FBITS));
    printf ("lowmsg=%d hghmsg=%d nummsg=%d curmsg=%d\n",
	mp->lowmsg, mp->hghmsg, mp->nummsg, mp->curmsg);
    printf ("lowsel=%d hghsel=%d numsel=%d\n",
	mp->lowsel, mp->hghsel, mp->numsel);
#ifndef	MTR
    printf ("lowoff=%d hghoff=%d\n", mp->lowoff, mp->hghoff);
#else
    printf ("lowoff=%d hghoff=%d msgbase=0x%x msgstats=0x%x\n",
	mp->lowoff, mp->hghoff, mp->msgbase, mp->msgstats);
#endif
}


/*
 * Print debugging info about all the SELECTED
 * messages and the sequences they are in.
 */
static void
seq_printdebug (struct msgs *mp)
{
    int msgnum;
    char buf[100];

    for (msgnum = mp->lowsel; msgnum <= mp->hghsel; msgnum++) {
	if (is_selected (mp, msgnum))
	    printf ("%*d: %s\n", DMAXFOLDER, msgnum,
		sprintb (buf, (unsigned) mp->msgstats[msgnum],
		seq_bits (mp)));
    }
}
