/*
 *	Program Name:	imap server (see rfc1064)
 *	Module:         imapsavebb.c
 *
 *	Author(s):	William J. Yeager
 *			Symbolic Systems Resources Group
 *			Knowledge Systems Laboratory
 *			Departments of Computer Science and Medicine
 *			Stanford University
 *			Stanford, CA  94305
 *
 *	Date:		December 29th, MCMLXXXVIII
 *
 *	Sponsorship:	This work was supported by the Biomedical Research
 *			Technology Program of the National Institutes of Health
 *			under grant number RR-00785.
 *
 *      Copyright (c) 1988 by The Leland Stanford Junior University.
 *
 *  "This program may be distributed without restriction for non-commercial 
 *   use. Any sale or use of this program or adaptations thereof for commercial
 *   purposes is prohibited except under license from the Stanford Office of
 *   Technology Licensing."
 *
 */
/*
 * This little beast reads a message from stdin and copies it
 * to the appropriate file given as a passed parameter.
 * (This file is in mail format)
 * If no file is supplied, the message goes into the bit bucket.
 *
 * It is actually designed to be invoked by sendmail via the
 * aliases file - eg -
 *     su-etc:"/usr/stanford/bin/saveimapbb /usr/spool/bboards/su-etc"
 *
 *     Will append the new mail for su-etc to the destination file su-etc
 *     and also append to su-etc.txt. If the latter succeeds, then the file
 *     su-etc is deleted.
 *
 *     If the output .txt file has more than BB_MAXMESSAGES, then
 *     both the unix mail file and mtxt file are trimed down to
 *     BB_TRIMEDMESSAGES
 */
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <pwd.h>
#include <grp.h>
#include <strings.h>
#include <stdio.h>
#include <ctype.h>
#include "sin.h"
#include "requests.h"
#include "memory.h"
#include "envelope.h"
#include "mailcache.h"
#include "mbox.h"

#define OK 1
#define SAVEPOINTER 0
#define ERROR -1
#define POINTERPREFIX ".pointer_"

/* #define DEBUG */
/* #define SMALLBB */
/*
 *  Mail file sizing criteria
 */
#ifdef DEBUG
#define BB_MAXMESSAGES 10
#define BB_TRIMEDMESSAGES 5
#else
#ifdef SMALLBB
#define BB_MAXMESSAGES 50
#define BB_TRIMEDMESSAGES 25
#else
#define BB_MAXMESSAGES 250
#define BB_TRIMEDMESSAGES 100
#endif /* SMALLBB */
#endif /* DEBUG */
#define MAXFILESIZE 1000000			  /* maximum size in bytes */
int bb_maxmsgs= BB_MAXMESSAGES;
int bb_trimedSize= BB_TRIMEDMESSAGES;
/*
 *  Below are not used except to satisfy the linking of
 *  this code.
 */
int currentmbox= 0;
int mboxqptr= 0;
FILE *logfile;
#define LOGFILENAME "/usr/adm/imapbboard.log"
int fromsplen;
char *fromsp= {"From "};
int TheUsersName;
char logstring[1024];
main(argc,argv)
 int argc;
 char *argv[];
{/*  MAIN
  *  We read in the spooling file name:
  *    o Set our uid, gid (root, daemon)
  *    o If it does not exist, then the mode bits
  *      are set to 0764 before appending the message.
  */
  struct passwd *pw;
  struct group *gp;
  char pathname[1024],
       ptrfilename[1024],
       invisible[1024],
       dir[1024];
  char *cptr;
  int setmode;
  FILE *f,*pf;
  long offset;
  MAILBOX mbox;
  int ncopied;

  if (argc <= 1) {
    readWriteInput(0);
    exit(1);
  }
  if (!(gp = (struct group *)getgrnam("nogroup"))) {
    readWriteInput(0);
    exit(1);
  }
  endgrent();
  if (!(pw = (struct passwd *)getpwnam("root"))) {
    readWriteInput(0);
    exit(1);
  }
  endpwent();
  /*
   * Read in the file pathname/ requires directory path
   */
  strcpy(pathname,argv[1]);
  cptr = rindex(pathname,'/');
  if (!cptr) {
    readWriteInput(0);
    exit(1);
  }
  /*
   *  If we cannot access it, assume it does not exist.
   *  cuz we should be the creator of the file.
   */
  if (access(pathname,F_OK) < 0) setmode = 1;
  else
    setmode = 0;
  /*
   *  Connect to the directory so all other accesses are path
   *  relative.
   */
  strncpy(dir,pathname,cptr - pathname + 1);	  /* save directory */
  if (chdir(dir) < 0) {				  /* connect there */
    readWriteInput(0);
    exit(1);
  }
  ++cptr;					  /* file name pointer */
  strcpy(invisible,".");
  strcat(invisible,cptr);                         /* .filename */
  if (!(f = fopen(invisible,"a+"))) {
    readWriteInput(0);    
    exit(1);
  }  
  setuid(pw->pw_uid);				  /* run as root */
  if (setmode) {
    fchmod(fileno(f),0604);
    fchown(fileno(f),pw->pw_uid,gp->gr_gid);
  }
  /*
   * Set and save the starting file position.
   */
  if (fseek(f,0,FROMENDOFFILE) < 0) exit(1);
  offset = ftell(f);
  ncopied = readWriteInput(f);			  /* read data/write file */
  /*
   * Seek offset of first appended message
   * (which may in fact be saved from a previous
   * SAVEPOINTER return (see below))
   */
  strcpy(ptrfilename,POINTERPREFIX);
  strcat(ptrfilename,cptr);
  if (pf = fopen(ptrfilename,"r")) {		  /* Ahh! A previous one. */
    long old_offset;

    fscanf(pf,"%d",&old_offset);
    fclose(pf);
    ncopied += offset - old_offset;		  /* add in the excess */
    offset = old_offset;
    sprintf(logstring,"\nUsing ptr offset %d",offset);
    logfilewrite(logstring);
  }
  /*
   * Point back to beginning of this text.
   */
  if (fseek(f,offset,0) < 0) exit(1);		  /* system failure ?? */
#ifdef DEBUG
  logfile = stderr;
#else
  logfile = fopen(LOGFILENAME,"a");
#endif
  fromsplen = strlen(fromsp);                     /* Set for "From:" search */
  /*
   *  We need to append ncopied bytes + any left over
   *  from the previous failed attempts(offset < size of file);
   */
  switch (appendBbdMbox(f,cptr,&mbox,&offset,ncopied,pw,gp,dir)) {
						  /* append to assoc mbox */
    case OK: 
       if (!CheckTheBarberShop(&mbox,f,cptr))	  /* Time for the barber? */
	 logfilewrite("\n-->EXPUNGE failed");
       /* 
	*  remove previous pointer file if it exists
	*/
       if (pf) {				  
	 fclose(pf);
	 unlink(ptrfilename);
       }
       /*
	* rubout invisible file. Not needed.
	*/
       fclose(f);
       unlink(invisible);
       f = 0;
       break;
    case SAVEPOINTER:
      /*
       *  Save a pointer to offset for next attempt in
       *  .pointer_<filename> unless such a pointer already
       *  exists.
       */
       if (!pf)
	 if (pf = fopen(ptrfilename,"w")) {	  /* Ahh! A previous one. */
	   fprintf(pf,"%d",offset);
	   fclose(pf);
	 }
       break;
     case ERROR:
       /*
	* Something was amis and we have done nothing. So, leave
	* state as it was.
	*/
       break;
     }
  if (mbox.stream) fclose(mbox.stream);
  if (mbox.istream) fclose(mbox.istream);
  if (f) fclose(f);
  if (pf) fclose(pf);
}
HDRHDR headerBuf;
appendBbdMbox(mf,nameptr,mbox,offset,ncopied,pw,gp,dir)
  FILE *mf;					  /* input mail file */
  char *nameptr;				  /* points to "<filename>" */
  MAILBOX *mbox;				  /* mail box of bboard */
  long *offset;					  /* starting byte pos */
  int ncopied;					  /* size of new messages */
  struct passwd *pw;
  struct group *gp;
  char *dir;
{
  int len,initHdr;
  struct stat sbuf;
  /*
   *   Build MAILBOX template:
   */
  strcpy(mbox->dir,dir);
  /*
   * Save mail box and index file names:
   */
  strcpy(mbox->name,nameptr);			  /* file name */
  strcat(mbox->name,".txt");			  /* mbox text file */
  strcpy(mbox->iname,DEFAULTIPREFIX);		  /* create index file name */
  strcat(mbox->iname,mbox->name);
  mbox->rname[0] = '\0';			  /* Ignore user RO field */
  /*
   *  Open/Create the associated "Mail Box" files.
   */
  if (!(mbox->stream = fopen(mbox->name,"a+"))) {
    sprintf(logstring,"\nCould not open %s",mbox->name);
    logfilewrite(logstring);
    return(SAVEPOINTER);
  }
  /*
   *  We need an exclusive flock on this file
   */
  if (!tryAndLockTheBeastUp(mbox->stream)) {
    fclose(mbox->stream);
    sprintf(logstring,"\nCould not lock %s",mbox->name);
    logfilewrite(logstring);
    return(SAVEPOINTER);
  }
  /*
   * If this is a new file, then we must create the
   * index file and init its first record.
   */
  if (fstat(fileno(mbox->stream),&sbuf) < 0) {
    fclose (mbox->stream);
    sprintf(logstring,"\nstat(%s,&sbuf) failed",mbox->name);
    logfilewrite(logstring);
    return(SAVEPOINTER);				  
  }
  if (sbuf.st_size == 0) {/* create index file header template */
    headerBuf.Last_Read = getitime();		  /* Assume current time */
    headerBuf.Number_Of_Messages = 0;
    headerBuf.Size_Of_Text_File = 0;
    initHdr = 1;
  } else
    initHdr = 0;
  /*
   * Open the index file(create it if it isn't there)
   */
  mbox->istream = fopen(mbox->iname,"a+");
  if (!mbox->istream) {			  /* System error */
    fclose(mbox->stream);
    sprintf(logstring,"\nCould not open %s",mbox->iname);
    logfilewrite(logstring);
    return(SAVEPOINTER);
  }
  /* 
   * If text exists, then Index should have data. If not, we'll
   * create it again.
   */
  if (!initHdr) {				  /* text file exists */
    if (fstat(fileno(mbox->istream),&sbuf) < 0) {
      fclose (mbox->stream);
      fclose (mbox->istream);
      sprintf(logstring,"\nfstat(%s,&sbuf) failed",mbox->iname);
      logfilewrite(logstring);
      return(SAVEPOINTER);				  
    }
    /*
     * our open may have just created this file
     */
    if (sbuf.st_size == 0)			  /* WE created index file */
      if (!createIndexFile(mbox)) {
	fclose(mbox->stream);
	sprintf(logstring,"\nCould not create %s",mbox->iname);
	logfilewrite(logstring);
	return(SAVEPOINTER);
      }
    fchmod(fileno(mbox->istream),0604);
    fchown(fileno(mbox->istream),pw->pw_uid,gp->gr_gid);
  }
  if (!tryAndLockTheBeastUp(mbox->istream)) {	  /* ours alone */
    fclose(mbox->stream);
    fclose(mbox->istream);
    sprintf(logstring,"\nCould not lock %s",mbox->iname);
    logfilewrite(logstring);
    return(SAVEPOINTER);
  }
  if (initHdr) {				  /* Initialize index header */
    if (1 != fwrite(&headerBuf,sizeof(HDRHDR),1,mbox->istream)) {
      fclose(mbox->stream);
      fclose(mbox->istream);
      return(SAVEPOINTER);
    }
    fchmod(fileno(mbox->stream),0604);
    fchown(fileno(mbox->stream),pw->pw_uid,gp->gr_gid);
    fchmod(fileno(mbox->istream),0604);
    fchown(fileno(mbox->istream),pw->pw_uid,gp->gr_gid);
  } else {					  /* Opened append, so 1st */
    if (fseek(mbox->istream,0,FROMBEGINNINGOFFILE) < 0) {
      fclose(mbox->stream);
      fclose(mbox->istream);
      return(SAVEPOINTER);
    }
						  /* Now read into global buf*/
    if (1 != fread(&headerBuf,sizeof(HDRHDR),1,mbox->istream)) {
      fclose(mbox->stream);
      fclose(mbox->istream);
      return(SAVEPOINTER);
    }
  }
  /*
   * So, let us now append the "unix" mail to the
   * "mail.txt" mail box in Mtxt format.
   */
  return(appendUmhToMtxt(mbox,mf,offset,ncopied));
  }
/*
 *	Append new mail here. We map in the portion of the
 *      Unix mail file we need (sml is pointing there)
 *	because we have to read ahead to make our special
 *	mm header. So, we save relevant pointers into our
 *	buffer as we munge along.
 */ 
char *findfromline();
char *bytecopy();
#define LINELEN 256
appendUmhToMtxt(mbox,sml,bposition,nbytes)
  register MAILBOX *mbox;
  register FILE *sml;				  /* the spooled mail */
  register *bposition;				  /* offset at start */
  register nbytes;				  /* number to copy */
  {
    HDRHDR thehdr;
    register HDRHDR *hp= &thehdr;
    MSGHDR themhdr;
    register MSGHDR *mp= &themhdr;
    char headerline[LINELEN+1],
	 fromline[LINELEN+1];
    register mfoffset;				/* n chars into smailf */
    char *mfpos,				/* start of mail file */
	 *headerpos,				/* start of mail header */
	 *bodypos;				/* start of body of msg */
    char *fmapptr;
    struct stat sbuf;
    register ntoscan;
    int nastyerror;
    char vbuf[1024];

    /*
     *  Save number of bytes we must scan.
     */
    if ((ntoscan = nbytes) == 0) return(SAVEPOINTER);
    /*
     * 	Get the current header of the headers from the global
     *  header buffer.
     */
    bytecopy(hp,&headerBuf,sizeof(HDRHDR));
    if (hp->Number_Of_Messages >=  MAXMESSAGES) {
      sprintf(logstring,"\n%d exceeded",MAXMESSAGES);
      logfilewrite(logstring);
      return(SAVEPOINTER);
    }
    /*
     *	Point to the end of our mail file.
     */
    if (fseek(mbox->stream,0,FROMENDOFFILE) < 0) return(SAVEPOINTER);
    /*
     *	Read in the spooled mail file
     */
    if (!(fmapptr = (char *)malloc(ntoscan+2))) {
      sprintf(logstring,"\nmalloc(%d) failed",ntoscan+2);
      logfilewrite(logstring);
      return(SAVEPOINTER);
    }
    mfpos = fmapptr;
    *(mfpos + ntoscan) = '\0';				/* fire wall */
    if (1 != fread(mfpos,ntoscan,1,sml)) {
	free(fmapptr);
	sprintf(logstring,"\nCould not map in mail file of size %d",ntoscan);
	logfilewrite(logstring);
	return(SAVEPOINTER);
    }
    /*
     * 	Hussle through the new mail
     */
    mfoffset = flushtillfirstmsg(mfpos,ntoscan);  /* just in case */
    if (mfoffset == -1) return(0);		  /* NO first msg */
    mfpos += mfoffset;				  /* should not change*/
    ntoscan -= mfoffset;
    while (ntoscan > 0) {
	int lastmessage;
	int len,
	    size,
	    unixinternaldate,
	    badformat;
	char *msgpos;

	if (hp->Number_Of_Messages >=  MAXMESSAGES) {
	    *bposition = (int)mfpos;		  /* update to current msg */
	    free(fmapptr);
	    return(SAVEPOINTER);
	}
	/*
	 *	Find body of message after saving fromline
	 */
        msgpos = mfpos;
	/*
	 * set mfpos to first char after LF terminating fromline.
	 */
	mfpos = (char *)ncopytillCH(msgpos,fromline,LINELEN,LF) + 1;
	headerpos = mfpos;			  /* first msg. line */
	len = mfpos - msgpos;			  /* from line len */
	mfoffset += len;
	ntoscan -= len;
	while (1) {/* body search -- rfc822 begins body with bare LF */
	    register char *linepos;
	    
	    linepos = index(mfpos,LF);
	    if (linepos == 0) {/* bad format - kiss off */
	        badformat = 1;
		break;
	    } else
		  badformat = 0;
	    ++linepos;					/* skip LF */
	    len = linepos - mfpos;			/* line length */
	    mfpos = linepos;                            /* current line */
	    ntoscan -= len;                             /* dec. total bytes */
	    if (len == 1 || ntoscan <= 0) {/* Bare LF or bad file */
		break;
	    }
	}
	bodypos = mfpos - 1;                            /* start with LF */
	/*
	 *	Now point to next message
	 *	len will be the number of skipped chars
         *	to the first char of the next fromline.
	 */
	if (!badformat) {
	    lastmessage = 0;
	    mfpos = findfromline(mfpos,&lastmessage,ntoscan);
	} else {
	    lastmessage = 1;
	}
	/*
	 * make the MM header line
	 */
	size = mfpos - headerpos;        /* header of previous to next msg */
	/* 
	 *  less body size (-1 for LF that starts body and 
	 *  has already been deducted)
	 */
	ntoscan -= (mfpos - bodypos - 1);
	mfoffset += size;
	len = makeMMheader(fromline,headerline,&unixinternaldate,size);
	/*
	 *	Update our header records
	 */
	hp->Size_Of_Text_File += len + size;
	hp->Number_Of_Messages += 1;
        mp->memmap.inited = 0;
	mp->Size = len + size;
	mp->Header_Size = len;
	mp->Body_Offset = bodypos - headerpos + len;
	mp->Flags = 0;
	mp->Keyword_Flags = 0;
	mp->Date = unixinternaldate;
	/*
	 *	Now write the files themselves.
	 *		(1) Mail file
	 *		(2) Header file(Header and index record)
	 */
	nastyerror = 0;
	if (1 != fwrite(headerline,len,1,mbox->stream)) {
		nastyerror = 1;
		break; /*MM header */
	}
	if (1 != fwrite(headerpos,size,1,mbox->stream)) {/*  MM HDR */
		nastyerror = 1;
		break;  /* text */
	}
        if (fseek(mbox->istream,0,FROMENDOFFILE) < 0) {
		nastyerror = 1;
		break;
	}
	if (1 != fwrite(mp,sizeof(*mp),1,mbox->istream)) {/* MSGHDR */
		nastyerror = 1;
		break;	/* index */
	}
	/*
	 *	We write the Header each time to be cautious about 
	 *	system crashes.
	 */
	if (fseek(mbox->istream,0,FROMBEGINNINGOFFILE) < 0) {
		nastyerror = 1;
		break;
	}
	if (1 != fwrite(hp,sizeof(*hp),1,mbox->istream)) {/* HDRHDR */
		nastyerror = 1;
		break;/* Header */
	}
	if (lastmessage) break;
    }
    free(fmapptr);
    if (nastyerror) {
      logfilewrite("\nNASTYERROR while appending");
      return(SAVEPOINTER);
    }
    else {
      bytecopy(&headerBuf,hp,sizeof(HDRHDR));	  /* set global headerBuf */
      return(OK);
    }
  }
/*
 * Validate an MM header line(more or less)
 */
validMMheaderline(str)
  register char *str;
{
  char *stopper= index(str,',');
  register i;

  if (!stopper) return(0);			  /* expected a "," */
  *stopper = '\0';
  if (internaldate(str,0,0) < 0)
    return(0);		                          /* expected valid */
						  /* date */
  ++stopper;					  /* check for size */
  str = index(stopper,';');
  if (!str) return(0);				  /* expected ";" */
  i = str - stopper;				  /* n chars */
  while (i-- > 0)
    if (!isdigit(*stopper++)) return(0);	  /* expected n digits */
  return(1);
}
/*
 *	Make a MM header line from a fromline and return the length
 *	of the MM header line including the LF, and set the unixinternal
 *	date.
 *
 * 	MM header line: "10-Aug-87 13:19:32-PDT,501;000000404011 
 * 	From line: "From yeager Thu Feb 18 13:13:01 1988"
 *			        0123456789x123456789x123
 */
int timezonesecs,daylight;
makeMMheader(fromline,headerline,unixinternaltime,size)
  char 	*fromline,
	*headerline;
  int	*unixinternaltime,
	size;
  {
    int zonesecs,
	daylt;
    char *date,
	 *year,
	 *timezonep,
	 *hlstart= headerline;
    /*
     *	Should be a ctime fixed format.
     */
    year = fromline + strlen(fromline) - 2;		/* 2 lsdigs year */
    date = year - 22;
    
    *unixinternaltime = internaldate(date,&zonesecs,&daylt);
    if (*unixinternaltime < 0) {/* something amiss in from line date */
	date = (char *)tod();
	*unixinternaltime = getitime();
	zonesecs = timezonesecs;
	daylt = daylight;
    }
    /*
     *	We need the time zone
     */
    timezonep = (char *)timezone(zonesecs,daylt);
    /*
     *	Now build the MM header line
     * 	MM header line: "10-Aug-87 13:19:32-PDT,501;000000404011 
     * 	From line: "From yeager Thu Feb 18 13:13:01 1988"
     *			        0123456789x123456789x123
     */
     date += 4;						/* skip "Thu " */
     headerline = bytecopy(headerline,date+4,2);	/* 18 */
     *headerline++ = '-';				/* 18- */
     headerline = bytecopy(headerline,date,3);		/* 18-Feb */
     *headerline++ = '-';				/* 18-Feb- */
     headerline = bytecopy(headerline,year,2);		/* 18-Feb-88 */
     headerline = bytecopy(headerline,date+6,9); /* 18-Feb-88 13:13:01*/
     /*
      *	Finally the time zone.
      */
     if (nstrEqualCI("GMT",timezonep,3)) {/* GMT+/-hh:mm  eg GMT+4:30 */
	int hh,mm;

	timezonep += 3;
	*headerline++ = *timezonep++;			/* + or - */
	hh = atoi(timezonep);
	timezonep = index(timezonep,':');	
	if (!timezonep) mm = 0;
	else
	    mm = atoi(timezonep+1);
	sprintf(headerline,"%02d%02d",hh,mm);		/* HHMM */
	headerline += 4;				/* skip zone */
      } else {
	  *headerline++ = '-';
	  headerline = bytecopy(headerline,timezonep,3);
      }
      /*
       *	size and flags.
       */
      sprintf(headerline,",%d;000000000000\n",size);
#ifdef DEBUG
      fprintf(stderr,"\nMake MM header:\n\"%s\"-->\n\"%s\"",fromline,hlstart);
#endif
      return(strlen(hlstart));
  }

/*
 *	Look for "from " discarding anything we see.
 *	return number of discarded characters.
 *
 *      Noting that spooled mail files have been found
 *      with leading garbage bytes, we do a free scan until
 *      "from" is found and proceed from there expecting
 *      a valid format.
 */

flushtillfirstmsg(fpos,size)
  register char *fpos;
  register size;
  {
    register n= 0;
    register matchlen;
    register char *cpos= fpos,
                  c;

    matchlen = fromsplen+1;		  /* add in LF */
    while (1) {
	if (nstrEqualCI(fromsp,cpos,fromsplen)) break;
	cpos += 1;
        ++n;
	if ((size - n) <= matchlen) return(-1);
      }
    if (!index(cpos,LF)) {
      return(-1);
    }
    return(n);
  }
/*
 *	Here we read lines until we have the fromsp starting
 *	a line. We return the number of skipped chars. Also,
 *	we make sure that the fromline is terminated by a LF
 *      and that it has a desired format.
 *
 *	This may be paranoid but who can trust software anyhow?
 *	If we don't find such a line, we assume we have reached
 *	the end-of-the file.
 */
char *findfromline(mfpos,lastmessage,nleft)
  register char *mfpos;		/* current mail file position */
  int *lastmessage;			/* set to 1 if we come up empty */
  int nleft;				/* max chars to scan */
  {
    char *nfpos= mfpos,
	 *lfpos= mfpos;
    register countdown= nleft;

    while (1) {
	if (nstrEqualCI(fromsp,lfpos,fromsplen)) {/* check line */
	    if (reasonableFromline(lfpos)) {/* slight paranoa check */
	        break;
	    }
	}
	nfpos = index(lfpos,LF);		       /* END of this line */
	if (!nfpos) {/* Hit the NULL firewall */
	    *lastmessage = 1;
	    return(lfpos + strlen(lfpos));
	}
	++nfpos;                                        /* skip LF */
	countdown -= (nfpos - lfpos);	 		/* chars skipped */
	lfpos = nfpos;	                       /* set line file position */
	if (countdown <= 0) {
	    *lastmessage = 1;
	    break;
	}
    }
    return(lfpos);
  }

/*
 *	Close, delete, and then create a new spooled mail file
 *	with the remaining messages.
 */
saveRemainingMail(sml,bufptr,nbytes,name)
  {
    return;
  }
tryAndLockTheBeastUp(f)
  FILE *f;
{
  int lockcount = 0;
  while (!setexclusivelock(f)) {
    sleep(1);
    ++lockcount;
    if (lockcount == 10) {/* give up */
      return(0);
    }
  }
  return(1);
}
#define EXSECSTOWAIT 3
setexclusivelock(s)
  FILE *s;
  {
     int wait_time= EXSECSTOWAIT;
 
     while (wait_time-- > 0) 
       if (lock_file(fileno(s),LOCK_EX) < 0) {
	   sleep(1);
	   continue;
        } else
	    return(1);
     return(0);
  }
static lock_file (fd, lock_id)
int fd;
{
    int status;
#ifdef FCNTL_LOCKS
    struct flock flk;
    flk.l_type = (lock_id == LOCK_EX) ? F_WRLCK : F_RDLCK;
    flk.l_whence = 0;
    flk.l_start = 0;
    flk.l_len = 0;
    status = fcntl (fd, F_SETLK, &flk);
#else
    status = flock (fd, LOCK_NB|lock_id);
#endif
    return status;
}

#ifdef DEBUG
/*
 * Reads from a file called "debugbbd" on the TEST directory.
 * 
 */
readWriteInput(f)
  FILE *f;
{
  register c;
  FILE *df= fopen("debugbbd","r");
  register nbytes= 0;

  if (!df) return(0);
  while ((c = fgetc(df)) != EOF) 
    if (f) {
      fputc(c,f);
      ++nbytes;
    }
  fclose(df);
  return(nbytes);
}
#else
readWriteInput(f)
  FILE *f;
{
  register c;
  register nbytes= 0;

  while ((c = getchar()) != EOF) 
    if (f) {
      fputc(c,f);
      ++nbytes;
    }
  return(nbytes);
}
#endif
/*
 * 	From line: "From yeager Thu Feb 18 13:13:01 1988"
 *			        0123456789x123456789x123
 */
extern char day_tab[],month_tab[];
extern errono;
reasonableFromline(fl)
  char *fl;
  {
    char *endpos;
    register char *datepos;

    endpos = index(fl,LF);
    if (!endpos) return(0);
    datepos = endpos - 24;

    if (!match(day_tab,datepos,3) || !match(month_tab,datepos+4,3))
        return(0);
    if (*(datepos+13) != ':' || *(datepos+16) != ':') return(0);
    /*
     *  What more can we ask?
     */
    return(1);
  }   
debugsyswrite(s)
  char *s;
  {
    return;
  }
/*
 *  Create the index file from the mail file.
 */
typedef struct memblock {
  char *mem;
  int size;
} MEMBLOCK;
/*
 * We create a new one because the text file exists but
 * the index file does not.
 */
createIndexFile(mbox)
  register MAILBOX *mbox;
  {
    FILE *f= mbox->istream;
    HDRHDR hdr,
	   *hp= &hdr;
    MSGHDR mhdr,
	   *mp= &mhdr;
    register i;
    char headerline[256];
    int strangeness= 0,
	offset= 0;
    MEMBLOCK tmp;
    struct timeval tp;
    struct timezone tz;
    struct stat sbuf;
    int now;

    /*
     *  Should  have a stream here ...
     */
    if (!f) return(0);
    /*
     *	Clear hdr block
     */
    hp->Last_Read = getitime();			/* Assume current time */
    hp->Number_Of_Messages = 0;
    hp->Size_Of_Text_File = 0;
    /*
     *	position to first message header by writing null header.
     */
    if (1 != fwrite(hp,sizeof(*hp),1,f)) {
	fclose(f);
	unlink(mbox->iname);
	return(0);
    }
    /*
     *	So, we whip through the mail file creating header
     *	entries.
     */
    i = 0;
    if (fseek(mbox->stream,0,FROMBEGINNINGOFFILE) < 0) {
	fclose(f);
	unlink(mbox->iname);
	return(0);
    }
    gettimeofday(&tp,&tz);
    now = tp.tv_sec;
    tmp.mem = 0;
    while (1) {
	int hlen,
	    size;
	char *sp,*tp;
	int headersize;
	/*
	 *	This read includes the LF
	 */
	if (!fgets(headerline, 256, mbox->stream)) break;
	sp = index(headerline,',');
#ifdef DEBUG
	fprintf(stderr,"%s",headerline);
#endif
	if (!sp) {
	    strangeness = 1;
    	    break;				/* something strange */
	}
	tp = sp;
	++sp;					/* skip the comma */
	size = atoi(sp);			/* message size */
	if (!size) {
	    strangeness = 1;
    	    break;				/* something strange */
	}
	++hp->Number_Of_Messages;		/* one more message */
	hlen = strlen(headerline);		/* includes LF */
	hp->Size_Of_Text_File += hlen + size;	/* total file size */
	/*
	 *	Now the message header
	 */
	mp->Size = size + hlen;
	mp->memmap.inited = mp->memmap.size = 0;
	mp->Header = 0;
	mp->Header_Size = hlen;
	/* 
	 *	OK. Now read a line at a time looking for the text
	 *	of the message.
	 */
	if (!getMem(&tmp,size)) {
	  strangeness = 1;
	  break;
	}
	headersize = hlen;
	/*
	 *	Set pointer to NEXT message. We are now at the first
	 *	character after the MM header line.
	 */
	offset = size;
	while (1) {/* Look for text - everthing after first NULL line */
	    int len;
	    char *cp;

	    if (!fgets(tmp.mem,size,mbox->stream)) break;	/* EOF */
	    if (len = nullline(tmp.mem)) {
		offset -= len;
		break;					/* emptiness */
	    }
	    len = strlen(tmp.mem);			/* line len(inc. LF)*/
	    headersize += len;				/* total chars */
	    offset -= len;				/* for next seek */
	}
	if (offset < 0) {
	    strangeness = 1;
	    break;
	}
	mp->Body_Offset = headersize;
	/*
	 *	Do flags and keywords. IF we don't have flags kiss off.
	 */
	if (!(sp = index(sp,';'))) {
	    strangeness = 1;
	    break;
	}
	flagstringtoFlags(mp,sp+1);
	/*
	 * 	Do internal date.
	 */
	*tp = 0;
	mp->Date = internaldate(headerline,0,0);
	if (mp->Date < 0) mp->Date = 0;
	/*
	 *	write the header file
	 */
	if (1 != fwrite(mp,sizeof(*mp),1,f)) {
	    strangeness = 1;
	    break;
	}
	/*
	 *	Seek next message
	 */
        if (fseek(mbox->stream,offset,FROMCURRENTPOS) < 0) break;
    } 
    /*
     *	A paranoia check
     */
    fstat(fileno(mbox->stream),&sbuf);
    if (hp->Size_Of_Text_File != sbuf.st_size) strangeness = 1;
    /*
     *	Free the memory assigned to tmp
     */    
    free(tmp.mem);
    if (strangeness) {/* some warped condition */
	fclose(f);
	unlink(mbox->iname);
	return(0);
    }
    /*
     *	Write the updated header. Postion to start, close and open
     *  "r+"
     */
    if (fseek(f,0,FROMBEGINNINGOFFILE) < 0) return(0);
    if (1 != fwrite(hp,sizeof(*hp),1,f)) {
	fclose(f);
	unlink(mbox->iname);
	return(0);
    }
    /*
     *	Put index file at its tail end(the mail file is already
     *  at EOF).
     */
    if (fseek(mbox->istream,0,FROMENDOFFILE) < 0) return(0);
    return(1);
      }
static char *octaldigs= {"01234567"};
static char flipper[8]= {0,4,2,6,1,5,3,7};
/*
 *	Take the 12 asciz chars in the MM header line and
 *	turn them into a binary message header field.
 */
flagstringtoFlags(mp,flags)
  MSGHDR *mp;
  register char *flags;
  {
    register staticOctades= NSTATICFLAGS/3;
    register nexcess= NSTATICFLAGS % 3;
    register keyOctades= (36 - NSTATICFLAGS + nexcess)/3;
    register c,
    	     binf= 0,
	     i;
    register char *fp= flags;
    /*
     *	OK. So, first we read the keyword flags from left
     *	to right. They are then stashed from right to
     *  left in the header field. Essentially,
     *
     *	Bit -->	Key Bit
     *   36        0
     *   35	   1
     *       etc ...
     */
     for (i=0; i<keyOctades; ++i) {
	c = *fp++;
	c -= '0';				/* to octal */
	c = flipper[c];				/* invert them */
	binf |= c << (i * 3);
    }
    mp->Keyword_Flags = binf;
    /*
     *	Now for the static flags.
     */
     fp = flags + 11;				/* least sig. char */
     binf = 0;
     for (i=0; i<staticOctades; ++i) {
	c = *fp--;				/* the ascii */
	c -= '0';				/* the octal */
	binf |= c << (i * 3);			/* the binary */
    }
    mp->Flags = binf;
  }
#define NULLLEN 1
nullline(str)
  register char *str;
  {/*
    *	A null line is defined as a LF
    */
    char c = *str++;

    if (c == LF && *str == '\0') return(NULLLEN);
    else
	return(0);
  }
/*
 *  If the barber is in, then we trim the files.
 *      We remove the appropriate number of messages
 *      From the mtxt(mail.txt format).
 */
char logbuf[1024];
char *mtxttmp= "mtxtXXXXXX";
char *minxtmp= "minxXXXXXX";
#define MAXFROMS (4*BB_MAXMESSAGES)
char *fromers[MAXFROMS];
CheckTheBarberShop(mtxt,mfile,mfilename)
  MAILBOX *mtxt;
  FILE *mfile;
  char *mfilename;
{
  register nmsgs,
           nTrimed= 0;
  FILE *tmpm,*tmpi,*tmpu;
  register i,saveInx,saveSize;
  int badshave;
  char *thisLine;
  MSGHDR mhdr;
  MEMBLOCK mb;   
  struct stat sbuf;
  int offset;
  int shaveIt= 0;
  int newCount;
  char *reason;
  /*
   *  See if we are still within the limits ...
   */
  if ((nmsgs = headerBuf.Number_Of_Messages) >= BB_MAXMESSAGES) {
    shaveIt = 1;
    nTrimed = nmsgs - BB_TRIMEDMESSAGES;
    newCount = BB_TRIMEDMESSAGES;
    reason = "NMSGS";
  }
  if (headerBuf.Size_Of_Text_File >= MAXFILESIZE) {
    int blade= shaveSomeFuzz(mtxt->istream);
    shaveIt = 1;
    if (blade > nTrimed) {
      nTrimed = blade;
      reason = "BSIZE";
      newCount = nmsgs - nTrimed;
    }
  }
  if (!shaveIt) return(1);			  /* All went OK */
  /*
   *  Log this information with a time stamp.
   */
  sprintf(logbuf,"\nTrim[%s] %s of %d msgs on %s",reason,
	  mtxt->name,nTrimed,(char *)tod());
  logfilewrite(logbuf);
  /*
   *  Open tmp files for caching copies of saved mail.
   */
  mktemp(mtxttmp);				  /* make unique file name */
  unlink(mtxttmp);				  /* Just in case not unique */
  tmpm = fopen(mtxttmp,"a");
  if (!tmpm) return(0);
  mktemp(minxtmp);
  unlink(minxtmp);				  /* Just in case  ditto */
  tmpi = fopen(minxtmp,"a");
  if (!tmpi) {
    fclose(tmpm);
    unlink(mtxttmp);
    return(0);
  }
  /*
   *  Calculate seek position of first mtxt msg to be
   *  saved by flipping through the index file.
   */
  if (fseek(mtxt->istream,sizeof(HDRHDR),FROMBEGINNINGOFFILE) < 0) {
    logfilewrite("-->SEEK failed on index file");
    return(cleartmps(tmpm,mtxttmp,tmpi,minxtmp));
  }
  offset = 0;
  for (i=0; i<nTrimed; ++i) {
    if (1 != fread(&mhdr,sizeof(MSGHDR),1,mtxt->istream)) {
      logfilewrite("-->READ failed on index file");
      return(cleartmps(tmpm,mtxttmp,tmpi,minxtmp));
    }
    offset += mhdr.Size;
  }
  /*
   *  Now, seek the first kept message in the mail file.
   */
  if (fseek(mtxt->stream,offset,FROMBEGINNINGOFFILE) < 0) {
    logfilewrite("-->SEEK failed on text file");
    return(cleartmps(tmpm,mtxttmp,tmpi,minxtmp));
  }
   /*
    *  We are now pointing at the First message to KEEP.
    *  Set up the HDRHDR to reflect the new status
    */
  headerBuf.Number_Of_Messages = 0;
  headerBuf.Size_Of_Text_File = 0;
						  /* WRITE dummy hdrhdr */
  if (1 != fwrite(&headerBuf,sizeof(HDRHDR),1,tmpi)) {
    logfilewrite("-->WRITE failed on index file");
    return(cleartmps(tmpm,mtxttmp,tmpi,minxtmp));
  }
  /*
   *  Read each index record and message. Write them to tmp files
   *  and update the header fields.
   */
  mb.mem = 0;
  for (i=0; i<newCount; ++i) {
						  /* read the next quantum */
    if (1 != fread(&mhdr,sizeof(MSGHDR),1,mtxt->istream)) {
      logfilewrite("-->READ(1) failed on index file");
      return(cleartmps(tmpm,mtxttmp,tmpi,minxtmp));
    }
    if (!getMem(&mb,mhdr.Size)) {
      logfilewrite("-->MALLOC failed");
      return(cleartmps(tmpm,mtxttmp,tmpi,minxtmp));
    }
    if (1 != fread(mb.mem,mhdr.Size,1,mtxt->stream)) {
      logfilewrite("-->READ failed on text file");      
      return(cleartmps(tmpm,mtxttmp,tmpi,minxtmp));
    }
						  /* write to tmp files */
    if (1 != fwrite(&mhdr,sizeof(MSGHDR),1,tmpi)) {
      logfilewrite("-->WRITE failed on tmp index file");      
      return(cleartmps(tmpm,mtxttmp,tmpi,minxtmp));
    }
    if (1 != fwrite(mb.mem,mhdr.Size,1,tmpm)) {
      logfilewrite("-->WRITE failed on tmp text file");      
      return(cleartmps(tmpm,mtxttmp,tmpi,minxtmp));
    }
						  /* update header block */
    headerBuf.Number_Of_Messages += 1;
    headerBuf.Size_Of_Text_File += mhdr.Size;
  }
  /*
   *  Write new header to tmp index file.
   */
  if (fseek(tmpi,0,FROMBEGINNINGOFFILE) < 0) {
    logfilewrite("-->SEEK(1) failed on index file");      
    return(cleartmps(tmpm,mtxttmp,tmpi,minxtmp));
  }
  if (1 != fwrite(&headerBuf,sizeof(HDRHDR),1,tmpi)) {
    logfilewrite("-->WRITE(1) failed on index file");      
    return(cleartmps(tmpm,mtxttmp,tmpi,minxtmp));
  }
  /*
   *  Delete orignal files and rename tmps ...
   */
  fclose(mtxt->stream);
  fclose(mtxt->istream);
						  /* remove old files */
  unlink(mtxt->name);
  unlink(mtxt->iname);

  fclose(tmpm); fclose(tmpi);			  /* rename tmps */
  link(mtxttmp,mtxt->name);
  link(minxtmp,mtxt->iname);
  /*
   * Now chmods
   */
  chmod(mtxt->name,0664);
  chmod(mtxt->iname,0664);
						  /* remove tmps */
  unlink(mtxttmp);
  unlink(minxtmp);
						  /* reassign streams */
  mtxt->stream = fopen(mtxt->name,"a+");
  mtxt->istream = fopen(mtxt->name,"a+");
  free(mb.mem);
  return(1);
}

logfilewrite(str)
     char *str;
{
  if (logfile) {
    fputs(str,logfile);
    fflush(logfile);
  }
}

getMem(mb,size)
  register MEMBLOCK *mb;
  register size;
{  
  if (!mb->mem) {				  /* Never allocated */
    mb->size = size;				  /* get first allocation */
    mb->mem = (char *)malloc(size);
    if (!mb->mem) return(0);			  /* malloc failure */
    else
      return(1);
  }
  if (mb->size < size) {			  /* Cannot reuse */
    mb->size = size;
    mb->mem = (char *)realloc(mb->mem,size);
    if (!mb->mem) return(0);
  }
  return(1);
}

cleartmps(f1,t1,f2,t2)
  FILE *f1,*f2;
  char *t1,t2;
{
  if (f1) {
    fclose(f1);
    unlink(t1);
  }
  if (f2) {
    fclose(f2);
    unlink(t2);
  }
  return(0);
}
debugwrite(s) { return; }			  /* dummy for linking */
/*
  Chase through mail index file until we find the message which
  makes the file less than 1/2 the Max file size */
int shaveSomeFuzz(hfile)
     FILE *hfile;
{
  MSGHDR mh;

  int whiskers= 0,
      fileSize= headerBuf.Size_Of_Text_File;
  /*
     seek to first message block */
  if (fseek(hfile, sizeof(HDRHDR), FROMBEGINNINGOFFILE) < 0) return(0);
  /*
     Discover how many messages we must trim to leave MAXFILESIZE/2
     bytes */
  while (1) {
    if (1 != fread(&mh, sizeof(mh), 1, hfile)) return(whiskers);
    whiskers += 1;
    fileSize -= mh.Size;			  /* decrement file size */
    if (fileSize <= MAXFILESIZE/2 || 
	whiskers >= headerBuf.Number_Of_Messages) break;
  }
  return(whiskers);
}
