;;;
;;; CTCP DCC module for ZenIRC client. 
;;;

;;; Copyright (C) 1993, 1994 Ben A. Mesander

;;; Author: Ben A. Mesander <ben@gnu.ai.mit.edu>
;;; Maintainer: ben@gnu.ai.mit.edu
;;; Keywords: extensions
;;; Created: 1994/01/23

;;; $Id: zenirc-dcc.el,v 1.13 1994/05/13 05:07:35 ben Exp $

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, you can either send email to this
;;; program's maintainer or write to: The Free Software Foundation,
;;; Inc.; 675 Massachusetts Avenue; Cambridge, MA 02139, USA.

;;; Commentary:

;;; This is still very unfinished.

;;; Code:

(defvar zenirc-verbose-dcc t)
;; zenirc-dcc-list looks like:
;; (("from_nick!user@host" "SEND" proc_or_nil ip_address port size 
;; filename)
;;  ("from_nick!user@host" "CHAT" proc_or_nil ip_address port))
;; port is 0 if the size isn't known
(defvar zenirc-dcc-list '())
(defvar zenirc-command-dcc-hook '(zenirc-command-dcc) 
  "Hook variable for /dcc command")
(defvar zenirc-ctcp-query-DCC-hook '(zenirc-ctcp-query-DCC)
  "Hook variable for CTCP DCC queries")

;;
;; zenirc-ctcp-query-DCC is the function called when a CTCP DCC
;; request is detected by the client. It examines the DCC subcommand,
;; and either calls the appropriate routine for that subcommand, or
;; sends a ctcp errmsg to the sender.
;;
(defun zenirc-ctcp-query-DCC (proc parsedctcp from to)
  (let* ((cmd (car (zenirc-parse-firstword (cdr parsedctcp)))))
    (cond 
     ((string= cmd "SEND") (zenirc-handle-ctcp-send proc parsedctcp from to))
     ((string= cmd "CHAT") (zenirc-handle-ctcp-chat proc parsedctcp from to))
     (t (progn
	  (zenirc-ctcp-errmsg 
	   nil from to (concat "DCC " (cdr parsedctcp))
	   (concat cmd " Is not a DCC subcommand known to this client") proc)
	  (if zenirc-verbose-ctcp
	      (zenirc-display-string
	       proc (format "[info] unknown dcc subcommand %s from %s\n"
			    cmd (zenirc-run-hook 'zenirc-format-nickuserhost-hook from)))))))))

;;
;; zenirc-handle-ctcp-send is called when a CTCP DCC SEND subcommand is
;; sent to the client. It extracts the information about the dcc request,
;; and adds it to zenirc-dcc-list.
;;
(defun zenirc-handle-ctcp-send (proc parsedctcp from to)
  (let* (localfile filename ip port size (str (cdr parsedctcp)))
    ; DCC SEND requests must be sent to you, and you alone.
    (if (not (string= (downcase to) (downcase zenirc-nick)))
	(zenirc-display-string
	 proc
	 (format "[dcc] bogus dcc from user %s\n"
		 (zenirc-run-hook 'zenirc-format-nickuserhost-hook from)))
;                          filename    ip address   port         size (opt)
      (string-match "^SEND \\([^ ]+\\) \\([0-9]+\\) \\([0-9]+\\) *\\([0-9]*\\)"
		    str)
      (setq filename (substring str (match-beginning 1) (match-end 1)))
      (setq ip       (substring str (match-beginning 2) (match-end 2)))
      (setq port     (substring str (match-beginning 3) (match-end 3)))
      (setq size     (substring str (match-beginning 4) (match-end 4)))
      (zenirc-display-string
       proc
       ; a warning really should also be sent 
       ; if the ip address != the host the dcc sender is on.
       (format "[dcc] file %s offered by user %s (size %s) %s\n"
	       filename (zenirc-run-hook 'zenirc-format-nickuserhost-hook from)
	       (if (string= "" size) "unknown" size)
	       (if (< (string-to-int port) 1025)
		   (concat "WARNING! Possibly bogus request: privledged port "
			   port)
		 "")))
      (setq zenirc-dcc-list 
	    (cons 
	     (list from "SEND" nil (string-to-int ip) (string-to-int port) 
		   (string-to-int size) filename)
	     zenirc-dcc-list)))))
;;
;; return a string containing the hostname from an argument of the
;; form `nick!user@hostname'. If no `@' is found in the string,
;; return nil.
;;
(defun zenirc-extract-host (nickuserhost)
  (let ((posn (string-match "@" nickuserhost)))
    (if posn 
	(substring nickuserhost (1+ posn) (length nickuserhost))
      nil)))
;;
;; zenirc-get-file does the work of setting up a transfer from the remote
;; client to the local one over a tcp connection. This involves setting
;; up a process filter and a process sentinel, and making the connection.
;;
(defun zenirc-get-file (entry localfile)
  (save-excursion
    (let ((zenirc-dcc-buffer 
	   (generate-new-buffer (concat "ZenIRC-DCC-" localfile))))
      (switch-to-buffer zenirc-dcc-buffer)
      (make-local-variable 'zenirc-dcc-bytecount)
      (setq zenirc-dcc-bytecount 0)
      (make-local-variable 'zenirc-dcc-from)
      (setq zenirc-dcc-from (car entry))
      (make-local-variable 'zenirc-dcc-port)
      (setq zenirc-dcc-port (nth 4 entry))
      (make-local-variable 'zenirc-dcc-ip)
      (setq zenirc-dcc-ip (nth 3 entry))
      (make-local-variable 'zenirc-dcc-size)
      (setq zenirc-dcc-size (nth 5 entry))
      (make-local-variable 'zenirc-dcc-localfile)
      (setq zenirc-dcc-localfile localfile)
      (make-local-variable 'zenirc-dcc-process)
      (unwind-protect
	  (setq zenirc-dcc-process 
		(open-network-stream "zenirc-dcc" zenirc-dcc-buffer
				     (zenirc-extract-host zenirc-dcc-from)
				     zenirc-dcc-port))
	(if (not zenirc-dcc-process)
	    (zenirc-dcc-get-sentinel zenirc-dcc-process 
				     "connection refused.")
	  (set-process-buffer zenirc-dcc-process zenirc-dcc-buffer)
	  (set-process-filter zenirc-dcc-process 'zenirc-dcc-get-filter)
	  (set-process-sentinel zenirc-dcc-process 'zenirc-dcc-get-sentinel)
	  (setcar (nthcdr 2 entry) zenirc-dcc-process))))))
;;
;; This is the process filter for transfers from other clients to this one.
;; It reads incoming bytes from the network and stores them in the DCC buffer,
;; and sends back the replies after each block of data per the DCC protocol
;; spec. Well not really. We write back a reply after each read, rather than
;; every 1024 byte block, but nobody seems to care.
;;
(defun zenirc-dcc-get-filter (proc str)
  (let ((data (match-data)))
    (unwind-protect
	(progn
	  (save-excursion
	    (set-buffer (process-buffer proc))
	    (goto-char (point-max))
	    (insert str)
	    (set-marker (process-mark proc) (point-max))
	    (setq zenirc-dcc-bytecount (+ (length str) zenirc-dcc-bytecount))
	    (if zenirc-verbose-dcc
		(message "[dcc] %s: %d bytes recieved" zenirc-dcc-localfile
			 zenirc-dcc-bytecount))
	    (if (and (/= zenirc-dcc-size 0)
		     (> zenirc-dcc-bytecount zenirc-dcc-size))
		(progn
		  (message 
		   "[dcc] %s: file is longer than sender claimed. Aborting."
		   zenirc-dcc-localfile)
		  (delete-process proc)))
	    (process-send-string proc 
				 (zenirc-packed-int zenirc-dcc-bytecount 4))))
      (store-match-data data))))

;;
;; msa wrote this nifty little frob to convert an n-byte integer to a packed
;; string.
;;
(defun zenirc-packed-int (value count)
  (if (> count 0)
      (concat (zenirc-packed-int (/ value 256) (1- count))
	      (char-to-string (% value 256))) 
    ""))
;;
;; This is the process sentinel for CTCP DCC SEND connections.
;; It shuts down the connection and notifies the user that the
;; transfer is complete.
;;
(defun zenirc-dcc-get-sentinel (proc event)
  (let* ((dcc 0) (dccval (nth dcc zenirc-dcc-list)))
    (save-excursion
      (if proc
	  (progn
	    (set-buffer (process-buffer proc))
	    (delete-process proc)))
      (set-visited-file-name zenirc-dcc-localfile)
      (goto-char (point-min))
      ; remove entry from zenirc-dcc-list
      (while dccval
	(if (and (string= (nth 1 dccval) "SEND")        ; connection type
		 (eq (nth 3 dccval) zenirc-dcc-ip)      ; ip address
		 (eq (nth 4 dccval) zenirc-dcc-port))   ; port number
	    (progn
	      (setq zenirc-dcc-list (delq dccval zenirc-dcc-list))
	      (setq dcc (length zenirc-dcc-list)))) ; set up to terminate loop
	(setq dcc (1+ dcc) dccval (nth dcc zenirc-dcc-list)))
      (message "[dcc] file %s recieved in buffer: %s" zenirc-dcc-localfile
	       (buffer-name)))))
;;
;; parser for /dcc command. This figures out the dcc subcommand and calls
;; the appropriate routine to handle it.
;;
(defun zenirc-command-dcc (proc parsedcmd)
  (let ((subcommand (zenirc-parse-firstword (cdr parsedcmd))))
    (cond
     ((string= (car subcommand) "list")
      (zenirc-do-dcc-list proc subcommand))
     ((string= (car subcommand) "get")
      (zenirc-do-dcc-get proc subcommand))
     ((string= (car subcommand) "chat")
      (zenirc-do-dcc-chat proc subcommand))
     ((string= (car subcommand) "close")
      (zenirc-do-dcc-close proc subcommand))
     (t (zenirc-display-string proc (format "[dcc] no such subcommand: %s\n"
					    (car subcommand)))))))

;; /dcc close type nick
;; need to figure out if closing an active connect should remove buffers, etc.
;; associated with it. probably not - let the users do that if they want to.
(defun zenirc-do-dcc-close (proc subcommand) 
  (let* ((tmp (zenirc-parse-firstword (cdr subcommand)))
	 (type (upcase (car tmp))) (nick (cdr tmp)) (next 0) 
	 (rec (nth next zenirc-dcc-list)))
    (while rec
      (if (string= type (nth 1 rec))
	  (if (string= nick "")
					; just delete all of this type
	      (progn
		(if (nth 2 rec)
					; process is active
		    (delete-process (nth 2 rec)))
		(setq zenirc-dcc-list (zenirc-delete rec zenirc-dcc-list)))
					; make sure nick matches
	    (if (string= (downcase nick) 
			 (downcase (zenirc-extract-nick (car rec))))
		(progn
		  (if (nth 2 rec)
					; process is active
		      (delete-process (nth 2 rec)))
		  (setq zenirc-dcc-list 
			(zenirc-delete rec zenirc-dcc-list)))))
	(setq next (1+ next)))
      (setq rec (nth next zenirc-dcc-list)))))
    
;;
;; this is the handler for the /dcc list command - it lists the current state
;; of zenirc-dcc-list in an easy to read manner.
;;
(defun zenirc-do-dcc-list (proc parsedcmd)
  (let* ((next 0) (rec (nth next zenirc-dcc-list)))
    (zenirc-display-string proc "[dcc] from      type active size filename\n")
    (while rec
      (zenirc-display-string 
       proc (format "[dcc] %-9s %4s %-6s %4s %s\n" 
		    (zenirc-extract-nick (car rec))
		    (nth 1 rec) (if (nth 2 rec) "yes" "no") 
		    (if (string= (nth 1 rec) "SEND")
			(if (eq 0 (nth 5 rec)) "unkn" (int-to-string 
						       (nth 5 rec)))
		      "")
		    (if (string= (nth 1 rec) "SEND")		    
			(nth 6 rec)
		      "")))
      (setq next (1+ next))
      (setq rec (nth next zenirc-dcc-list)))))

;; /dcc get nick file
(defun zenirc-do-dcc-get (proc parsedcmd) 
  (let* ((next 0) (rec (nth next zenirc-dcc-list))
	 (tmp (zenirc-parse-firstword (cdr parsedcmd))) (nick (car tmp)) 
	 (filename (cdr tmp)) (localfile) (found nil))
    (while rec
      (if (and (string= (downcase (zenirc-extract-nick (car rec))) nick)
	       (string= (nth 1 rec) "SEND")
	       (string= (nth 6 rec) filename))
	  (progn
	    (setq found t)
	    (setq localfile
		  (read-string "[dcc] local filename: " filename))
	    (if (file-exists-p localfile)
		(if (yes-or-no-p 
		     (format "[dcc] file %s already exists. Overwrite? "
			     filename))
		    (zenirc-get-file rec localfile)
		  (message "[dcc] %s not accepted for dcc from %s." filename
			   (zenirc-extract-nick (car rec))))
	      (zenirc-get-file rec localfile))))
      (setq next (1+ next))
      (if found
	  (setq rec nil)
	(setq rec (nth next zenirc-dcc-list))))
    (if (not found)
	(message "[dcc] %s not found." filename))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; DCC CHAT handling
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun zenirc-handle-ctcp-chat (proc parsedctcp from to)
  (let* ((ip nil) (port nil) (next 0) (rec (nth next zenirc-dcc-list))
	 (already nil) (str (cdr parsedctcp)))
    ; DCC CHAT requests must be sent to you, and you alone.
    (if (not (string= (downcase to) (downcase zenirc-nick)))
	(zenirc-display-string
	 proc
	 (format "[dcc] bogus dcc chat from user %s\n"
		 (zenirc-run-hook 'zenirc-format-nickuserhost-hook from)))
;                               ip address   port
      (string-match "^CHAT chat \\([0-9]+\\) \\([0-9]+\\)" str)
      (setq ip       (substring str (match-beginning 1) (match-end 1)))
      (setq port     (substring str (match-beginning 2) (match-end 2)))
      (while rec
	(if (and (string= (downcase from) (downcase (car rec)))
		 (string= "CHAT" (nth 1 rec)))
	    (progn
	      (zenirc-display-string
	       proc (format 
		     "[dcc] Already have dcc chat request from %s, ignoring one just recieved.\n"
		       (zenirc-run-hook 'zenirc-format-nickuserhost-hook from)))
	      (setq already t)
	      (setq next (length zenirc-dcc-list))))
	(setq next (1+ next))
	(setq rec (nth next zenirc-dcc-list)))
      (if (not already)
	  (progn
	    (zenirc-display-string
	     proc
             ; a warning really should also be sent 
             ; if the ip address != the host the dcc sender is on.
	     (format "[dcc] chat offered by user %s %s\n"
		     (zenirc-run-hook 'zenirc-format-nickuserhost-hook from)
		     (if (< (string-to-int port) 1025)
			 (concat 
			  "WARNING! Possibly bogus request: privledged port "
			  port)
		       "")))
	    (setq zenirc-dcc-list 
		  (cons 
		   (list from "CHAT" nil (string-to-int ip) 
			 (string-to-int port))
		   zenirc-dcc-list)))))))

(defun zenirc-do-dcc-chat (proc subcommand)
  (let* ((next 0) (rec (nth next zenirc-dcc-list))
	 (nick (cdr subcommand)) (found nil))
    (while rec
      (if (and (string= (downcase (zenirc-extract-nick (car rec))) 
			(downcase nick))
	       (string= (nth 1 rec) "CHAT"))
	  (progn
	    (setq found t)
	    (zenirc-dcc-chat rec)))
      (setq next (1+ next))
      (setq rec (nth next zenirc-dcc-list)))
    (if (not found)
	(message "[dcc] chat request from %s not found." nick))))

(defun zenirc-dcc-chat (entry)
  (save-excursion
    (let* ((zenirc-dcc-buffer 
	    (generate-new-buffer (concat "ZenIRC-DCC-" (zenirc-extract-nick 
							(car entry))))))
      (switch-to-buffer zenirc-dcc-buffer)
      (make-local-variable 'zenirc-dcc-from)
      (setq zenirc-dcc-from (car entry))
      (make-local-variable 'zenirc-dcc-port)
      (setq zenirc-dcc-port (nth 4 entry))
      (make-local-variable 'zenirc-dcc-ip)
      (setq zenirc-dcc-ip (nth 3 entry))
      (make-local-variable 'zenirc-partialline) ; this is not right
      (setq zenirc-partialline "")              ; see zenirc-mail folder
      (make-local-variable 'zenirc-dcc-process)
      (unwind-protect
	  (setq zenirc-dcc-process 
		(open-network-stream "zenirc-dcc" zenirc-dcc-buffer
				     (zenirc-extract-host zenirc-dcc-from)
				     zenirc-dcc-port))
	(if (not zenirc-dcc-process)
	    (zenirc-dcc-chat-sentinel zenirc-dcc-process
				      "connection refused.")
	  (set-marker (process-mark zenirc-dcc-process) (point-max))
	  (set-process-buffer zenirc-dcc-process zenirc-dcc-buffer)
	  (set-process-filter zenirc-dcc-process 'zenirc-dcc-chat-filter)
	  (set-process-sentinel zenirc-dcc-process 'zenirc-dcc-chat-sentinel)
	  (setcar (nthcdr 2 entry) zenirc-dcc-process)
	  (zenirc-dcc-mode))))))

(defun zenirc-dcc-chat-filter (proc str)
  (let ((data (match-data)))
    (unwind-protect
	(progn
	  (save-excursion
	    (set-buffer (process-buffer proc))
	    (setq zenirc-partialline (zenirc-dcc-parselines
				      proc str zenirc-partialline))))
      (store-match-data data))))

(defun zenirc-dcc-parselines (proc string zenirc-partialline)
  (let ((clntstr (concat zenirc-partialline string)) eol clntline)
    (while (setq eol (string-match "\n" clntstr))
      (setq clntline (substring clntstr 0 eol))
      (setq clntstr (substring clntstr (1+ eol) (length clntstr)))
      (zenirc-display-string 
       proc (format "=%s= %s\n" (zenirc-run-hook 
				 'zenirc-format-nickuserhost-hook 
				 zenirc-dcc-from)
		    clntline)))))

(defvar zenirc-dcc-mode-map '()
  "Sparse keymap for zenirc-dcc-mode")
(cond ((not zenirc-dcc-mode-map)
       (setq zenirc-dcc-mode-map (make-sparse-keymap))
       (define-key zenirc-dcc-mode-map "\n" 'zenirc-dcc-send-line)
       (define-key zenirc-dcc-mode-map "\C-m" 'zenirc-dcc-send-line)))

(defun zenirc-dcc-mode ()
  "Major mode for wasting time via DCC chat."
  (interactive)
  (setq mode-name "ZenIRC-DCC")
  (setq major-mode 'zenirc-dcc-mode)
  (use-local-map zenirc-dcc-mode-map))

(defun zenirc-dcc-send-line ()
  "Send current line to other client."
  (interactive)
  (end-of-line)
  (insert "\n")
  (let* ((proc (get-buffer-process (current-buffer)))
	 (proc-mark (process-mark proc))
	 (string (buffer-substring proc-mark (point))))
    (set-marker proc-mark (point))
    (process-send-string proc string)))

(defun zenirc-dcc-chat-sentinel (proc event)
  (let* ((dcc 0) (dccval (nth dcc zenirc-dcc-list)))
    (save-excursion
      (if proc 
	  (progn
	    (set-buffer (process-buffer proc))
	    (delete-process proc)))
      (goto-char (point-min))
      ; remove entry from zenirc-dcc-list
      (while dccval
	(if (and (string= (nth 1 dccval) "CHAT")        ; connection type
		 (eq (nth 3 dccval) zenirc-dcc-ip)      ; ip address
		 (eq (nth 4 dccval) zenirc-dcc-port))   ; port number
	    (progn
	      (setq zenirc-dcc-list (delq dccval zenirc-dcc-list))
	      (setq dcc (length zenirc-dcc-list)))) ; set up to terminate loop
	(setq dcc (1+ dcc) dccval (nth dcc zenirc-dcc-list)))
      (message "[dcc] chat with %s ended: %s" (zenirc-run-hook
					       'zenirc-format-nickuserhost-hook
					       zenirc-dcc-from) event))))

;;; zenirc-dcc.el ends here
