/*
    Filter Manager - Management software for the Drawbridge package
    Copyright (C) 1993 David K. Hess, Douglas Lee Schales, David R. Safford

    Please see the file `COPYING' for the complete copyright notice.

    comm.c - Version 1.0 - 4/21/93
*/

/*
 *  Communications interface file.
 */
#include "fm.h"

struct ether_addr currTarget;
unsigned char currKey[128];

/* 
 * Buffer for an acknowledgement.
 */
static unsigned char acknowledgement[1536];
static unsigned char message[1536];
static unsigned long fmSeq;
static unsigned long filterSeq;
static int nitFd;

/*
 * This function initializes the communications interface.
 */
void initCommunications(char *interface)
{
    nitFd = nit_open(interface,ETHERPROTO);

    if (nitFd == -1) {
	fprintf(messages,"Could not open nit interface\n");
	exit(1);
    }
}

/* 
 * This starts the sequence number sending in preparation for a series
 *   of messages to the current target.
 */
int startTransaction(void)
{
    SyncPacket *packet;
    SyncPacket *replyPacket;
    DESauth tempAuth;
    ErrorPacket *errorPacket;
    struct timeval waitTime;
    fd_set nitSet;
    int result;
    int retries;

    packet = (SyncPacket *) message;

    /*
     * Send a SYNC and wait for a SYNCACK
     */
    filterSeq = 0;
    fmSeq = rand();

    tempAuth.filterSeq = 0;
    tempAuth.fmSeq = fmSeq;

    longSwap((unsigned long *)&tempAuth,2);

    memcpy(packet->header.edst,&currTarget,sizeof(currTarget));
    packet->header.etype = ETHERPROTO;
    packet->header.type  = FM_M_SYNC;
    packet->header.flags = keySet;

    /*
     * Encrypt the sequence numbers if in that mode.
     */
    if (keySet == YES)
        encrypt((unsigned char *)&packet->header.auth,
	        (unsigned long *)currKey,
		(unsigned char *)&tempAuth);
    else
	packet->header.auth = tempAuth;

    /*
     * Send the packet.
     */
    result = nit_write(nitFd,(char *) packet,60);

    if (result < 0) {
	fprintf(messages,"nit write: %s\n",sys_errlist[errno]);
	return -1;
    }

    /* 
     * Wait for the response.
     */
    FD_ZERO(&nitSet);
    FD_SET(nitFd,&nitSet);

    waitTime.tv_sec = TIMEOUT;
    waitTime.tv_usec = 0;

    retries = 0;

    while (retries < RETRIES) {
	result = select(getdtablesize(),
			&nitSet,
			(fd_set *) NULL,
			(fd_set *) NULL,
			&waitTime);
	
	if (result == 0) {
	    result = nit_write(nitFd,(char *) packet,60);

	    if (result < 0) {
		fprintf(messages,"nit write: %s\n",sys_errlist[errno]);
		return -1;
	    }

	    ++retries;
	}
	else {
	    /*
	     * Packet must have arrived.
	     */
	    result = read(nitFd,acknowledgement,sizeof(acknowledgement));

	    if (result < 0) {
		fprintf(messages,"nit read: %s\n",sys_errlist[errno]);
		return -1;
	    }
	    else {
		/*
		 * Check the packet.
		 */
		replyPacket = (SyncPacket *) acknowledgement;

		/*
		 * Make sure the packet is coming from the filter and that it is the correct
		 *  protocol. Silently drop any that are not.
		 */
		if (memcmp(replyPacket->header.esrc,
			   (char *)&currTarget,
			   sizeof(currTarget)) == 0 &&
		    replyPacket->header.etype == ETHERPROTO) {
		    if (replyPacket->header.type == FM_M_SYNCACK) {
			/*
			 * SYNC succeeded. Pull out the sequence numbers.
			 */
			if (keySet)
			    decrypt((unsigned char *)&tempAuth,
				    (unsigned long *)currKey,
				    (unsigned char *)&replyPacket->header.auth);
			else
			    tempAuth = replyPacket->header.auth;

			longSwap((unsigned long *) &tempAuth,2);

			/* 
			 * Make sure that the sequence number coming back is ok.
			 */
			if (tempAuth.fmSeq != fmSeq + 1) {
			    fprintf(messages,"filter authentication failed: bad sequence number\n");
			    return -1;
			}

			++fmSeq;
			filterSeq = tempAuth.filterSeq;

			break;
		    }
		    else if (replyPacket->header.type == FM_M_ERROR) {
			/*
			 * We had some kind of error. Print it and return.
			 */
			errorPacket = (ErrorPacket *) replyPacket;
			printError(errorPacket->errorCode,output);
			return -1;
		    }
		}
	    }
	}

	/* 
	 * Set up for the next select call.
	 */
	FD_ZERO(&nitSet);
	FD_SET(nitFd,&nitSet);
	
	waitTime.tv_sec = TIMEOUT;
	waitTime.tv_usec = 0;
    }

    if (retries == RETRIES) {
	fprintf(messages,"target is not responding\n");
	return -1;
    }

    return 0;
}

/*
 *  This function sends a message to the current target and returns the
 *    acknowledgement in a static buffer (which will be overwritten by
 *    the next call).
 */
unsigned char *sendMessage(unsigned char type,void *data,unsigned long size)
{
    DESauth tempAuth;
    fd_set nitSet;
    FiltHeader *filtHeader;
    FiltHeader *replyFiltHeader;
    ErrorPacket *errorPacket;
    struct timeval waitTime;
    int result;
    int retries;
    int packetLength;

    filtHeader = (FiltHeader *) message;

    /*
     * Setup the sequence numbers (these were established in the previous
     *   startTransaction() call.
     */
    tempAuth.filterSeq = ++filterSeq;
    tempAuth.fmSeq     = ++fmSeq;

    longSwap((unsigned long *)&tempAuth,2);
 
    memcpy(filtHeader->edst,&currTarget,sizeof(currTarget));
    filtHeader->etype = ETHERPROTO;
    filtHeader->type  = type;
    filtHeader->flags = keySet;

    /*
     * Encrypt the sequence numbers if in that mode.
     */
    if (keySet == YES)
        encrypt((unsigned char *)&filtHeader->auth,
	        (unsigned long *)currKey,
		(unsigned char *)&tempAuth);
    else
	filtHeader->auth = tempAuth;

    /*
     * Copy the data portion of the packet over.
     */
    if (size)
	memcpy(message + sizeof(FiltHeader),data,size);

    /*
     * Send the packet.
     */
    packetLength = sizeof(FiltHeader) + size < 60 ? 60 : 
	                                            sizeof(FiltHeader) + size;

    result = nit_write(nitFd,(char *) message,packetLength);

    if (result == -1) {
	fprintf(messages,"nit write: %s\n",sys_errlist[errno]);
	return (char *) NULL;
    }

    /* 
     * Wait for the response.
     */
    FD_ZERO(&nitSet);
    FD_SET(nitFd,&nitSet);

    waitTime.tv_sec = TIMEOUT;
    waitTime.tv_usec = 0;

    retries = 0;

    while (retries < RETRIES) {
	result = select(getdtablesize(),
			&nitSet,
			(fd_set *) NULL,
			(fd_set *) NULL,
			&waitTime);
	
	if (result == 0) {
	    /*
             * Timed out so send the request again.
             */
	    if (nit_write(nitFd,(char *) message,packetLength) == -1) {
		fprintf(messages,"nit write: %s\n",sys_errlist[errno]);
		return (char *) NULL;
	    }

	    ++retries;
	}
	else {
	    /*
	     * Packet must have arrived.
	     */
	    result = read(nitFd,acknowledgement,sizeof(acknowledgement));

	    if (result <= 0) {
		fprintf(messages,"nit read: %s\n",sys_errlist[errno]);
		return (char *) NULL;
	    }
	    else {
		/*
		 * Check the packet. Silently drop any packets that do not have the
		 *   correct target address or the wrong protocol.
		 */
		replyFiltHeader = (FiltHeader *) acknowledgement;

		if (memcmp(replyFiltHeader->esrc,
			   (char *)&currTarget,
			   sizeof(currTarget)) == 0 &&
		    replyFiltHeader->etype == ETHERPROTO) {
		    
		    /*
		     * Check the sequence numbers to see if this is a valid
		     *   reply.
		     */
		    if (replyFiltHeader->type == FM_M_ERROR) {
			/*
			 * We had some kind of error.
			 */
			errorPacket = (ErrorPacket *) replyFiltHeader;
			printError(errorPacket->errorCode,output);
			return (char *) NULL;
		    }

		    /*
		     * Packet was ok.
		     */
		    if (keySet)
			decrypt((unsigned char *)&tempAuth,
				(unsigned long *)currKey,
				(unsigned char *)&replyFiltHeader->auth);
		    else
			tempAuth = replyFiltHeader->auth;

		    longSwap((unsigned long *) &tempAuth,2);

		    /* 
		     * Make sure that the sequence number coming back is ok.
		     */
		    if (tempAuth.fmSeq != fmSeq + 1 || tempAuth.filterSeq != filterSeq + 1) {
			fprintf(messages,"filter authentication failed: bad sequence number\n");
			return (char *) NULL;
		    }

		    /*
		     * Update the sequence numbers.
		     */
		    ++fmSeq;
		    ++filterSeq;
		    
		    break;
		}
	    }
	}

	/* 
	 * Set up for the next select call.
	 */
	FD_ZERO(&nitSet);
	FD_SET(nitFd,&nitSet);
	
	waitTime.tv_sec = TIMEOUT;
	waitTime.tv_usec = 0;
    }

    if (retries == RETRIES) {
	fprintf(messages,"target is not responding\n");
	return (char *) NULL;
    }

    return acknowledgement;
}
