/*
 * snmpdelta.c - Monitor delta's in integer valued SNMP variables
 *
 */
/***********************************************************************
	Copyright 1988, 1989, 1991, 1992 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
************************************************************************/
#include <sys/types.h>
#include <netinet/in.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>


#include "snmp.h"
#include "asn1.h"
#include "snmp_impl.h"
#include "snmp_api.h"
#include "snmp_client.h"
#include "party.h"

extern int  errno;
int	snmp_dump_packet = 0;

log(file, message)
    char *file;
    char *message;
{
    FILE *fp;
    long timeofday;
    char buf[30];

    fp = fopen(file, "a");
    if (fp == NULL){
        fprintf(stderr, "Couldn't open %s\n", file);
        return;
    }
    fprintf(fp, "%s\n", message);
    fclose(fp);
}

usage() {
    printf("usage: snmpdelta gateway-name srcParty dstParty [-f] [-s] [-t] [-S] [-p period] object-identifier [object-identifier ...]\n");
}

main(argc, argv)
    int	    argc;
    char    *argv[];
{
    struct snmp_session session, *ss;
    struct snmp_pdu *pdu, *response;
    struct variable_list *vars;
    int	arg;
    char *gateway = NULL;
    char *community = NULL;
    int	count, current_name = 0;
    char *names[128];
    long values[128];
    float maxes[128];
    long value;
    float printvalue;
    int period = 1;
    int deltat = 0, timestamp = 0, fileout = 0, dosum = 0, printmax = 0;
    long last_time = 0;
    long this_time;
    long delta_time;
    long sum;
    float printsum, maxprintsum;
    char *cp, buf[256];
    oid name[MAX_NAME_LEN];
    int name_length;
    struct timeval tv;
    struct tm tm;
    char timestring[64], label[256], outstr[256];
    int status;

    int port_flag = 0;
    int dest_port = 0;
    oid src[MAX_NAME_LEN], dst[MAX_NAME_LEN];
    int srclen = 0, dstlen = 0;
    struct partyEntry *pp;

    
    init_mib();
    /*
     * usage: snmpdelta gateway-name srcParty dstParty [-f] [-s] [-t] [-S] [-p period]
     variable list ..
     */
    for(arg = 1; arg < argc; arg++){
	if (argv[arg][0] == '-'){
	    switch(argv[arg][1]){
	      case 'd':
		snmp_dump_packet++;
		break;
	      case 'p':
		period = atoi(argv[++arg]);
		break;
	      case 't':
		deltat = 1;
		break;
	      case 's':
		timestamp = 1;
		break;
	      case 'S':
		dosum = 1;
		break;
	      case 'm':
		printmax = 1;
		break;
	      case 'f':
		fileout = 1;
		break;
	      default:
		printf("invalid option: -%c\n", argv[arg][1]);
		break;
	    }
	    continue;
	}
	if (gateway == NULL){
	    gateway = argv[arg];
	} else if (srclen == 0){

	    if (!read_party_database("/etc/party.conf")){
		fprintf(stderr,
			"Couldn't read party database from /etc/party.conf\n");
		exit(0);
	    }
	    if (!read_acl_database("/etc/acl.conf")){
		fprintf(stderr,
			"Couldn't read access control database from /etc/acl.conf\n");
		exit(0);
	    }
	    party_scanInit();
	    for(pp = party_scanNext(); pp; pp = party_scanNext()){
		if (!strcasecmp(pp->partyName, argv[arg])){
		    srclen = pp->partyIdentityLen;
		    bcopy(pp->partyIdentity, src, srclen * sizeof(oid));
		    break;
		}
	    }
	    if (!pp){
		srclen = MAX_NAME_LEN;
		if (!read_objid(argv[arg], src, &srclen)){
		    printf("Invalid source party: %s\n", argv[arg]);
		    srclen = 0;
		    usage();
		    exit(1);
		}
	    }
	} else if (dstlen == 0){
	    dstlen = MAX_NAME_LEN;
	    party_scanInit();
	    for(pp = party_scanNext(); pp; pp = party_scanNext()){
		if (!strcasecmp(pp->partyName, argv[arg])){
		    dstlen = pp->partyIdentityLen;
		    bcopy(pp->partyIdentity, dst, dstlen * sizeof(oid));
		    break;
		}
	    }
	    if (!pp){
		if (!read_objid(argv[arg], dst, &dstlen)){
		    printf("Invalid destination party: %s\n", argv[arg]);
		    dstlen = 0;
		    usage();
		    exit(1);
		}
	    }
	} else {
	    names[current_name++] = argv[arg];
	}
    }
    
    if (!(gateway && srclen && dstlen && current_name > 0)){
	usage();
	exit(1);
    }
    
    bzero((char *)&session, sizeof(struct snmp_session));
    session.peername = gateway;

    session.version = 2;

    session.srcParty = src;
    session.srcPartyLen = srclen;
    session.dstParty = dst;
    session.dstPartyLen = dstlen;
    

    session.retries = SNMP_DEFAULT_RETRIES;
    session.timeout = SNMP_DEFAULT_TIMEOUT;
    session.authenticator = NULL;
    snmp_synch_setup(&session);
    ss = snmp_open(&session);
    if (ss == NULL){
	printf("Couldn't open snmp\n");
	exit(-1);
    }
    
    for(count = 0; count < current_name; count++){
	values[count] = 0;
	maxes[count] = 0;
    }
    maxprintsum = 0;
    
    while(1){
	pdu = snmp_pdu_create(GET_REQ_MSG);
	
	if (deltat){
	    name_length = MAX_NAME_LEN;
	    if (!read_objid(".iso.org.dod.internet.mgmt.mib.system.sysUptime.0", name, &name_length)){
		printf("Invalid object identifier: %s\n", "system.sysUptime.0");
	    }
	    snmp_add_null_var(pdu, name, name_length);
	}	
	for(count = 0; count < current_name; count++){
	    name_length = MAX_NAME_LEN;
	    if (!read_objid(names[count], name, &name_length)){
		printf("Invalid object identifier: %s\n", names[count]);
	    }
	    
	    snmp_add_null_var(pdu, name, name_length);
	}
	
retry:
	status = snmp_synch_response(ss, pdu, &response);
	if (status == STAT_SUCCESS){
	    if (response->errstat == SNMP_ERR_NOERROR){
		vars = response->variables;
		if (deltat){
		    if (!vars){
			printf("Missing variable in reply\n");
		    } else {
			delta_time = *(vars->val.integer) - last_time;
			this_time = *(vars->val.integer);
		    }
		    vars = vars->next_variable;
		} else {
		    this_time = 1;
		}
		sum = 0;
		for(count = 0; count < current_name; count++,
		    vars = vars->next_variable){
		    if (!vars){
			printf("Missing variable in reply\n");
			break;
		    }
		    value = *(vars->val.integer) - values[count];
		    values[count] = *(vars->val.integer);
		    if (last_time == 0)
			continue;
		    sum += value;

		    if (deltat)
			printvalue = ((float)value * 100) / delta_time;
		    else
			printvalue = value;
		    if (printvalue > maxes[count]){
			maxes[count] = printvalue;
		    }

		    sprint_objid(buf, vars->name, vars->name_length);
		    for(cp = buf; *cp; cp++)
			;
		    while(cp >= buf){
			if (isalpha(*cp))
			    break;
			cp--;
		    }
		    while(cp >= buf){
			if (*cp == '.')
			    break;
			cp--;
		    }
		    cp++;
		    if (cp < buf)
			cp = buf;

		    if (timestamp){
			gettimeofday(&tv, (struct timezone *)0);
			bcopy(localtime(&tv.tv_sec), &tm, sizeof(tm));
			sprintf(timestring, "%d:%02d:%02d %d/%d", tm.tm_hour,
				tm.tm_min, tm.tm_sec, tm.tm_mon+1, tm.tm_mday);
			sprintf(label, "[%s] %s", timestring, cp);
		    } else {
			sprintf(label, "%s", cp);
		    }

		    if (deltat){
			if (printmax)
			    sprintf(outstr, "%s /sec: %.2f (Max: %.2f)",
				    label,
				    printvalue, maxes[count]);
			else
			    sprintf(outstr, "%s /sec: %.2f", label,
				    printvalue);
		    } else {
			if (printmax)
			    sprintf(outstr, "%s: %.0f (Max: %.0f)", label,
				    printvalue, maxes[count]);
			else
			    sprintf(outstr, "%s: %.0f", label, printvalue);
		    }

		    if (fileout){
			log(cp, outstr);
		    } else {
			printf("%s\n", outstr);
			fflush(stdout);
		    }
		}
		if (dosum && last_time){
		    if (deltat)
			printsum = ((float)sum * 100) / delta_time;
		    else
			printsum = sum;
		    if (printsum > maxprintsum)
			maxprintsum = printsum;

		    if (timestamp){
			gettimeofday(&tv, (struct timezone *)0);
			bcopy(localtime(&tv.tv_sec), &tm, sizeof(tm));
			sprintf(timestring, "%d:%02d:%02d %d/%d", tm.tm_hour,
				tm.tm_min, tm.tm_sec, tm.tm_mon+1, tm.tm_mday);
			sprintf(label, "[%s] %s", timestring, "Sum");
		    } else {
			sprintf(label, "%s", "Sum");
		    }
		    if (deltat){
			if (printmax)
			    sprintf(outstr, "%s /sec: %.2f (Max: %.2f)",
				    label,
				    printsum, maxprintsum);
			else
			    sprintf(outstr, "%s /sec: %.2f", label,
				    printsum);
		    } else {
			if (printmax)
			    sprintf(outstr, "%s: %.0f (Max: %.0f)", label,
				    printsum, maxprintsum);
			else
			    sprintf(outstr, "%s: %.0f", label, printsum);
		    }

		    if (fileout){
			log("Sum", outstr);
		    } else {
			printf("%s\n", outstr);
			fflush(stdout);
		    }
		}
	    } else {
		printf("Error in packet.\nReason: %s\n", snmp_errstring(response->errstat));
		if (response->errstat == SNMP_ERR_NOSUCHNAME){
		    printf("This name doesn't exist: ");
		    for(count = 1, vars = response->variables; vars && count != response->errindex;
			vars = vars->next_variable, count++)
			;
		    if (vars)
			print_objid(vars->name, vars->name_length);
		    printf("\n");
		}
		exit(1);
		if ((pdu = snmp_fix_pdu(response, GET_REQ_MSG)) != NULL)
		    goto retry;
	    }
	    
	} else if (status == STAT_TIMEOUT){
	    printf("No Response from %s\n", gateway);
	} else {    /* status == STAT_ERROR */
	    printf("An error occurred, Quitting\n");
	    break;
	}
	
	if (response)
	    snmp_free_pdu(response);
	last_time = this_time;
	sleep(period);
    }
    snmp_close(ss);
}

