/*
 * snmpd.c - send snmp GET requests to a network entity.
 *
 */
/***********************************************************
	Copyright 1988, 1989 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/
#include <sys/types.h>
#include <netinet/in.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <errno.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <nlist.h>

#include "snmp.h"
#include "asn1.h"
#include "snmp_impl.h"
#include "snmp_api.h"
#include "m2m.h"
#include "party.h"
#include "alarm.h"

u_long get_myaddr();

extern int  errno;
int	snmp_dump_packet = 0;

#ifndef FD_SET
#include <sys/param.h>
typedef long    fd_mask;
#define NFDBITS (sizeof(fd_mask) * NBBY)        /* bits per mask */

#define FD_SET(n, p)    ((p)->fds_bits[(n)/NFDBITS] |= (1 << ((n) % NFDBITS)))
#define FD_CLR(n, p)    ((p)->fds_bits[(n)/NFDBITS] &= ~(1 << ((n) % NFDBITS)))
#define FD_ISSET(n, p)  ((p)->fds_bits[(n)/NFDBITS] & (1 << ((n) % NFDBITS)))
#define FD_ZERO(p)      bzero((char *)(p), sizeof(*(p)))

#endif

main(argc, argv)
    int	    argc;
    char    *argv[];
{
    int	arg;
    int sd, sdlist[32], portlist[32], sdlen = 0, index;
    struct sockaddr_in	me;
    int port_flag = 0;
    u_short dest_port = 0;
    struct partyEntry *pp;
    u_long myaddr;


    /*
     * usage: snmpd
     */
    for(arg = 1; arg < argc; arg++){
	if (argv[arg][0] == '-'){
	    switch(argv[arg][1]){
		case 'd':
		    snmp_dump_packet++;
		    break;
                case 'p':
                    port_flag++;
                    dest_port = atoi(argv[++arg]);
                    break;
		default:
		    printf("invalid option: -%c\n", argv[arg][1]);
		    break;
	    }
	    continue;
	}
    }
    init_snmp();
/*    agent_party_init(); */
    init_mib();
    if (!read_party_database("/etc/party.conf")){
	fprintf(stderr, "Couldn't read party database from /etc/party.conf\n");
	exit(0);
    }
    if (!read_acl_database("/etc/acl.conf")){
	fprintf(stderr, "Couldn't read acl database from /etc/acl.conf\n");
	exit(0);
    }
    if (!read_view_database("/etc/view.conf")){
	fprintf(stderr, "Couldn't read view database from /etc/view.conf\n");
	exit(0);
    }
    
    myaddr = get_myaddr();

    party_scanInit();
    printf("Opening ports: ");
    fflush(stdout);
    for(pp = party_scanNext(); pp; pp = party_scanNext()){
	if ((pp->partyTDomain != DOMAIN1351
	     && pp->partyTDomain != DOMAINSMPUDP)
	    || bcmp((char *)&myaddr, pp->partyTAddress, 4))
	    continue;	/* don't listen for non-local parties */
	
	dest_port = 0;
	bcopy(pp->partyTAddress + 4, &dest_port, 2);
	for(index = 0; index < sdlen; index++)
	    if (dest_port == portlist[index])
		break;
	if (index < sdlen)  /* found a hit before the end of the list */
	    continue;
	printf("%u ", dest_port);
	fflush(stdout);
	/* Set up connections */
	sd = socket(AF_INET, SOCK_DGRAM, 0);
	if (sd < 0){
	    perror("socket");
	    return 1;
	}
	me.sin_family = AF_INET;
	me.sin_addr.s_addr = INADDR_ANY;
	/* already in network byte order (I think) */
	me.sin_port = dest_port;
	if (bind(sd, (struct sockaddr *)&me, sizeof(me)) != 0){
	    perror("bind");
	    return 2;
	}
	sdlist[sdlen] = sd;
	portlist[sdlen] = dest_port;
	if (++sdlen == 32){
	    printf("No more sockets... ignoring rest of file\n");
	    break;
	}	
    }
    printf("\n");
/*    snmp_trap(sd, (u_long)0x8002E804, 2, 0); */
    receive(sdlist, sdlen);
    return 0;
}

receive(sdlist, sdlen)
    int sdlist[];
    int sdlen;
{
    int numfds, index;
    fd_set fdset;
    struct timeval  timeout, *tvp = &timeout;
    int count, block;

    while(1){
	numfds = 0;
	FD_ZERO(&fdset);
	tvp->tv_sec = 0;
	tvp->tv_usec = 500000L;
	for(index = 0; index < sdlen; index++){
	    if (sdlist[index] + 1 > numfds)
		numfds = sdlist[index] + 1;
	    FD_SET(sdlist[index], &fdset);
	}
        block = 0;
        snmp_select_info(&numfds, &fdset, tvp, &block);
        if (block == 1)
            tvp = NULL; /* block without timeout */
	count = select(numfds, &fdset, 0, 0, tvp);
	if (count > 0){
	    for(index = 0; index < sdlen; index++){
		if(FD_ISSET(sdlist[index], &fdset)){
		    snmp_read_packet(sdlist[index]);
		    FD_CLR(sdlist[index], &fdset);
		}
	    }
	    snmp_read(&fdset);
	} else switch(count){
	    case 0:
	        gettimeofday(tvp);
		alarmTimer(tvp);
		eventTimer(tvp);
                snmp_timeout();
		break;
	    case -1:
		if (errno == EINTR){
		    continue;
		} else {
		    perror("select");
		}
		return -1;
	    default:
		printf("select returned %d\n", count);
		return -1;
	}
    }
}

snmp_read_packet(sd)
    int sd;
{
    struct sockaddr_in	from;
    int length, out_length, fromlength;
    u_char  packet[1500], outpacket[1500];

    fromlength = sizeof from;
    length = recvfrom(sd, packet, 1500, 0, (struct sockaddr *)&from,
		      &fromlength);
    if (length == -1)
	perror("recvfrom");
    if (snmp_dump_packet){
	printf("recieved %d bytes from %s:\n", length,
	       inet_ntoa(from.sin_addr));
	xdump(packet, length, "");
	printf("\n\n");
    }
    out_length = 1500;
    if (snmp_agent_parse(packet, length, outpacket, &out_length,
			 from.sin_addr.s_addr)){
	if (snmp_dump_packet){
	    printf("sent %d bytes to %s:\n", out_length,
		   inet_ntoa(from.sin_addr));
	    xdump(outpacket, out_length, "");
	    printf("\n\n");
	}
	if (sendto(sd, (char *)outpacket, out_length, 0,
		   (struct sockaddr *)&from,
	    sizeof(from)) < 0){
		perror("sendto");
		return 0;
	}

    }
    return 1;
}

struct nlist nl[] = {
    { "_boottime" },
    { "" }
};

/* deals with replies from remote alarm variables, and from inform pdus */
int
snmp_input(op, session, reqid, pdu, magic)
    int op;
    struct snmp_session *session;
    int reqid;
    struct snmp_pdu *pdu;
    void *magic;
{
    struct get_req_state *state = (struct get_req_state *)magic;
    
    if (op == RECEIVED_MESSAGE) {
	if (pdu->command == GET_RSP_MSG) {
	    if (state->type == EVENT_GET_REQ) {
		/* this is just the ack to our inform pdu */
		return 1;
	    }
	    return alarmGetResponse(pdu, state, op, session);
	}
    }
    else if (op == TIMED_OUT) {
	if (state->type == ALARM_GET_REQ) {
	    return alarmGetResponse(pdu, state, op, session);
	}
    }
    return 1;
}


#define NUM_NETWORKS    32   /* max number of interfaces to check */

#ifndef IFF_LOOPBACK
#define IFF_LOOPBACK 0
#endif
#define LOOPBACK    0x7f000001
u_long
get_myaddr(){
    int sd;
    struct ifconf ifc;
    struct ifreq conf[NUM_NETWORKS], *ifrp, ifreq;
    struct sockaddr_in *in_addr;
    int count;
    int interfaces;             /* number of interfaces returned by ioctl */

    if ((sd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
        return 0;
    ifc.ifc_len = sizeof(conf);
    ifc.ifc_buf = (caddr_t)conf;
    if (ioctl(sd, SIOCGIFCONF, (char *)&ifc) < 0){
        close(sd);
        return 0;
    }
    ifrp = ifc.ifc_req;
    interfaces = ifc.ifc_len / sizeof(struct ifreq);
    for(count = 0; count < interfaces; count++, ifrp++){
        ifreq = *ifrp;
        if (ioctl(sd, SIOCGIFFLAGS, (char *)&ifreq) < 0)
            continue;
        in_addr = (struct sockaddr_in *)&ifrp->ifr_addr;
        if ((ifreq.ifr_flags & IFF_UP)
            && (ifreq.ifr_flags & IFF_RUNNING)
            && !(ifreq.ifr_flags & IFF_LOOPBACK)
            && in_addr->sin_addr.s_addr != LOOPBACK){
                close(sd);
                return in_addr->sin_addr.s_addr;
            }
    }
    close(sd);
    return 0;
}

/*
 * Returns uptime in centiseconds(!).
 */
long get_uptime(){
    struct timeval boottime, now, diff;
    int kmem;

    if ((kmem = open("/dev/kmem", 0)) < 0)
        return 0;
    nlist("/vmunix", nl);
    if (nl[0].n_type == 0){
        close(kmem);
        return 0;
    }

    lseek(kmem, (long)nl[0].n_value, L_SET);
    read(kmem, &boottime, sizeof(boottime));
    close(kmem);

    gettimeofday(&now, 0);
    now.tv_sec--;
    now.tv_usec += 1000000L;
    diff.tv_sec = now.tv_sec - boottime.tv_sec;
    diff.tv_usec = now.tv_usec - boottime.tv_usec;
    if (diff.tv_usec > 1000000L){
        diff.tv_usec -= 1000000L;
        diff.tv_sec++;
    }
    return ((diff.tv_sec * 100) + (diff.tv_usec / 10000));
}

snmp_trap(sd, destAddr, trapType, specificType)
    int sd;
    u_long  destAddr;
    int     trapType;
    int     specificType;
{
#define MAX_TRAP_LEN 1500
    u_char          out_packet[MAX_TRAP_LEN];
    register u_char *out_data;
    int             out_length;
    struct sockaddr_in to;
    u_long  myaddr;
    long uptime;
    struct trapVar var[2], *vp;
    static u_char string[] = "Unix SNMPd";

    static oid      objid_enterprise[] = {1, 3, 6, 1, 4, 1, 3, 1, 1};
    static oid      sysDescrOid[] = {1, 3, 6, 1, 2, 1, 1, 1, 0};

    out_data = out_packet;
    out_length = MAX_TRAP_LEN;

    myaddr = get_myaddr();
    uptime = get_uptime();
    vp = var;
    vp->varName = sysDescrOid;
    vp->varNameLen = sizeof(sysDescrOid)/sizeof(oid);
    vp->varType = ASN_OCTET_STR;
    vp->varLen = strlen("UNIX SNMPD");
    vp->varVal = string;
    vp->next = vp + 1;
    vp++;

    vp->varName = sysDescrOid;
    vp->varNameLen = sizeof(sysDescrOid)/sizeof(oid);
    vp->varType = ASN_OCTET_STR;
    vp->varLen = strlen("UNIX SNMPD");
    vp->varVal = string;
    vp->next = NULL;

    out_length = snmp_build_trap2(out_data, &out_length, objid_enterprise, sizeof(objid_enterprise) / sizeof(oid), myaddr, trapType, specificType, uptime, var);
    if (out_length > 0){
        to.sin_family = AF_INET;
        to.sin_port = htons(SNMP_TRAP_PORT);
        to.sin_addr.s_addr = destAddr;
	if (snmp_dump_packet){
	    printf("sent %d bytes to %s:\n", out_length, inet_ntoa(to.sin_addr));
	    xdump(out_packet, out_length, "");
	    printf("\n\n");
	}

        if (sendto(sd, (char *)out_packet, out_length, 0, (struct sockaddr *)&to,
            sizeof(to)) < 0){
                perror("sendto");
                return;
        }
    } else {
printf("Bad trap\n");
    }
}
