/*
 * $Header: /home/vikas/netmgt/nocol/src/tpmon/RCS/main.c,v 2.5 1994/06/09 21:15:50 vikas Exp $
 */

/*
 *
 * DESCRIPTION: throughput monitor
 *
 *   This program connects to the 'discard' port on a remote host and
 *   sends random data to the remote host and calculates the throughput
 *   (in bits per second).
 *
 *   For interfacing with 'nocol', it opens the data file supplied by 
 *   the user and then creates a 'tpmon-output' file for use with netmon.
 *   It then directly reads and writes from this file.  If it gets SIGHUP,
 *   it rescans the data file before continuing.
 *
 *   Derived in part from the pingmon code, RCS v1.11
 *
 *   S. Spencer Sun, Princeton Univ., June 1992
 *   Updates and changes: Vikas Aggarwal, vikas@navya.com
 *
 * MODIFICATIONS
 *
 * $Log: main.c,v $
 * Revision 2.5  1994/06/09  21:15:50  vikas
 * Deleted the definition for inet_addr(), it is defined in arpa/inet.h
 *
 * Revision 2.4  1994/05/23  03:39:24  vikas
 * Rewrite. Now uses the library nocol_startup() function.
 *
 * Revision 2.3  1994/01/21  17:08:04  aggarwal
 * Needed an extra '2' characters in the malloc for pidfilename.
 *
 * Revision 2.2  1994/01/10  20:53:29  aggarwal
 * Added typecast for malloc() and changed fgetline to fgetLine()
 *
 * Revision 2.1  1993/10/30  03:27:42  aggarwal
 * Deleted watchdog info. Fixed defines of the signal handlers
 *
 * Revision 2.0  1992/06/18  21:20:43  aggarwal
 * Just increased the revision number for rcsid[] and releasing.
 *
 * Revision 1.3  1992/06/14  23:38:56  aggarwal
 * Had to increment 'sender' after 'strrchr'
 *
 * Revision 1.2  1992/06/12  04:15:47  aggarwal
 * Fixed scanning of keyword 'pollinterval'
 *
 * Revision 1.1  1992/06/12  04:06:46  aggarwal
 * Initial revision
 *
 *
 */

#ifndef lint
 static char rcsid[] = "$RCSfile: main.c,v $ $Revision: 2.5 $ $Date: 1994/06/09 21:15:50 $" ;
#endif

#include "nocol.h"			/*	common structures	*/

#include <string.h>			/* For strcat() definitions	*/
#include <sys/file.h>
#include <signal.h>			/* For signal numbers		*/

#define GLOBALS				/* for global variables */
#include "tpmon.h"			/* program specific defines	*/
#undef GLOBALS


/*+ 
 * Variables definitions
 */
static char	*configfile, *datafile;
static char	*sender;

main (ac, av)
     int ac;
     char **av;
{
    extern char *optarg;
    extern int optind;
    int	fdout = 0;			/* File desc for output data file */
    register int c ;
    char *p;
    time_t starttm, polltime ;

    prognm = av[0] ;				/* Save the program name */

#ifdef SENDER
    sender = SENDER ;
#else						/* delete the directory name */
    if ((sender = (char *)strrchr (prognm , '/')) == NULL)
      sender = prognm ;				/* no path in program name */
    else
      sender++ ;				/* skip leading '/' */
#endif

    while ((c = getopt(ac, av, "do:")) != EOF)
      switch (c) {
        case 'd':
	  debug++ ;
	  break ;
        case 'o':				/* output datafile */
	  datafile = optarg ;
	  break ;
        case '?':
        default:
	  fprintf (stderr, "%s: Unknown flag: %c\n", prognm, optarg);
	  help() ;
	  goto Cleanup ;
      }

    switch (ac - optind) {
      case 0:					/* default input file */
	break ;
      case 1:
	configfile = av[optind];
	break ;
      default:
	fprintf (stderr, "%s Error: Too many 'hosts' files\n\n", prognm);
	help() ;
	goto Cleanup;
    }

    nocol_startup(&configfile, &datafile);

    if (debug)
      fprintf(stderr,
	"(debug) %s: NODESFILE= '%s', DATAFILE= '%s'\n",
	  prognm, configfile, datafile) ;
    
    if (fdout != 0)			/* Say, recovering from longjmp	*/
      close(fdout);

    openeventlog() ;			/* noclogd for logging */

    if ( (fdout = open(datafile, O_RDWR|O_CREAT|O_TRUNC, DATAFILE_MODE)) < 0) {
      fprintf(stderr, "(%s) ERROR in open datafile ", prognm);
      perror (datafile);
      goto Cleanup ;
    }

    if (init_sites(fdout, configfile) == -1 )
      goto Cleanup ;

    if (!pollinterval)
      pollinterval = POLLINTERVAL ;		/* default value */


    /* poll_sites makes one complete pass over the list of nodes */
    while (1)                                   /* forever */
    {
        starttm = time((time_t *)NULL) ;        /* time started this cycle */
        if (poll_sites(fdout) == -1)            /* Polling error */
          break ;

        if ( (polltime = time((time_t *)NULL) - starttm) < pollinterval)
          sleep ((unsigned)(pollinterval - polltime));
    }

    /* HERE ONLY IF ERROR */
  Cleanup:
    nocol_done();

}	/***************** End of main *******************/
    
/*+ 
** FUNCTION:
** 	Brief usage
**/
help ()
{
    static char usage[] = " [-d (debug)] [-o <output file>] [<config file>]\n";

    fprintf(stderr, "\nUSAGE: %s %s\n\n", prognm, usage);
    fprintf(stderr, "\tThis program tests the throughput to sites by connecting\n");
    fprintf(stderr, "\tto the discard port (TCP port 9) and writing data into\n");
    fprintf(stderr, "\tit as fast as it can and measuring the transfer rate.\n");

    fprintf(stderr,"\tBy default, the list of sites to monitor is in %s\n",
	    configfile);
    fprintf(stderr,"\tbut can be changed on the command line.\n");

    fprintf(stderr,"\tThe 'config file' format is:\n");
    fprintf(stderr,"\t\t <hostname> <ip-address>  <threshold> [TEST]\n");
    fprintf(stderr,"\t\t POLLINTERVAL <number of secs>\n");
    fprintf(stderr,"\tThe program writes its pid in %s.pid and \n",prognm);
    fprintf(stderr,"\tif a new process starts, it kills the earlier one.\n");
    return (1);
}


/*
** FUNCTION
**
**	init_sites
**
**	This function writes to the LSTFILE. All sites in the NODESFILE
**	file are set to UNINIT status.
**
**	Careful while using 'localtime': the calue of the month varies from
**	0 - 11 and hence has to be incremented for the correct month.
*/

init_sites(fdout, cfigfile)
     int fdout ;			/* Output file descriptor	*/
     char *cfigfile;			/* Filename of the config file	*/
{
    extern char sigtoprog[];		/* In pingmon.h			*/
    FILE *p_nodes ;
    EVENT v;				/* Defined in NOCOL.H		*/
    char record[MAXLINE];
    struct tm *loctime ;
    time_t locclock ;			/* Careful, don't use 'long'	*/

    if ((p_nodes = fopen(cfigfile, "r")) == NULL)
    {
	fprintf(stderr, "%s error (init_sites) ", prognm) ;
	perror (cfigfile);
	return (-1);
    }

    /*
     * Fill in the static data stuff
     */
    bzero (&v, sizeof(v)) ;
    locclock = time((time_t *)0);
    loctime = localtime((long *)&locclock);

    v.mon = loctime->tm_mon + 1;        v.day = loctime->tm_mday;
    v.hour = loctime->tm_hour;          v.min = loctime->tm_min;

    /*
     * in the following strncpy's, the NULL is already appended because
     * of the bzero, and copying one less than size of the arrays.
     */
    strncpy (v.sender, sender, sizeof(v.sender) - 1);

    strncpy (v.var.name, VARNM, sizeof (v.var.name) - 1);
    strncpy (v.var.units, VARUNITS, sizeof (v.var.units) - 1);
    v.var.value = 0 ; v.var.threshold = 0 ;	/* threshold reset below */
    v.nocop = SETF_UPDOUN (v.nocop, n_UNKNOWN); /* Set all to UNKNOWN   */
    v.severity = E_INFO ;

    while(fgetLine(p_nodes,record,MAXLINE) > 0 ) 
    {
	char *p;		/* for strtol */
	u_long thresh;
	char w1[MAXLINE], w2[MAXLINE], w3[MAXLINE], w4[MAXLINE];
		/* Confg words	*/
	int rc;						/* return code	*/

	v.nocop = 0 ;				/* Init options to zero	*/
	*w1 = *w2 = *w3 = *w4 = NULL ;
        /* line should look like "<name>  <IP#>  <threshold>  [TEST]" */
	rc = sscanf(record,"%s %s %s %s", w1, w2, w3, w4);
	if (rc == 0 || *w1 == NULL || *w1 == '#')  /* Comment or blank 	*/
	  continue;

	if (strncmp(w1, "POLLINT", 7) == 0 || strncmp(w1, "pollint", 7) == 0)
	{
	    pollinterval = (u_long)strtol(w2, &p, 0) ;
	    if (p == w2)
	    {
		fprintf(stderr,"(%s): Error in format for POLLINTERVAL '%s'\n",
			prognm, w2) ;
		pollinterval = 0 ;		/* reset to default above */
	    }
	    continue ;
	}

	strncpy(v.site.name, w1, sizeof(v.site.name) - 1);
	strncpy(v.site.addr, w2, sizeof(v.site.addr) - 1);	/* no checks */

	if (inet_addr(w2) == -1)	/* bad address */
	{
	    fprintf(stderr,
		    "(%s): Error in address '%s' for site '%s', ignoring\n",
		    prognm, w2, w1);
	    continue ;
	}

        thresh = (u_long)strtol(w3, &p, 0);
        if (p == w3) { 			/* strtod() couldn't convert */
          fprintf(stderr,
            "(%s): Error in format of threshold '%s' for site '%s'\n", 
		  prognm, w3, w1);
	  fprintf(stderr,
            "\t(should be a long integer representing bits per second)\n");
        } else
          v.var.threshold = thresh;
          	
	if (*w4 != NULL)			/* Some other keyword	*/
	{
	    if (strcmp(w4, "test") == 0 || strcmp(w3, "TEST") == 0)
	      v.nocop = v.nocop | n_TEST ;
	    else
	      fprintf(stderr, "%s: Ignoring unknown keyword- %s\n", prognm,w3);
	}

	write (fdout, (char *)&v, sizeof(v)) ;

    }					/* end: while			*/
    fclose (p_nodes);			/* Not needed any more		*/    
    return(1);				/* All OK			*/
}		/* end:  init_sites()		*/

