/* $Header: /home/vikas/netmgt/nocol/src/portmon/RCS/portmon.c,v 1.5 1994/05/16 02:12:55 vikas Exp $  */

/* Copyright 1994 Vikas Aggarwal, vikas@navya.com */

/*+ 		portmon
 * FUNCTION:
 * 	Monitor TCP ports and reponses for NOCOL. It can send a string
 * and then parse the responses against a list. Each response can be
 * assigned a severity. Checks simple port connectivity if given a NULL
 * response string.
 *
 * CAVEATS:
 *	1) Uses 'strstr' and not a real regular expression. Case sensitive
 *
 *  	Vikas Aggarwal, -vikas@navya.com
 */


/*
 * $Log: portmon.c,v $
 * Revision 1.5  1994/05/16  02:12:55  vikas
 * bugfix from David Meyer (meyer@phloem.uoregon.edu) to distinguish
 * between the alram goign off and read error (n=0 vs. n<0)
 *
 * Revision 1.4  1994/05/09  01:03:37  vikas
 * Rewrite to use new nocol library 'startup_nocol' function.
 *
 * Revision 1.3  1994/01/10  20:52:20  aggarwal
 * Added typcast for malloc. Changed fgetline() to fgetLine()
 * Fixed strdup to handle NULL strings. Added comments if just checking
 * connectivity.
 *
 * Revision 1.2  1993/11/03  20:51:03  aggarwal
 * Added ifdef for h_addr (defined in netdb.h) in case its defined.
 *
 * Revision 1.1  1993/10/30  03:39:04  aggarwal
 * Initial revision
 *
 */

/*  */
#ifndef lint
 static char rcsid[] = "$Id: portmon.c,v 1.5 1994/05/16 02:12:55 vikas Exp $" ;
#endif

#include "portmon.h"

char	*prognm;
int	debug;

static int 	maxseverity, fdout, pollinterval ;
static int 	gotalarm, sock ;
static char	*configfile, *datafile;
static char 	*sender;
static void	alarm_handler();

extern char	*skip_spaces() ;		/* in libnocol */

main(ac, av)
     int ac;
     char **av;
{
    extern char *optarg;
    extern int  optind;
    int         c ;

    if ((prognm = (char *)strrchr (av[0], '/')) == NULL)
      prognm = av[0] ;                        /* no path in program name */
    else
      prognm++ ;                                /* skip leading '/' */

#ifdef SENDER
    sender = SENDER ;
#else                                           /* delete the directory name */
    sender = prognm ;                         /* no path in program name */
#endif


    pollinterval = POLLINTERVAL ;

    while ((c = getopt(ac, av, "df:")) != EOF)
      switch(c)
      {
       case 'd':
          debug++;
          break;
       case 'f':
          configfile = optarg ;
          break ;
       case '?':
       default:
          fprintf(stderr, "%s: Unknown flag: %c\n", prognm, optarg);
          fprintf(stderr, "Usage: %s [-d] [-f <config file>\n] ", prognm);
          exit (1);
      }

    nocol_startup(&configfile, &datafile);

    if ( (fdout = open(datafile, O_RDWR|O_CREAT|O_TRUNC, DATAFILE_MODE)) < 0)
    {
        fprintf(stderr, "(%s) ERROR in open datafile ", prognm);
        perror (datafile);
	nocol_done();
    }

    if (readconfig(fdout) == -1)
      nocol_done();

 redo:
    lseek(fdout, (off_t)0, SEEK_SET);	/* rewind file */
    c = 0;
    while (hostarray[c].hname)
    {
	int status ;
	EVENT v;

	read(fdout, &v, sizeof v);
	status = checkports(&(hostarray[c])) ;
	if (status == -1)
	  fprintf (stderr, "%s: Error in checkports, skipping site %s\n",
		   prognm, hostarray[c].hname);
	else
	{
	    update_event(&v, status, /* value */ (u_long)status, maxseverity) ;
	    lseek(fdout, -(off_t)sizeof(v), SEEK_CUR);
	    write(fdout, (char *)&v, sizeof(v));
	}
	++c;
    }
    lseek(fdout, (off_t)0, SEEK_SET);	/* rewind file */

    if (debug)
      fprintf(stderr, "(debug) %s: sleeping for %d...zzz\n", 
	      prognm,pollinterval);
#ifdef SVR4
    bsdsignal(SIGALRM, SIG_DFL);
#else
    signal(SIGALRM, SIG_DFL);		/* wake-up, in case blocked */
#endif
    sleep (pollinterval);
    goto redo ;

}	/* end:  main()  */


/*+ 
 * FUNCTION:
 * 	Checks the port for the structure _harray passed to it. Sets
 * global maxseverity. Return value is the 'status' to be used in
 * update_event().
 */
checkports(h)
     struct _harray  *h;
{
    int status, n;
    char readbuf[BUFSIZ];
    struct sockaddr_in sin;

    if (debug)
      fprintf (stderr, "(debug) %s: Checking site '%s'/%d\n",
	       prognm, h->hname, h->port);

    sock = socket(AF_INET, SOCK_STREAM, 0) ;
    if (sock < 0)
    {
	perror("socket");
	return (-1);
    }

    bzero(&sin, sizeof(sin));
    sin.sin_family = AF_INET;

    if (isdigit(*(h->ipaddr)) ) 	/* given an address, not a name */
      sin.sin_addr.s_addr = inet_addr(h->ipaddr);
    else
    {
        struct hostent  *hp ;
        if ((hp = gethostbyname(h->ipaddr)) == NULL)
        {
            perror(prognm);
            fprintf(stderr, "gethostbyname() failed for %s\n", h->ipaddr);
            return(-1);
        }
#ifdef h_addr	/* in netdb.h */
        sin.sin_addr.s_addr = inet_addr(hp->h_addr) ;
#else
        sin.sin_addr.s_addr = inet_addr(hp->h_addr_list[0]) ;
#endif
    }

    if (sin.sin_addr.s_addr == -1)      /* error */
    {
        perror("inet_addr() failed");
	return(-1);			/* fatal error */
    }


    sin.sin_port = htons(h->port);		/* port number to use */
    
    if (connect(sock, (struct sockaddr *)&sin, sizeof(sin)) < 0)
    {
        if (debug)
	  perror("client: AF_INET connect() failed");
	close(sock);
	maxseverity = h->maxseverity;
        return(0);
    }

    if (h->send && *(h->send) != NULL)		/* if something to send */
    {
	if ((n = timeout_read (readbuf, sizeof readbuf)) < 0)
	{		/* n=0 for alarm going off, -1 for read error */
	    alarm(0);
	    if (debug)
	    {
		fprintf(stderr, "readstring error for '%s' port '%d'",
			h->hname, h->port);
		if (gotalarm)
		  fprintf (stderr, "- timeout\n");
		else {
		    perror("read");
		    close (sock);
		}
	    }
	    maxseverity = h->maxseverity;
	    return(0);		/* down */
	}

	if (debug)
	  fprintf(stderr, " (debug) %s: ignoring connect string '%s'\n",
		  prognm, readbuf );

	if (write(sock, h->send, strlen(h->send)) != strlen(h->send))
	{
	    if (debug)
	    {
		fprintf(stderr, "Sendstring error for '%s' port '%d'\n",
			h->hname, h->port);
		perror("write");
		close (sock);
		maxseverity = h->maxseverity;
		return(0);	/* mark as down */
	    }
	}	/* end if (couldn't write)  */

	write(sock, "\n", 1);			/* send newline */
	if (debug)
	  fprintf(stderr, " (debug) %s: sent string '%s' to %s...\n",
		  prognm, h->send, h->hname) ;

    }	/* end:  if (something to send) */


    if (h->responses[0].response)	/* responses to check */
    {
	if (timeout_read(readbuf, sizeof readbuf) <= 0)
	{
	    if (debug)
	    {
		fprintf(stderr, "readstring error for '%s' port '%d'",
			h->hname, h->port);
		if (gotalarm)
		  fprintf(stderr, "- timeout\n");
		else
		{
		    perror("read");
		    close (sock);
		}
	    }
	    maxseverity = h->maxseverity;
	    return(0);	/* down  */
	}
	maxseverity = check_resp(readbuf, h->responses);
    }
    else
      maxseverity = E_INFO ;	/* Just had to check a 'connect' */

    if (maxseverity == -1)	/* something failed, use default severity */
      maxseverity = h->maxseverity ;

    close(sock);
    if (debug)
      fprintf(stderr, " (debug): returning severity %d\n", maxseverity) ;
    if (maxseverity == E_INFO)
      return(1);
    else
      return(0);		/* status is down */

}	/* end:  checkports()  */

/*+ 
 * FUNCTION:
 * 	Read from socket into buffer and put terminating NULL.
 * Set alarm for timeout. Note that it is reading from 'global' sock
 * descriptor (since the alarm closes it).
 * Returns the number of bytes read.
 *
 */
timeout_read(buf, bufsiz)
     char *buf;
     int bufsiz ;
{
    int n = 0, len = bufsiz ;
    char *cp;

    cp = buf ; 

    gotalarm = 0;
#ifdef SVR4
    bsdsignal(SIGALRM, alarm_handler);
#else
    signal(SIGALRM, alarm_handler);
#endif
    alarm(RTIMEOUT);

    /* read a LINE from global socket (until newline) */
    while (len != 0 && (n = read (sock, cp, len)) > 0)
    {
	cp += (n - 1);	/* increment pointer to last byte read */
	len -= n;
	if (*cp == '\n'  || *cp == '\0')
	  break ;
	else
	  cp++ ;	/* increment past last byte read */
    }

#ifdef SVR4
    bsdsignal (SIGALRM, SIG_IGN);
#else
    signal (SIGALRM, SIG_IGN);
#endif
    alarm (0);				/* remember to unblock alarm */

    if (n <= 0)
      len = n ;				/* some error or got alarm */
    else
      len = bufsiz - len ;		/* Total number bytes read */

    if (debug)
      fprintf(stderr, "(debug) timeout_read: Read %d bytes from socket\n",len);

    buf[len > 0 ? len:0] = '\0';		/* terminate with NULL */

    return (len);			/* number bytes read else negative */

}	/* end:  timeout_read() */

     
/*+ 
 * FUNCTION:
 * 	Check the list of responses
 */
check_resp(readstr, resarr)
     char *readstr;
     struct _response  *resarr ;
{

    if (debug)
      fprintf(stderr, " (debug) %s: Checking response '%s'\n", prognm,readstr);

    while (resarr->response)
    {
	if (strstr(readstr, resarr->response) != NULL)
	{
	    if (debug)
	      fprintf(stderr, " (debug) %s: Matched '%s'\n",
		      prognm, resarr->response);

	    return(resarr->severity);
	}

	++resarr ;
    }

    if (debug)
      fprintf (stderr, "%s: No response matched for site\n", prognm);

    return(-1);		/* No response matched given list */
}	/* check_resp()  */


/*+ 
 * FUNCTION:
 * 	Duplicate a string. Can handle a NULL ptr.
 */

char *Strdup(s)
     char *s ;
{
    char *t ;

    if (s == NULL)
    {
	t = (char *)malloc(1);
	*t = '\0';
    }
    else
    {
	t = (char *)malloc(strlen(s) + 1);
	if (t != NULL)
	  (char *)strcpy(t, s);
    }

    return (t);
}

/*+ 
 * FUNCTION:
 * 	Read the config file.
 * POLLINTERVAL ppp
 * HOST  <hostname>  <address>  <var>  <port> <failseverity>  <send string>
 * <severity>	response1 till end of this line
 * <severity>   response2 till end of this line
 *
 */
#define NEXTTOK  (char *)skip_spaces(strtok(NULL, " \t"))

readconfig(fdout)
     int fdout ;			/* output data filename */
{
    int i = 0, j=0;			/* array indexes */
    int maxseverity ;
    char *j1, *j2 ;			/* temp string pointers */
    FILE *cfd ;
    EVENT v;                            /* Defined in NOCOL.H */
    char record[MAXLINE] ;
    struct tm *loctime ;
    time_t locclock ;                   /* Careful, don't use 'long'    */

    if ((cfd = fopen(configfile, "r")) == NULL)
    {
        fprintf(stderr, "%s error (init_sites) ", prognm) ;
        perror (configfile);
        return (-1);
    }

    /*
     * Fill in the static data stuff
     */
    bzero ((char *)hostarray, sizeof(hostarray));
    bzero (&v, sizeof(v)) ;

    locclock = time((time_t *)0);
    loctime = localtime((long *)&locclock);

    v.mon = loctime->tm_mon + 1;        v.day = loctime->tm_mday;
    v.hour = loctime->tm_hour;          v.min = loctime->tm_min;

    strncpy (v.sender, sender, sizeof(v.sender) - 1);

    strncpy (v.var.units, VARUNITS, sizeof (v.var.units) - 1);
    v.var.threshold = 0 ;

    v.nocop = SETF_UPDOUN (0, n_UNKNOWN); /* Set all to UNKNOWN   */
    v.severity = E_INFO ;

    /*
     * Now parse the config file
     */

    while(fgetLine(cfd,record,MAXLINE) > 0 ) 	/* keeps the \n */
    {
	static int skiphost;
	int port ;

	record[strlen(record) -1] = '\0' ;		/* chop off newline */
	if (*record == '#' || *(skip_spaces(record)) == NULL)
	  continue ;

	if (strncasecmp(record, "POLLINTERVAL", strlen("POLLINTERVAL")) == 0)
	{
	    strtok(record, " \t");
	    j1 = (char *)skip_spaces(strtok(NULL, "")) ;
	    if (j1 == NULL || *j1 == NULL || (pollinterval = atoi(j1)) <= 0)
		fprintf(stderr, "%s: bad or missing pollinterval value\n",
			prognm);

	    pollinterval = (pollinterval > 0 ? pollinterval : POLLINTERVAL);
	    if (debug)
	      fprintf (stderr, "(debug) %s: Pollinterval = %d\n", 
		       prognm, pollinterval);

	    continue;
	}


	if (strncasecmp(record, "HOST", 4) != 0)	/* Not HOST line */
	{
	    if (skiphost)
	      continue ;

	    j1 = strtok(record," \t") ;			/* severity level */
	    j1 = (char *)skip_spaces(j1) ;
	    switch(tolower(*j1))
	    {
	     case 'c': case '1': maxseverity = E_CRITICAL; break;
	     case 'e': case '2': maxseverity = E_ERROR; 	break;
	     case 'w': case '3': maxseverity = E_WARNING; break;
	     case 'i': case '4': maxseverity = E_INFO; 	break;
	     default:  maxseverity = E_CRITICAL; break ;
	    }
	    /* remember that the 'i' index has been incremented */
	    hostarray[i-1].responses[j].severity = maxseverity ;
	    j1 = (char *)skip_spaces(strtok(NULL, "")) ;
	    if (j1 == NULL)
	    {
		fprintf(stderr, "%s: missing response line, skipped\n",prognm);
		fprintf(stderr, "For null responses, dont put any lines\n");
		continue;
	    }
	    hostarray[i-1].responses[j].response = (char *)Strdup(j1);
	    j++ ;

	    continue ;	/* next input line */
	}

	/*
	 * Here if parsing a HOST line
	 */
	j = 0;			/* and reset the response array index */
	
	skiphost = 0;		/* assume valid config */
	strtok(record, " \t");	/* skip HOST keyword */	
	    
	strncpy(v.site.name, NEXTTOK, sizeof(v.site.name) - 1);
	strncpy(v.site.addr, NEXTTOK, sizeof(v.site.addr) - 1);
	if (inet_addr(v.site.addr) == -1)        /* bad address */
	{
	    fprintf(stderr,
		    "(%s): Error in addr '%s' for site '%s', ignoring\n",
		    prognm, v.site.addr, v.site.name);
	    skiphost = 1;
	    continue ;
	}
	strncpy(v.var.name, NEXTTOK, sizeof(v.var.name) - 1);
	
	if ((port = atoi(NEXTTOK)) == 0)
	{
	    fprintf(stderr,
		    "(%s): Error in port for site '%s', ignoring\n",
		    prognm, v.site.name);
	    skiphost = 1;
	    continue ;
	}
	
	j1 = NEXTTOK;	/* severity level */
	if (j1 == NULL)
	  j1 = "c";
	switch(tolower(*j1))
	{
	 case 'c': maxseverity = E_CRITICAL; break;
	 case 'e': maxseverity = E_ERROR; break;
	 case 'w': maxseverity = E_WARNING; break;
	 case 'i': maxseverity = E_INFO; break;
	 default:  maxseverity = E_CRITICAL ; break ;
	}
	
	if (debug)
	  fprintf (stderr, "Name: %s %s, VAR: %s, Port %d SEV: %d\n",
		   v.site.name,v.site.addr,v.var.name, 
		   port,maxseverity);
	
	/* the rest of string is SEND */
	j1 = (char *)skip_spaces(strtok(NULL, ""));  
	if (j1)
	  hostarray[i].send = (char *)Strdup(j1) ;
	else
	  hostarray[i].send = NULL ;		/* no SEND string */

	hostarray[i].port = port;
	hostarray[i].maxseverity = maxseverity ;
	hostarray[i].hname = (char *)Strdup(v.site.name);
	hostarray[i].ipaddr = (char *)Strdup(v.site.addr);
	bzero (hostarray[i].responses, sizeof (hostarray[i].responses));
	
	if (write (fdout, (char *)&v, sizeof(v)) != sizeof(v))
	{
	    fprintf(stderr, "%s (write): %s\n", prognm,sys_errlist[errno]);
	    nocol_done() ;
	}
	++i;			/* increment the index, in case */

    }	/* end: while */

    fclose (cfd);       		/* Not needed any more */    
    return(1);                          /* All OK  */

}  /* end:  init_sites() */

/*+ 
 * FUNCTION:
 * 	Handle timeouts for sockets. Since it doesn't interrupt system
 *	calls (like read() ), close the socket so that read fails.
 */
static void alarm_handler()
{
    if (debug)
      fprintf (stderr, "(debug) %s: recvd ALARM\n", prognm);
    gotalarm = 1;
    close (sock);
}


