/*
 * $Header: /nocol/src/tpmon/RCS/main.c,v 2.0 1992/06/18 21:20:43 aggarwal Exp $
 *
 * DESCRIPTION: throughput monitor
 *
 *   This program connects to the 'discard' port on a remote host and
 *   sends random data to the remote host and calculates the throughput
 *   (in bits per second).
 *
 *   For interfacing with 'nocol', it opens the data file supplied by 
 *   the user and then creates a 'tpmon-output' file for use with netmon.
 *   It then directly reads and writes from this file.  If it gets SIGHUP,
 *   it rescans the data file before continuing.
 *
 *   It sends a SIGUSR1 signal to the 'watchdog' program if SEND_WSIGNAL
 *   is defined during compile.
 *
 *   Derived in part from the pingmon code, RCS v1.11
 *
 *   S. Spencer Sun, Princeton Univ. / JvNCnet, June 1992
 *
 * MODIFICATIONS
 *
 * $Log: main.c,v $
 * Revision 2.0  1992/06/18  21:20:43  aggarwal
 * Just increased the revision number for rcsid[] and releasing.
 *
 * Revision 1.3  1992/06/14  23:38:56  aggarwal
 * Had to increment 'sender' after 'strrchr'
 *
 * Revision 1.2  1992/06/12  04:15:47  aggarwal
 * Fixed scanning of keyword 'pollinterval'
 *
 * Revision 1.1  1992/06/12  04:06:46  aggarwal
 * Initial revision
 *
 *
 */
/*  Copyright 1992 JvNCnet

 Permission to use, copy, modify and distribute this software and its
 documentation for any purpose is hereby granted without fee, provided that
 the above copyright notice appear in all copies and that both that copyright
 notice and this permission notice appear in supporting documentation, and
 that the name of JvNCnet not be used in advertising or publicity pertaining
 to distribution of the software without specific, written prior permission.
 JvNCnet makes no representations about the suitability of this software for
 any purpose.  It is provided "as is" without express or implied warranty.

 JvNCnet DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL JvNCnet
 BE LIABLE FOR ANY DAMAGES WHATSOEVER, INCLUDING DAMAGES RESULTING FROM LOSS
 OF USE, DATA OR PROFITS, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/

/* ^L */

#ifndef lint
 static char rcsid[] = "$RCSfile: main.c,v $ $Revision: 2.0 $ $Date: 1992/06/18 21:20:43 $" ;
#endif



#include "nocol.h"			/*	common structures	*/

#include <strings.h>			/* For strcat() definitions	*/
#include <sys/file.h>
#include <signal.h>			/* For signal numbers		*/
#include <setjmp.h>

#define GLOBALS				/* for global variables */
#include "tpmon.h"			/* program specific defines	*/
#undef GLOBALS


/*+ 
 * Variables definitions
 */

jmp_buf	env;				/* For longjmp and setjmp 	*/

main (ac, av)
     int ac;
     char **av;
{
    extern char *prognm ;                       /* in pingmon.h  */
    extern int debug ;
    extern char *optarg;
    extern int optind;
    int	fdout = 0;			/* File desc for output data file */
    register int c ;
    time_t starttm, polltime ;
    void *done(), *restart();			/* For the signals	*/

    prognm = av[0] ;				/* Save the program name */

#ifdef SENDER
    sender = SENDER ;
#else						/* delete the directory name */
    if ((sender = (char *)strrchr (prognm , '/')) == NULL)
      sender = prognm ;				/* no path in program name */
    else
      sender++ ;				/* skip leading '/' */
#endif

    /* the output data filename */    
    sprintf(datafile,  "%s/%s%s\0", DATADIR,  sender, TPMONOUTPUTEXT); 
    sprintf(ipnodes, "%s\0", NODESFILE);    /* the name of the config file */

    while ((c = getopt(ac, av, "do:")) != EOF)
      switch (c) {
        case 'd':
	  debug++ ;
	  break ;
        case 'o':				/* output datafile */
	  sprintf(datafile, "%s\0", optarg) ;
	  break ;
        case '?':
        default:
	  fprintf (stderr, "%s: Unknown flag: %c\n", prognm, optarg);
	  help() ;
	  goto Cleanup ;
      }

    switch (ac - optind) {
      case 0:					/* default input file */
	break ;
      case 1:
	sprintf(ipnodes, "%s\0", av[optind]) ;
	break ;
      default:
	fprintf (stderr, "%s Error: Too many 'hosts' files\n\n", prognm);
	help() ;
	goto Cleanup;
    }

    if (standalone(prognm) == -1) { 	/* Kill prev running process	*/
      fprintf(stderr, "%s: Error in standalone...exiting\n", prognm);
      exit (1);
    }

    if (debug)
      fprintf(stderr,
	"(debug) %s: NODESFILE= '%s', DATAFILE= '%s'\n",
	  prognm, ipnodes, datafile) ;
    
    setjmp (env);			/* Save the environment in case	*/

    signal (SIGQUIT, done);		/* Delete pid file while dying	*/
    signal (SIGTERM, done);
    signal (SIGINT, done);
    signal (SIGHUP, restart);		/* Re-read the ipnodes file	*/

    if (getenv(DEBUG_FLG) != NULL)      /* see ifdebug mode desired     */
      debug = debug ^ 1 ;

    umask (002);			/* write access to the group	*/
    if (fdout != 0)			/* Say, recovering from longjmp	*/
      close(fdout);

    if ( (fdout = open(datafile, O_RDWR | O_CREAT | O_TRUNC, 0664)) < 0) {
      fprintf(stderr, "(%s) ERROR in open datafile ", prognm);
      perror (datafile);
      goto Cleanup ;
    }

    if (init_sites(fdout, ipnodes) == -1 )
      goto Cleanup ;

    if (!pollinterval)
      pollinterval = POLLINTERVAL ;		/* default value */


    /* poll_sites makes one complete pass over the list of nodes */
    while (1)                                   /* forever */
    {
        starttm = time((time_t *)NULL) ;        /* time started this cycle */
        if (poll_sites(fdout) == -1)            /* Polling error */
          break ;

        if ( (polltime = time((time_t *)NULL) - starttm) < pollinterval)
          sleep ((unsigned)(pollinterval - polltime));
    }

    /* HERE ONLY IF ERROR */
  Cleanup:
    done();

}	/***************** End of main *******************/
    
/*+ 
** FUNCTION:
** 	Brief usage
**/
help ()
{
    static char usage[] = " [-d (debug)] [-o <output file>] [<config file>]\n";

    fprintf(stderr, "\nUSAGE: %s %s\n\n", prognm, usage);
    fprintf(stderr, "\tThis program tests the throughput to sites by connecting\n");
    fprintf(stderr, "\tto the discard port (TCP port 9) and writing data into\n");
    fprintf(stderr, "\tit as fast as it can and measuring the transfer rate.\n");

    fprintf(stderr,"\tBy default, the list of sites to monitor is in %s\n",
	    ipnodes);
    fprintf(stderr,"\tbut can be changed on the command line.\n");

    fprintf(stderr,"\tThe 'config file' format is:\n");
    fprintf(stderr,"\t\t <hostname> <ip-address>  <threshold> [TEST]\n");
    fprintf(stderr,"\t\t POLLINTERVAL <number of secs>\n");
#ifdef SEND_WSIGNAL	/* send signal to watchdog */
    fprintf(stderr,"\t\t SIGNAL <program name>\n\n");
#endif
    fprintf(stderr,"\tThe program writes its pid in %s.pid and \n",prognm);
    fprintf(stderr,"\tif a new process starts, it kills the earlier one.\n");
    fprintf(stderr,"\tIf a HUP signal is sent to the process, it rescans ");
    fprintf(stderr,"the config file. \n");
#ifdef SEND_WSIGNAL	/* send signal to watchdog */
    fprintf(stderr,"\tThe SIGNAL keyword sends a SIGUSR1 signal to the pid\n");
    fprintf(stderr,"\tstored in the <program name>.pid whenever a site changes\n");
    fprintf(stderr,"\tseverity level to WARNING.\n\n");
#endif
    return (1);
}


/*
** FUNCTION
**
**	init_sites
**
**	This function writes to the LSTFILE. All sites in the NODESFILE
**	file are set to UNINIT status.
**
**	Careful while using 'localtime': the calue of the month varies from
**	0 - 11 and hence has to be incremented for the correct month.
*/

init_sites(fdout, ipnodes)
     int fdout ;			/* Output file descriptor	*/
     char *ipnodes;			/* Filename of the ipnodes file	*/
{
    extern char sigtoprog[];		/* In pingmon.h			*/
    extern char *sender ;
    FILE *p_nodes ;
    EVENT v;				/* Defined in NOCOL.H		*/
    char record[MAXLINE];
    struct tm *loctime ;
    time_t locclock ;			/* Careful, don't use 'long'	*/

    if ((p_nodes = fopen(ipnodes, "r")) == NULL)
    {
	fprintf(stderr, "%s error (init_sites) ", prognm) ;
	perror (ipnodes);
	return (-1);
    }

    /*
     * Fill in the static data stuff
     */
    bzero (&v, sizeof(v)) ;
    locclock = time((time_t *)0);
    loctime = localtime((long *)&locclock);

    v.mon = loctime->tm_mon + 1;        v.day = loctime->tm_mday;
    v.hour = loctime->tm_hour;          v.min = loctime->tm_min;

    /*
     * in the following strncpy's, the NULL is already appended because
     * of the bzero, and copying one less than size of the arrays.
     */
    strncpy (v.sender, sender, sizeof(v.sender) - 1);

    strncpy (v.var.name, VARNM, sizeof (v.var.name) - 1);
    strncpy (v.var.units, VARUNITS, sizeof (v.var.units) - 1);
    v.var.value = 0 ; v.var.threshold = 0 ;	/* threshold reset below */
    v.nocop = SETF_UPDOUN (v.nocop, n_UNKNOWN); /* Set all to UNKNOWN   */
    v.severity = E_INFO ;

    while(fgetline(p_nodes,record,MAXLINE) > 0 ) 
    {
	u_long inet_addr() ;
	char *p;		/* for strtol */
	u_long thresh;
	char w1[MAXLINE], w2[MAXLINE], w3[MAXLINE], w4[MAXLINE];
		/* Confg words	*/
	int rc;						/* return code	*/

	v.nocop = 0 ;				/* Init options to zero	*/
	*w1 = *w2 = *w3 = *w4 = NULL ;
        /* line should look like "<name>  <IP#>  <threshold>  [TEST]" */
	rc = sscanf(record,"%s %s %s %s", w1, w2, w3, w4);
	if (rc == 0 || *w1 == NULL || *w1 == '#')  /* Comment or blank 	*/
	  continue;
#ifdef SEND_WSIGNAL
	if (strcmp(w1, "signal") == 0 || strcmp(w1, "SIGNAL") == 0)
	{
	    strcpy(sigtoprog, w2) ;		/* Prog to get signal	*/
	    strcat(sigtoprog, ".pid");		/* append suffix	*/
	    continue ;
	}
#endif /* SEND_WSIGNAL */

	if (strncmp(w1, "POLLINT", 7) == 0 || strncmp(w1, "pollint", 7) == 0)
	{
	    pollinterval = (u_long)strtol(w2, &p, 0) ;
	    if (p == w2)
	    {
		fprintf(stderr,"(%s): Error in format for POLLINTERVAL '%s'\n",
			prognm, w2) ;
		pollinterval = 0 ;		/* reset to default above */
	    }
	    continue ;
	}

	strncpy(v.site.name, w1, sizeof(v.site.name));
	strncpy(v.site.addr, w2, sizeof(v.site.addr));	/* no checks */

	if (inet_addr(w2) == -1)	/* bad address */
	{
	    fprintf(stderr,
		    "(%s): Error in address '%s' for site '%s', ignoring\n",
		    prognm, w2, w1);
	    continue ;
	}

        thresh = (u_long)strtol(w3, &p, 0);
        if (p == w3) { 			/* strtod() couldn't convert */
          fprintf(stderr,
            "(%s): Error in format of threshold '%s' for site '%s'\n", 
		  prognm, w3, w1);
	  fprintf(stderr,
            "\t(should be a long integer representing bits per second)\n");
        } else
          v.var.threshold = thresh;
          	
	if (*w4 != NULL)			/* Some other keyword	*/
	{
	    if (strcmp(w4, "test") == 0 || strcmp(w3, "TEST") == 0)
	      v.nocop = v.nocop | n_TEST ;
	    else
	      fprintf(stderr, "%s: Ignoring unknown keyword- %s\n", prognm,w3);
	}

	write (fdout, (char *)&v, sizeof(v)) ;

    }					/* end: while			*/
    fclose (p_nodes);			/* Not needed any more		*/    
    return(1);				/* All OK			*/
}		/* end:  init_sites()		*/


/*+ 		restart
** FUNCTION:
**
** 	If a SIGHUP is sent, it rescans the ipnodes file and starts all
** over without exiting from the program.
**/

void *restart ()
{
    longjmp (env, 1);
}

/*+             done
** FUNCTION:
**
** Delete the PID and data file.
** Called just before exiting. Only if error. Exits since it can be called
** on recieving a signal.
**/
void *done ()
{
    char pidfile[MAXLINE] ;

    fprintf (stderr, "%s: removing pid, data file.... ", prognm);
    sprintf (pidfile, "%s.pid\0", prognm);
    unlink (pidfile);                           /* remove the PID file  */
    unlink (datafile);                          /* delete the data file */
    fprintf (stderr, "Done\n");
    exit (1);
}		/* end  done() */
