/*
 * $Header: /nocol/src/netmon/RCS/utils.c,v 1.3 1992/06/18 21:09:22 aggarwal Exp $
 */

/* Copyright 1992 JvNCnet, Princeton */

/*+
 ** Utility functions:
 **
 **      get_reply: prompt and get user's response.
 **	 setuserenviron: get terminal type and extract bold strings
 **      lock_file, unlock_file: do file locking
 **	 outchar:	function to print out char (uses macro putc)
 **
 **/

#ifndef lint
 static char rcsid[] = "$RCSfile: utils.c,v $ $Revision: 1.3 $ $Date: 1992/06/18 21:09:22 $" ;
#endif

#include  "netmon.h"

#ifdef NFS
#include <fcntl.h>				/* for lock_file()	*/
#endif


/*+ 		get_reply
** FUNCTION:
** 	This function prompts a user and gets a response from him. The
** 'prompt' & 'default' are supplied to the function - it returns a
** pointer to a static character string which is the 'reply'.
**/

#ifndef C_ANY
*
 * Define types of user responses for get_reply()
 */
#define C_ANY		0x0	/* Any character (printable or non-) */
#define C_ALPHA		0x1	/* A-z */
#define C_DIGIT		0x2	/* 0-9 */
#define C_SPACE		0x4	/* SPACE, TAB */
#define C_PUNCT		0x8	/* All punctuation characters */

#endif	/* C_ANY */

char *get_reply (prompt, deflt, response_type)
     char *prompt;
     char *deflt ;
     int  response_type ;		/* types defined above */
{
    static char reply[MAXLINE];
    char *r ;				/* temp pointer */
    int nodefault = 0, invalid = 0;

    reply[sizeof(reply) - 1] = '\0' ;		/* terminate with a NULL */

    if (deflt == NULL || *deflt == NULL)
      nodefault = 1 ;				/* No default value	*/

    printf("%s ", prompt);
    if (nodefault)
      printf(": ");
    else
      printf("[%s]: ", deflt) ;
    if (strlen(prompt) > 70)				/* long string	*/
      putchar('\n');

 again:
    r = (char *)reply ;
    gets(reply) ;
    if ( *reply == NULL )
    {
	if ( nodefault )
	{
	    printf("Invalid NULL response!! Enter again: ");
	    goto again;
	}
	else	/* default has been supplied */
	{
	    strncpy(reply, deflt, sizeof(reply) - 1) ;
	    return (reply) ;
	}
    }
    
    /*
     * Now check the user's response against the desired response_type
     */

    if (response_type == C_ANY)
      return (reply);

    invalid = 0;
    while (!invalid && *r)
    {
	register int ch_type = classify (*r++) ;

	if ( (response_type & ch_type) == 0)
	{
	    fprintf (stderr, "Invalid reply. ");
	    if ( isprint(*(--r)) )
		fprintf(stderr, "(character '%c' illegal)\n", *r);
	    fprintf(stderr, " Enter again: ");
	    invalid = 1 ;
	    break ;
	}
    }	/* end while */

    if (invalid)
      goto again;
    else
      return (reply);

}	/* end get_reply	*/

/*
 * Return class of character 'c' - one of C_ALPHA, C_DIGIT, ... Used
 * by get_reply()
 */
classify (c)
  int c ;
{

    if (isalpha(c))
      return (C_ALPHA) ;

    else if (isdigit(c))
      return (C_DIGIT) ;

    else if (isspace(c))
      return (C_SPACE) ;

    else if (ispunct(c))
      return (C_PUNCT);

    else
      return (C_ANY);

}	/* end classify */
      
/*  */
/*+		setuserenviron 
** FUNCTION
**
**	This function attempts to set the environment variables
** relevant to the program (like TERM). It modifies the structure
** 'environ' which is a pointer to the strings containing the
** variables. If an essential environment variable is not set, then
** it sets the variable by prompting or by a default value.
**
** 'putenv' requires that the storage for the variable be static.
**
** Lastly gets the value of the termcap variable to enable 'bold'
** on the terminals. 'md' = bold, 'mh' = half-intensity, 'me' = end all
**
**/

setuserenviron ()
{
    char termtype[MAXLINE], bp[1024] ;	     	/* needed by tgetent	*/
    char *get_reply() ;
    static char newvar[MAXLINE];		/* has to be static	*/
    
    bzero(termtype, sizeof (termtype)) ;
    strcpy (termtype,getenv("TERM"));		/* retrieve TERM type	*/
    
    if (check_terminal(termtype) == -1)       	/* Extract terminal type */
      return(-1) ;
    
    sprintf (newvar, "TERM=%s\0",(char *)termtype);
    if (putenv(newvar) != 0)			/* add the new string  */
      return (-1) ;
    
    /*
     * Extract the bold string from the '/etc/termcap' ("md")
     */
    if ( tgetent(bp, termtype) != 1 )		/* some error   */
    {
	bolds[0] = '\0' ; bolde[0] = '\0' ;	/* Null strings	*/
	clscr[0] = '\0' ; bellstr[0] = '\0' ;
    }
    else					/* bp has valid data	*/
    {
	char *s = (char *)bolds  ;
	tgetstr("md", &s) ;			/* extract bold string	*/
	s = (char *)bolde ;
	tgetstr("me", &s) ;			/* end all attr string	*/
	s = (char *)clscr ;
	tgetstr("cl", &s);
	s = (char *)bellstr ;
	if (tgetstr("bl", &s) == NULL)
	  strcpy(bellstr, "\007") ;
    }
    
}	/* end setuserenviron	*/


/*+ 		check_terminal
** FUNCTION:
** 	Check the supplied terminal string against the arrays
** 'good_term' and 'bad_terminals'. Sets the value of the terminal type.
** Return 1 if ok, -1 if error.
**/
check_terminal(ptermtype)
     char *ptermtype ;
{
    register char **p ;
    char bp[1024];		/* to extract 'tgetent' entry  */
    int tries = 2 , badterm = 1 ;

    /* Always prompt  so user can see the terminal type */
    strcpy(ptermtype, get_reply("Terminal type", ptermtype,
				C_ALPHA|C_DIGIT|C_PUNCT));
    for ( p = good_terminals ; **p != NULL ; ++p )
      if (strcmp (*p, ptermtype) == 0)
	return(1) ;

    while (tries--)
    {				
	fprintf (stderr, "Searching for terminal type '%s'...", ptermtype);

	if ( tgetent(bp, ptermtype)  == 1 )
	{
	    badterm = 0 ;			/* Okay terminal type */
	    fprintf(stderr, "\n") ;
	    /*
	     * Now check against unacceptable terminals
	     */
	    for ( p = bad_terminals ; **p != NULL ; ++p )
	      if (strcmp (*p, ptermtype) == 0)
	      {
		  badterm = 1 ;
		  break ;
	      }
	}
	else				/* Couldn't get term entry */
	{
	    badterm = 1 ;
	    fprintf(stderr, "not found\n") ;
	}

	if (!badterm)
	  return (1) ;

	if (tries)
	  strncpy(ptermtype, get_reply("Terminal type", "vt100",
				      C_ALPHA|C_DIGIT|C_PUNCT), MAXLINE -1);
	else
	  break ;

    }		/* end while() */

    strcpy(ptermtype, "dumb") ;		/* set terminal type to dumb	*/
    fprintf(stderr, "Setting terminal type to '%s'\n", ptermtype) ;
    sleep (2) ;
    if ( tgetent(bp, ptermtype) <= 0 )		/* can't even find dumb */
    {
	fprintf(stderr, "(%s) ERROR: tgetent cannot find entry for '%s'\n",
		prognm, ptermtype);
	return (-1) ;				/* error */
    }
    else
      return(1);
}
     
/*  */

/*+ 		lock_file
 ** FUNCTION:
 ** 	Locks entire file as "read" or "write". Needs the file
 ** descriptor, the type of lock desired ("r" or "w"), whether it should
 ** try to remove the lock forcibly after waitsecs.
 ** If NFS is not defined, 'flock' is used instead of 'fcntl'.
 ** Also gets the PID of the locking process.
 **
 ** Returns '-1' if couldn't lock file anyway.
 +*/

lock_file(fd, ltype, force, waitsecs)
     int fd, force, waitsecs ;
     char *ltype;
{
    void *wakeup() ;				/* On getting ALARM  */
    int lockvalue;				/* make things easier	*/
#ifdef NFS					/* In NFS files		*/
    struct flock lk ;				/* To lock the file	*/
    
    if ( *ltype == 'w' || *ltype == 'W')
      lk.l_type = F_WRLCK;
    else
      lk.l_type = F_RDLCK ;
    lk.l_whence = 0, lk.l_start = 0 ;		/* from start of file	*/
    lk.l_len = 0 ;				/* lock entire file	*/

    if(fcntl(fd, F_SETLK, &lk) == -1)
    {
	struct flock glk ;	/* for getting info about the lock  */

	fcntl(fd, F_GETLK, &glk) ;	/* Get description of lock  */
	if (glk.l_type == F_UNLCK)	/* Its now unlocked	*/
	  ;				/* so continue */
	else
	{
	    fprintf(stderr, "(%s) fcntl: ", prognm) ;
	    fprintf(stderr, "file locked by another (pid %ld), waiting...\n",
		    (long)glk.l_pid);

	}

	signal(SIGALRM, wakeup);		/* Wait for waittime secs */
	alarm(waitsecs);			/* ... and retry	*/
	if (fcntl(fd, F_SETLKW, &lk) == -1)	/* second try at locking */
	{				       	/* ... fails on SIGALRM */
	    perror("lock_file");
	    if (force)			/* Unlock  */
	    {
		fprintf(stderr, "forcibly removing lock\n");
		if (unlock_file(fd) == -1)
		  return(-1);
		else
		  return( fcntl(fd, F_SETLK, &lk) );	/* Return what hap */
	    }
	    else
	      return(-1) ;		/* Couldn't lock, and not force */

	}      	/* end second try at locking */
    }		/* end if( fcntl failed ) */


#else						/* In BSD 4.3 systems	*/

    if ( *ltype == 'w' || *ltype == 'W')	
      lockvalue = LOCK_EX ;
    else
      lockvalue = LOCK_SH ;			/* shared lock for read	*/
    
    if (flock(fd, lockvalue | LOCK_NB) == -1 )	/* see if locked already */
    {
	if (errno == EWOULDBLOCK)		/* Would block 		*/
	{				
	    fprintf(stderr,
		    "(%s) flock: file locked by another, waiting...\n",
		    prognm);
	}
	else					/* Some other error 	*/
	{
	    perror("lock_file");
	    return(-1);
	}

	signal(SIGALRM, wakeup);		/* Wait for waittime secs */
	alarm(waitsecs);			/* ... and retry	*/

	if (flock (fd, lockvalue) == -1)	/* Second try, with waiting */
	{					/* ... fails on SIGALRM */
	    perror("lock_file");
	    return(-1) ;
	}
    }		/* end if LOCK_NB	*/

#endif
    return(0) ;

}	/* end: lock_file	*/


/*+ 		unlock_file
** FUNCTION:
** 	Unlocks file.
**/

unlock_file(fd)
     int fd;
{
#ifdef NFS					/* In NFS files		*/
	struct flock lk ;			/* To lock the file	*/

	lk.l_type = F_UNLCK ;
	lk.l_whence = 0, lk.l_start = 0 ;	/* from start of file	*/
	lk.l_len = 0 ;				/* unlock entire file	*/
	if (fcntl(fd, F_SETLK, &lk) == -1)
	{
	    perror("(unlock_file) fcntl");
	    return(-1) ;
	}
#else
	if (flock(fd, LOCK_UN))			/* In BSD 4.3 systems	*/
	{
	    perror("(unlock_file) flock");
	    return(-1) ;
	}
#endif
	return(0) ;
} 	/* end: unlock_file	*/

/*+ 
** FUNCTION:
** 	Just wakeup from the ALARM signal. Do nothing.
**/
void *wakeup()
{
    signal(SIGALRM, SIG_DFL);		/* restore default action */
    if (options & debug)
      fprintf(stderr, "(debug) wakeup: reset SIGALRM to SIG_DFL") ;
}

/*  */

/*+ 
 ** FUNCTION:
 ** 	Prints out character to the terminal. Used by tputs, and hence
 ** cannot be a macro
 **/
outchar(c)
     char c ;
{
    putc (c, stdout) ;
}

