/*
 * $Header: /nocol/src/netmon/RCS/build_display.c,v 1.7 1992/06/18 21:02:56 aggarwal Exp $
 */

/* Copyright 1992 JvNCnet, Princeton */

/*+ 
** It builds the 'display' screen by calling routines which write to the
** various windows (wmsg, whdr, wtitle, wprompt).
**
**	1) smile
**	2) display static data
**	3) event display
**	4) msgs display
**	4) refresh
**	5) poll and parse user input
**	6) never return
**
** displaying the messages needs a small trick - in the debug mode,
** the new mesage would overwrite the previous message because the
** debug message was writing independent of a file.
** Thus I used 'msg_on' which is a small toggle. The variable is
** turned off in 'msg_dpy()' at the end of all the files. The function
** is then not called by build_display() right away, but it is called
** after one skip. Seems to be a bit muddly, but pretty effective.
**
** RETURN VALUES
**
**	-1 on severe error.
**
**/

/*+ 
 *
 *	$Log: build_display.c,v $
 * Revision 1.7  1992/06/18  21:02:56  aggarwal
 * Cleaned up for releasing.
 *
 *
 * Revision 1.3  1990/05/23  17:19:44  aggarwal
 * Added msg_on so that the function 'msg_dpy()' is skipped once
 * after displaying all the files.
 *
 * Revision 1.1  90/03/09  13:04:36  aggarwal
 * Initial revision
 * 
 */

#include 	"netmon.h"
#include 	<sys/dir.h>			/* needed by 'open_file' */
#include 	<sys/file.h>			/* for raw i/o */

#define WFULL(w)  	(w->_cury == (w->_maxy - 1)) ? 1:0

build_display ()
{
    extern struct allwins aw;			/* defined in netmon.h	*/
    extern int level, options, page;		/* defined in netmon.h	*/
    extern int msg_on ;				/* defined in netmon.h	*/

    for (; ;)
    {
	hdr_dpy(aw.whdr);			/* display header	*/
	msgtitle_dpy(aw.wmsgtitle);		/* display msg title	*/
	if (event_dpy() == -1)			/* resets page number	*/
	  return(-1);
	title_dpy (aw.wtitle, page);		/* variable page number	*/
	++page ;		  		/* increase after displaying */
	if (msg_on)				/* reset in msg_dpy()	*/
	{
	    if (msg_dpy() == -1)
	      return(-1);
	}
	else
	  msg_on = 1 ;				/* Turn it on		*/

	if (display_screenful() == -1) 		     /* display */
	  return (-1);

	parse_input(poll_input()); 		/* process user input 	*/
    }						/* end:  endless for	*/
}					       	/* end: build_display	*/


/***********************  FUNCTIONS  ***********************************/

/*+ 
** The following functions write static information to the various 
** windows.
**/

title_dpy(wtitle,screen)			/* Main title		*/
     WINDOW *wtitle;
     int screen;				/* to show screen num	*/
{
    char	ptime[27];			/* present time		*/
    time_t	stamp ;				/* time stamp		*/

    stamp = time ((time_t *)0) ;
    strncpy (ptime, ctime ((long *)&stamp), 26);

    mvwprintw (wtitle, 0, 0, "%-3.2d", screen);
    mvwprintw (wtitle, 0, (int)(COLS/2 - 8), "%s", "JvNCnet NOCOL");
    mvwprintw (wtitle, 0, (int)(COLS - 30), "%26s", ptime);
    return(0);
}						/* end:  title_dpy	*/


hdr_dpy(whdr)					/* field headers	*/
     WINDOW *whdr ;
{
    extern int	options ;

    if (options & emode)			/* extended mode	*/
      mvwprintw (whdr, 0, 0, EHDR);
    else
      mvwprintw (whdr, 0, 0, SHDR);		/* short header		*/
    return(0) ;
}						/* end:  hdr_dpy	*/

msgtitle_dpy(wmsgtitle)				/* Messages title	*/
     WINDOW *wmsgtitle ;
{
    extern bool frozen ;			/* in netmon.h		*/
    extern int  isdefined_pattern;		/* in read_filter.c */
    extern char pattern[] ;			/* in read_filter.c */

    wmove(wmsgtitle, 0, 0) ;
    if (frozen)				       	/* on first line of win	*/
    {					    	/* in reverse video	*/
	wstandout (wmsgtitle);
	wprintw (wmsgtitle, "SCREEN FROZEN") ;
	wstandend (wmsgtitle);
    }
    else
      wclrtoeol (wmsgtitle) ;			/* erase line		*/

    /*
     * display the pattern after the 'screen locked' message (about 20 chars
     * long). The show_display_level should overwrite in case this gets too
     * long to display.
     */
    if (isdefined_pattern)
    {
	mvwprintw(wmsgtitle, 0, (int)max(COLS/4,  20), "FILTER ON: ") ;
	wprintw (wmsgtitle, "%.*s..", COLS/2, pattern);
    }

    wclrtoeol (wmsgtitle) ;			/* erase rest of line	*/

    show_display_level(wmsgtitle, level);	/* on first line of win	*/
    mvwprintw (wmsgtitle, 1, (int)(COLS/2 - 4), "MESSAGES");
    return(0) ;
}						/* end:  msgtitle_dpy	*/

/*+ 		show_display_level
** FUNCTION:
** 	Write out the display level in the first line in the message
** title window for information.
**/

show_display_level(wmsgtitle,level)		/* on first line of win	*/
     WINDOW *wmsgtitle;
     int level;
{
    mvwprintw (wmsgtitle, 0, COLS - 25, "  Display Level: ");
    if (level == E_CRITICAL)
      wprintw (wmsgtitle, "%-8.8s", "CRITICAL");
    else if (level == E_ERROR)
      wprintw (wmsgtitle,  "%-8.8s", "ERROR");
    else if (level == E_WARNING)
      wprintw (wmsgtitle,  "%-8.8s", "WARNING");
    else if (level == E_INFO)
      wprintw (wmsgtitle,  "%-8.8s", "INFO");
    else
      wprintw (wmsgtitle,  "%-8.8s", "UNKNOWN");
    wclrtoeol (wmsgtitle) ;
    return(0); 
}						/* end:  show_dpy_level	*/

/*
 * Display the prompts
 */
aprompt_dpy (wprompt)		/* Prompt at end of a full screen	*/
     WINDOW *wprompt;
{
    mvwprintw (wprompt, 1, 0, PROMPTA);		/* blank line above	*/
    wclrtoeol (wprompt);
    return(0);
}						/* end: aprompt_dpy	*/

bprompt_dpy (wprompt)	   /* Prompt after displaying all files		*/
     WINDOW *wprompt ;
{
    mvwprintw (wprompt, 1, 0, PROMPTB);		/* blank line above	*/
    wclrtoeol (wprompt);
    return(0) ;
}						/* end: bpromt_dpy	*/

/*+ 		display_screenful
**
** FUNCTION
**
**	Display screen of 'wmain' on the terminal. Forces cursor
** location by setting wmain's current location to that of the
** prompt windows current location.
**
** I tried a lot of variations (all logical according to me) on the
** screen update part - like get rid of touchwin() and call
** refresh(promptw) at the end and enable leaveok() etc. Tried it
** between BSD + Ultrix + Sun, and the 'subtle' differances forced me
** to add touchwin()  and  force the cursor location be turning
** leaveok off + moving the current wmain location to the actual
** wprompt location.
**
**/

display_screenful ()
{
    extern struct allwins aw;		/* defined in netmon.h		*/
    extern int options ;

    int x, y ;					/* To move the cursor	*/
    if (WFULL(aw.wevent))
      aprompt_dpy(aw.wprompt);			/* display full  prompt	*/
    else
      bprompt_dpy(aw.wprompt);			/* display help prompt	*/

    if (options & debug)
    {
	wprintw(aw.wmsg,
		"\n(DEBUG): frozen = %s, options = 0x%x\n", 
		frozen ? "YES" : "NO", options);
	wprintw(aw.wmsg,		
		"(DEBUG): Term= %s, LINES= %d, COLS= %d, Datadir= %s", ttytype,
		LINES, COLS, datadir) ;	 	/* ttytype in curses.h */
	wclrtoeol(aw.wmsg);
    }
    
    /*
     * Am calling touchwin() since its foolproof (see discussion above)
     */
    touchwin (aw.wmain);
    y = aw.wprompt->_cury + aw.wprompt->_begy ;	/* force cursor loc..	*/
    x = aw.wprompt->_curx + aw.wprompt->_begx ;	/* ..to prompt window */
    wmove(aw.wmain, y, x);
    wrefresh (aw.wmain);			/* Onto screen */
    return (0) ;
}			/*** end:    display_screenful  ***/
