/*
 * $Header: /noc/network/netlog/src/RCS/create_entry.c,v 2.3 1992/05/15 22:55:29 aggarwal Exp $
 */

/* Copyright 1992 JvNCnet, Princeton University */

/*+ 
 ** This module creates an entry in the netlog and also updates the
 ** 'key' index file. The format of the key index file is described
 ** elsewhere. 
 **
 ** It creates the OPENTICKETS file if it does not exist (adds a
 ** header to it to make it look pretty). Also adds a RECSEP as a record
 ** delimiter so that the header is retained when GREPREC runs and deletes
 ** the first entry).
 **
 ** Logic behind this adding an entry is simple - 
 **	1) ...
 **	2) Create OPENTICKET file if not existent
 **	3) Fill ticket entry structure
 **		- date + time
 **		- get initials by using the saved 'ruid'
 **		- if ticket is U or C, then get open-ticket number
 **	4) Edit the new entry
 **		- ask for who to mail
 **	5) Fork - parent return's to main, child continues
 **	6) Update netlog
 **		- if O or X, get PREVTKT (lock the file)
 **		- format the entry
 **	     	- append to netlog (create dir + today's file if needed)
 **	   	- update (add/delete) OpenTickets file
 **		- mail the entry out
 **
 ** Updating netlog can be done in the background by a forked process.
 **
 ** CAVEATS
 **
 ** 
 ** RETURN VALUE
 **	0 for entry added (reached fork stage)
 **	1 for entry not added
 **	-1 for fatal error
 **/

/*
 *
 *	$Log: create_entry.c,v $
 * Revision 2.3  1992/05/15  22:55:29  aggarwal
 * Added code in locking to identify the host and locker of
 * the locked file.
 *
 * Revision 2.2  1992/05/11  19:29:33  aggarwal
 * During the prompt for additional person to mail it to, get_reply()
 * should have been called with C_ANY.
 *
 * Revision 2.1  1992/05/10  17:27:41  aggarwal
 * Added line that said 'could not update log' in the save_by_email mail.
 *
 * Revision 2.0  1992/05/10  16:28:26  aggarwal
 * Cleaned up and restructured for releasing 'netlog v2.0'
 *
 * Revision 1.14  1992/03/10  00:02:17  aggarwal
 * 
 * Revision 1.10  90/04/17  13:13:31  aggarwal
 * The part where the mail addresses were prompted for, was
 * appending a space even if nothing was to be mailed out.
 * Thus, the check later on for a NULL at the start
 * of the string bombed out.
 * 
 * Revision 1.3  90/03/02  13:19:51  aggarwal
 * Made sure that all the sprintf() commands have a '\0' at the end.
 * 
 * Revision 1.2  90/03/01  17:34:22  network
 * The mail was not appending a space between the names. So modified 
 * get_mail_rec() so that it appends a space between names.
 * 
 * Revision 1.1  90/03/01  15:48:31  network
 * Initial revision
 * 
 */

/*  */
#ifndef lint
  static char rcsid[] = "$RCSfile: create_entry.c,v $ $Revision: 2.3 $ $Date: 1992/05/15 22:55:29 $" ;
#endif

#include "netlog.h"
#include <pwd.h>			/* To get the username from uid	*/
#include <math.h>			/* for atoi() functions		*/

static int do_fork = DO_FORK ;		/* whether to fork or not */


create_entry()
{
    struct ticket_entry te ;			/* defined in netlog.h	*/
    char temp[MAXLINE], 			/* Temp for misc	*/
         *reply ;				/* pointer to the reply	*/
    register i ;				/* index		*/
    int childpid;				/* for the 'fork' call	*/

    bzero((char *)&te, sizeof(te)) ;		/* zero out the struct	*/

    /*
    ** The following tests if the user has write permission to the 
    ** essential files/dirs. If not, then return.
    */
    if (create_opentkt_file() == -1) 		/* create hdr file	*/
      return(-1) ;

    for (i = 0 ; *write_dirs[i] != NULL ; ++i )
      if (access (write_dirs[i], W_OK) == -1)
      {
	  fprintf(stderr, "create_entry (write access) ") ;
	  perror (write_dirs[i]) ;
	  return(-1) ;
      }					/* need write permission */
    for (i = 0 ; *write_files[i] != NULL ; ++i )
      if(strchr (write_files[i], '/') != NULL)	/* if pathname qualified */
	if (access (write_files[i], W_OK) == -1)
	{
	    Fprintf("create_entry (write access) ") ;
	    perror (write_files[i]) ;
	    return(-1);
	}


    get_user_info(&te) ;			/* username, host, time */

    if (get_tentry_data(&te) != 1)		/* Prompt for frills */
      return(-1) ;
    strncpy (te.stage, "Gotten ticket entry data", strlen(te.stage - 1 ));

    if (edit_new_entry(&te) != 1)		/* Get text of ticket. */
      return(-1) ;

    if (debug || !do_fork)			/* don't fork */
    {
	printf("Updating netlog\n");
	return (update_netlog(&te)) ;
    }

    else					/* Fork off */
    {
	if ((childpid = fork()) == 0)	/* successfully created child	*/
	{
	    struct ticket_entry cte ;	/* Child uses new structure	*/

	    bcopy ((char *)&te, (char *)&cte, (int)sizeof(te));
	    close(0), close(1) ;		/* No  stdout/stdin in child */
	    exit (update_netlog(&te));		/* Update and die  */
	}

	else if (childpid < 0)		/* couldn't fork successfully	*/
	{			     	/* ..so parent has to update */
	    perror("create_entry - fork") ;
	    printf("\nPlease wait while entry is being added\n") ;
	    return (update_netlog(&te)) ;
	}
    }

    return (0) ;				/* if ever reached */

}		/* end:  create_entry()  */


/*+*******		create_opentkt_file		**********
** FUNCTION:
** 	If the file does not exist, create it and copy the header over
** Also append a RECSEP after the header (for technical reasons to
** offset the way that GREPREC works). Only useful the first time when
** the openticket file does not exist. Needed so that the openticket
** file has RECSEP automatically.
**
*/
create_opentkt_file()
{
    int opentfd = -1, nhdrfd = -1,		/* File descriptors	*/
         rv = 0 ;				/* return value		*/

    if (access(Opentickets, W_OK) == 0)		/* file exists & write	*/
      return (0);
    if ((opentfd = creat(Opentickets, 0666)) == -1)
    {					
	fprintf(stderr, "ERROR: creat %s ", Opentickets) ;
	rv = -1;
	goto done ;
    }
    lock_file (opentfd, "w", 0, 2) ;	/* don't force, wait for 2 sec */

    if ((nhdrfd = open(Netloghdr, O_RDONLY)) < 0)	/* no hdr file	*/
    {
	fprintf(stderr, "ERROR: open %s", Netloghdr) ;
	rv = -1;
	goto done ;
    }
    if (copy_using_open_files_desc(nhdrfd, opentfd) == -1)
    {
	rv = -1;
	goto done ;
    }

    if (write(opentfd, (char *)RECSEP, strlen(RECSEP)) == -1 )
    {
	Fprintf("write ") ;
	rv = -1;
	goto done ;
    }
    write(opentfd, "\n", 1) ;			/* add a newline	*/
 done:
    if (rv == -1)			/* return value indicates error	*/
      perror("create_opentkt_file") ;

    unlock_file(opentfd) ;

    if ( opentfd >= 0 )
      close (opentfd) ;
    if ( nhdrfd >= 0 )
      close (nhdrfd) ;
    return (rv);
}					/* end: create_opentkts_file	*/


/*+*****		get_tentry_data			*********
** FUNCTION:
** 	Fills in the structure ticket_entry. Calls get_open_tkt_number()
** to get the open ticket number if ticket status is CLOSE or UPDATE.
** If status is X or O, fills in zero in the 'tnum' value. The ticket
** number should then be gotten from PREVTKT somewhere else. Also asks
** for the mail recipients and fills in the 'pte->mailrec' string.
**
** Returns 1 if the ticket is to be further edited, 0 if the ticket is to
** be ignored (if we get a zero from get_open_tkt_number ).
**/

get_tentry_data(pte)
     struct ticket_entry *pte ;		/* ptr to the struct filled	*/
{
    char *reply,
         temp[MAXLINE] ;			/* temporary storage	*/
    time_t stamp ;				/* for getting the time	*/
    struct tm *timest ;				/* returned pointer	*/

    stamp = time((long *)NULL);
    timest = localtime ((long *)&stamp) ;
    
    /*
    ** Fill in the date and logdir value. This value should not be
    ** entered by the user since he/she can really mess the system up.
    */
    sprintf (temp,"%02d%02d\0", timest->tm_mon + 1,timest->tm_year);
    pte->tlogdir = atoi(temp) ;			/* Log dir mmyy		*/
    sprintf(temp, "%02d%02d%02d\0", 
	    timest->tm_mon + 1, timest->tm_mday, timest->tm_year) ;
    pte->tdate = atoi(temp) ;			/* convert to integer	*/
    
    /* 
    ** Now fill in the query data (time, status, initials). Put a NULL at
    ** end of the strings, and copy only 'sizeof - 1' characters to string.
    */
    
    pte->tinit[sizeof(pte->tinit) - 1] = '\0' ;	/* Null char at end	*/
    pte->tsite[sizeof(pte->tsite) - 1] = '\0' ;

    sprintf(temp, "%02d%02d\0", timest->tm_hour, timest->tm_min) ;
    reply = get_reply("Enter time", temp, C_DIGIT) ;
    sscanf(reply, "%d", &(pte->ttime)) ;
    pte->ttime = (pte->ttime == 0)? atoi(temp):pte->ttime;  /* Non zero	*/

    get_initials(pte->whoami, pte->tinit) ;
    strncpy (pte->tinit, get_reply("Enter initials", pte->tinit, C_ANY), 
	     sizeof(pte->tinit) - 2) ;

    strncpy (pte->tsite, get_reply("Enter site", (char *)NULL, C_ANY),
	     sizeof(pte->tsite) - 2);

    reply= get_reply("Ticket status (O=open, U=update, C=close, X=info)", "X",
		     C_ALPHA );
    pte->status[0] = toupper(reply[0]) ;

    if (pte->status[0] == 'U' || pte->status[0] == 'C')
    {
	pte->tnum = get_open_ticket_number() ;	/* displays open tkts	*/
	if (pte->tnum == 0)
	  return (0);			/* User is an idiot. Dont add	*/
    }
    else if (pte->status[0] == 'O')	/* open new ticket		*/
      pte->tnum = 0 ;			/* get new ticket number later	*/
    else
    {
	pte->status[0] = 'X';		/* default value is 'Inform	*/
	pte->tnum = 0 ;			/* get new ticket number later	*/
    }
    return (1) ;			/* entry to be further edited	*/
}					/* end: get_tentry_data		*/


/*+ 			get_initials
** FUNCTION:
** 	Gets the initials of the user from the real uid. Must pass it
** the 'init' character which should be large enuf to hold 5 characters.
**
**/
get_initials(name, initials)
     char *name, *initials ;			/* 5 char array	storage	*/
{
    static char sinitial[sizeof(initials)] ;
    char cmd[MAXLINE];
    FILE *pcmd ;				/* for popen		*/

    if (*sinitial == NULL)		     	/* not gotten ever	*/
    {
	sprintf(cmd, "%s %s %s\0", EGREP, name, Initialsfile);
	if ((pcmd = popen(cmd, "r")) == NULL)
	  perror("get_initials (popen)") ;
	else
	{
	    fscanf(pcmd, "%*s %s", sinitial) ;
	    pclose(pcmd) ;
	}
    }					/* end: if not gotten ever	*/
    strncpy(initials, sinitial, sizeof(initials) - 1);
    return(0);
}			/* end:  get_initials()	*/

/*+ 		get_open_ticket_number
** FUNCTION:
** 	Prompts user and gets the ticket number of an open ticket by
** displaying the OPENTICKETS file and getting a ticket number which
** is already present in the file (for UPDATE or CLOSE entry). Returns
** the ticket number if user's entry was in the OPENTICKETS file else 
** returns 0.
**
** Gives two chances for the user to enter a valid open ticket number. 
**/
get_open_ticket_number()
{
    char  temp[MAXLINE], grepcmd[MAXLINE],
          morecmd[MAXLINE], *reply ;
    int i, valid = 0 ;			/* temporary index	*/
    int tkt ;

    /* write the command to display file,,, */
    sprintf(morecmd, "%s %s | %s\0", CAT, Opentickets, MORE);
    for (i = 0 ; i < 2 && !valid ; ++i )	/* give two chances	*/
    {
	printf("Following is the list of open tickets.\n");
	printf("Enter\n\t 'f' \t to move forward.\n");
	printf("\t 'b' \t to move backward.\n") ;
	printf("\t 'q' \t to return to main menu\n\n");
	printf("Hit RETURN to continue: "), gets(temp) ;
	
	system(morecmd) ;
	reply = get_reply("\nEnter ticket number you are updating/closing", 
			  (char *)NULL, C_DIGIT | C_ALPHA);
	if (tolower(*reply) == 'q')		/* user quits		*/
	  return(0) ;

	sscanf(reply, "%d", &tkt) ;		/* get ticket number	*/
	sprintf(grepcmd,
		"%s '\t#%d :'  %s >/dev/null\0", EGREP, tkt, Opentickets);
	if (system(grepcmd) == 0)		/* match found		*/
	  valid = 1;
	else
	  printf("\n No such open ticket !\n") ;
    }						/* end: for 2 chances	*/

    if (valid)					/* ticket number found	*/
      return (tkt) ;
    else
      return (0) ;				/* User is an idiot	*/
}						/* end: get_open_tkt_#	*/

/*+****** 		edit_new_entry			*********
** FUNCTION:
** 	Edits the new entry, and asks if to be added to netlog. If added,
** then returns 1; if not added, then deletes the edited entry and 
** returns 0. Also fills in the mail_rec field in the structure.
**/

edit_new_entry(pte)
     struct ticket_entry *pte ;
{
    char *reply,
         editcmd[MAXLINE];	  	/* the command to edit file	*/
    char edtfile[MAXLINE] ;	     	/* new edited ticket entry	*/

    fflush(stdout) ;			/* Clear all buffered data	*/
    strcpy(edtfile, "/tmp/Ne.XXXXXX") ;	/* Create unique file name	*/
    mktemp(edtfile) ;			
    sprintf (editcmd, "%s %s\0", editor, edtfile);
    system (editcmd) ;			/* edit the new entry		*/

    strncpy(pte->stage, "edit_new_entry", sizeof(pte->stage) - 1) ;

    reply = get_reply("Add entry to netlog", "y", C_ALPHA);
    if (toupper(*reply) != 'Y')
    {
	reply = get_reply("Are you sure you want to lose entry","n", C_ALPHA) ;
	if (toupper(*reply) != 'N')
	{
	    unlink(edtfile);
	    return(0);			/* Entry not added to netlog	*/
	}
    }
    strcpy(pte->edtfile, edtfile);	/* save the edited filename	*/
    get_mail_rec(pte) ;		       	/* Fill in names of mail rec	*/
    return (1) ;			/* Entry to be added to netlog	*/
}					/* end: edit new entry		*/


/*+ 		get_mail_rec
** FUNCTION:
** 	Prompts the user for the names of people to send the entry to
** and fills in the names into the passed string.
**/
get_mail_rec(pte)
     struct ticket_entry *pte;	
{
    int maxstr = sizeof(pte->mailrec) ;	/* max length of string		*/
    char *reply, temp[MAXLINE] ;
    register char *tomail = pte->mailrec ;

    --maxstr ;				/* save a space for NULL char	*/
    *tomail = '\0';			/* If null, then nothing mailed	*/
    *temp = '\0' ;

#ifdef PROMPTMAIL1
    strcat (strcpy(temp, "Mail entry to  "), PROMPTMAIL1) ;
    reply = get_reply(temp, "y", C_ALPHA);
    if (toupper(*reply) == 'Y')
    {
	strncat(tomail, PROMPTMAIL1, maxstr - strlen(tomail) ) ;
	strncat(tomail, " ", maxstr - strlen(tomail)) ;	/* add a space	*/
    }
#endif

#ifdef PROMPTMAIL2
    strcat (strcpy(temp, "Mail entry to  "), PROMPTMAIL2) ;
    reply = get_reply(temp, "n", C_ALPHA);
    if (toupper(*reply) == 'Y')
    {
	strncat(tomail, PROMPTMAIL2, maxstr - strlen(tomail) ) ;
	strncat(tomail, " ", maxstr - strlen(tomail)) ;	/* add a space	*/
    }
#endif
    reply = get_reply("Mail entry to any other address", "n", C_ANY);
    if (toupper(*reply) == 'Y')
    {					/* allow entering any address	*/
	do				/*... but check length		*/
	{
	    reply = get_reply("Enter address(s)", (char *)NULL, C_ANY) ;
	    if (strlen(reply) > (maxstr - strlen(tomail)) )
	      printf("Address length too long. Enter shorter address\n");
	    else
	      break ;
	} while (1) ;
	strncat(tomail, reply, maxstr - strlen(tomail) ) ;
    }			       	/* end : if mail to any other address	*/

    tomail[maxstr] = '\0' ;	/* for safety sake			*/

}		/* end : get_mail_rec()	*/



/*+****** 		update_netlog			*********
** FUNCTION:
** 	This formats the entry (getting appropriate new ticket number
** if necessary from PREVTKT), adds header to netlog, updates index and
** deletes a closed entry from the OPENTICKETS file.
**
** Call this routine ONLY if everything is OK - entry is edited and it
** is to be added. This routine can be called from a forked child, and
** the child might have no way of writing out errors. Check out the
** accessibility of the NROFF, GREPREC etc. before calling this routine.
** Since this runs in background, write errors to stderr or to a log file
** Treat updating log a serious matter :-)
**
** Blocks if any of the files are in use (PREVTKT, NETLOG).
** Mails out the entry in the end.
**
** If there is any error, this calls save_by_mail_entry()
**/

update_netlog(pte)
     struct ticket_entry *pte ;
{
    char  temp[MAXLINE] ;
    int rv = 0 ; 				/* return value */

    strncpy(pte->stage, "update_netlog", sizeof(pte->stage) - 1) ;

    if (pte->tnum == 0)				/* get new tkt number	*/
      if ((pte->tnum = get_new_tkt(pte)) < 0)
      {
	  Fprintf("(update_netlog) new ticket number less than 0\n");
	  rv = -1 ; goto done ;
      }

    if (format_entry(pte) == -1)		/* format new entry	*/
    {
	rv = -1 ;
	goto done ;
    }

    if (append_to_netlog(pte) == -1)		/* error in appending..	*/
    {
	rv = -1 ; 
	goto done ;	
    }

    if (toupper(pte->status[0]) == 'C')		/* only after appending	*/
      if (delete_open_ticket(pte) == -1)	/* ..delete from open tkt */
	fprintf(stderr, 
		"(update-netlog) Error in deleting #%d from %s, do manually\n",
		pte->tnum, Opentickets) ;

    if (toupper(pte->status[0]) == 'O')		/* Add to list of open	*/
      if (add_new_open_tkt(pte) == -1)
	fprintf(stderr,
		"(update-netlog)Error in adding #%d to %s, do manually\n",
		pte->tnum, Opentickets);

    update_ot_hdrs(pte);			/* Open tkts hdr file	*/

    mail_entry(pte) ;				/* Mail out entry	*/

 done:
    if (rv == -1)				/* Some error */
      save_by_mail_entry (pte, sys_errlist[errno]) ;
    unlink(pte->edtfile) ;			/* delete the file	*/
    return (rv) ;				/* all okay		*/
}		/* end: update_netlog	*/

/*+ 		get_new_tkt
** FUNCTION:
** 	gets new ticket number after adding 1 to the PREVTKT file.
** To make sure that the ticket wrapping is handled well, the program
** zeroes out the next MAXTKT/2 entries in the index file when it
** reaches MAXTKT/2 or MAXTKT. It also sends a mail to the OWNER
** when it wraps the ticket numbers.
**
** Returns the new ticket number.
**/
get_new_tkt(pte)
     struct ticket_entry *pte ;
{
    int tfd, ptlock ;				/* file descriptors	*/
    int prevtkt, newtkt ;
    char temp[MAXLINE];

    tfd = open(Prevtkt, O_CREAT | O_RDWR, 0666);
    ptlock = open(Ptlock, O_WRONLY | O_CREAT, 0666);

    if (lock_file(ptlock, "w", 0, 10) == -1)
    {
	struct ticket_entry t ;

	Fprintf("(get_new_tkt) Error in locking PREVTKT ptlock\n") ;
	if (read(ptlock, &t, sizeof(t)) == sizeof(t))
	  fprintf(stderr, ".. locked by: %s @ %s since %s", t.whoami, t.host,
		  t.asctime);
	unlock_file(ptlock) ;
	close(tfd), close(ptlock);
	return(-1);
    }
    else
      write(ptlock, pte, sizeof(struct ticket_entry)) ;	/* info about locker */

    if (read(tfd, temp, MAXLINE) <= 0)
      prevtkt = -1 ;
    else
      sscanf(temp, "%d", &prevtkt) ;		/* get previous ticket	*/

    newtkt = (prevtkt + 1) % MAXTKT ;		/* update prev tkt	*/
    sprintf(temp, "%d\0", newtkt);

    ftruncate(tfd, (off_t)0);				/* zero file out */
    lseek(tfd, (off_t)0, L_SET) ;		/* rewind the file	*/
    write(tfd, temp, strlen(temp)) ;		/* have NULL at str end	*/
    fsync(tfd);					/* make sure out to disk */
    close(tfd) ;				/* close ticket file	*/
    ftruncate(ptlock, (off_t)0);		/* null out information */
    unlock_file(ptlock) ;
    close(ptlock);

    /*
    ** Now make sure that the previous ticket is not at any stage
    ** where wrapping might cause a problem. If so then zero out
    ** the index file for the next MAXTKT/2 values
    */
    if (prevtkt == MAXTKT || prevtkt == (int)(MAXTKT * WRAP_THRESHOLD))
    {					
	register i ;
	int indexfd ;
	char mailcmd[MAXLINE] ;

	if ((indexfd = open(Indexfile, O_RDWR)) < 0)
	{
	    Fprintf ("(get_new_tkt) open ") ;
	    perror(Indexfile) ;
	    return(-1) ;
	}
	if (prevtkt == MAXTKT)			/* zero from 0 - THRES	*/
	  for (i = 0; i < (int)(MAXTKT * WRAP_THRESHOLD) ; ++i)
	    delete_index_entry(i, (char *)NULL, indexfd) ;

	else					/* zero from THRES - MAX */
	  for (i = (int)((MAXTKT * WRAP_THRESHOLD) + 1); i <= MAXTKT ; ++i)
	    delete_index_entry(i, (char *)NULL, indexfd) ;

	close(indexfd);
	sprintf(mailcmd, "%s -s \"%s: %s %d, %s  %d-%d\" %s < /dev/null\0",
		MAIL,  prognm, "Reset ticket numbers at ", 
		prevtkt, "Archive old logs", 
		(prevtkt == MAXTKT ? 0 : (int)(MAXTKT * WRAP_THRESHOLD)),
		(prevtkt == MAXTKT ?(int)(MAXTKT * WRAP_THRESHOLD)+1 : MAXTKT),
		 MAINTMAIL);
	system(mailcmd);
    }
	
    return(newtkt) ;
    
}		/* end: got new tkt num	*/


/*+ 		format_entry
** FUNCTION:
** 	Formats the file passed to it using NROFF. It makes a temp
** file and then renames the formatted file to the original filename.
** Appends beginning of log file's header (Netloghdr) and also the
** time + ticket number + status on the first line. Adds a RECSEP
** line at the bottom.
**
** Emacs leaves no blank line at the end and that screws up NROFF. Thus
** we add an extra blank line at the bottom of the edited entry.
**
** THE FORMAT OF THE TOPMOST LINE IN AN ENTRY IS IMPORTANT SINCE THE
** TICKET NUMBERS ARE SEARCHED ASSUMING THIS FORMAT. THIS FORMAT IS:
**
** 	<time>\t#<tkt number> :<status>\t<initials>
**/

format_entry(pte)
     struct ticket_entry *pte;
{
    char fmtcmd[MAXLINE], buf[MAXLINE];
    char fmtfile[MAXLINE] ;		/* Temp filename for formatted	*/
    FILE *fmtp, *fptr; 			/* fomatted fp, edited fp	*/

    if (debug)
      fprintf(stderr, "\nFormatting entry..\n ");

    if ((fptr = fopen(pte->edtfile,"a+")) == NULL)
    {					/* Append a blank line at end..	*/
	Fprintf("format_entry (fopen)") ;
	perror(pte->edtfile);
	return (-1);
    }
    fprintf(fptr,"\n");			/* ..of newly edited entry and.	*/
    fclose(fptr);			/* .. close the file		*/

    strcpy (fmtfile, "/tmp/Nf.XXXXXX");	/* Create unique file name	*/
    mktemp(fmtfile);
    sprintf(fmtcmd, "%s %s %s | %s > %s\0", 
	    CAT, Nroffhdr, pte->edtfile, NROFF, fmtfile);
    system(fmtcmd) ;

    fmtp = fopen(fmtfile, "r") ;
    if ((fptr = fopen(pte->edtfile,"w")) == NULL)	/*  truncate	*/
    {						/* truncate orig file..	*/
	fclose(fmtp) ;
	Fprintf("format_entry (fopen)") ;
	perror(pte->edtfile);			/* ..and make it a ..	*/
	unlink(fmtfile);		       	/* nicely fmted entry	*/
	return(-1);
    }
    /*
     * The format of the top line in an entry is set here...
     */
    fprintf(fptr, "%04.4d\t#%d :%c\t\t%s\t%s\n\n",
	    pte->ttime, pte->tnum, *(pte->status), pte->tinit, pte->tsite) ;
    while (fgets(buf, MAXLINE, fmtp) != NULL)
      fputs(buf, fptr) ;			/* copy formatted entry */
    fputs (RECSEP, fptr);			/* Put RECSEP at end	*/
    fputs ("\n", fptr);				/* add a newline at end	*/
    fclose(fptr), fclose(fmtp);			/* close all files	*/
    unlink(fmtfile);				/* deleted the tmp file	*/

    return(0);
}			/* end: format entry	*/

/*+ 		append_to_netlog
** FUNCTION:
** 	Appends new entry to today's netlog. Locks the file while updating
** it, waits if it is in use. Forms the netlog filename from the date
** in the 'ticket_entry' structure. Also updates the INDEX file.
**
** Locks the files during update using LOCK. That is because the system
** programs "cp" and "cat" have been used and in case they don't lock.
**
*/
append_to_netlog(pte)
     struct ticket_entry *pte ;
{
    char netlogf[MAXLINE],			/* netlog filename	*/
         syscmd[MAXLINE] ;			/* system command	*/
    int lockfd ;				/* Lock file descriptor	*/
    time_t stamp ;				/* for the time stamp	*/

    if (debug)
      fprintf(stderr, "\nAppending to netlog..\n  ");

    strncpy(pte->stage, "append_to_netlog", sizeof(pte->stage) - 1) ; 
   /*
    ** Make sure that the directory for the month exists
    */
    sprintf(netlogf,"%s/%04.4d\0", LOGDIR, pte->tlogdir);
    if (access(netlogf, F_OK) != 0 )	/* No directory	*/
      if (errno != ENOENT || mkdir(netlogf, 0777) == -1) /* Cant create dir */
      {
	  Fprintf("(append_to_netlog) ERROR- access/mkdir ");
	  perror(netlogf) ;
	  return (-1);
      }
    
    sprintf(netlogf,"%s/%04.4d/netlog.%06d\0",LOGDIR,pte->tlogdir, pte->tdate);

    /*
     * If file doesn't exist, create one and copy the header over.
     */
    if (access(netlogf, F_OK) != 0)		/* file doesn't exist	*/
    {
	int rv, nread, nhdrfd, nlogfd ;	     	/* hdr & log file descrip */
	char buffer[BUFSIZ] ;
	
	if ((nlogfd = creat(netlogf, 0666)) == -1)	/* create file */
	{					
	    Fprintf("(append_to_netlog) creat ");
	    perror(netlogf);
	    return (-1) ;
	}
	if ((nhdrfd = open(Netloghdr, O_RDONLY)) < 0)	/* no hdr file	*/
	{
	    Fprintf("(append_to_netlog) open ") ;
	    perror(Netloghdr);
	    close (nlogfd) ;			/* close earlier opened file */
	    return (-1) ;
	}

	stamp = time((long *)NULL) ;
	sprintf(buffer, "%s\0", ctime((long *)&stamp));	/* nice time	*/
	nread = strlen(buffer);

	do					/* copy header file over */
	  if (write(nlogfd, buffer, nread) == -1)
	  {
	      Fprintf("(append_to_netlog) write ");
	      perror(netlogf) ;
	      close (nlogfd) , close (nhdrfd) ;
	      return (-1) ;
	  }
	while ((nread = read(nhdrfd, buffer, BUFSIZ)) > 0) ;

	sprintf(buffer, "%s\n", RECSEP) ;	/* add a record separator */
	write(nlogfd, buffer, strlen(buffer)) ;

	fsync (nlogfd) ;
	close (nlogfd), close (nhdrfd) ;

    }	/* end: if (netlogf dont exist) */

    /*
     * Now simply cat the formatted entry over to the existing log file
     */
    lockfd = open (Loglock, O_RDWR | O_CREAT, 0666) ;
    if (lock_file (lockfd, "w", 0, 60) == -1)	/* wait 60 seconds */
    {
	struct ticket_entry t ;

	Fprintf("(append_to_netlog) ERROR- couldn't lock LOGLOCK\n");
	if (read(lockfd, &t, sizeof(t)) == sizeof(t))
	  fprintf(stderr, "..locked by '%s' @ %s since %s\n",
		  t.whoami, t.host, t.asctime);
	strncpy(pte->stage, "append_to_netlog- could not lock log",
		sizeof(pte->stage) - 1) ;
	return(-1) ;
    }
    else
      write(lockfd, pte, sizeof(struct ticket_entry)) ;	/* info about locker */

    sprintf(syscmd, "%s %s >> %s\0", CAT, pte->edtfile, netlogf );
    system(syscmd);				/* Append the entry	*/
    ftruncate(lockfd, (off_t)0);	       	/* null out lock info */
    unlock_file(lockfd) ;
    close(lockfd);				/* Auto deletes lock	*/

    add_index_entry(pte->tnum, pte->tdate, Indexfile, 0) ;

    return(0);					/* All okay		*/
}				/* end:append_to_netlog	*/

/*+		add_new_open_tkt
** FUNCTION:
** 	Adds the newly edited entry to the file of open tkts (OPENTICKETS).
** Add a date stamp (assumes that the file has been created (in init() ).
**/
add_new_open_tkt(pte)
      struct ticket_entry *pte ;
{
    char buffer[BUFSIZ];			/* defined in stdio.h	*/
    int newtfd, opentfd ;			/* new tkt+open tkt fd	*/
    int otlock;					/* Open ticket lock	*/
    register int nread ;			/* num of char read	*/

    if (debug)
      fprintf(stderr,"\nAdding to Open-Tickets..\n  ");

    strncpy(pte->stage, "add_new_open_tkt", sizeof(pte->stage) - 1) ;

    otlock = open(Otlock, O_CREAT | O_RDWR, 0666);
    if (lock_file(otlock, "w", 0, 30) == -1)
    {
	struct ticket_entry t ;

	Fprintf("(add_new_open_tkt) ERROR- couldn't lock OTLOCK\n");
	if (read(otlock, &t, sizeof(t)) == sizeof(t))
	  fprintf(stderr, "..locked by '%s' @ %s since %s\n",
		  t.whoami, t.host, t.asctime);
	close(otlock) ;
	return(-1) ;
    }
    else
      write(otlock, pte, sizeof(struct ticket_entry)) ;	/* info about locker */

    if ((newtfd = open((char *)pte->edtfile, O_RDONLY)) < 0)
    {					/* open newly fmted entry	*/
	perror ("add_new_open_tkt (Open new entry file)") ;
	ftruncate(otlock, 0) , close(otlock) ;
	return (-1);
    }
    if ((opentfd = open(Opentickets, O_RDWR | O_APPEND)) < 0)
    {				
	close (newtfd) ;
	perror ("add_new_open_tkt (open tkts file)") ;
	ftruncate(otlock, 0) , close(otlock) ;
	return (-1);
    }
    
    sprintf(buffer, "%06.6d\n", pte->tdate) ;	/* grab  date 		*/
    nread = strlen(buffer) ;			/* get str len		*/

    do						/* append new entry	*/
      if (write(opentfd, buffer, nread) == -1)
      {
	  perror ("add_new_open_tkt (write open tkt file)") ;
	  break ;
      }
    while ((nread = read(newtfd, buffer, BUFSIZ)) > 0) ;
    fsync(opentfd);
    close (newtfd), close (opentfd) ;	/* Close up opened descriptors	*/
    ftruncate(otlock, 0) , close(otlock) ;
}					/* end: add_new_open_tkt()	*/

/*		delete_open_ticket
** FUNCTION:
**     	Deletes the ticket entry from the open-tkt file. Uses the program
** 'grep_record' to delete the entry. Assumes that OPENTICKETS exists
** and is writable (makes no attempt to create if non-existent).
**/

delete_open_ticket (pte)
     struct ticket_entry *pte ;
{
    int otlock;				/* Locking OTLOCK		*/
    int tkt = pte->tnum ;		/* ticket num to be deleted */
    char grepcmd[MAXLINE*4],
         cpycmd[MAXLINE],		/* Cannot use rename, use cp	*/
         tmpotf[MAXLINE] ;		/* temp open ticket file	*/

    if (debug)
      fprintf(stderr,"\nDeleting open ticket..\n  ");

    if (access(Opentickets, W_OK) == -1)
    {
	Fprintf("delete_open_ticket (access) ");
	perror(Opentickets);
	return (-1);
    }

    strcpy (tmpotf,"/tmp/No.XXXXXX");	/* Create unique file name	*/
    mktemp (tmpotf) ;
    sprintf(grepcmd, "%s -vs '%s' -e '%s' '\t#%d :' %s > %s\0",
	    GREPREC, RECSEP, RECSEP, tkt, Opentickets, tmpotf) ;
    otlock = open(Otlock, O_CREAT | O_RDWR, 0666);
    if (lock_file(otlock, "w", 0, 30) == -1)
    {
	struct ticket_entry t ;

	Fprintf("(add_new_open_tkt) ERROR- couldn't lock OTLOCK\n");
	if (read(otlock, &t, sizeof(t)) == sizeof(t))
	  fprintf(stderr, "..locked by '%s' @ %s since %s\n",
		  t.whoami, t.host, t.asctime);
	close(otlock) ;
	return(-1) ;
    }
    else
      write(otlock, pte, sizeof(struct ticket_entry)) ;	/* info about locker */

    system(grepcmd);			/* remove the extra entry	*/
    sprintf(cpycmd, "%s %s %s\0", COPY, tmpotf, Opentickets) ;
    if (system(cpycmd) == -1)		/* copy tmp file over		*/
    {
	perror(cpycmd);			/* some fatal error in copying	*/
	ftruncate(otlock, 0) , close(otlock);
	return(-1);
    }
    unlink(tmpotf);			/* Not deleted if uncopied	*/
    unlock_file(otlock) ;
    ftruncate(otlock, 0) , close(otlock);
    return (0);
}					/* end: delete open ticket	*/     

/*+ 		update_ot_hdrs
** FUNCTION:
** 	Update the netlog open ticket header file. This 'extra' effort
** simply creates/updates a file of the Open Ticket Headers so that
** they can be displayed in NETMON. Lesser processing here than having
** to grep the headers later from the list of Open Tickets.
**/
update_ot_hdrs(pte)
     struct ticket_entry *pte;
{
    int othlock;

    if (toupper(pte->status[0]) == 'U' || toupper(pte->status[0]) == 'X')
      return(0);

    strncpy(pte->stage, "update_ot_hdrs", sizeof(pte->stage) - 1) ;

    if ((othlock = open(Othlock, O_RDWR | O_CREAT, 0666)) < 0)
    {
	perror(Othlock);
	return(-1);
    }
    lock_file(othlock, "w", 0, 30);
    if (lock_file(othlock, "w", 0, 30) == -1)
    {
	struct ticket_entry t ;

	Fprintf("(add_new_open_tkt) ERROR- couldn't lock OTHLOCK\n");
	if (read(othlock, &t, sizeof(t)) == sizeof(t))
	  fprintf(stderr, "..locked by '%s' @ %s since %s\n",
		  t.whoami, t.host, t.asctime);
	close(othlock) ;
	return(-1) ;
    }
    else
      write(othlock, pte, sizeof(struct ticket_entry));	/* info about locker */

    if (toupper(pte->status[0]) == 'O')
    {
	FILE *fp;
	if ((fp = fopen(Opentickethdrs, "a")) == NULL)
	{
	    perror(Opentickethdrs);
	    return(-1);
	}
	fprintf(fp,"NETLOG: %06.6d %04.4d\t#%d :%c\t\t%s\t%s\n",
		pte->tdate, pte->ttime, pte->tnum, *(pte->status), 
		pte->tinit, pte->tsite) ;
	fclose (fp);
    }					/* end: if Open status	*/

    else if (toupper(pte->status[0]) == 'C')
    {
	char tmpfil[MAXLINE], syscmd[MAXLINE];	

	strcpy(tmpfil, "/tmp/Noth.XXXXXX");
	mktemp(tmpfil);
	sprintf(syscmd, "%s -v '\t#%d :' %s > %s\0",
		EGREP, pte->tnum, Opentickethdrs, tmpfil);
	system(syscmd);
	sprintf(syscmd, "%s %s %s\0", COPY, tmpfil, Opentickethdrs);
	system(syscmd);
	unlink(tmpfil);			/* delete temp file	*/
    }

    ftruncate(othlock, 0);
    unlock_file(othlock);
    close(othlock);
}			/* end  update_ot_hdrs	*/

      
/*+ 			mail_entry
** FUNCTION:
** 	Mails out the entry (name is in the pte->edtfile) to all the
** names specified in the pte->mailrec string. The names of recipients
** is filled in elsewhere.
**
** Also creates a suitable header file 'mailhdr' which is the NETLOGHEADER
** alongwith necessary information (date, etc.). This is appended to the
** ticket entry before mailing out.
**/
mail_entry(pte)
     struct ticket_entry *pte ;
{
    struct passwd *pwent ;			/* For signature	*/
    char mailcmd[MAXLINE*3],			/* long mail command	*/
         cpycmd[MAXLINE],			/* for system-ing	*/
         name[MAXLINE],				/* user's name		*/
         mailhdrf[MAXLINE] ;			/* file for header	*/
    FILE *hdrfp ;				/* header file ptr	*/

    if (debug)
      fprintf(stderr,"\nMailing out the entry..\n  ") ;

    if (*(pte->mailrec) == NULL)		/* nothing mailed out	*/
      return(0) ;

    if ((pwent = getpwuid(ruid)) == NULL)
      perror("Netlog mail (getpwuid)");
    if (pwent == NULL)
      sprintf(name, "UID %d\0", ruid) ;			/* User's uid	*/
    else 
      expand_geco(name, pwent);			/* Ampersand expansion	*/

    strcpy(mailhdrf, "/tmp/Nm.XXXXXX");
    mktemp(mailhdrf);
    if (create_blank_file(mailhdrf,0666) == -1)	/* preserve mode	*/
      return (-1) ;
    sprintf (cpycmd, "%s %s %s\0", COPY, Netloghdr, mailhdrf) ;
    system (cpycmd);				/* copy hdr file over	*/
    if ((hdrfp = fopen(mailhdrf, "a")) == NULL)
    {
	Fprintf("Netlog mail");
	perror(mailhdrf);
	return(-1);
    }
    fprintf (hdrfp, "Date: %06.6d\n",pte->tdate); /* Add date stamp	*/
    fclose(hdrfp);				/* now have a file	*/

    sprintf (mailcmd,
	     "%s %s %s | %s -s \"(JvNCnet NETLOG) %s (From: %s)\" %s\0",
	     CAT, mailhdrf, pte->edtfile, MAIL, pte->tsite, name,pte->mailrec);

    system (mailcmd) ;			/* mail out the entry		*/
    unlink (mailhdrf);		      	/* delete the pretty header	*/
}			/* end:  mail_entry */

/*+ 		expand_geco
** FUNCTION:
**	Expands the ampersand in the gecos field and gets rid of funny
** and extraneous ',' characters. Sampled from 'finger'.
**/

expand_geco(name, pwd)
     char *name;			/* should have enough storage	*/
     struct passwd *pwd ;
{

#define ASTERISK	'*'		/* ignore this in real name 	*/
#define COMMA		','		/* separator in pw_gecos field	*/
#define AMPERSAND	'&'		/* repeat login name in real	*/

    register char *bp, *gp, *lp;	/* Temp pointers	*/

    gp = pwd->pw_gecos;
    bp = name;
    if (*gp == ASTERISK)
      gp++;
    while (*gp && *gp != COMMA)
      if (*gp == AMPERSAND)
      {
	  lp = pwd->pw_name;			/* The login name	*/
	  *bp = toupper(*lp);			/* First character	*/
	  bp++, lp++ ;				/* Now skip first char	*/
	  while (*bp++ = *lp++)			/* Copy the login name	*/
	    ;
	  bp--;					/* Skip the end NULL	*/
	  gp++;					/* Skip the ampersand	*/
      } 
      else
	*bp++ = *gp++;
    *bp++ = 0;					/* Throw in a NULL	*/
    
}						/* end:  expand_geco	*/

/*+		get_user_info
** FUNCTION:
** 	Fill in the user and host information and time stamp.
**/
get_user_info(pte)
     struct ticket_entry *pte ;
{
    struct passwd *passwd ;
    char tmp[256] ;			/* hostname, etc. */
    time_t clock = time((long *)NULL) ;	/* extract time in secs */
    struct tm *tm = localtime(&clock) ;

    passwd = getpwuid(ruid) ;		/* get invoker's information */
    strncpy (pte->whoami, passwd->pw_name, sizeof(pte->whoami) - 1);
    pte->whoami[sizeof(pte->whoami) - 1] = '\0' ;

    gethostname(tmp, sizeof(tmp) );
    strncpy (pte->host, tmp, sizeof(pte->host) - 1) ;
    pte->host[sizeof(pte->host) - 1] = '\0' ;

    sprintf (pte->asctime, "%2.2d:%2.2d:%2.2d\0",
	     tm->tm_hour, tm->tm_min, tm->tm_sec) ;

    return (0) ;
}		/* end get_user_info */

/*+ 
** FUNCTION:
** 	Mails the entry out to the person making the entry in case of
** any major errors. This prevents entry from being lost.
**/
save_by_mail_entry(pte, reason)
     struct ticket_entry *pte ;
     char *reason ;
{
    char syscmd[MAXLINE] ;

    if (pte->edtfile == NULL)
      return (0) ;			/* No filename */

    sprintf(syscmd, "%s -s \"COULD NOT UPDATE LOG (%s at %s)\" %s %s <%s\0", 
	    MAIL, reason, pte->stage, pte->whoami, MAINTMAIL, pte->edtfile);

    fprintf(stderr,
	    "(save_by_mail): LOG ENTRY NOT UPDATED, emailing to '%s & %s'\n",
	    pte->whoami, MAINTMAIL) ;
    return (system(syscmd)) ;

}		/* save_by_mail_entry */
