/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 * PROBLEM FILE management routines
 */

#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <pwd.h>
#include <sys/param.h>

#include "pathnames.h"
#include "CommonDefs.h"
#include "problem.h"

#define MAXNOTES 	MAXNODES
#define MAXCOMMENT 	30

struct ConfigType {
	char ICare;
	char Name[MAXNODENAME];
	char Comment[MAXCOMMENT];
} Config[MAXNOTES];
static int NumConfig = 0;

int NumProblems = 0;
struct ProblemType ProblemArray[MAXPROBLEMS];

static char *DefaultPingkyDir = DEFAULT_PINGKY_DIR;
static char *PingkyDir;

static char Problem_File[MAXPATHLEN]; 
static char Log_File[MAXPATHLEN];
static char Config_File[MAXPATHLEN];

static char buffer[BUFSIZ];		/* static buffer for misc uses */

static time_t ProblemFileLastUpdated;
static time_t ConfigFileLastUpdated;

static int ProblemFileChanged = 0;

static int Add_Problem_Entry(), AddProblem(), DelProblem();
static void ReadConfigFile();
static char *Comment();

static int BatchCount=0, BatchMode=0;
static struct BatchType {
	char NodeName[ MAXNODENAME ];
	char UniqueId[ MAXUNIQUEID ];
	char TestType[ MAXTESTNAME ];
	int Type;	/* ADD/DELETE etc */
} Batch[ MAXNODES ];

#ifdef STANDALONE
char *thisprogname;
#endif

static char ProblemPrefix[ MAXNODENAME ];

/*
 *	SetProblemPrefix() - Allow the caller to provide a prepended
 *			character sequence to all names being added to the
 *			problem file.
 *
 */
int SetProblemPrefix( Prefix )
char * Prefix;
{
	if ( Prefix == NULL ) return 1;
	if ( strlen( Prefix ) >= MAXNODENAME ) return 1;
	strcpy( ProblemPrefix, Prefix );
	return 0;
}

/****************************************************************
 *   Problem_Manager: 						*
 *	cmd:							*
 *		ADD_PROBLEM:  Add a problem to the list		*
 *				( If not already there )	*
 *		ADD_PRIMARY_PROBLEM: Add a problem deleteing	*
 *			all other problems with this node 	*
 *		DELETE_PROBLEM: Remove this problem		*
 *				( if there is one )		*
 *		UPDATE_PROBLEM: Update this problem's StatusLine*
 *				( if there is one )		*
 *		CLAIM_PROBLEM: Claim this problem		*
 *				( just like UPDATE PROBLEM )    *
 *  								*
 *		FUTURIZE_PROBLEM: Advance the TimeStamp to	*
 *				somewhere in future		*
 *  								*
 ****************************************************************/

Problem_Manager(cmd, type, Node, UniqueID, status)
int cmd;		/* Add_Problem, Del_Problem, Add_Problem_Primary*/
char *type;		/* Type of Problem - string text		*/
char *Node;		/* name of node we are reporting 	 	*/
char *UniqueID;		/* Unique Identifier for this node(like IP addr)*/
char *status;		/* Status Line - if cmd is UPDATE_PROBLEM	*/
{
	int i, changed = 0;
	extern char *getenv(), *progname();
	time_t TimeNow,t;
	static int virgin = 1;
	struct passwd *mypass;
	char *name;
	char node[ MAXNODENAME ];	/* Copy of caller's node name */
	char uid[ MAXUNIQUEID ];	/* Copy of caller's node name */

	if ( Node != NULL ) {
		strcpy( node, ProblemPrefix );	/*Copy in Problem Prefix*/
		strncat( node, Node, MAXNODENAME - strlen(ProblemPrefix) -1 );
		node[ MAXNODENAME-1 ] = '\0';	/* NULL TERMINATE NAME */
	} else node[0]='\0';

	if ( UniqueID != NULL ) {
		strncpy( uid, UniqueID, MAXUNIQUEID-1 );
		uid[ MAXUNIQUEID-1 ] = '\0';	/* NULL TERMINATE NAME */
	} else uid[0]='\0';

	ProblemFileChanged = 0;
	if (virgin) {	
		/* set-up config files we will access */
		if ((PingkyDir = getenv("PINGKYDIR")) == NULL)
			PingkyDir = DefaultPingkyDir;
		sprintf(Problem_File, "%s/%s", PingkyDir, PROBLEM_FILE);
		sprintf(Log_File, "%s/%s", PingkyDir, PROBLEM_LOG);
		sprintf(Config_File, "%s/%s", PingkyDir, PROBLEM_MGR_CONF);
		virgin = 0;
	}

	/* reread config file if it has changed */
	if (FileChanged(Config_File, &ConfigFileLastUpdated)) {
		ReadConfigFile(Config_File);
		FileChanged(Config_File, &ConfigFileLastUpdated);
	}
	/* reread problem file if it has changed */
	if (FileChanged(Problem_File, &ProblemFileLastUpdated)) {
		NumProblems = ReadProblemFile(ProblemArray, Problem_File);
		FileChanged(Problem_File, &ProblemFileLastUpdated);
		/*ProblemFileChanged = 1;@wbn*/
		ProblemFileChanged = 1;  /* Why??? @wbn*/
	}

	/****************** Handle Caller's Command *******************/
	switch(cmd) {

	case ADD_PRIMARY_PROBLEM:
	changed = 0;
	for (i = 0; i < NumProblems; i++)
		if (strcmp(ProblemArray[i].UniqueID,uid) == 0)
			if (strcmp(ProblemArray[i].TestName,type) !=0 ) {
				ProblemArray[i].Name[0] = '\0';
				changed = 1;
			}
	if (changed) {
		WriteProblemFile(ProblemArray, Problem_File, NumProblems);
		NumProblems = ReadProblemFile(ProblemArray, Problem_File);
		ProblemFileChanged = 1;
	}
	/* now fall through to the next case */

	case ADD_PROBLEM:
	NumProblems = AddProblem(node, uid, type, ProblemArray, NumProblems);
	break;

	case DELETE_PROBLEM:
	NumProblems = DelProblem(node, uid, type, ProblemArray, NumProblems);
	break;

	case UPDATE_PROBLEM:
	case CLAIM_PROBLEM:

	if ((i = Problem_Exists(uid,type,ProblemArray,NumProblems)) == -1)
		/* XXXX error message here? */
		break;

	strncpy(ProblemArray[i].StatusLine, status, MAXSTATUSLINE - 1);
	ProblemArray[i].StatusLine[MAXSTATUSLINE - 1] = '\0';
	/* NCS - pkn - We need to know who updating or claiming the problem. */
	if ((mypass = getpwuid(getuid())) == NULL)
		name = "Unknown_Person";
	else
		name = mypass->pw_name;

	if (cmd == UPDATE_PROBLEM)
		sprintf(buffer," !%s!%s!%s Update: %s (%d mins elapsed)\n",
			ProblemArray[i].Name, ProblemArray[i].UniqueID,
			name, ProblemArray[i].StatusLine,
			(time(&TimeNow) - ProblemArray[i].TimeStamp) / 60);
	else
		sprintf(buffer," !%s!%s!%s Claim: %s (%d mins elapsed)\n",
			ProblemArray[i].Name, ProblemArray[i].UniqueID,
			name, ProblemArray[i].StatusLine,
			(time(&TimeNow) - ProblemArray[i].TimeStamp) / 60);
	Log(buffer, Log_File);
	ProblemFileChanged = 1;
	WriteProblemFile(ProblemArray, Problem_File, NumProblems);
	NumProblems = ReadProblemFile(ProblemArray, Problem_File);
	break;

	case CHECK:
	break;

	case FUTURIZE_PROBLEM:
	if ((i = Problem_Exists(uid,type,ProblemArray,NumProblems)) == -1)
		/* XXXX error message here? */
		break;

	/* NCS - pkn - We need to know who updating or claiming the problem. */
	if ((mypass = getpwuid(getuid())) == NULL)
		name = "Unknown_Person";
	else
		name = mypass->pw_name;

	t = atol(status);
	sprintf(buffer," !%s!%s!%s Futurize: %s til %s\n",
			ProblemArray[i].Name, ProblemArray[i].UniqueID,
			name, ProblemArray[i].StatusLine, ctime(&t));
	Log(buffer, Log_File);
	ProblemArray[i].TimeStamp=atol(status);
	ProblemFileChanged = 1;
	WriteProblemFile(ProblemArray, Problem_File, NumProblems);
	NumProblems = ReadProblemFile(ProblemArray, Problem_File);
	break;

	case ADD_BATCH_PRIMARY_PROBLEM:
	case DELETE_BATCH_PRIMARY_PROBLEM:
		strcpy( Batch[BatchCount].NodeName, node );
		strcpy( Batch[BatchCount].UniqueId, uid );
		strcpy( Batch[BatchCount].TestType, type );
		Batch[BatchCount].Type=cmd;
		BatchCount++;
		break;
	case FLUSH_BATCH_PRIMARY_PROBLEM:
		BatchMode=1;	/* BatchMode - Don't read/write to disk */
		for( i=0; i<BatchCount; i++ ) {
			if ( Batch[i].Type == ADD_BATCH_PRIMARY_PROBLEM ) {
				NumProblems = AddProblem(Batch[i].NodeName, 
					Batch[i].UniqueId, Batch[i].TestType, ProblemArray, NumProblems);
			}
			else 
			if ( Batch[i].Type == DELETE_BATCH_PRIMARY_PROBLEM) {
				NumProblems = DelProblem(Batch[i].NodeName, 
					Batch[i].UniqueId, Batch[i].TestType, ProblemArray, NumProblems);
			}
		}
		BatchCount=0;
		BatchMode=0;
		if ( ProblemFileChanged ) {
			WriteProblemFile(ProblemArray, Problem_File, NumProblems);
			NumProblems = ReadProblemFile(ProblemArray,Problem_File);
		}
		break;

	default:
	Log("Problem_Manager- invalid cmd type!\n", Log_File);
	panic("Problem_Manager- invalid cmd type!\n");
	/*NOTREACHED*/
	}

	return(ProblemFileChanged);
}

/**************************************************************************
 * Add_Problem_Entry:  Add a problem to the Problem structure and to disk *
 * To Do: Open this file with the append flag                             *
 **************************************************************************/
static int Add_Problem_Entry(Node,UniqueID,ProblemType,Problem, n)
char *Node, *UniqueID, *ProblemType;
struct ProblemType *Problem;
int n;
{
	char *sl = NULL;	/* status line */

	(void) time(&Problem[n].TimeStamp);
	strncpy(Problem[n].Name, Node, MAXNODENAME - 1);
	Problem[n].Name[MAXNODENAME - 1] = '\0';
	strncpy(Problem[n].UniqueID, UniqueID, MAXUNIQUEID - 1);
	Problem[n].UniqueID[MAXUNIQUEID - 1] = '\0';
	strncpy(Problem[n].TestName, ProblemType, MAXTESTNAME - 1);
	Problem[n].TestName[MAXTESTNAME - 1] = '\0';

	sl = Comment(Node);
	if (strlen(sl) == 0) 
		sl = Comment(UniqueID);
	if (strlen(sl) > 0 ) {
		(void) strncpy(Problem[n].StatusLine, sl, MAXSTATUSLINE - 1);
		Problem[n].StatusLine[MAXSTATUSLINE - 2] = '\0';
		sprintf(buffer, " AUTO COMMENT FOR NODE %s %s = %s",
			Node, UniqueID, sl);
#ifdef DEBUG
		printf( " AUTO COMMENT FOR NODE %s %s = %s\n", Node, UniqueID,
									sl);
#endif
		Log(buffer, (char *) NULL);
	}
	else 
		Problem[n].StatusLine[0] = '\0';
	Problem[n].StatusLine[MAXSTATUSLINE - 1] = '\0';

#ifdef DEBUG
	printf("Add_Problem_Entry: ADDING %lu %s %s %s_%s_%s %s\n",
		Problem[n].TimeStamp,
		_Day(Problem[n].TimeStamp),
		_Time(Problem[n].TimeStamp),
		Problem[n].Name,Problem[n].UniqueID,
		Problem[n].TestName, 
		Problem[n].StatusLine);
#endif

	n++;
	if ( ! BatchMode ) {
		WriteProblemFile(Problem, Problem_File, n);
		n = ReadProblemFile(ProblemArray,Problem_File);
	}
	ProblemFileChanged = 1;
	return(n);
}

/********************************************************************
 * AddProblem:   Add a problem to the Problem File if necessary     *
 *  								    *
 *  Variables:							    *
 *	Node: The name of the node that is having problems.	    *
 *  	UniqueID: The IP UniqueIDess of the node that is having the problem *
 *	Service: The service that is failing (PING,TELNET,etc)	    *
 *	Problem_File: Name of the Problem File containing Problems  *
 ********************************************************************/
static int AddProblem(node, uid, service, problem, n)
char *node, *uid, *service;	/* node name, unique id, service */
struct ProblemType *problem;	/* problem pointer */
int n;				/* number of problems */
{
	if (TestNode(uid) || TestNode(node))
		return(n);
	if (Problem_Exists(uid, service, problem, n) == -1) {
		sprintf(buffer,"ADDING PROBLEM %s!%s!%s!%s\n",
			node, uid, service, Comment(node));
		Log(buffer, Log_File);
		n = Add_Problem_Entry(node, uid, service, problem,n);
	}
	return(n);
}

/********************************************************************
 * DelProblem:   Delete a problem in the Problem File if necessary  *
 *  								    *
 *  Variables:							    *
 *	Node: The name of the node that is now behaving.	    *
 *  	UniqueID: The IP UniqueIDess of the node that is behaving correctly *
 *	Service: The service that works now (PING,TELNET,etc)	    *
 *	Problem_File: Name of the Problem File containing Problems  *
 ********************************************************************/
/* XXXX node unused in this function */
static int DelProblem(node,uid,Service,Problem,n)
char *node,*uid;
char *Service;
struct ProblemType *Problem;
int n;
{
	int index;
	time_t TimeNow;

	(void) time(&TimeNow);
	if  ((index = Problem_Exists(uid, Service, Problem, n)) != -1) {
		sprintf(buffer,"DELETING PROBLEM %s!%s!%s (%d mins elapsed)\n",
			Problem[index].Name, Problem[index].UniqueID,
			Service,((TimeNow - Problem[index].TimeStamp) / 60));
		Log(buffer, Log_File);
		Problem[index].Name[0] = '\0';
		if ( ! BatchMode ) {
			WriteProblemFile(Problem, Problem_File, n);
			n = ReadProblemFile(Problem, Problem_File);
		}
		ProblemFileChanged = 1;
	}
	return(n);
}

/************************************************************************
 *	AUTOMATIC STATUS MESSAGES / TEST NODE SECTION			*
 ************************************************************************/

/****** ReadConfigFile - Default comments, and TestName Nodes ******/
static void ReadConfigFile( file )
char *file;
{
	FILE *fp;
	int i = 0, j, k;


	if ((fp = fopen(file,"r")) == NULL) {
		fprintf(stderr,"ReadConfig: config file %s unreadable! ...will try to create it\n",file);
		Empty_File( file );
		if ((fp = fopen(file,"r")) == NULL) {
			fprintf(stderr,"ReadConfig: config file %s unaccessible!\n",file);
			return;
		}
	}

	while((fgets(buffer, sizeof(buffer), fp) != NULL) && (i < MAXNOTES)) {
		if (buffer[0] == '#')
			continue;
		j = 0;
		if (buffer[j] == '*') {	
			Config[i].ICare = 0;	/* ignore */
			j++;
		}
		else 
			Config[i].ICare = 1;

		/*  Zero out the Name array in case any garbage currently 
		    exists.  Before, the old name may have left remnants */
		bzero( Config[i].Name, MAXNODENAME );  /* Thanks Dennis */

		for (k = 0; j < MAXNODENAME && !isspace(buffer[j]); j++, k++)
			Config[i].Name[k] = buffer[j];
		Config[i].Name[MAXNODENAME - 1] = '\0';

		for(j++, k = 0; j < strlen(buffer) && k < MAXCOMMENT; k++, j++)
			if (isprint(buffer[j]))
				Config[i].Comment[k] = buffer[j];
		Config[i].Comment[MAXCOMMENT - 1] = '\0';

#ifdef DEBUG
		printf("Name[%d]=%s Comment=%s ICare=%d\n", i, Config[i].Name,
					Config[i].Comment, Config[i].ICare);
#endif
		i++;
	}
	fclose(fp);
	NumConfig = i;
}

/***************************************************************************
 * IsConfig() - Is this node configured in the ProblemManager.config file? *
 *		Return index if it is					   *
 *		Return -1 if this node is a Test Node (Do Not Show)	   *
 *		Return -2 if there is a comment for this node		   *
 ***************************************************************************/
static IsConfig(node)
char *node;
{
	register int i;

	if (FileChanged(Config_File, &ConfigFileLastUpdated)) {
		ReadConfigFile(Config_File);
		FileChanged(Config_File, &ConfigFileLastUpdated);
	}
	for(i = 0; i < NumConfig; i++) {
		if (strcmp(Config[i].Name, node ) == 0) {
			if (Config[i].ICare == 0 ){
				return(-1);	/* TestNode Machine */
			}
			else 
				return(i);	/* index of node */
		}
	}
	return(-2); /* node not found */
}

/****************************************************************
 * TestNode() - Return whether or not this is a test node (0or1)*
 ****************************************************************/
int TestNode( node )
char *node;
{
	if (IsConfig(node) == -1) 
		return(1);		/* TestNode */
	return(0);			/* Non-TestNode */
}

/************************************************************************
 * Comment() - Return a comment associated with this node entry, or ""	*
 ************************************************************************/
static char *Comment( node )
char *node;
{
	int i;

	if ((i = IsConfig(node)) == -2)
		return("");
#ifdef DEBUG
	printf("%s Comment=%s\n",node, Config[i].Comment);
#endif
	return(Config[i].Comment);
}

#ifdef STANDALONE

main(argc, argv)
int argc;
char *argv[];
{
	extern void exit();

	thisprogname = argv[0];

	if ( argc < 5 ) {
		printf("%s Usage(): ADD|DELETE|UPDATE ProblemType NodeName NodeUniqueID [ status ]\n",argv[0] );
		exit(1);
	}

	if (strcmp( argv[1],"ADD")==0) 
		Problem_Manager(ADD_PROBLEM,argv[2],argv[3],argv[4],argv[5]);
	else if (strcmp( argv[1],"DELETE")==0) 
		Problem_Manager(DELETE_PROBLEM,argv[2],argv[3],argv[4],argv[5]);
	else if (strcmp( argv[1],"UPDATE")==0) 
		Problem_Manager(UPDATE_PROBLEM,argv[2],argv[3],argv[4],argv[5]);
	else printf("Unknown command: %s %s %s %s %s\n",
			argv[1],argv[2],argv[3],argv[4],argv[5]);

#ifdef SAMPLE
	Problem_Manager(ADD_PROBLEM,"PING","NODE1","NODE1ID",NULL);
	Problem_Manager(ADD_PROBLEM,"PING","NODE2","NODE2ID",NULL);
	Problem_Manager(ADD_PROBLEM,"PING","NODE3","NODE3ID",NULL);
	PrintProblems(NumProblems,ProblemArray);
	sleep(10);

	Problem_Manager(DELETE_PROBLEM,"PING","NODE2","NODE2ID",NULL);
	PrintProblems(NumProblems,ProblemArray);
	sleep(10);

	Problem_Manager(ADD_PRIMARY_PROBLEM,"DOWN","NODE1","NODE1ID",NULL);
	PrintProblems(NumProblems,ProblemArray);
	sleep(10);

	Problem_Manager(DELETE_PROBLEM,"PING","NODE3","NODE3ID",NULL);
	Problem_Manager(UPDATE_PROBLEM,"PING","NODE1","NODE1ID","AN UPDATE");
#endif
	PrintProblems(NumProblems,ProblemArray);
	exit(0);
}

#endif
