/*
 * snmp_api.c - 
 *
 *	This code provides the glue between the Merit Fast SNMP API
 *	and the CMU SNMP Code.
 *
 *	Thanks to Woody Huang (Merit) for doing this port to the CMU SNMP base
 *
 */
#include <sys/types.h>
#include <netinet/in.h>
#include <stdio.h>
#include <string.h>
#include <string.h>

#include "snmptable2.h"
#include "snmp.h"
#include "snmp_impl.h"
#include "asn1.h"
#include "snmpapi.h"
#include "mib_descio.h"

#define	MIB_DESC	"/etc/mib_desc"
#define	PACKET_LEN 1024 

int io_debug=0;

void fastsnmpsyserr(str)
     char *str;
{
	fprintf(stderr, "fastsnmpsyserr(): %s\n", str);
	perror("");
	exit(0);
}

/***************************************************************************
 *  make_SNMP_request() - Assemble a SNMP packet:  version:community:PDU   *
 *								           *
 *	Use caller's buffer and create a SNMP packet to caller's specs	   *
 *	Return length of SNMP packet.					   *
 ***************************************************************************/
static u_long 	Reqid;
static char 	ret_buf[100];	
extern 	int  errno;
int	snmp_dump_packet = 0;

int make_SNMP_request(type, community, num_reqs, preq, sva, packet, len)
     int type;
     char *community;
     int num_reqs;
     char **preq;
     long int sva;
     char *packet;
     int len;
{
	u_char  buf[PACKET_LEN];
	register u_char  *cp;
	int     i, j, length, req_len, comm_len;
	long    zero = 0;
	long	reqid, errstat, errindex;
	int     totallength;
	oid	objid[MAX_NAME_LEN];
        extern  int verbose;

	int IDtoOID();	
	void fastsnmpsyserr();

	length = len;
	cp = (u_char *) packet;

	for (i = 0; i < num_reqs; i++) {
		if (IDtoOID(preq[i], objid, &req_len) != 0) 
			fastsnmpsyserr("make_SNMP_request: IDtoOID error");
		cp = snmp_build_var_op(cp, (oid *) objid, &req_len, 
			ASN_NULL, 0, (u_char *) NULL, &length);
		if (cp == NULL)
		return 3;
	};

	totallength = (char *) cp - packet;

	length = PACKET_LEN;

	cp = asn_build_header(buf, &length, 
		(u_char) (ASN_SEQUENCE | ASN_CONSTRUCTOR), totallength);
	if (cp == NULL)
		return 1;

	bcopy((char *)packet, (char *)cp, totallength);
	totallength += cp - buf;
	length = len;

	reqid = (long) Reqid;
	cp = asn_build_int(packet, &length, 
		(u_char) (ASN_UNIVERSAL | ASN_PRIMITIVE | ASN_INTEGER), 
		&reqid, sizeof(reqid));
	if (cp == NULL)
		return 3;
	Reqid++;

	/* error status */
	errstat = 0;
	cp = asn_build_int(cp, &length,
		(u_char) (ASN_UNIVERSAL | ASN_PRIMITIVE | ASN_INTEGER),
		&errstat, sizeof(errstat));
	if (cp == NULL)
		return 3;

	/* error index */
	errindex = 0;
	cp = asn_build_int(cp, &length,
		(u_char) (ASN_UNIVERSAL | ASN_PRIMITIVE | ASN_INTEGER),
		&errindex, sizeof(errindex));
	if (cp == NULL)
		return 3;

	if (length < totallength)
		return 1;

	bcopy((char *)buf, (char *)cp, totallength);
	totallength += (char *) cp - packet;
	length = PACKET_LEN;

	cp = asn_build_header(buf, &length, 
		(type == 1)? GET_REQ_MSG : GETNEXT_REQ_MSG, totallength);
	if (cp == NULL)
		return 1;

	if (length < totallength)
		return 1;

	bcopy((char *)packet, (char *)cp, totallength);
	totallength += cp - buf;
	length = len;

	comm_len = strlen(community);
	cp = snmp_auth_build(packet, &length, (u_char *) community, 
			     &comm_len, &zero, totallength);
	if (cp == NULL)
		return 3;

	if ((len - ((char *) cp - packet)) < totallength)
		return 1;

	bcopy((char *)buf, (char *)cp, totallength);

	totallength += (char *) cp - packet;

	if (io_debug) {
		printf("Prepared SNMP packet for transmission: \n");
		for (i=0; i<totallength; i++ ) {
			printf("%2.2x ",packet[i] );
			if ((i+1) % 16 == 0 ) {
				printf("  ");
				for( j=i-15; j<=i; j++)
					if ( isprint(packet[j]) )
						printf("%c",packet[j]);
					else printf(".");
				printf("\n");
			}
		}
		printf("\n");
	}

	return totallength;
}

/************************************************************************
 * parse_SNMP_packet() - digest received SNMP packet and chain varbind	*
 *			to caller's list.				*
 *	return 0 or error status					*
 ************************************************************************/
int parse_SNMP_packet(packet, len, head, errvar)
     char *packet;
     int len;
     struct bindings **head;
     char **errvar;
{
	u_char	msg_type, type, *var_val, community[128], var_type;
	u_long	uptime;
	long	reqid, errstat, errindex, version, var_int;
	oid	*enterprise, *op, objid[MAX_NAME_LEN], var_objid[MAX_NAME_LEN];
	int	enterprise_len, trap_type, specific_type;
	int	community_length, four, i, j;
	int	var_name_len, var_val_len, datalen;
	struct 	sockaddr_in	agent_addr;
	struct 	variable_list *vp;
        extern  int verbose;
        char    *var_name_buf, *value_buf, value[PACKET_LEN];
	unsigned char	ipaddr[5],*ipaddrptr;
	static char var_name[200];	/* space for variable OID @wbn */
	int varbindindex;

	void 	fastsnmpsyserr();
	void	link_binding();

	if (io_debug) {
		printf("Received packet: \n");
		for (i=0; i<len; i++ ) {
			printf("%2.2x ",packet[i] );
                       if ((i+1) % 16 == 0 ) {
                                printf("  ");
                                for( j=i-15; j<=i; j++)
                                        if ( isprint(packet[j]) )
                                                printf("%c",packet[j]);
                                        else printf(".");
                                printf("\n");
                        }

		}
		printf("\n");
	}

	*head = NULL;	
	community_length = 128;

	packet = snmp_auth_parse(packet, &len, community, 
			&community_length, &version);
	if (packet == NULL) {
		return -1;
	};

	if (version != SNMP_VERSION_1) {
		fprintf(stderr, "Wrong version: %d\n", version);
		fprintf(stderr, "Continuing anyway\n");
	};

	packet = asn_parse_header(packet, &len, &msg_type);
	if (packet == NULL) {
		return -1;
	};

	if (msg_type != TRP_REQ_MSG) {
		packet = asn_parse_int(packet, &len, &type, (long *)&reqid, 
				sizeof(reqid));
		if (packet == NULL) {
			 return -1;
		};

		packet = asn_parse_int(packet, &len, &type, (long *)&errstat, 
				sizeof(errstat));
		if (packet == NULL) {
			return -1;
		};

		packet = asn_parse_int(packet, &len, &type, (long *)&errindex, 
				sizeof(errindex));
		if (packet == NULL) {
			return -1;
		};

		if ( errstat != 0 ) {		/* @wbn */
			if (verbose)
				fprintf(stderr,
					"parse_SNMP_packet(): rc=%d %s index=%d\n",
					errstat,SNMP_errormsg(errstat),errindex);
			/*return( errstat ); don't return yet*/
		}

	} else {
		enterprise_len = MAX_NAME_LEN;
		packet = asn_parse_objid(packet, &len, &type, objid, 
					&enterprise_len);
		if (packet == NULL) {
			return -1;
		};

		enterprise = (oid *)malloc(enterprise_len * sizeof(oid));

		bcopy((char *)objid, (char *)enterprise, 
			enterprise_len * sizeof(oid));

		four = 4;

		packet = asn_parse_string(packet, &len, &type, 
				(u_char *)&agent_addr.sin_addr.s_addr, &four);
		if (packet == NULL) {
			return -1;
		};

		packet = asn_parse_int(packet, &len, &type, 
				(long *)&trap_type, sizeof(trap_type));
		if (packet == NULL) {
			return -1;
		};

		packet = asn_parse_int(packet, &len, &type, 
				(long *)&specific_type, sizeof(specific_type));
		if (packet == NULL) {
			return -1;
		};

		packet = asn_parse_int(packet, &len, &type, (long *)&uptime, 
				sizeof(uptime));
		if (packet == NULL) {
			return -1;
		};
	}

	packet = asn_parse_header(packet, &len, &type);
	if (packet == NULL) {
		return -1;
	};

	if (type != (u_char) (ASN_SEQUENCE | ASN_CONSTRUCTOR)) {
		return -1;
	};

	vp = NULL;

	varbindindex=1;
	while((int)len > 0) {
		var_name_len = MAX_NAME_LEN;
		packet = snmp_parse_var_op(packet, objid, &var_name_len, &var_type,
				&var_val_len, &var_val, (int *)&len);
		if (packet == NULL) {
			return -1;
		};

		var_name_buf = var_name;
		for (i = 0; i < var_name_len; i++) {
			sprintf(var_name_buf, "%lu", objid[i]);
			if (i == (var_name_len - 1))
				break;
			strcat(var_name_buf, ".");
			var_name_buf = (char *) strchr(var_name_buf, '.');
			var_name_buf++;
		};

		if ( errstat && (varbindindex==errindex)) {
			*errvar=var_name;
			return(errstat);
		}
		varbindindex++;

		datalen = PACKET_LEN;
		switch( var_type ) {
			case ASN_INTEGER:
			case COUNTER:
			case GAUGE:
			case TIMETICKS:
				asn_parse_int(var_val, &datalen, &var_type, 
					      &var_int, sizeof(var_int));
				sprintf( value,"%u",var_int);
				break;
			case ASN_OCTET_STR:
			case OPAQUE:
				asn_parse_string(var_val, &datalen, &var_type, 
						 value, &var_val_len);
				value[var_val_len] = '\0';
				break;
			case IPADDRESS:
				ipaddrptr=ipaddr;
				asn_parse_string(var_val, &datalen, &var_type, 
						 ipaddrptr, &var_val_len);
				ipaddr[var_val_len] = '\0';
				sprintf(value,"%d.%d.%d.%d",ipaddr[0],ipaddr[1],ipaddr[2],ipaddr[3]);
			/*	printf("ipaddress=%s\n",value);*/
				break;
			case ASN_OBJECT_ID:
				var_val_len = MAX_NAME_LEN;
				asn_parse_objid(var_val, &datalen, &var_type, 
						var_objid, &var_val_len);
				value_buf = value;
				for (i = 0; i < var_val_len; i++) {
					sprintf(value_buf, "%lu", var_objid[i]);
					if (i == (var_val_len - 1))
						break;
					strcat(value_buf, ".");
					value_buf = (char *) strchr(value_buf, '.');
					value_buf++;
				};
				break;
			case ASN_NULL:
				sprintf( value,"NULL");
				break;
			default:
				fprintf( stderr,"Unknown Type %d", (int) var_type);
				sprintf( value,"Unknown Type");
			break;
		};
		/* link it - we have a valid varbind */
		link_binding(var_name, value, head);
		if (verbose)
			fprintf(stderr, "parse_SNMP_packet(): varname = %s, value = %s\n", var_name, value);
	};
	return 0;
}

/************************************************************************
 *  link_binding() - add this variable to caller's bind list		*
 ************************************************************************/
static void link_binding(var_name, value, head)
     char *var_name;
     char *value;
     struct bindings **head;
{
	static	struct bindings *b;
	struct bindings *b2;

	void fastsnmpsyserr();

	if ((b = (struct bindings *) malloc(sizeof(struct bindings))) 
	     == NULL)
                fastsnmpsyserr("parse_SNMP_packet(): bindlist malloc() failed");
	if ((b->instance = (char *) malloc(strlen(var_name)+1)) == NULL)
                fastsnmpsyserr("parse_SNMP_packet(): instance malloc() failed");
	if ((b->value = (char *) malloc(strlen(value)+1)) == NULL)
                fastsnmpsyserr("parse_SNMP_packet(): value malloc() failed");
	if (*head == NULL)
		*head = b;
	else {
		for(b2 = *head; b2->Next != NULL; b2 = b2->Next);
		b2->Next = b;
	};
	strcpy(b->instance, var_name);
	strcpy(b->value, value);
	b->Next = NULL;
}


/*************************************************************************
 * SNMP_errormsg() - return a string that corresponds to the error status*
 *************************************************************************/
char *SNMP_errormsg(rc)
     int rc;
{
	static char	errbuf[80];

	switch (rc) {
		case 0:	sprintf(errbuf, "No error");
			break;
		case 1:	sprintf(errbuf, "Too big");
			break;
		case 2:	sprintf(errbuf, "No such name");
			break;
		case 3:	sprintf(errbuf, "Bad value");
			break;
		case 4:	sprintf(errbuf, "Read only");
			break;
		case 5:	sprintf(errbuf, "Unsupported or Unauthorized operation");
			break;
		case -1: sprintf(errbuf, "Malformed SNMP packet");
			break;
		default:	
			sprintf(errbuf, "Unrecognized return code %d", rc);
			break;
	};
	return errbuf;
}

/************************************************************************
 * IDtoOID() - return OID givne a text name				*
 ************************************************************************/
int IDtoOID(instance, objid, len)
     char *instance;
     oid *objid;
     int *len;
{
	int	i;
	char 	*ptr, *j, *id;
        extern  int verbose;
  
	if (instance[strlen(instance)-1] == '.')
		instance[strlen(instance)-1] = '\0';
  
    	i = 0;
    	if ((id = (char *) malloc(strlen(instance) + 3)) == NULL)
		fastsnmpsyserr("IDtoOID(): id malloc failed");

    	bcopy(instance, id, strlen(instance)+1);
  
/*	if (id[strlen(id)-1] != '0')
		strcat(id, ".0");
*/

    	if (*id == '.')
       		*id = '0';

    	while ((ptr = (char *) strchr(id, '.')) != NULL) {
        	*ptr = '\0';
        	for (j = id; j < ptr; j++)
            	if (!isdigit(*j))
               		return 3;
        	objid[i++] = (oid) atol(id);
        	id = ptr+1;
    	};

    	objid[i++] = (oid) atol(id);
    	*len = i;
	return 0;
}

#ifdef OLD

/************************************************************************
 *	IDtoName() - return the OID given a name 			*
 ************************************************************************/
char *IDtoName(char* instance)
{
	FILE	*mib_desc;
	char 	var_name[100], name[100], id[100], type[100], num[100];
	int	var_len, i;
	char	buf[256], inst_buf[100], *str1, *str2, *ret_str;

	var_len = 0;
	strcpy(inst_buf, instance);

	mib_desc = fopen(MIB_DESC, "r");

	while (fgets(buf, sizeof(buf), mib_desc) != NULL) {
		sscanf(buf, "%s %s %s %s\n", name, id, type, num);
		if (strstr(inst_buf, id) != NULL) {
			var_len = strlen(id);
			strcpy(var_name, name);
		} else {
			str1 = inst_buf;
			str2 = id;
			i = 0;
			while (*str1++ == *str2++)
				i++;
			if (i < var_len)
				break;
		};
	};
	fclose(mib_desc);
	strcat(var_name, &inst_buf[var_len-1]);
	strcpy(ret_buf, var_name);

	return ret_buf;
}

/************************************************************************
 *  lookup_SNMP_name() - return the name that matches the longest	*
 *			OID subidentifier string			*
 ************************************************************************/
char *lookup_SNMP_name(char *var)
{
	FILE	*mib_desc;
	char	buf[256], *str, *str1, *str2;
	char 	name[100], id[100], type[100], num[100];

	if ((str = (char *) malloc(strlen(var)+1)) == NULL)
		fastsnmpsyserr("lookup_SNMP_name(): str malloc failed");

	strcpy(str, var);
	str1 = str;

	while (*str1) {
		if (isupper(*str1)) 
			*str1 = *str1 + 32;
		str1++;
	};
	
	mib_desc = fopen(MIB_DESC, "r");

	while (fgets(buf, sizeof(buf), mib_desc) != NULL) {
		sscanf(buf, "%s %s %s %s\n", name, id, type, num);
		str2 = name;
		while (*str2) {
			if (isupper(*str2)) 
				*str2 = *str2 + 32;
			str2++;
		};
		if (strcmp(str, name) == 0) {
			fclose(mib_desc);
			strcpy(ret_buf, id);
			return(ret_buf);
		};
	};
	fclose(mib_desc);
	return(NULL);
}
#endif
