/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

/************************************************************************
 *		Link I/O Routines and Data Manipulation Routines	*
 *									*
 *	Link structure I/O / addition / deletion etc.	routines	*
 *									*
 ************************************************************************/
 
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <malloc.h>
#include <sys/param.h>
#include "CommonDefs.h"
#include "pathnames.h"
#include "ctools.h"
#include "Linkfileio.h"

extern void exit();

int debug = 0;

struct LinkDetail *LinkDetailHead=NULL;		/* Head of Link Structures */
static int POP2CircuitFileExists;
static int IPtoPOPFileExists;

		/**************************************
		 *  In-Core mappings from Node<->POP  *
		 * and POPpair<->Circuit	      *
		 **************************************/
static struct POP_TO_CIRCUIT_MAP {	/* Place to store POP-Circuit Mapping */
	char POP1[ MAXPOP ];		/* One end of MCI Link 		    */
	char POP2[ MAXPOP ];		/* Other end of MCI Link 	    */
	char CIRCUIT[ MAXCIRCUIT ];	/* MCI Circuit Identifier 	    */
	struct POP_TO_CIRCUIT_MAP *Next;
} *POP_TO_CIRCUIT_MAPHead=NULL;		/* Head of Structure list of Mapping */

static struct IP_TO_POP_MAP {		/* Place to store Node-POP Mapping  */
	char IP[ MAXIPADDRESS ];	/* Address of Node 		    */
	char POP[ MAXPOP ];		/* MCI POP or Customer name 	    */
	struct IP_TO_POP_MAP *Next;
} *IP_TO_POP_MAPHead=NULL;		/* Head of structure of Mappings    */


/************************************************************************
 * AddLinkDetail() - Add a Link to the chain of LinkDetail		*
 *		See dsuchk.h for details on the Link Structure		*
 ************************************************************************/
struct LinkDetail * AddLinkDetail(LinkType, LocalTimeStamp, node, ifaddr, dsu,
	RemoteTimeStamp, remotenode, remoteifaddr, remotedsu, TTLastOpened,
	TTNextAction)
int LinkType;
time_t LocalTimeStamp;
char *node, *ifaddr;
int dsu;
time_t RemoteTimeStamp;
char *remotenode, *remoteifaddr;
int remotedsu;
time_t TTLastOpened, TTNextAction;
{
	register struct LinkDetail *ldp;
	struct LinkDetail *L;
	int ValidPOPs = 1;
	char *ip_to_pop(), *pops_to_circuit();
	extern time_t time();
	time_t TimeNow = time(&TimeNow);

	if (debug) 
		fprintf(stderr,
		"AddLinkDetail(%d, %u, %s, %s, %d, %u, %s, %s, %d, %u, %u)\n", 
		LinkType, LocalTimeStamp, node, ifaddr, dsu, RemoteTimeStamp,
		remotenode, remoteifaddr, remotedsu, TTLastOpened,
		TTNextAction);
#ifdef IGNOREOLDLINKS
	/*
	 *  Ignore links that have a timestamp older than four (4) days.
	 */
	if ((( TimeNow - LocalTimeStamp ) > FOUR_DAYS ) ||
	    (( TimeNow - RemoteTimeStamp ) > FOUR_DAYS )) {
		printf("Ignoring link with OLD time stamp: \n\
TimeNow=%ld LocalTimeStamp=%ld LocalNode=%s LocalAddr=%s LocalDSU=%d\n\
TimeNow=%ld RemoteTimeStamp=%ld RemoteNode=%s RemoteAddr=%s RemoteDSU=%d\n",
		TimeNow,LocalTimeStamp,node,ifaddr,dsu,
		TimeNow,RemoteTimeStamp,remotenode,remoteifaddr,remotedsu);
		return((struct LinkDetail *) NULL);
	}
#endif
		
	/* ignore loopbacks */
	if (strcmp(node, remotenode) == 0) 
		return((struct LinkDetail *) NULL);

	/* get some space via malloc() and fill in the structure */
	L = (struct LinkDetail *) malloc(sizeof(struct LinkDetail));
	if ( L == NULL ) {
		perror("malloc");
		exit(-1);
		/*NOTREACHED*/
	}
	L->Dirty = 0;
	L->Valid = 1;
	L->LinkType = LinkType;
	L->LocalTimeStamp = LocalTimeStamp;
	strcpy(L->LocalNode, node);
	strcpy(L->LocalIPAddr, ifaddr);
	L->LocalDSUNumber = dsu;
	strcpy(L->LocalDSUSerialNumber, "");
	L->LocalDSUSerialNumberVerified = 0;
	L->RemoteDSUSerialNumberVerified = 0;
	strcpy(L->RemoteDSUSerialNumber, "");
	strcpy(L->RemoteNode, remotenode);
	strcpy(L->RemoteIPAddr, remoteifaddr);
	L->RemoteDSUNumber = remotedsu;
	L->RemoteTimeStamp = RemoteTimeStamp;
	strcpy(L->LocalPOP, ip_to_pop(L->LocalNode));
	if (strlen(L->LocalPOP) <= 0) {
		strcpy(L->LocalPOP, "???");
		ValidPOPs = 0;
		if ( IPtoPOPFileExists==1 )
		fprintf(stderr,"No mapping from %s to any POP\n",L->LocalNode);
	}
	strcpy(L->RemotePOP, ip_to_pop(L->RemoteNode));
	if (strlen(L->RemotePOP) <= 0) {
		strcpy(L->RemotePOP, "???");
		ValidPOPs = 0;
		if ( IPtoPOPFileExists==1 )
		fprintf(stderr,"No mapping from %s to any POP\n",L->RemoteNode);
	}
	if (ValidPOPs) {
		if (strcmp(L->LocalPOP, L->RemotePOP) == 0) 
			strcpy(L->Circuit, "------------");
		else {
			strcpy(L->Circuit, 
				pops_to_circuit( L->LocalPOP, L->RemotePOP));
			if (strlen(L->Circuit) <= 0) {
				if ( POP2CircuitFileExists==1 )
				fprintf(stderr, "Circuit ID for %s %s - %s %s Not configured in file\n", L->LocalPOP, L->LocalNode,L->RemoteNode,L->RemotePOP);
				strcpy(L->Circuit, "????????????");
			}
		}
	}
	else 
		strcpy(L->Circuit, "????????????");
	L->TTLastOpened = TTLastOpened;
	L->TTNextAction = TTNextAction;
	L->Next = NULL;

	/*  attach to list */
	if (LinkDetailHead == NULL) {
		LinkDetailHead = L;
	}
	else {
		for (ldp = LinkDetailHead; ldp->Next != NULL; ldp = ldp->Next)
			;
		ldp->Next = L;
	}
	return(LinkDetailHead);
}

/***************************************************************************
 * FindLinkDetail() - Find a LinkDetail structure given an IP Address and  *
 *		      DSU Index   					   *
 * 		Return LinkDetail structure or NULL if not found           *
 ***************************************************************************/
struct LinkDetail * FindLinkDetail( IP, dsunum )
char *IP;
int dsunum;
{
	static struct LinkDetail *ldp;

	if ( debug )
		fprintf(stderr, "FindLinkDetail(%s, %d)\n", IP, dsunum);
	for ( ldp = LinkDetailHead; ldp != NULL; ldp = ldp->Next ) 
		if ((( strcmp( ldp->LocalNode, IP ) == 0 ) && 
		     ( dsunum == ldp->LocalDSUNumber )) ||
		    (( strcmp( ldp->RemoteNode, IP ) == 0 ) && 
		     ( dsunum == ldp->RemoteDSUNumber )))
			return( ldp );
	return( (struct LinkDetail *) NULL );
}


/************************************************************************
 *	FindLinkDetailAbsolute() - COnvenience routine when abs match needed	*
 ************************************************************************/

struct LinkDetail * FindLinkDetailAbsolute( LocalNode, LocalIPAddr, LocalDSU,
		RemoteDSU, RemoteIPAddr, RemoteNode)
char *LocalNode, *LocalIPAddr; 
int LocalDSU, RemoteDSU;   
char *RemoteIPAddr, *RemoteNode;
{
	static struct LinkDetail *LPtr;

	if ( debug )
		fprintf(stderr, "FindLinkDetailAbsolute(%s, %s, %d, %d, %s, %s)\n", LocalNode, LocalIPAddr, LocalDSU, RemoteDSU, RemoteIPAddr, RemoteNode);
	if (( LocalDSU == 0 ) || (RemoteDSU == 0 )) 
		return ((struct LinkDetail *) NULL); 

	if ((LPtr = FindLinkDetail( LocalNode, LocalDSU )) == NULL) 
		return ((struct LinkDetail *) NULL); 

	if ( strcmp(LPtr->LocalNode, LocalNode) == 0 ) {
		if ( strcmp( LPtr->LocalNode, LocalNode ) != 0 ) 
			return ((struct LinkDetail *) NULL); 
  		if ( strcmp( LPtr->LocalIPAddr, LocalIPAddr ) != 0 ) 
			return ((struct LinkDetail *) NULL); 
		if ( LPtr->LocalDSUNumber != LocalDSU) 
			return ((struct LinkDetail *) NULL); 
		if ( LPtr->RemoteDSUNumber != RemoteDSU) 
			return ((struct LinkDetail *) NULL); 
		if ( strcmp( LPtr->RemoteIPAddr, RemoteIPAddr ) != 0 ) 
			return ((struct LinkDetail *) NULL); 
		if ( strcmp( LPtr->RemoteNode, RemoteNode) != 0 ) 
			return ((struct LinkDetail *) NULL); 
		return( LPtr );		/*** All Fields matched ! ***/
	}

	if ( strcmp( LPtr->RemoteNode, LocalNode ) != 0 ) 
		return ((struct LinkDetail *) NULL); 
  	if ( strcmp( LPtr->RemoteIPAddr, LocalIPAddr ) != 0 ) 
		return ((struct LinkDetail *) NULL); 
	if ( LPtr->RemoteDSUNumber != LocalDSU) 
		return ((struct LinkDetail *) NULL); 
	if ( LPtr->LocalDSUNumber != RemoteDSU) 
		return ((struct LinkDetail *) NULL); 
	if ( strcmp( LPtr->LocalIPAddr, RemoteIPAddr ) != 0 ) 
		return ((struct LinkDetail *) NULL); 
	if ( strcmp( LPtr->LocalNode, RemoteNode) != 0 ) 
		return ((struct LinkDetail *) NULL); 
	return( LPtr );		/*** All Fields matched ! ***/
}

/************************************************************************
 * UpdateLink() - Call this routine to update/verify the link config.	*
 *									*
 *		  Paramters with a 'r' prefix are REMOTE.		*
 *		  Paramters with a 'l' prefix are LOCAL.		*
 ************************************************************************/
void UpdateLink(lNode, lAddr, lDSU, rDSU, rAddr, rNode)
char *lNode, *lAddr;
int lDSU, rDSU;
char *rAddr, *rNode;
{
	static struct LinkDetail *LPtr;
	extern time_t time();
	time_t TimeNow = time( &TimeNow );

	if ( debug )
		fprintf(stderr, "UpdateLink(%s, %s, %d, %d, %s, %s)\n",
			lNode, lAddr, lDSU, rDSU, rAddr, rNode);

	/* ignore loopbacks */
	if ( strcmp( lNode, rNode) == 0 ) 
		return;

	if ((LPtr = FindLinkDetailAbsolute( lNode, lAddr, lDSU, rDSU,
                                rAddr, rNode )) != NULL) {
		LPtr->LocalTimeStamp = TimeNow;
		LPtr->RemoteTimeStamp = TimeNow;
		return;
	}

	if (( lDSU == 0 ) || ( rDSU == 0)) {
		fprintf(stderr, "Error: One end didn't respond");
		if ( lDSU != 0 ) {
			if ((LPtr = FindLinkDetail( lNode, lDSU )) == NULL)
				fprintf( stderr, "Ignoring link: one side didn't answer, and the other isn't in DB:\n");
			else 
				LPtr->LocalTimeStamp = TimeNow;
		} else {
			if ((LPtr = FindLinkDetail( rNode, rDSU )) == NULL)
				fprintf( stderr, "Ignoring link: one side didn't answer, and the other isn't in DB:\n");
			else 
				LPtr->RemoteTimeStamp = TimeNow;
		}
		return;
	}
	fprintf(stderr, "UpdateLink(%s, %s, %d, %d, %s, %s)\n",
		lNode, lAddr, lDSU, rDSU, rAddr, rNode);
	if (AddLinkDetail( UNKNOWN, TimeNow, lNode, lAddr, lDSU, 
		TimeNow, rNode, rAddr, rDSU, (time_t) 0L, (time_t) 0L) == NULL)
		fprintf(stderr, "AddLinkDetail failed\n");
}


/************************************************************************
 *  FindPtPtLink( Node, Node )						*
 * returns a pointer to the specified link or NULL if it doesn't exist  *
 ************************************************************************/
struct LinkDetail *FindPtPtLink( Node1, Node2 )
char *Node1, *Node2;
{
	register struct LinkDetail *LPtr;

	if ( debug )
		fprintf(stderr, "FindPtPtLink(%s, %s)\n", Node1, Node2);
	for (LPtr = LinkDetailHead; LPtr != NULL; LPtr = LPtr->Next) {
		if (( strcmp(LPtr->LocalNode, Node1) == 0) ||
		    ( strcmp(LPtr->LocalIPAddr, Node1) == 0)) 
			if (( strcmp(LPtr->RemoteNode, Node2) == 0) ||
			    ( strcmp(LPtr->RemoteIPAddr, Node2) == 0) )
				return( LPtr );
			else continue;
		else 
			if (( strcmp(LPtr->RemoteNode, Node1) == 0) ||
		    	    ( strcmp(LPtr->RemoteIPAddr, Node1) == 0)) 
				if (( strcmp(LPtr->LocalNode, Node2) == 0) ||
			    	    ( strcmp(LPtr->LocalIPAddr, Node2) == 0))
				return( LPtr );
			else continue;
	}
	return( (struct LinkDetail *) NULL );
}

/************************************************************************
 * UpdateLinkType() - Update the link infomation table with ifType	*
 ************************************************************************/
void UpdateLinkType( Node, ifnum, ifType )
char *Node;
int ifnum, ifType;
{
	static struct LinkDetail *LPtr;

	if ( debug )
		fprintf(stderr, "UpdateLinkType(%s, %d, %d)\n", Node, ifnum,
			ifType);
	if ( (ifType != T1 ) && (ifType != T3 ) ) 
		return;
	if (( LPtr = FindLinkDetail( Node, ifnum)) == NULL ) 
		return;
	if ( LPtr->LinkType != ifType ) {
		if ( debug )
			fprintf( stderr, 
			"UpdateLinkType(): Link Type changing from %d to %d\n",
			LPtr->LinkType, ifType);
		LPtr->LinkType = ifType;
	}
}

void UpdateDSUSerialNumbers( Node, Index, num )
char *Node, *Index, *num;
{
	int ifnum,Topo=0;
	char *p;
	struct LinkDetail *LPtr;

	
	if ( debug )
		fprintf(stderr, "UpdateDSUSerialNumbers(%s, %s, %s)\n",
			Node, Index, num );

	ifnum = atoi(Index);
	if ((p = strtok( Index, ". \t\n")) == NULL) {
		fprintf( stderr, "No '.' in DSUSerialNumber...err\n");
		return;
	}
	if ((p = strtok((char *) NULL, ". \t\n")) == NULL) {
		fprintf( stderr, "No Local/Remote in DSUSerialNumber...err\n");
		return;
	}
	Topo = atoi(p);
	if ((LPtr = FindLinkDetail( Node, ifnum )) == NULL ) {
		fprintf( stderr,"Can't find link for %s %d\n", Node, ifnum);
		return;
	}
	switch ( Topo ) {
	case LOCALDSU:
		if ( strcmp( LPtr->LocalNode, Node ) == 0 ) {
			if ( strcmp( num, LPtr->LocalDSUSerialNumber ) == 0 ) {
				LPtr->LocalDSUSerialNumberVerified = 1;
				return;
			}
			if ( strlen( LPtr->LocalDSUSerialNumber ) == 0 ) {
				strcpy( LPtr->LocalDSUSerialNumber, num );
				return;
			}
			fprintf( stderr,"Local DSU Configuration ERROR: %s #%d (%s OR %s ??) #%d %s\nRemoving link\n", 
				LPtr->LocalNode, LPtr->LocalDSUNumber, 
				LPtr->LocalDSUSerialNumber, num,
				LPtr->RemoteDSUNumber, LPtr->RemoteNode );
			LPtr->Valid = 0;
			return;
		}
		if ( strcmp( num, LPtr->RemoteDSUSerialNumber ) == 0 ) {
			LPtr->RemoteDSUSerialNumberVerified = 1;
			return;
		}
		if ( strlen( LPtr->RemoteDSUSerialNumber ) == 0 ) {
			strcpy( LPtr->RemoteDSUSerialNumber, num );
			return;
		}
		fprintf( stderr,"Remote DSU configuration ERROR: %s #%d (%s OR %s ??) #%d %s\nRemoving link\n", 
			LPtr->LocalNode, LPtr->LocalDSUNumber, 
			LPtr->RemoteDSUSerialNumber, num,
			LPtr->RemoteDSUNumber, LPtr->RemoteNode );
		LPtr->Valid = 0;
		break;

	case REMOTEDSU:
		if ( strcmp( LPtr->LocalNode, Node ) == 0 ) {
			if ( strcmp( num, LPtr->RemoteDSUSerialNumber ) == 0 ) {
				LPtr->RemoteDSUSerialNumberVerified = 1;
				return;
			}
			if ( strlen( LPtr->RemoteDSUSerialNumber ) == 0 ) {
				strcpy( LPtr->RemoteDSUSerialNumber, num );
				return;
			}
			fprintf( stderr,"Remote DSU Configuration ERROR: %s #%d (%s OR %s ??) #%d %s\nRemoving link\n", 
				LPtr->LocalNode, LPtr->LocalDSUNumber, 
				LPtr->RemoteDSUSerialNumber, num,
				LPtr->RemoteDSUNumber, LPtr->RemoteNode );
			LPtr->Valid = 0;
			return;
		}
		if ( strcmp( num, LPtr->LocalDSUSerialNumber ) == 0 ) {
			LPtr->LocalDSUSerialNumberVerified = 1;
			return;
		}
		if ( strlen( LPtr->LocalDSUSerialNumber ) == 0 ) {
			strcpy( LPtr->LocalDSUSerialNumber, num );
			return;
		}
		fprintf( stderr,"Local DSU configuration ERROR: %s #%d (%s OR %s ??) #%d %s\nRemoving link\n", 
			LPtr->LocalNode, LPtr->LocalDSUNumber, 
			LPtr->LocalDSUSerialNumber, num,
			LPtr->RemoteDSUNumber, LPtr->RemoteNode );
		LPtr->Valid = 0;
		break;

	default:
		fprintf( stderr, "Weird Local/Remote Spec: %d...err\n",Topo);
		return;
		/*NOTREACHED*/
	}
}

/************************************************************************
 * UpdateLinkTimeStamp() - Update the Local or Remote TimeSTamp		*
 ************************************************************************/
void UpdateLinkTimeStamp( Node, ifnum )
char *Node;
int ifnum;
{
	static struct LinkDetail *LPtr;

	if ( debug )
		fprintf(stderr, "UpdateLinkTimeStamp(%s, %d)\n", Node, ifnum);

        if (( LPtr = FindLinkDetail( Node, ifnum)) == NULL )  {
               fprintf( stderr, "Can't find Node:DSU pair in PrintReport...Should never happen!\n");
		return;
        }
	if ( strcmp( LPtr->LocalNode, Node ) == 0 )
		LPtr->LocalTimeStamp = time( &LPtr->LocalTimeStamp );
	else
		LPtr->RemoteTimeStamp = time( &LPtr->RemoteTimeStamp );
}


/************************************************************************
 * NodeDSUResponded() - For reference counting - obsolete now		*
 *			since this can be derived from the datestamp	*
 ************************************************************************/
NodeDSUResponded( IP, dsunum )
char *IP;
int dsunum;
{
	struct LinkDetail *LPtr;

	if ( debug )
		fprintf(stderr, "NodeDSUResponded(%s, %d)\n", IP, dsunum);

	if (( LPtr = FindLinkDetail( IP, dsunum )) == NULL ) {
		fprintf( stderr, "NodeDSUResponded(): - Can't find associated link %s:%d\n",IP,dsunum);
		return(0);
	}
	if ( strcmp( LPtr->LocalNode, LPtr->RemoteNode ) == 0 ) {
		fprintf( stderr, "NodeDSUResponded(): - Loopback link-ignoring %s %d\n",IP, dsunum );
		return(0);
	}
	if (( strcmp( LPtr->LocalNode, IP ) == 0 ) && 
	        ( dsunum == LPtr->LocalDSUNumber )) {
		LPtr->LocalTimeStamp = time( &LPtr->LocalTimeStamp );
	}
	else
		LPtr->RemoteTimeStamp = time( &LPtr->RemoteTimeStamp );
	return( 1 );
}

/************************************************************************
 * MarkLinkDirty() - Set the "Dirty" flag for this LinkDetail given Node IP	*
 *			address and DSU Index.				*
 ************************************************************************/
void MarkLinkDirty( IP, dsunum )
char *IP;
int dsunum;
{
	struct LinkDetail *LPtr;

	if ( debug )
		fprintf(stderr, "MarkLinkDirty(%s, %d)\n", IP, dsunum);

	if (( LPtr = FindLinkDetail( IP, dsunum )) == NULL ) {
		fprintf( stderr, "MarkLinkDirty(): - Can't find link %s %d\n",
								IP,dsunum);
		return;
	}
	if ( strcmp( LPtr->LocalNode, LPtr->RemoteNode ) != 0 ) 
		LPtr->Dirty = 1;
}

/************************************************************************
 * PrintLinkDetail() - Print the linked list of LinkDetail to the screen*
 ************************************************************************/
void PrintLinkDetail()
{
	struct LinkDetail *LPtr;

	if ( debug )
		fprintf(stderr, "PrintLinkDetail()\n");

	for ( LPtr = LinkDetailHead; LPtr != NULL; LPtr = LPtr->Next )
		fprintf(stderr, "POP=%s node=%s ifaddr=%s dsu=%d Circuit=%s remotedsu=%d remoteifaddr=%s remotenode=%s remotePOP=%s\n", LPtr->LocalPOP, LPtr->LocalNode, LPtr->LocalIPAddr, LPtr->LocalDSUNumber, LPtr->Circuit, LPtr->RemoteDSUNumber, LPtr->RemoteIPAddr, LPtr->RemoteNode,  LPtr->RemotePOP );

}

static int virgin=1;
static char Links_File[ MAXPATHLEN ];		/* LINKS Filename */
static char POP2Circuit_File[ MAXPATHLEN ];	/* POP Pair->Circuit */
static char IP2POP_File[ MAXPATHLEN ];		/* IP -> POP Mapping */
/************************************************************************
 * Read_LinkDetail_File() - Read the LinkDetail file and store in a 	*
 *			linked list of LinkDetail structures.		*
 ************************************************************************/
struct LinkDetail * Read_LinkDetail_File(network)
char *network;
{
	FILE *stream;
	char buffer[BUFSIZ];
	char *ltimestamp, *lNode, *lAddr, *lDSU;
	char *rtimestamp, *rNode, *rAddr, *rDSU;
	char *lastopen, *nextaction, *LinkType, *pingkydir, *msg = "";
	int line = 0, error = FALSE;
	extern char * getenv();
	extern long atol();
	extern void Read_IP_TO_POP_MAP(), Read_POP_TO_CIRCUIT_MAP_File();

	if (debug)
		fprintf(stderr, "Read_LinkDetail_File(%s)\n", network);

	if (virgin) {
		if ((pingkydir = getenv("PINGKYDIR")) == NULL) 
			pingkydir = DEFAULT_PINGKY_DIR;
		sprintf(Links_File, "%s/%s.LINKS", pingkydir, network);
		sprintf(POP2Circuit_File, "%s/%s.POP2CIRCUIT", 
							pingkydir, network);
		sprintf(IP2POP_File, "%s/%s.IP2POP", pingkydir, network);
		virgin = 0;
	}

	Read_POP_TO_CIRCUIT_MAP_File(POP2Circuit_File);
	Read_IP_TO_POP_MAP(IP2POP_File);

        if ((stream = fopen(Links_File, "r")) == NULL) {
                fprintf(stderr, "Can't open %s LinkDetail file\n", Links_File);
		return((struct LinkDetail *) NULL);
        }

        while(fgets(buffer, sizeof(buffer), stream ) != NULL) {
		if (error == TRUE) {
			fprintf(stderr, "%s: Line %d:  bad %s (ignored)\n", 
						Links_File,line, msg);
		}
		error = FALSE;
		line++;
		if (( LinkType = strtok(buffer, DELIMITERS)) == NULL ) {
			msg = "link type";
			error = TRUE;
			continue;
		}
		if ((ltimestamp = strtok((char *) NULL, DELIMITERS)) == NULL) {
			msg = "local timestamp";
			error = TRUE;
			continue;
		}
		if ((lNode = strtok((char *) NULL, DELIMITERS)) == NULL) {
			msg = "local node";
			error = TRUE;
			continue;
		}
		if ((lAddr = strtok((char *) NULL, DELIMITERS)) == NULL) {
			msg = "local IP address";
			error = TRUE;
			continue;
		}
		if ((lDSU = strtok((char *) NULL, DELIMITERS)) == NULL) {
			msg = "local DSU number";
			error = TRUE;
			continue;
		}
		if ((rDSU = strtok((char *) NULL, DELIMITERS)) == NULL) {
			msg = "remote DSU number";
			error = TRUE;
			continue;
		}
		if ((rAddr = strtok((char *) NULL, DELIMITERS)) == NULL) {
			msg = "remote IP address";
			error = TRUE;
			continue;
		}
		if ((rNode = strtok((char *) NULL, DELIMITERS)) == NULL) {
			msg = "remote node";
			error = TRUE;
			continue;
		}
		if ((rtimestamp = strtok((char *) NULL, DELIMITERS)) == NULL) {
			msg = "remote timestamp";
			error = TRUE;
			continue;
		}
		if ((lastopen = strtok((char *) NULL, DELIMITERS)) == NULL) {
			msg = "last opened timestamp";
			error = TRUE;
			continue;
		}
		if ((nextaction = strtok((char *) NULL, DELIMITERS)) == NULL ) {
			msg = "next action timestamp";
			error = TRUE;
			continue;
		}
		if (AddLinkDetail(atoi(LinkType), (time_t) atol(ltimestamp), 
			lNode, lAddr, atoi(lDSU), (time_t) atol(rtimestamp), 
			rNode, rAddr, atoi(rDSU), (time_t) atol(lastopen), 
			(time_t) atol(nextaction)) == NULL)
			fprintf(stderr, "AddLinkDetail failed\n");
        }
        fclose(stream); 
	if (debug) 
		PrintLinkDetail();
	return(LinkDetailHead);
}

/************************************************************************
 *               Write_LinkDetail_File() - Write Link DB to disk		*
 ************************************************************************/
void Write_LinkDetail_File( network )
char *network;
{
	register struct LinkDetail *L;
	FILE *fp;
	char *pingkydir;
	extern char * getenv();
	extern void Read_POP_TO_CIRCUIT_MAP_File();

	if (debug)
		fprintf(stderr, "Write_LinkDetail_File(%s)\n", network);

	if (virgin) {
		if ((pingkydir = getenv("PINGKYDIR")) == NULL) 
			pingkydir = DEFAULT_PINGKY_DIR;
		sprintf(Links_File, "%s/%s.LINKS", pingkydir, network );
		sprintf(POP2Circuit_File, "%s/%s.POP2CIRCUIT", 
							pingkydir, network );
		sprintf(IP2POP_File, "%s/%s.IP2POP", pingkydir, network );
		virgin = 0;
	}
	Read_POP_TO_CIRCUIT_MAP_File(POP2Circuit_File);
	if ((fp = fopen(Links_File, "w")) == NULL) {
		fprintf(stderr, "fopen %s failed\n", Links_File);
		return;
	}
	for (L = LinkDetailHead; L != NULL; L = L->Next ) {
		if (L->Valid)
			fprintf(fp, "%d %ld %s %s %d %d %s %s %ld %ld %ld\n",
				L->LinkType, L->LocalTimeStamp, L->LocalNode, 
				L->LocalIPAddr,L->LocalDSUNumber,
				L->RemoteDSUNumber, L->RemoteIPAddr, 
				L->RemoteNode, L->RemoteTimeStamp,
				L->TTLastOpened, L->TTNextAction);
	}
	fclose(fp);
}

/************************************************************************
 * pops_to_circuit() - Return the Circuit ID given the two MCI POP names*
 ************************************************************************/
char *pops_to_circuit( POP1, POP2 )
char *POP1, *POP2;
{
	register struct POP_TO_CIRCUIT_MAP *P;

	if (debug)
		fprintf(stderr, "pops_to_circuit(%s, %s)\n", POP1, POP2);

	if ( POP_TO_CIRCUIT_MAPHead == NULL ) {
		fprintf( stderr, "No POP to Circuit Mappings exist.. Err!\n");
		return( "" );
	}
	for( P=POP_TO_CIRCUIT_MAPHead; P!=NULL; P=P->Next )
		if (( (strcmp( POP1, P->POP1 ) == 0 ) &&
		      (strcmp( POP2, P->POP2 ) == 0 ) ) ||
		    ( (strcmp( POP1, P->POP2 ) == 0 ) &&
		      (strcmp( POP2, P->POP1 ) == 0 ) ) )
			return( P->CIRCUIT );

	return( "" );
}

/************************************************************************
 * AddPOP_TO_CIRCUIT_MAP() - Add a POP_TO_CIRCUIT_MAP structure to the  *
 *			current list of Mappings.			*
 ************************************************************************/
static void AddPOP_TO_CIRCUIT_MAP( POP1, POP2, CIRCUIT )
char *POP1, *POP2, *CIRCUIT;
{
	struct POP_TO_CIRCUIT_MAP *P;
	register struct POP_TO_CIRCUIT_MAP *pp;

	if ( debug ) 
		fprintf(stderr,"AddPOP_TO_CIRCUIT_MAP(%s, %s, %s)\n", 
							POP1, POP2, CIRCUIT );
	if ((P = (struct POP_TO_CIRCUIT_MAP *) malloc(sizeof(struct POP_TO_CIRCUIT_MAP))) == NULL) {
		perror("malloc");
		exit(-1);
		/*NOTREACHED*/
	}
	strcpy(P->POP1, POP1 );
	strcpy(P->POP2, POP2 );
	strcpy(P->CIRCUIT, CIRCUIT );
	P->Next = NULL;
	if (POP_TO_CIRCUIT_MAPHead == NULL) {
		POP_TO_CIRCUIT_MAPHead=P;
		return;
	}
	for (pp = POP_TO_CIRCUIT_MAPHead; pp->Next != NULL; pp = pp->Next)
		;
	pp->Next = P;
}

/************************************************************************
 * PrintPOP_TO_CIRCUIT_MAP() - Print all POP_TO_CIRCUIT mappings to 	*
 *				the screen.				*
 ************************************************************************/
static void PrintPOP_TO_CIRCUIT_MAP()
{
	struct POP_TO_CIRCUIT_MAP *PPtr;

	if (debug)
		fprintf(stderr, "PrintPOP_TO_CIRCUIT_MAP()\n");
	for (PPtr = POP_TO_CIRCUIT_MAPHead; PPtr != NULL; PPtr = PPtr->Next)
		fprintf(stderr, "POP1=%s POP2=%s CIRCUIT=%S\n", 
					PPtr->POP1, PPtr->POP2, PPtr->CIRCUIT);
}

/************************************************************************
 * Read_POP_TO_CIRCUIT_MAP_File() - Read the POP-CIRCUIT mappings from  *
 *				the configuration file specified.	*
 *				Build a list of structures.		*
 ************************************************************************/
void Read_POP_TO_CIRCUIT_MAP_File( filename )
char *filename;
{
	FILE *stream;
	char buffer[BUFSIZ];
	char *POP1, *POP2, *CIRCUIT;
	int line = 0;

        if ((stream = fopen( filename, "r")) == NULL) {
                /*fprintf(stderr, "Can't open %s POP_TO_CIRCUIT_MAP file\n",
								filename);*/
                return;
        }
	POP2CircuitFileExists=1;
        while (fgets(buffer, sizeof(buffer), stream) != NULL) {
		line++;
                if ((POP1 = strtok(buffer, "- \t\n")) == NULL) {
                        fprintf(stderr, "Error on line %d of %s- Bad POP1.. line ignored \n", line, filename);
                        continue;
                }
                if ((POP2 = strtok((char *) NULL," -\t\n")) == NULL) {
                        fprintf(stderr,"Error on line %d of %s- Bad POP2.. line ignored \n", line, filename);
                        continue;
                }
                if ((CIRCUIT = strtok((char *) NULL," -\t\n")) == NULL) {
                        fprintf(stderr,"Error on line %d of %s- Bad CIRCUIT.. line ignored \n", line, filename);
                        continue;
                }
                AddPOP_TO_CIRCUIT_MAP( POP1, POP2, CIRCUIT );

        }
        fclose(stream); 
	if (debug) 
		PrintPOP_TO_CIRCUIT_MAP();
}

/************************************************************************
 * ip_to_pop() - Return the MCI POP Name given the IPAddr of the node	*
 ************************************************************************/
char * ip_to_pop( ipaddr )
char *ipaddr;
{
	register struct IP_TO_POP_MAP *I;
	
	if (debug)
		fprintf(stderr, "ip_to_pop(%s)\n", ipaddr);

	if ( IP_TO_POP_MAPHead == NULL )
		return("");
	for( I=IP_TO_POP_MAPHead; I!=NULL; I=I->Next )
		if ( strcmp( ipaddr, I->IP ) == 0 ) 
			return( I->POP );
	return( "" );
}

/************************************************************************
 * AddIP_TO_POP_MAP() - Add a mapping structure to the list.		*
 *			We are building a linked list of IPAddr-POP	*
 *			mappings.					*
 ************************************************************************/
static void AddIP_TO_POP_MAP( IP, POP )
char *IP, *POP;
{
	struct IP_TO_POP_MAP *I, *IPtr;

	if ( debug ) 
		fprintf(stderr, "AddIP_TO_POP_MAP(%s, %s)\n", IP, POP );
	I = (struct IP_TO_POP_MAP*) malloc(sizeof(struct IP_TO_POP_MAP));
	if ( I == NULL ) {
		fprintf(stderr,"malloc failed in AddIP_TO_POP_MAP()\n");
		exit(1);
	}
	strcpy(I->IP, IP );
	strcpy(I->POP, POP );
	I->Next=NULL;
	if (IP_TO_POP_MAPHead==NULL) {
		IP_TO_POP_MAPHead=I;
	}
	else {
		for(IPtr=IP_TO_POP_MAPHead; IPtr->Next!=NULL; IPtr=IPtr->Next);
		IPtr->Next=I;
	}
}

/************************************************************************
 * PrintIP_TO_POP_MAP() - Print the IP-POP mappings			*
 ************************************************************************/
static void PrintIP_TO_POP_MAP()
{
	struct IP_TO_POP_MAP *IPtr=IP_TO_POP_MAPHead;

	if (debug)
		fprintf(stderr, "PrintIP_TO_POP_MAP()\n");

	for (IPtr = IP_TO_POP_MAPHead; IPtr != NULL; IPtr = IPtr->Next)
		fprintf(stderr, "IP=%s POP=%s\n", IPtr->IP, IPtr->POP);
}

/************************************************************************
 * Read_IP_TO_POP_MAP() - Read the config file and build a list of 	*
 *			IP-POP mappings.				*
 ************************************************************************/
void Read_IP_TO_POP_MAP( filename )
char *filename;
{
	FILE *stream;
	char buffer[BUFSIZ];
	char *IP, *POP;
	int line = 0;

	if (debug)
		fprintf(stderr, "Read_IP_TO_POP_MAP(%s)\n", filename);
        if ((stream=fopen( filename, "r" ) ) == NULL ) {
                /*fprintf(stderr,"Can't open %s IP_TO_POP file\n",filename);*/
                return;
        }
	IPtoPOPFileExists=1;
        while( fgets( buffer, sizeof(buffer), stream ) != NULL ) {
		line++;
                if (( IP=strtok(buffer," -\t\n")) == NULL ) {
                        fprintf(stderr,"Error on line %d of %s - Bad IP.. line ignored \n", line, filename );
                        continue;
                }
                if (( POP=strtok((char *) NULL," -\t\n")) == NULL ) {
                        fprintf(stderr,"Error on line %d of %s - Bad POP.. line ignored \n", line, filename );
                        continue;
                }
                AddIP_TO_POP_MAP( IP, POP );
        }
        fclose(stream); 
	if (debug) 
		PrintIP_TO_POP_MAP();
}

#ifdef STANDALONE
char *progname;

main( argc, argv )
int argc;
char *argv[];
{
	struct LinkDetail *LinkHead;

	progname=argv[1];
	if ( argc !=2 ) {
		printf("Usage %s <Network>\n",progname);
		exit(1);
	}
	LinkHead=Read_LinkDetail_File( argv[1] );
	PrintLinkDetail();
}
#endif
