/*
 * Copyright (c) 1989, 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <sys/wait.h>
#include <sys/file.h>
#include "CommonDefs.h"
#include "problem.h"
#include "netquery.h"
#include "hostfile.h"

static int nq_errlog;
static char cmd[100];	/* for sprintf()ing before a system() call */

/* 
 *	Add services here and update netquery.h
 */
static char *NetProto[] = {	
	"NONE", "PING", "SGMP", "SNMP", "TELNET",
	"FTP", "SMTP", "NAMED", "TROUBLE", "Empty",
	"ATPING", "ATLOOK", "FINGER", "DIXIE","GOPHER",
	"GENERIC"};
#define MAX_NUMBER_OF_SERVICES (sizeof(NetProto) / sizeof(char *))

static char *NetStatus[]={
	"ITBROKE", "ITWORKS", "ITDEAD"};
#define MAX_NUMBER_OF_STATUS (sizeof(NetStatus) / sizeof(char *))

NetService( Service )
char *Service;
{
	char service[100];
	int i;

	if (Service == NULL) 
		return(NONE);
	for (i = 0; isalpha(Service[i]); i++)
		service[i] = (isupper(Service[i]) ? Service[i] : toupper(Service[i]) );
	service[i]='\0';

	for (i = 0; i < MAX_NUMBER_OF_SERVICES; i++)
		if (strcmp(service, NetProto[i]) == 0) 
			return(i);
	return(GENERIC);
}

/****************************************************************************
 *	All modules should return ITWORK, ITBROKE, or ITERROR(fatal)        *
 ****************************************************************************/
NetQuery(addr, Test)
char *addr;			/* Return ITBROKE if it Doesn't Function */
struct TestType *Test;		/* Network Management protocol */
{

	if (Test == NULL) 
		panic(" NetQuery: NULL Test\n");
#ifdef DEBUG
	printf("NetQuery(addr=%s,service=%s) entering\n", addr,service);
	printf("netquery(): argv[0]=%s argv[1]=%s argv[2]=%s argv[3]=%s\n",
	    Test->argv[0],Test->argv[1],Test->argv[2],Test->argv[3]);
#endif

	switch (NetService(Test->argv[0])) {

	case PING:   
		return(TestPING(addr));
	case SGMP:   
		return(TestSGMP());
	case SNMP:   
		return(TestSNMP());
	case TELNET: 
		return(TestPORT("telnet", addr));
	case FTP:    
		return(TestPORT("ftp", addr));
	case SMTP:   
		return(TestPORT("smtp", addr));
	case FINGER: 
		return(TestPORT("finger", addr));
	case NAMED:  
		return(TestNAMED(addr));
	case TROUBLE:  
		return(TestTROUBLE());
	case ATPING:    
		return(TestATPING(Test));
	case ATLOOK:    
		return(TestATLOOK(Test));
	case DIXIE:    
		return(TestDIXIE(addr));
	case GOPHER:    
        	if (Test->argv[1] == NULL)
			return(TestPORT( "70", addr));	/* Std GOPHER */
		else return(TestPORT( Test->argv[1], addr)); /*Custom GOPHER*/
	case NONE:   
		return(ITWORKS);
	case ERROR:
	case GENERIC:
	default:
		return(TestGENERIC(Test));
	}
}


char *NetServiceString(i)
int i;
{
	if (i > MAX_NUMBER_OF_SERVICES)
		return("ERROR");
	return(NetProto[i]);
}

char *NetStatusString(i)
int i;
{
	if (i > MAX_NUMBER_OF_STATUS) 
		return("ITERROR");
	return(NetStatus[i]);
}

void init_netquery()
{
	if ((nq_errlog = open("/dev/null", O_RDWR | O_CREAT, 0666)) < 0)
		syserr("Opening error log: ");
}

/****************************************************************
 *       TestPING:    Test to see if a node is up               *
 *                    Return ITBROKE if node is DOWN            *
 *                    Return ITWORK if node is UP               *
 *								*
 *   This deserves some explaination.   The pingd now handles   *
 * all pings, and adds/deletes them from the PROBLEM.FILE.	*
 * This returns whether the PING() test has failed or succeeded	*
 * (based on whther or not the problem exists in the		*
 *  PROBLEM.FILE).   Then the secondary tests will be performed	*
 * if appropriate.						*
 *    This code actually adds very little overhead.  If the 	*
 * problem has been added, this is just a memory lookup.	*
 ****************************************************************/
static TestPING(address)
char *address; 
{
	extern struct ProblemType ProblemArray[];
	extern int NumProblems;
	register int i;

	for (i = 0; i < NumProblems; i++) {
		if ((strcmp(ProblemArray[i].UniqueID,address) == 0) &&
		    (strcmp(ProblemArray[i].TestName,"PING") == 0))
			return(ITBROKE);
	}
	return(ITWORKS);
}

/****************************************************************
 *       TestDIXIE:   Test to see if X.500 Dixie is up          *
 *                    Return 0 if node is DOWN                  *
 *                    Return Non-Zero if node is UP             *
 ****************************************************************/
static TestDIXIE( address )
char *address;          /* character string representing address      */
{
	char	*newav[5];
	extern char **environ;
	int	pid;
	union wait status;

#ifdef OLDSTYLE
	char buffer[100];

	/*sprintf( buffer, "./dxiup %s dixie",address);	 OLD */
	sprintf( buffer, "./dxiup %s dsa",address);
	if (system( buffer ) == 0 ) 
		return( ITWORKS);
	return(ITBROKE);
#endif

#ifndef BUILTIN
	switch ( pid = vfork() ) {
	case -1 :	/* Fork failed */
		syserr( "vfork: ");
		/*NOTREACHED*/
	case 0 :	/* Child process */
		newav[ 0 ] = "dxiup";
		newav[ 1 ] = address;
		newav[ 2 ] = "dixie";
		newav[ 3 ] = NULL;
		dup2( nq_errlog, 0 );
		dup2( nq_errlog, 1 );
		dup2( nq_errlog, 2 );
		(void) close( nq_errlog );
#define DIXIECMD "./dxiup"
		execve( DIXIECMD, newav, environ );
		syserr( "execve: " );
		/*NOTREACHED*/
	default :	/* Parent process */
		if ( pid != wait( (int *) &status ) )
			fprintf( stderr, "Unknown child (pid=%d) returned to me!\n",pid );
		if ( status.w_retcode != 0 ) {
			return( ITBROKE );
		} else {
			return( ITWORKS );
		}
	}
#else
	int newac;

	newav[ 0 ] = "ping (pingky)";
	newav[ 1 ] = address;
	newav[ 2 ] = "5";
	newav[ 3 ] = "5";
	newav[ 4 ] = NULL;
	newac = 5;
	return(ping( newac, newav ));

#endif
}

static TestTROUBLE() /* Test MERIT TROUBLE MAIL */
{
	/* sprintf(cmd,"rsh merit.edu -n 'ls /usr/spool/mail/down' 2>/dev/null | grep down  >/dev/null");*/
	/*sprintf(cmd,"rsh knock.aa.ans.net -n 'ls /usr/spool/mail/down' 2>/dev/null | grep down  >/dev/null");*/
	sprintf(cmd,"rsh noc.ns.itd.umich.edu -n 'ls /usr/spool/mail/down' 2>/dev/null | grep down  >/dev/null");

#ifdef DEBUG
	printf("TROUBLE COMMAND: %s\n", cmd);
#endif
	/* XXXX This test's conditions are backwards!! */
	if (system(cmd))
		return(ITWORKS);
	return(ITBROKE);
}

static TestATPING( Test ) /*  AppleTalk Pinger @wbn1*/
struct TestType *Test;
{
	sprintf(cmd, "atpinger %s@%s | grep 'Okay'  >/dev/null",
	    Test->argv[2],Test->argv[1]);
#ifdef DEBUG
	printf("ATPING COMMAND: %s\n",cmd);
#endif
	if (system(cmd))
		return(ITBROKE);
	return(ITWORKS);
}

static TestATLOOK( Test ) /*  AppleTalk Look @wbn1*/
struct TestType *Test;
{
	sprintf(cmd, "atlook %s | grep '%s'  >/dev/null",
	    Test->argv[1],Test->argv[2]);
#ifdef DEBUG
	printf("ATLOOK COMMAND: %s\n",cmd);
#endif
	if (system(cmd))
		return(ITBROKE);
	return(ITWORKS);
}

static TestGENERIC( Test )
struct TestType *Test;
{
	struct stat buf;

	if (Test->argv[1] == NULL) 
		return(ITWORKS);
	if (stat(Test->argv[1], &buf) == 0)
		return(ITBROKE);
	return(ITWORKS);
}

static TestNAMED( addr )
char *addr;
{
extern int NAMEDTest();
#ifdef OLD
	sprintf(cmd, "./ns %s",addr);
	if (system(cmd))
		return(ITBROKE);
	return(ITWORKS);
#endif
	if ( NAMEDTest( addr ) )
		return( ITBROKE );
	else return( ITWORKS );
}

static TestSGMP( )
{
	return(ITWORKS);
}

static TestSNMP( )
{
	return(ITWORKS);
}
