/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 *		M E M O R Y    M A N A G E M E N T    C O D E
 *
 *     The idea is to malloc what you need, and free to our own linked
 * list of free structures.  Recycle these structures before you go and
 * MyMalloc() some more for yourself.
 *
 *	I was unable to find the memory leak... The X application grows
 * in size, and all of the mallocs are printed out.  I think the X code is 
 * growing.  I'll leave the debugs in here just the same.
 */

#include <stdio.h>
#include <malloc.h>

#include "CommonDefs.h"

#ifdef MEMLEAKDEBUG
static int NumLinksMalloced = 0, NumNodesMalloced = 0, NumUnknownMallocs = 0;
#endif

static int FreeNodeCount = 0, FreeLinkCount = 0;
static struct NodeType *FreeNodeHead = NULL;
static struct LinkType *FreeLinkHead = NULL;

/************************************************************************
 * MyMalloc() - My Memory Allocator.  Use this stub as a debugging hook	*
 *		to catch memory allocation problems.			*
 ************************************************************************/
char *MyMalloc( size )
int size;
{

#ifdef MEMLEAKDEBUG
	if (size == sizeof(struct LinkType)) 
		NumLinksMalloced++;
	else if (size == sizeof(struct NodeType)) 
		NumNodesMalloced++;
	else 
		NumUnknownMallocs++;
	fprintf(stderr, "MyMalloc MallocedLinks=%d MallocedNodes=%d \n",
				NumLinksMalloced, NumNodesMalloced);
#endif
	return(malloc((unsigned) size));
}

/************************************************************************
 *	AllocNode-Allocate a Node Structure				*
 ************************************************************************/
struct NodeType *AllocNode()
{
	struct NodeType *Ptr;

	/*
	 *  Fetch some more memory if necessary; otherwise, just pull
	 *  off some meory from the free list we maintain ourselves.
	 */
	if (FreeNodeHead == NULL)
		return((struct NodeType *) MyMalloc(sizeof(struct NodeType)));
	Ptr = FreeNodeHead;
	FreeNodeHead = Ptr->Next;
	FreeNodeCount--;
#ifdef MEMLEAKDEBUG
	printf("AllocNode() FreeNodeStructcount=%d      \n",FreeNodeCount);
#endif
	return( Ptr );
}

/************************************************************************
 * AllocLink-Allocate a Link Structure					*
 ************************************************************************/
struct LinkType *AllocLink()
{
	struct LinkType *Ptr;

	if (FreeLinkHead == NULL) {
		if ( FreeLinkCount != 0)
			fprintf(stderr,"Lost some link memory!!!\n");
		return((struct LinkType *) MyMalloc(sizeof(struct LinkType)));
	}
	Ptr = FreeLinkHead;
	FreeLinkHead = Ptr->Next;
	FreeLinkCount--;
#ifdef DEBUGMEMLEAK
	printf("AllocLink() FreeLinkStructcount=%d    \n",FreeLinkCount);
#endif
	return(Ptr);
}

/************************************************************************
 * FreeNode() - Return this node structure to our internal linked list.	*
 ************************************************************************/
static void FreeNode( n )
struct NodeType *n;
{
	if (n == NULL) {
		fprintf(stderr, "FreeNode(): can't free NULL!\n");
		return; 
	}
	FreeNodeCount++;
#ifdef DEBUGMEMLEAK
	printf("FreeNode() FreeNodeStructcount=%d      \n",FreeNodeCount);
#endif
	if (FreeNodeHead == NULL) {
		FreeNodeHead = n;
		n->Next = NULL;
	}
	else {
		n->Next = FreeNodeHead;
		FreeNodeHead = n;
	}
}

/************************************************************************
 * FreeLink() - Return this link structure to our internal linked list.	*
 ************************************************************************/
static void FreeLink( l )
struct LinkType *l;
{
	if (l == NULL) { 
		fprintf(stderr, "FreeLink(): can't free NULL!\n");
		return; 
	}
	FreeLinkCount++;
#ifdef DEBUGMEMLEAK
	printf("FreeLink() FreeLinkStructcount=%d       \n",FreeLinkCount);
#endif
	if (FreeLinkHead == NULL) {
		FreeLinkHead = l;
		l->Next = NULL;
	}
	else {
		l->Next = FreeLinkHead;
		FreeLinkHead = l;
	}
}

/************************************************************************
 * FreeLinkChain() - Free all Link structures in this chain.		*
 *		Call FreeLink() to do the actual work.			*
 ************************************************************************/
void FreeLinkChain( LinkChain )
struct LinkType *LinkChain;
{
	struct LinkType *L;

	while (LinkChain != NULL) {
		L = LinkChain->Next;
		FreeLink(LinkChain);
		LinkChain = L;
	}	
}

/************************************************************************
 * FreeNodeChain() - Free all Node structures in this chain.		*
 *		Call FreeNode() to do the actual work.			*
 ************************************************************************/
void FreeNodeChain( NodeChain )
struct NodeType *NodeChain;
{
	struct NodeType *N;

	while (NodeChain != NULL) {
		N = NodeChain->Next;
		FreeNode(NodeChain);
		NodeChain = N;
	}	
}
