/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 * This file contains modules concerned with Creating and Altering
 * the Network Structure, which is required in Processing the Neighbor 
 * Table received from the Network Node we queried.
 */
 
#include <stdio.h>
#include <string.h>
#include <memory.h>
#include "CommonDefs.h"
#include "StatusFileio.h"
#include "ProcessNeighborTable.h"


/****************************************************************
 * HandleMERITSpecialCases - Remove DOWN SCP Links if the SCP	*
 *			is in fact up elsewhere.		*
 ****************************************************************/
static struct NetworkType *HandleMERITSpecialCases( Net )
struct NetworkType *Net;
{
struct NodeType *n;
struct LinkType *l,*lprev;
char Buf[BUFSIZ];
extern char *Error_Log;

        for (l = Net->LinkHead; l != NULL; l = l->Next ) {
		if (( strlen(l->Node1)==2 )  && (strlen(l->Node2)==2))
			continue;	/* forget about PCP-PCP links */
		if ( l->State == DOWN ) {
			if ( strlen(l->Node1) == 4 )
	                	n = FindNode(Net, l->Node1);
			else
	                	n = FindNode(Net, l->Node2);
				
			if (( n == NULL ) || (n->State==UP)) {
			
                        	sprintf(Buf,"%s LINK BEING REMOVED: %s %s\n",
                                PrintLinkType(l->Type),l->Node1,l->Node2);
				Log(Buf, Error_Log);

				/* no free is done here-sloppy, but
				we will be exitting soon anyway */
				if ( l == Net->LinkHead )
					Net->LinkHead=l->Next;
				else 
					lprev->Next=l->Next;
			}
		}
		lprev=l;				
	}
	return( Net );
}

/************************************************************************
 *	MergeNetworks - Note differences				*
 ************************************************************************/
struct NetworkType *MergeNetworks( NEWNet, OLDNet )
struct NetworkType *NEWNet, *OLDNet;
{
	struct NodeType *n,*n2;
	struct LinkType *l,*l2;
	char Buf[BUFSIZ];
	extern char *Error_Log;
	extern void PreProcessNetwork();

	PreProcessNetwork(NEWNet);

	/*
	 *  Announce the differences.
	 */
        for (n = NEWNet->NodeHead; n != NULL; n = n->Next) {
		n2 = FindNode(OLDNet, n->IPAddress);
		if (n2 == NULL) {
			/*if (strlen(n->Name) < 2 || strlen(n->IPAddress) < 2) 
				continue;*/
			sprintf(Buf, "NEW %s NODE %s %s %s\n",
					PrintNodeType(n->Type), n->Name,
					n->IPAddress, PrintState(n->State));	
			AddNode(OLDNet, n->Name, n->Type, n->Queryable, 
					n->State, n->TimeStamp, n->IPAddress);
			Log(Buf, Error_Log);
		}
		else if (n2->State != n->State) {
			sprintf(Buf,"%s NODE State Transition: %s %s %s->%s\n",
				PrintNodeType(n2->Type),n->Name, n->IPAddress, 
				PrintState(n2->State), PrintState(n->State));
			n2->State = n->State;
			n2->TimeStamp = n->TimeStamp;
			Log(Buf, Error_Log);
		}
        }

	/* 
	 *  Process all newly found links.
	 */
        for (l = NEWNet->LinkHead; l != NULL; l = l->Next ) {
		l2 = FindLink(OLDNet, l->Node1, l->Node2, l->Type);
		if (l2 == NULL) { 
			sprintf(Buf, " NEW %s LINK: %s %s %s\n",
				PrintLinkType(l->Type), l->Node1, l->Node2,
				PrintState(l->State));
		Log( Buf, Error_Log );
		AddLink(OLDNet, l->Node1, l->Node2, l->State, l->TimeStamp, 
								l->Type);
		}
		else if (l2->State != l->State) {
                        sprintf(Buf,"%s LINK State Transition: %s %s %s->%s\n",
                                PrintLinkType(l->Type),l->Node1,l->Node2,
				PrintState(l2->State),
				PrintState(l->State)
				);
		Log( Buf, Error_Log );
		AddLink(OLDNet, l->Node1, l->Node2, l->State, l->TimeStamp, 
								l->Type);
		}
	}
	if (OLDNet->Type == MERIT )
		OLDNet=HandleMERITSpecialCases(OLDNet);
	return(OLDNet);
}

/************************************************************************
 * DiffNetworks - We use this routine with a callback telling us of any	*
 *		transitions that occured in the network.		*
 ************************************************************************/
DiffNetworks( NEWNet, OLDNet, callback )
struct NetworkType *NEWNet, *OLDNet;
void (* callback)();
{
	struct NodeType *n, *n2;
	struct LinkType *l, *l2;
	static struct TransitionCallback TransitionStruct;

	/* invoke the callback describing the differences */
	for (n = NEWNet->NodeHead; n != NULL; n = n->Next ) {
		n2 = FindNode(OLDNet, n->IPAddress);
		if (n2 == NULL) {	/* Exists in NEW but not OLD */
			/*** NEW NODE ***/
			TransitionStruct.Type=NEWNODE;
			memcpy((char *) &TransitionStruct.cptr.N, (char *) n, 
						sizeof(struct NodeType));
			TransitionStruct.OldState = n->State;
			TransitionStruct.NewState = n->State;
			TransitionStruct.PleaseDoIt = 1;
			callback(&TransitionStruct);	
		}
		else if (n2->State != n->State) {
			TransitionStruct.Type=NODETRANSITION;
			memcpy((char *) &TransitionStruct.cptr.N, (char *) n,
						sizeof(struct NodeType));
			TransitionStruct.OldState = n2->State;
			TransitionStruct.NewState = n->State;
			TransitionStruct.PleaseDoIt = 1;
			callback(&TransitionStruct);	
		}
        }

	/* process all newly found links and link state changes */
        for (l = NEWNet->LinkHead; l != NULL; l = l->Next) {
		l2 = FindLink(OLDNet, l->Node1, l->Node2, l->Type);
		if (l2 == NULL) {   
			TransitionStruct.Type = NEWLINK;
			memcpy((char *) &TransitionStruct.cptr.L, (char *) l,
						sizeof(struct LinkType));
			TransitionStruct.OldState = l->State;
			TransitionStruct.NewState = l->State;
			TransitionStruct.PleaseDoIt = 1;
			callback(&TransitionStruct);	
		}
		else if (l2->State != l->State) {
			TransitionStruct.Type = LINKTRANSITION;
			memcpy((char *) &TransitionStruct.cptr.L, (char *) l,
						sizeof(struct LinkType));
			TransitionStruct.OldState = l2->State;
			TransitionStruct.NewState = l->State;
			TransitionStruct.PleaseDoIt = 1;
			callback(&TransitionStruct);	
		}
	}
}

#ifdef USED
/************************************************************************
 * ProcessNeighborTable - We just got a Neighbor Table from the Network *
 *			  Convert it from the Network's NeighTable 	*
 *			  format into our required NeighTable Format.	*
 *									*
 *	Algorithm							*
 *	1) Call the node queried "UP"					*
 *	2) For each neighbor, call the Link to the Neighbor "UP"	*
 *	                      If the Neighbor is Queryable, upgrade the *
 *			      Node status to "BUSY"			*
 *									*
 ************************************************************************/
void ProcessNeighborTable(net, np, nt)
struct NetworkType *net;
struct NodeType *np;
struct NeighborTableType *nt;
{
	register int i;
	time_t TimeNow;
	extern char *Error_Log;

	time(&TimeNow);
	AddNode(net, np->Name, np->Type, QUERYABLE, UP, TimeNow, np->IPAddress);
	for (i = 0; (i < MAXNEIGHBORS) && (nt[i].Name[0] != '\0'); i++) {
		if (strcmp(nt[i].Name, np->Name) == 0) 
			continue;
		if (nt[i].State == UP) {
			AddLink(net, np->IPAddress, nt[i].IPAddress, UP, 
						TimeNow, nt[i].LinkType);
			if (nt[i].Queryable)
				AddNode(net, nt[i].Name, nt[i].Type, QUERYABLE, 
					BUSY, TimeNow, nt[i].IPAddress);
			else
				AddNode(net, nt[i].Name, nt[i].Type, 
					NONQUERYABLE, UP, TimeNow, 
					nt[i].IPAddress);
		}
		else {	/* Definative answer - link is down */
			AddNode(net, nt[i].Name, nt[i].Type, nt[i].Queryable, 
						NR, TimeNow, nt[i].IPAddress);
			AddLink(net, np->IPAddress, nt[i].IPAddress, DOWN, 
						TimeNow, nt[i].LinkType);
		}
	}
}
#endif
/************************************************************************
 *   ProcessNeighborLine() - This code deals with the getneighbors2	*
 *				output.					*
 *									*
 *   Syntax of the line is:						*
 *		ReportingNode NODE/LINK TYPE Name iface iface Name State*
 ************************************************************************/
void ProcessNeighborLine(Network, buf)
struct NetworkType *Network;
char *buf;
{
#ifdef HACK
	char dbuffer[BUFSIZ];
#endif
	time_t TimeNow;
	extern char *Error_Log;
	char *p, *ReportingNode, *Type, *UniqueID, *NodeName, *State, *Side1,
			*Side2;
	int isNode = 0, isLink = 0, type = 0, state = 0;
	char echoline[BUFSIZ];

	/*
	 *  Save away 'buf' since strtok() will fiddle around with it.
	 */
	TimeNow = time(&TimeNow);
	strcpy(echoline, buf);

	/*
	 *  Determine if it is a NODE or a LINK.
	 */
	if ((ReportingNode = strtok(buf, DELIMITERS)) == NULL) {
		fprintf(stderr, "%s NULL Neighbor Line\n", echoline);
		return;
	}
	if ((p = strtok((char *) NULL, DELIMITERS)) == NULL)  {
		fprintf(stderr, "%s NULL NODE/LINK Line\n", echoline);
		return;
	}
	else if ((strcmp(p, "NODE") == 0)) 
		isNode = 1;
	else if ((strcmp(p, "LINK") == 0)) 
		isLink = 1;
	else {
		fprintf(stderr, "%s Unrecognized NODE/LINK Line\n", echoline);
		return;
	}
	if ((Type = strtok((char *) NULL, DELIMITERS)) == NULL) {
                fprintf(stderr, "%s NULL Type Line\n", echoline);
		return;
	}
	else if (isNode) {
		if ((type = Str_To_NodeType(Type)) == 0) {
			fprintf(stderr, "%s Unrecognized Node Type (%s)\n",
							echoline, Type);
			return;
		}
	}
	else if (isLink) {
		if ((type = Str_To_LinkType(Type)) == 0) {
			fprintf(stderr, "%s Unrecognized Link Type (%s)\n",
							echoline, Type);
			return;
		}
	}

	/*
	 *  It is a LINK.
	 */
	if (isLink) {
	        if ((Side1 = strtok((char *) NULL, DELIMITERS)) == NULL) {
                	fprintf(stderr, "%s NULL Side1 Line\n", echoline);
			return;
		}
	        if (strtok((char *) NULL, DELIMITERS) == NULL) {
                	fprintf(stderr, "%s NULL Side1 IF Line\n", echoline);
			return;
		}
	        if (strtok((char *) NULL, DELIMITERS) == NULL) {
                	fprintf(stderr, "%s NULL Side2 IF Line\n", echoline);
			return;
		}
	        if ((Side2 = strtok((char *) NULL, DELIMITERS)) == NULL) {
                	fprintf(stderr, "%s NULL Side2 Line\n", echoline);
			return;
		}
		if (strcmp(Side1,Side2) == 0) {
			return;
		}
		if ((State = strtok((char *) NULL, DELIMITERS)) == NULL) {
                        fprintf(stderr, "%s %s NULL State Line\n",
						echoline, ReportingNode);
			return;
		}
		if ((state = Str_To_LinkState(State)) == 0) {
			fprintf(stderr, "%s Invalid State Line (%s)\n",
						echoline, State);
			return;
		}
		if ((state == UP) || FindNode(Network, Side2))
			AddLink(Network, Side1, Side2, state, TimeNow, type);
		switch(type) {
		case INPLINK:
			if (state == UP)
				if ( strlen(Side2) == 2 )
        				AddNode(Network, Side2, PCP, 1, 
						BUSY, TimeNow, Side2);
				else if ( strlen(Side2) == 4 )
        				AddNode(Network, Side2, SCP, 1, 
						UP, TimeNow, Side2);
				else fprintf(stderr,"Unidentified remote node:%s\n",Side2);
			break;
		case ISISLINK:
			if (state == UP)
        			AddNode(Network, Side2, NSS, 1, BUSY, 
							TimeNow, Side2);
			break;
		case ASLINK: 
			if (state == UP)
        			AddNode(Network, Side2, AS, 0, UP, 
							TimeNow,Side2);
			break;
		case GENLINK: 
			if (state == UP)
        			AddNode(Network, Side2, GENNODE, 0, BUSY, 
							TimeNow,Side2);
			break;
		default: 
			break;	
		}
	}

	/*
	 *  It is a node.
	 */
	if (isNode) {
	        if ((UniqueID = strtok((char *) NULL, DELIMITERS)) == NULL) {
                	fprintf(stderr, "%s NULL UniqueID Line\n", echoline);
			return;
		}
	        if ((NodeName = strtok((char *) NULL, DELIMITERS)) == NULL) {
                	fprintf(stderr, "%s NULL NodeName Line\n", echoline);
			return;
		}
		if (strcmp(NodeName, "()") == 0) 
			NodeName = NULL;
		if ((State = strtok((char *) NULL, DELIMITERS)) == NULL) {
                        fprintf(stderr, "%s %s NULL State Line\n", 
						echoline, ReportingNode);
			return;
		}
		if ((state = Str_To_NodeState(State)) == 0) {
			fprintf(stderr, "%s Invalid State Line (%s)\n",
						echoline, State);
			return;
		}

#ifdef HACK
		/*
		 *		HACK
		 *
		 *  Deal with ASs that are reachable but not via EGP.  
		 *  Attach to local node.
		 */
		sprintf(dbuffer, 
	"HACK ReportingNode=%s UniqueID=%s NodeName=%s State=%s state=%d\n",
				 ReportingNode,UniqueID,NodeName,State,state);
		Log(dbuffer, Error_Log);
		if ((strcmp(ReportingNode, "Unknown") == 0)) {
				if (Network->Type == NSFNETT3) 
					NodeName="140.222.131.62";	
				else if (Network->Type == NSFNET) 
					NodeName="129.140.17.1";	
                	AddNode(Network, NULL, type, CanQuery(type), state, 
							TimeNow,UniqueID);
                	AddLink(Network, NodeName, UniqueID, state, TimeNow, 
							ASLINK );
		}
		else
#endif
                AddNode(Network, NodeName, type, CanQuery(type), state, 
							TimeNow,UniqueID);
	}
	if ( strcmp(ReportingNode,"Unknown") != 0 ) 
	switch (Network->Type ) {
	case MERIT:
		switch( strlen(ReportingNode) ) {
        	case 2:
			AddNode(Network, ReportingNode, PCP, 1, UP, TimeNow,ReportingNode);
			break;
        	case 4:
			AddNode(Network, ReportingNode, SCP, 1, UP, TimeNow,ReportingNode);
			break;
		default:
			fprintf(stderr,"Unknown Merit Reporting Node type\n");
			/*AddNode(Network, ReportingNode, NSS, 1, UP, TimeNow,ReportingNode);*/
			break;
		}
	case GENERICNET:
		AddNode(Network, ReportingNode, GENNODE, 1, UP, TimeNow,ReportingNode);
		break;
	case NSFNETT3:
		AddNode(Network, ReportingNode, NSS, 1, UP, TimeNow,ReportingNode);
		break;
	default:
		fprintf(stderr,"Unknown Network type: %d\n",Network->Type);
		/*AddNode(Network, ReportingNode, NSS, 1, UP, TimeNow,ReportingNode);*/
		break;
	}
}

/************************************************************************
 *	PreProcessNetwork - Migrate Nodes/Links to Down if attached to  *
 *				something up.  Handle SCP links that    *
 *				move.					*
 ************************************************************************/
static void PreProcessNetwork( Network )
struct NetworkType *Network; 
{
	struct NodeType *n, *n2;
	register struct LinkType *l;
	char Buf[BUFSIZ];
	time_t TimeNow;
	extern char *Error_Log;

        /*
         * Migrate Links DOWN if attached to anything UP.
	 *
	 * A MOVED link is a link that is NR and migrates to DOWN,
	 * BUT BOTH Ends are UP ands one end is a SCP.
	 *
	 *  If either end is UP, then migrate the link to DOWN.
         */
        time(&TimeNow);
        for (l = Network->LinkHead; l != NULL; l = l->Next ) {
		/*
		 *  If the link state is not NOT RESPONDING, then
		 *  move on to the next one.
		 */
                if (l->State != NR)
			continue;

		/*
		 *  Lookup the node in the network; if it isn't
		 *  found, move on to the next one.
		 */
		if ((n = FindNode(Network, l->Node1)) == NULL)
			continue;

		/*
		 *  If the node state is not UP, move on to the next one.
		 */
		if (n->State != UP)
			continue;

		l->State = DOWN;
		l->TimeStamp = TimeNow;
		if ((n2 = FindNode(Network, l->Node2)) != NULL) {
#ifdef MIGRATENODES
			if (n2->State == NR) {
				n2->State=DOWN;
				n2->TimeStamp=TimeNow;
			}
#endif
			if (n2->State == UP) {
				if ((n->Type == SCP) || (n2->Type == SCP)) {
					/* MOVED LINK */
					sprintf(Buf,
			"Link MOVED : %s-%s no longer exists; Both UP!\n",
							l->Node1, l->Node2);
					printf(Buf);
					Log(Buf, Error_Log);
					l->State = MOVED;
				}
			}
			continue;
		}
		
		if ((n = FindNode(Network, l->Node2)) == NULL)
			continue;
		if (n->State != UP)
			continue;
		l->State = DOWN;
		l->TimeStamp = TimeNow;
		if ((n2 = FindNode(Network, l->Node1)) != NULL) {
#ifdef MIGRATENODES
			if (n2->State == NR) {
				n2->State = DOWN;
				n2->TimeStamp = TimeNow;
			}
#endif
			if (n2->State == UP) {
				if ((n->Type == SCP) || (n2->Type == SCP)) {
					sprintf(Buf,
			"Link MOVED : %s-%s no longer exists; Both UP!\n",
							l->Node1, l->Node2);
					printf(Buf);
					Log(Buf, Error_Log);
					l->State = MOVED;
				}
			}
		}
	}
}
