/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

/************************************************************************
 *   Tape Player Code - This code deals with the playback mechanism     *
 *                      in the graphical display.                       *
 *                                                                      *
 *      This code works hand in glove with the scan code, and the       *
 *      xmap code.                                                      *
 ************************************************************************/
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>
#include "motif.h"
#include "CommonDefs.h"
#include "xmap.h"
#include "ProcessNeighborTable.h"

time_t TimeStamp;
Widget toplevel,TimeText,rate;
int CurrentAction = STOP;
int PlaybackDialogActive = 0;

extern void doaction();
struct TransitionCallback *ParseLogEntry();

/************************************************************************
 * PlayBack() - This gets called for all playback mechanism callbacks	*
 ************************************************************************/
/*ARGSUSED*/
static void PlayBack(w, action, call_data)
Widget w;
int action;
XtCallbackList *call_data;
{
	extern void exit();

	switch (action) {
	case FASTBACKWARD:
			if (CurrentAction == FASTBACKWARD) 
				return;
			break;
	case BACKWARD:
			if (CurrentAction == BACKWARD) 
				return;
			break;
	case STOP:
			break;
	case FORWARD:
			if (CurrentAction == FORWARD) 
				return;
			break;
	case FASTFORWARD:
			if (CurrentAction == FASTFORWARD) 
				return;
			break;
	default:
			printf("Internal Error - unknown PlayBack type\n");
			exit(1);
			/*NOTREACHED*/
	}
	CurrentAction = action;
	doaction(w, (XtIntervalId) 0);
}

/************************************************************************
 *  Invoke action - playback action - reading log file and playback 	*
 *			network event					*
 *									*
 *	Playback by *REAL* Network Transitions.  This means batch	*
 * up transitions that occur in the same minute, into a meaningful	*
 * network state transition.						*
 *									*
 * if a redraw is pending and its a NewMinute, that is a Network Event  *
 * if a transition occured and its a NewMinute, that is a Network Event *
 ************************************************************************/
/*ARGSUSED*/
static void doaction(w, id)
Widget w;
XtIntervalId id;
{
	long next_action;
	char *dataptr;
	int ratevalue;
	Arg myArgs[MAX_X_ARGS];
	struct TransitionCallback *transition;
	extern void ObjectStateTransition();
	time_t LastDate=0;
	char TimeString[26];
	extern int ReDrawNecessary;
	extern char * getrecord();
	int TransitionReported = 0;

	switch (CurrentAction) {
	case FASTBACKWARD:
		if ((dataptr = getrecord(FASTBACKWARD)) == NULL) {
			CurrentAction = STOP;
			break;
		}
		if ((transition = ParseLogEntry(BACKWARD, dataptr))==NULL){
			fprintf(stderr,"FastBckwd: NULL parse of dataptr\n");
			break;
		};
		strcpy(TimeString, transition->timestr);
		next_action = 0;
		CurrentAction = STOP;
		ClearAndDrawMap();	
		break;
	case BACKWARD:
	case FORWARD:
		for (;;) {
			if ((dataptr = getrecord(CurrentAction )) == NULL) {
				CurrentAction = STOP;
				break;
			}

			if ((transition = ParseLogEntry(CurrentAction, dataptr))==NULL){
				fprintf(stderr,"Fwd/Bwd: NULL parse of dataptr\n");
				continue;
			};
			if ( transition->Type != UNKNOWN_TRANSITION ) {
				ObjectStateTransition( transition ); /*DO IT*/
				TransitionReported = 1;
			}

/*printf("TimeStamp=%lu LastDate=%lu\n",transition->TimeStamp,LastDate);*/
			if ( LastDate == 0 ) LastDate=transition->TimeStamp;
			strcpy(TimeString, transition->timestr );


			if ( CurrentAction==FORWARD )
				if (transition->TimeStamp - LastDate > 60 ) {
					/*printf("Forward event\n");*/
					break;
				}
			if ( CurrentAction==BACKWARD )
				if ( LastDate - transition->TimeStamp > 60 ) {
					/*printf("Backward Event\n");*/
					break;	
				}
			XmTextSetString( TimeText, TimeString);
			/* Same Minute - batch these transitions up */
			/*  Don't redraw multiple transitions 
			   that occured in the same minute	*/
			if (TransitionReported) {
				printf("Transition reported\n");
				break;
			}
			if (ReDrawNecessary) {
				ClearAndDrawMap();	
				break;   /* Redraws pending */
			}
		}
		XtSetArg( myArgs[0], XmNvalue, &ratevalue );
		XtGetValues( rate, myArgs, 1 );
		next_action = ratevalue;
		break;
	case STOP:
		if ( ReDrawNecessary ) {
			ClearAndDrawMap();	
		}
		return;
		break;
	case FASTFORWARD:
		if ((dataptr=getrecord( FASTFORWARD ))==NULL) {
			CurrentAction=STOP;
			break;
		}
		if ((transition=ParseLogEntry( FORWARD, dataptr ))==NULL){
			fprintf(stderr,"FastFwd: NULL parse of dataptr\n");
			break;
		};
		strcpy( TimeString, transition->timestr);
		next_action=0;
		CurrentAction=STOP;
		ClearAndDrawMap();	
		break;
	}
	XmTextSetString( TimeText, TimeString);
	/*if ( next_action != 0 )*/
	if ( CurrentAction != STOP )
		XtAddTimeOut(next_action, doaction, w);
	else printf("STOPPED\n");
}

/************************************************************************
 *   ClosePlaybackDialog() - User is done with playback function	*
 ************************************************************************/
/*ARGSUSED*/
static void ClosePlaybackDialog(w, shell, call_data)
Widget w;
Widget shell;
XtCallbackList *call_data;
{
	extern void CheckStatusFile(), ReDrawMap();

    	CurrentAction = STOP;
    	doaction(w, (XtIntervalId) 0);
	XtDestroyWidget(shell);
	PlaybackDialogActive = 0;

#ifndef STANDALONE
	printf("Restting status file to current status\n");
	ReDrawMap();
	XtAddTimeOut((long) 0, CheckStatusFile, (Widget) NULL);
#endif
}

/************************************************************************
 *  PlaybackDialog() - Bring up and install playback code		*
 ************************************************************************/
void PlaybackDialog()
{
	Widget 	shell, MainAppBB,stoppb,forwardpb,backwardpb,rewindpb,
		fastforwardpb, returntoprog;
	Arg	myArgs[MAX_X_ARGS];
	char 	mytimestr[26];
	extern time_t time();

	if (PlaybackDialogActive == 1) {
		fprintf(stderr,"Playback dialog is already active\n");
		return;
	}
	PlaybackDialogActive = 1;
	XtSetArg(myArgs[0], XmNx, 400);
	XtSetArg(myArgs[1], XmNy, 400);
	shell = XtCreateApplicationShell("PlaybackDialog", 
				transientShellWidgetClass, myArgs, 2);
	MainAppBB = XtCreateManagedWidget( "MainAppBB",
                        xmBulletinBoardWidgetClass, shell, myArgs, 0);
	TimeText = XtCreateManagedWidget( "time", xmTextWidgetClass, MainAppBB, 
			myArgs,0);
	TimeStamp = time(&TimeStamp);
	strcpy(mytimestr, ctime(&TimeStamp));
	mytimestr[24] = '\0';
	XmTextSetString(TimeText, mytimestr);

#ifdef FIXED
	XtSetArg(myArgs[0], XmNx, 0);
	XtSetArg(myArgs[1], XmNy, 40);
	rewindpb = XtCreateManagedWidget( "<<", xmPushButtonWidgetClass,
				MainAppBB, myArgs, 2);
	XtAddCallback(rewindpb, XmNactivateCallback, PlayBack, 
						(caddr_t) FASTBACKWARD);
#endif
	XtSetArg(myArgs[0], XmNx, 40);
	XtSetArg(myArgs[1], XmNy, 40);
	backwardpb = XtCreateManagedWidget( "<", xmPushButtonWidgetClass,
				MainAppBB, myArgs, 2);
	XtAddCallback(backwardpb, XmNactivateCallback, PlayBack, 
						(caddr_t) BACKWARD);

	XtSetArg(myArgs[0], XmNx, 60);
	XtSetArg(myArgs[1], XmNy, 40);
    	stoppb = XtCreateManagedWidget( "o", xmPushButtonWidgetClass,
				MainAppBB, myArgs, 2);
    	XtAddCallback(stoppb, XmNactivateCallback, PlayBack, (caddr_t) STOP);

	XtSetArg(myArgs[0], XmNx, 80);
	XtSetArg(myArgs[1], XmNy, 40);
    	forwardpb = XtCreateManagedWidget( ">", xmPushButtonWidgetClass,
				MainAppBB, myArgs, 2);
    	XtAddCallback(forwardpb, XmNactivateCallback, PlayBack, 
						(caddr_t) FORWARD);
#ifdef FIXED
	XtSetArg(myArgs[0], XmNx, 120);
	XtSetArg(myArgs[1], XmNy, 40);
    	fastforwardpb = XtCreateManagedWidget( ">>", xmPushButtonWidgetClass,
				MainAppBB, myArgs, 2);
    	XtAddCallback(fastforwardpb, XmNactivateCallback, PlayBack, 
						(caddr_t) FASTFORWARD);
#endif

	XtSetArg(myArgs[0], XmNx, 60);
	XtSetArg(myArgs[1], XmNy, 160);
    	returntoprog = XtCreateManagedWidget("returntoprog", 
				xmPushButtonWidgetClass, MainAppBB, myArgs, 2);
    	XtAddCallback(returntoprog, XmNactivateCallback, ClosePlaybackDialog, 
				(caddr_t) shell);

	XtSetArg(myArgs[0], XmNx, 0);
	XtSetArg(myArgs[1], XmNy, 80);
	XtSetArg(myArgs[2], XmNshowValue, True);
	XtSetArg(myArgs[3], XmNtitleString, 
			XmStringCreateLtoR("Rate",XmSTRING_DEFAULT_CHARSET));
	XtSetArg(myArgs[4], XmNheight, 80);
	XtSetArg(myArgs[5], XmNwidth, 10);
	XtSetArg(myArgs[6], XmNvalue, 200);
	XtSetArg(myArgs[7], XmNmaximum, 1500);
	XtSetArg(myArgs[8], XmNminimum, 100);
    	rate = XtCreateManagedWidget( "rate", xmScaleWidgetClass, MainAppBB, 
								myArgs, 9);
	XtRealizeWidget(shell);
}

#ifdef STANDALONE
char progname[];

/************************************************************************
 *	main() - test bench the code first				*
 ************************************************************************/
main(argc, argv)
int argc;
char *argv[];
{
	Widget activate;

	progname=argv[0];
	toplevel = XtInitialize(argv[0],		/* ShellName	*/
				"Ff",			/* Application Class */
				NULL,			/* Options	*/
				0,			/* Num Options	*/
				&argc,			/* Num Args 	*/
				argv );			/* Arg List	*/


    	activate = XtCreateManagedWidget( "Activate", xmPushButtonWidgetClass,
				toplevel, NULL ,0);
    	XtAddCallback(activate, XmNactivateCallback, PlaybackDialog, 
				(caddr_t) NULL);
	XtRealizeWidget(toplevel);	/* Bring up & manage toplevel widget */
	XtMainLoop();			/* Main Event handler loop	*/
}

#endif

static char *months[] =
{
        "Jan",
        "Feb",
        "Mar",
        "Apr",
        "May",
        "Jun",
        "Jul",
        "Aug",
        "Sep",
        "Oct",
        "Nov",
        "Dec",
        0
};


/*********************************************************************
 *  ParseEntryLog() - Parse the rover log entry.  Return a Transition*
 *			struct or NULL if it isn't identifyable      *
 *********************************************************************/
struct TransitionCallback *ParseLogEntry (direction, Entry)
int direction;
char *Entry;
{
	char workarea[2 * BUFSIZ], *p, *NodeLinkType, *Node1, *Node2,
		*OldState,*NewState;
	int New = 0, Type, x, y, i;
	static struct TransitionCallback trans;
	static char LastDate[26];
	extern int verbose;
	extern void exit();
	char s[BUFSIZ];
	struct tm tm;

	if ( strlen(Entry) < 26 ) {
		return(NULL);
	}
	strcpy(s, Entry);

	/* terminate strings in place */
	s[3] = s[7] = s[10] = s[13] = s[16] = s[19] = s[24] = '\0';
	tm.tm_sec = atoi(&s[17]);
	tm.tm_min = atoi(&s[14]);
	tm.tm_hour = atoi(&s[11]);
	tm.tm_mday = atoi(&s[8]);
	tm.tm_year = atoi(&s[20]) - 1900;
	tm.tm_isdst = 0;

	for (i = 0; months[i]; i++)
                if (strcmp(&s[4], months[i]) == 0)
                        break;
	if (months[i])
	        tm.tm_mon = i;
	else
	        fprintf(stderr, "Error: \"%s\"--no such month.\n");
	trans.TimeStamp= mktime(&tm);


	trans.Type = UNKNOWN_TRANSITION;
	strncpy(trans.timestr, Entry, 25);
	trans.timestr[25] = '\0';
	if ((atoi(&Entry[20]) == 0) || (atoi(&Entry[11]) == 0)) {
		strncpy(trans.timestr, LastDate, 25);
		trans.Type = UNKNOWN_TRANSITION;
		return(&trans);
	}
	if (verbose) 
		printf("ParseLogEntry(): echo %s \n", Entry);
	strncpy(LastDate, Entry, 25);
	strcpy(workarea, Entry);

	if ((p = strtok(workarea, DELIMITERS)) == NULL) {
		fprintf(stderr,"ParseLogEntry(): Bad Day\n"); 
		return(&trans);
	}
	if ((p = strtok((char *) NULL, DELIMITERS)) == NULL) {
		fprintf(stderr,"ParseLogEntry(): Bad Month\n"); 
		return(&trans);
	}
	if ((p = strtok((char *) NULL, DELIMITERS)) == NULL) {
		fprintf(stderr,"ParseLogEntry(): Bad Date\n"); 
		return(&trans);
	}
	if ((p = strtok((char *) NULL, DELIMITERS)) == NULL) {
		fprintf(stderr,"ParseLogEntry(): Bad time\n"); 
		return(&trans);
	}
	if ((p = strtok((char *) NULL, DELIMITERS)) == NULL) {
		fprintf(stderr,"ParseLogEntry(): Bad Year\n"); 
		return(&trans);
	}
	if ((p = strtok((char *) NULL, DELIMITERS)) == NULL) {
		fprintf(stderr,"ParseLogEntry(): Bad TYPE\n"); 
		return(&trans);
	}
	if (strcmp(p, "NEW") == 0) {
		New=1;
		return(&trans);
	}

	NodeLinkType = p;
	if (strcmp("NSS", p) == 0)
		printf("NSS Node transition\n");

	if ((p=strtok((char *) NULL, DELIMITERS)) == NULL) {
		fprintf(stderr, "ParseLogEntry(): NO Node or LINK TYPE\n"); 
		return(&trans);
	}
	if (strcmp(p, "NODE") == 0) 
		Type = NODE;
	else if (strcmp(p, "LINK") == 0)
		Type = LINK;
	else {
		/*fprintf(stderr,
			"ParseLogEntry(): Unrecognized Node or LINK TYPE: %s\n",Entry); */
		return(&trans);
	}

	if (((p = strtok((char *) NULL, DELIMITERS)) == NULL) || 
					(strcmp(p, "State") != 0)) {
		fprintf(stderr,"ParseLogEntry(): No State keyword\n"); 
		return(&trans);
	}
	if (((p = strtok((char *) NULL, DELIMITERS)) == NULL) || 
					(strcmp(p, "Transition:") != 0)) {
		fprintf(stderr,"ParseLogEntry(): No Transition keyword\n"); 
		return(&trans);
	}
	if ((Node1 = strtok((char *) NULL, DELIMITERS)) == NULL) {
		fprintf(stderr,"ParseLogEntry(): No Node1 specified\n"); 
		return(&trans);
	}
	if ((Node2 = strtok((char *) NULL, DELIMITERS)) == NULL) {
		fprintf(stderr,"ParseLogEntry(): No Node2 specified\n"); 
		return(&trans);
	}

	if ((OldState = strtok((char *) NULL, " ->\t\n")) == NULL) {
		fprintf(stderr,"ParseLogEntry(): No OldState specified\n"); 
		return(&trans);
	}

	if ((NewState = strtok((char *) NULL, " ->\t\n")) == NULL) {
		fprintf(stderr,"ParseLogEntry(): No NewState specified\n"); 
		return(&trans);
	}

	/*
	 *  Play backwards?
	 *  State transition is reversed.
	 */
	trans.OldState = Str_To_NodeState(OldState);
	trans.NewState = Str_To_NodeState(NewState);
	if (direction == BACKWARD) {
		x = trans.OldState;
		trans.OldState = trans.NewState;
		trans.NewState = x;
	}

	switch (Type) {
	case NODE:
		if (GetXY(Node2, &x, &y) == 0)
			return(&trans);
		if (New) 
			trans.Type = UNKNOWN_TRANSITION;
		else 
			trans.Type = NODETRANSITION;
		trans.cptr.N.TimeStamp = 0;
		strcpy(trans.cptr.N.Name, Node1);
		strcpy(trans.cptr.N.IPAddress, Node2);
    		trans.cptr.N.State = trans.NewState;
		trans.cptr.N.Type = Str_To_NodeType(NodeLinkType);
		break;

	case LINK:
		if ((GetXY(Node1, &x, &y) == 0) || (GetXY(Node2, &x,&y) == 0))
			return(&trans);
		if (New) 
			trans.Type = UNKNOWN_TRANSITION;
		else 
			trans.Type = LINKTRANSITION;
		trans.cptr.L.TimeStamp = 0;
		strcpy(trans.cptr.L.Node1, Node1);
		strcpy(trans.cptr.L.Node2, Node2);
    		trans.cptr.L.State = trans.NewState;
		trans.cptr.L.Type = Str_To_LinkType(NodeLinkType);
		break;

	default:
		fprintf(stderr,
			"ParseLogEntry(): Parse didn't determine type\n");
		exit(1);
		/*NOTREACHED*/
	}
	trans.PleaseDoIt = 1;
	return(&trans);
}
