/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       api.c
**     SYSTEM   NAME:       mib
**     ORIGINAL AUTHOR(S):  Jan van Oorschot
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       
**
** DESCRIPTION: This file contains the API calls supported my the
**              MIB library.
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#include <stdio.h>
#include <string.h>
#include <err.h>
#include "mibinc.h"

#ifndef MIBHOME
#define	MIBHOME "/usr/local/lib/mibhome"
#endif

/**************************************************************
** NAME:        MibInit
** SYNOPSIS:    iRet = MibInit();
**              
** DESCRIPTION: Initialises the MIB system
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int MibInit(void)
{
    MibInitialise();
    return 0;
}
/**************************************************************
** NAME:        MibFullMib
** SYNOPSIS:    iRet = MibFullMib();
**              
** DESCRIPTION: Reads the file $MIBHOME/mibs and interprets each
**		line as the name of a MOSI mib file in $MIBHOME
**		Puts these
**		files in the tree.
**		MIBHOME should be something like 
**
**			/usr/local/lib/mibhome/
**
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int MibFullMib(void)
{
    FILE *fp1,*fp2;
    char  fname[265],pname[256];
    int len,iret;

    strcpy(pname,MIBHOME);
    strcat(pname,"/mibs");

    fp1 = fopen(pname,"r");
    if(fp1 == NULL)
    {
	ERR_ERR(
		1,
		ERR_CRITICAL,
		"Could not open MIB catalog %s\n",
		pname
		);
    }
    while(!feof(fp1))
    {
	strcpy(pname,MIBHOME);
	fgets(fname,256,fp1);
	len = strlen(fname);
	if(len > 1)
	{
	    if(fname[len-1] == '\n')
	      fname[len-1] = '\0';
	    strcat(pname,"/");
	    strcat(pname,fname);
	    fp2 = fopen(pname,"r");
	
	    if(fp2 != NULL)
	    {
		iret = MibReadMib(fp2);
	    }
	    else
	    {
		ERR_ERR(
			1,
			ERR_WARNING,
			"could not read mib file <%s>\n",
			pname
			);
	    }
	}
    }
    return 0;
}

/**************************************************************
** NAME:        MibReadMib
** SYNOPSIS:    iRet = MibReadMib(fp);
**		FILE *fp;
**              
** DESCRIPTION: Reads a MIB definition from file-stream <fp>
**		up to the EOF.  The MIB read from this file
**		is stored in the global MIB tree.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int MibReadMib( FILE *fp)
{
    char line[256];
    MIBNODE *node;

    while( !feof(fp))
    {
	fgets(line,256,fp);
	node = MibScanLine(line);
    }
    return 0;
}
/**************************************************************
** NAME:	MibName2ID        
** SYNOPSIS:    int = MibName2ID(id,name);
**		int	*id;
**		char	*name;
**              
** DESCRIPTION: Translates the <name> of a MIB variable 
**		in the <id> of that variable.
** RETURNS:     <0 	error
**		else	length of id
**************************************************************/
int MibName2ID(int *id, char *name)
{
    MIBNODE *node;
    /* find the correct node, and trace it back to get it's id */
    node = MibFindNode(name);
    if( node == NULL)
      {
	  ERR_ERR(
		  1,	
		  ERR_CRITICAL,
		  "MIB variable %s doesn't exist\n",
		  name
		  );
      }
    return MibFillID(node,id);
}


/**************************************************************
** NAME:        MibID2Name
** SYNOPSIS:    cPtr = MibID2Name(id,nr);
**		int	*id;
**		int	nr;
**              
** DESCRIPTION: Translates the given <id> with length <nr>
**		to the corresponding logical MIB name. Returns
**		a pointer to that name.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
char * MibID2Name(int *id, int nr)
{
    
    MIBNODE *node;

    node = MibFindID(id,nr);
    if( node == NULL) return NULL;
    return node->name;
}
/**************************************************************
** NAME:        MibID2Name2
** SYNOPSIS:    cPtr = MibID2Name2(id,nr,nrmatched);
**		int	*id;
**		int	nr;
**		int	*nrmatched
**              
** DESCRIPTION: Translates the given <id> with length <nr>
**		to the corresponding logical MIB name. Returns
**		a pointer to that name and fills <nrmatched>
**		with the number of matched sub-identifiers
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
char * MibID2Name2(int *id, int nr,int *nrmatched)
{
    
    MIBNODE *node;

    node = MibFindID2(id,nr,nrmatched);
    if( node == NULL) return NULL;
    return node->name;
}
/**************************************************************
** NAME:        MibIDType
** SYNOPSIS:    iRet =  MibIDType(id,nr);
**              int	*id;
**              int	iRet,nr;
** DESCRIPTION: Returns the type if the MIB variable identified
**		by <id> of lenght <nr>
** RETURNS:     <0	error
**              else    mode
**************************************************************/
int MibIDType( int *id, int nr)
{
    MIBNODE *node;
    node = MibFindID(id,nr);
    if( node == NULL) return -1;
    return node->type;
}

/**************************************************************
** NAME:        MibIDMode
** SYNOPSIS:    iret = MibIDMode(id,nr);
**              int *id;
**              int nr,iRet;
** DESCRIPTION: Returns the mode of the MIB variable identified
**		by <id> of length <nr>
** RETURNS:     <0	error
**		mode
**************************************************************/
int MibIDMode( int *id, int nr)
{
    MIBNODE *node;
    node = MibFindID(id,nr);
    if( node == NULL) return -1;
    return node->mode;
}

/**************************************************************
** NAME:        MibIDStatus
** SYNOPSIS:    iRet = MibIDStatus(id,nr);
**		int 	*id;
**		int	nr,iRet;
** DESCRIPTION: Returns the status of the MIB varaible 
**		identified by <id> of length <nr>.
** RETURNS:     <0	error
**              status
** RETURNS:     void
**************************************************************/
int MibIDStatus( int *id, int nr)
{
    MIBNODE *node;
    node = MibFindID(id,nr);
    if( node == NULL) return -1;
    return node->status;
}

/**************************************************************
** NAME:        MibNameType
** SYNOPSIS:    iRet = MibNameType(name)
**		char	*name;
**
** DESCRIPTION: Returns the type of the MIB varaible 
**		identified by <name>
** RETURNS:     <0	error
**              type
** RETURNS:     void
**************************************************************/
int MibNameType( char *name)
{
    MIBNODE *node;
    node = MibFindNode(name);
    if( node == NULL) return -1;
    return node->type;
}

/**************************************************************
** NAME:        MibNameMode
** SYNOPSIS:    iRet = MibNameMode(name)
**		char	*name;
**
** DESCRIPTION: Returns the mode of the MIB varaible 
**		identified by <name>
** RETURNS:     <0	error
**              mode
** RETURNS:     void
**************************************************************/
int MibNameMode( char *name)
{
    MIBNODE *node;
    node = MibFindNode(name);
    if( node == NULL) return -1;
    return node->mode;
}

/**************************************************************
** NAME:        MibNameStatus
** SYNOPSIS:    iRet = MibNameStatus(name)
**		char	*name;
**
** DESCRIPTION: Returns the status of the MIB varaible 
**		identified by <name>
** RETURNS:     <0	error
**              status
** RETURNS:     void
**************************************************************/
int MibNameStatus( char *name)
{
    MIBNODE *node;
    node = MibFindNode(name);
    if( node == NULL) return -1;
    return node->mode;
}
/**************************************************************
** NAME:        MibGetFirstChild
** SYNOPSIS:    char * MibGetFirstChild(name);
**		char	*name;
** DESCRIPTION: return pointer to name of first 
**		child of <name>
** RETURNS:     NULL	no first child
**              ptr to name first child
** RETURNS:     void
**************************************************************/
char *MibGetFirstChild(char *name)
{
    MIBNODE *node,*fchild;

    node = MibFindNode(name);
    if(node==NULL) return NULL;
    fchild=node->child1;
    if(fchild==NULL) return NULL;
    return fchild->name;
}
/**************************************************************
** NAME:        MibGetNextChild
** SYNOPSIS:    char * MibGetNextChild(name);
**		char	*name;
** DESCRIPTION: return pointer to name of next 
**		child from child <name>
** RETURNS:     NULL	no next child
**              ptr to name next child
** RETURNS:     void
**************************************************************/
char *MibGetNextChild(char *name)
{
    MIBNODE *node,*fchild;

    node = MibFindNode(name);
    if(node==NULL) return NULL;
    fchild=node->brother;
    if(fchild==NULL) return NULL;
    return fchild->name;
}







