/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       mib.c
**     SYSTEM   NAME:       Sage
**     ORIGINAL AUTHOR(S):  Jan van Oorschot
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       
**
** DESCRIPTION: Main MIB handling routine.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#include <stdio.h>
#include "mibinc.h"

static MIBNODE * mibRoot = NULL;
static MIBHASH * mibHash = NULL;


VOID MibInitialise()
{
    int i;
    /* initialise the tree */
    mibRoot = (MIBNODE *) malloc(sizeof(MIBNODE));
    mibRoot->name = "iso";
    mibRoot->index = 1;
    mibRoot->type = 0;
    mibRoot->mode = 0;
    mibRoot->status = 0;
    mibRoot->father = mibRoot->brother = mibRoot->child1 = NULL;
    mibHash = (MIBHASH *) malloc(MIBHASHSIZE * sizeof(MIBHASH));

    /* initialise the hash table */
    if(mibHash == NULL)
    {
	perror("Can't alloc Hash Table");
	exit(1);
    }
    for(i=0;i<MIBHASHSIZE;i++)
    {
	mibHash[i].next = NULL;
	mibHash[i].contents = NULL;
    }

    /* add the tree root to the hash table */
    mibHash[MibNameHash(mibRoot->name)].contents = mibRoot;
}

int MibNameHash(char *name)
{
    int j=0;
    while (*name)
      j=(j*3+*name++) % MIBHASHSIZE;
    return j;
}

VOID MibAddHash(char *name,MIBNODE *ptr)
{
    MIBHASH *mptr,*nptr;

    /* mptr points to the first of the list */
    mptr = &(mibHash[MibNameHash(name)]);
    if(mptr->contents == NULL)
    {
	mptr->contents = ptr;
    } else
    {
	nptr = (MIBHASH *) malloc(sizeof(MIBHASH));
	nptr->next = mptr->next;
	nptr->contents = ptr;
	mptr->next = nptr;
    }	    
}

MIBNODE *MibResolveHash(char *name)
{
    MIBHASH *mptr;
    /* simple implementation, liniear search in hash entry */
    mptr = &(mibHash[MibNameHash(name)]);
    if (mptr->contents == NULL) 
      return NULL;
    while(strcmp(mptr->contents->name,name) != 0)
      if((mptr = mptr->next) == NULL)
	return NULL;
    return mptr->contents;
}

MIBNODE *MibFindChild(MIBNODE *fPtr, int index)
{
    MIBNODE *cPtr;
    
    cPtr = fPtr->child1;

    while ( cPtr != NULL)
    {
	if( cPtr->index == index)
	  return cPtr;
	cPtr = cPtr->brother;
    }
    return NULL;
}

MIBNODE *MibFindNode(char *name)
{
    char *nPtr,*cPtr,fName[128];
    MIBNODE *fPtr,*rPtr;
    int cIndex;

    /* find the node with the name <name> */
    /* name can be:
                         name
                         name.1
			 name.1.2.3
			 1.2.3
    */
    /* try to resolve, if this works, return imidiate */
    rPtr = MibResolveHash(name);
    if(rPtr != NULL)
      return rPtr;

    /* it doesn't exists, it can still be a dotted representation */
     
    nPtr = fName;
    for(cPtr = name; 
	*cPtr != '.' && *cPtr!='\0' && isalnum(*cPtr);
	cPtr++,nPtr++)
      *nPtr = *cPtr;
    
    *nPtr = '\0'; /* add a string end  */
    if(*cPtr != '\0')
      cPtr++;

    /* the rest of the string contains numbered list */
    /* if fName is "1", this is a full numbered id */
    if(strcmp(fName,"1") == 0)
    {
	fPtr = mibRoot;
    } 
    else
    {
	if( ( fPtr = MibResolveHash(fName)) == NULL)
	  return NULL;
    }

    /* follow the father */
    while( *cPtr != '\0')
    {
	nPtr = fName;
	while(isdigit(*cPtr))
	{
	    *nPtr++ = *cPtr++;
	}
	/* there should only be digits */
	if( *cPtr != '.' && *cPtr != '\0')
	  return NULL;
	if( *cPtr == '.')
	  cPtr++;

	cIndex = atoi(fName);
	  
	fPtr  = MibFindChild(fPtr, cIndex);
	if(fPtr == NULL)
	  return NULL;
    }

    return fPtr;
}    
	


int MibAddNode(MIBNODE *node,MIBNODE *fnode)
{
    MIBNODE *bnode;
    int bfound;

    /* node should contain a valid name and index */
    if (fnode == NULL)
      return -1;
    /* check if this is the first child */
    if(fnode->child1 == NULL)
    {
	fnode->child1 = node;
	node->father = fnode;
	node->brother = NULL;
	node->child1 = NULL;
	MibAddHash(node->name,node);
	return 0;
    }

    /* check if this node should be the first child of father */
    if( fnode->child1->index == node ->index)
      return -2;
    if(fnode->child1->index > node->index)
    {
	node->brother = fnode->child1;
	fnode->child1 = node;
	node->father = fnode;
	node->child1 = NULL;
	MibAddHash(node->name,node);
	return 0;
    }

    /* insert node in brother-list */
    bnode = fnode->child1; bfound = 0;
    while(! bfound )
      {
	  if(bnode->brother == NULL) 
	    bfound = 1;
	  else
	    if(bnode->brother->index >= node->index)
	      if(bnode->brother->index == node->index)
		return -1;
	      else
		bfound = 1;
	  else
	    bnode = bnode->brother;
      }
    /* insert in brother-list */
    node->brother = bnode->brother;
    node->father = fnode;
    node->child1 = NULL;
    bnode->brother = node;
    MibAddHash(node->name,node);

    return 0;
}

int MibFillID(MIBNODE *node,int *id)
{
    int indexnr;

    if( node->father == NULL)
    {
	/* at top , fill id and return sequence number */
	*id = node->index;
	return 1;
    }
    else
    {
	/* recursive call, then fill */
	indexnr = MibFillID(node->father,id);
	*(id + indexnr) = node->index;
	return ++indexnr;
    }
}

MIBNODE *MibFindID(int *id, int nr)
{
    MIBNODE *node;
    int cntr;
    
    node = mibRoot;

    for(cntr=1;cntr<nr;cntr++)
    {
	if(node == NULL) return NULL;
	node = MibFindChild(node,id[cntr]);
    }

    return node;
}

MIBNODE *MibFindID2(int *id, int nr, int *nrmatched)
{
    MIBNODE *node,*lastnode;
    int cntr;
    
    node = lastnode = mibRoot;

    for(cntr=1;cntr<nr;cntr++)
    {
	node = MibFindChild(lastnode,id[cntr]);
	if(node == NULL)
	{
	    *nrmatched = cntr;
	    return lastnode;
	}
	else
	{
	    lastnode = node;
	}
    }
    return node;
}


void MibDumpTree(MIBNODE *first)
{
    MIBNODE *cptr;
    /* dump the dump the tree using a recursive depth breath first search */
    cptr = first->child1;
    while(cptr != NULL)
    {
	MibDumpTree(cptr);
	cptr = cptr->brother;
    }
    printf("%s:%d ",first->name,first->index);
    if(first->father != NULL)
      printf("%s\n",first->father->name);
    else
      printf("none\n");
}

void MibDumpHash()
{
    int cntr;
    MIBHASH *hptr;

    for(cntr=0;cntr<MIBHASHSIZE;cntr++)
    {
	if( mibHash[cntr].contents != NULL)
	{
	    printf("%d ",cntr);
	    hptr = &(mibHash[cntr]);
	    while(hptr != NULL)
	    {
		printf("%s ",hptr->contents->name);
		hptr = hptr->next;
	    }
	    printf("\n");
	}
    }
}


