/*
 * snmp_auth.c -
 *   Authentication for SNMP (RFC 1067).  This implements a null
 * authentication layer.
 *
 *
 */
/**********************************************************************
	Copyright 1988, 1989, 1991, 1992 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

#ifdef KINETICS
#include "gw.h"
#include "fp4/cmdmacro.h"
#endif

#if (defined(unix) && !defined(KINETICS))
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#ifndef NULL
#define NULL 0
#endif
#endif

#ifdef vms
#include <in.h>
#endif

#include "asn1.h"
#include "snmp.h"
#include "snmp_impl.h"
#include "party.h"
#include "md5.h"
#include "acl.h"

u_char *
snmp_auth_parse(data, length, sid, slen, version)
    u_char          *data;
    int             *length;
    u_char          *sid;
    int             *slen;
    long            *version;
{
    u_char    type;

    data = asn_parse_header(data, length, &type);
    if (data == NULL){
        ERROR("bad header");
        return NULL;
    }
    if (type != (ASN_SEQUENCE | ASN_CONSTRUCTOR)){
        ERROR("wrong auth header type");
        return NULL;
    }
    data = asn_parse_int(data, length, &type, version, sizeof(*version));
    if (data == NULL){
        ERROR("bad parse of version");
        return NULL;
    }
    data = asn_parse_string(data, length, &type, sid, slen);
    if (data == NULL){
        ERROR("bad parse of community");
        return NULL;
    }
    sid[*slen] = '\0';
    return (u_char *)data;
}

u_char *
snmp_secauth_parse(data, length,  pi, srcParty, srcPartyLength,
		dstParty, dstPartyLength, pass)
    u_char	    *data;
    int		    *length;
    struct packet_info *pi;
    oid		    *srcParty, *dstParty;
    int		    *srcPartyLength, *dstPartyLength;
    int		    pass;
{
    u_char    type;
    oid	dstParty2[64];
    int dstParty2Length = 64, authMsgLen, authMsgInternalLen;
    u_long authSrcTimeStamp, authDstTimeStamp;
    long authNonce;
    u_char authDigest[16], digest[16];
    int authDigestLen;
    u_char *authMsg, *digestStart, *digestEnd;
    struct partyEntry *srcp, *dstp;
    int biglen, ismd5 = 0;
    struct timeval now;

    data = asn_parse_header(data, length, &type);
    if (data == NULL){
	ERROR("bad header");
	return NULL;
    }
    if (type != (ASN_CONTEXT | ASN_CONSTRUCTOR | 1)){
        ERROR("wrong auth header type");
        return NULL;
    }
    data = asn_parse_objid(data, length, &type, dstParty, dstPartyLength);
    if (data == NULL){
	ERROR("bad parse of dstParty");
	return NULL;
    }
    dstp = party_getEntry(dstParty, *dstPartyLength);
    if (!dstp)
	return NULL;
    /* check to see if TDomain and TAddr match here.
     * If they don't, discard the packet
     * This might be best handled by adding a user-supplied
     * function to the API that would validate the address.
     */
    
    data = asn_parse_header(data, length, &type);
    if (data == NULL || type != (ASN_CONTEXT | 1)){
	ERROR("bad parse of privData");
	return NULL;
    }
    /* check privacy algorithm for this party.  If dESPriv,
       decrypt from "data" here.  This is also the beginning of
       the digested portion of the packet.
       */
    if (dstp->partyPrivProtocol == DESPRIVPROT && (pass & FIRST_PASS)){
	/* privacy not implemented */
	return NULL;
    }
    authMsg = data;
    data = asn_parse_header(data, length, &type);
    if (data == NULL || type != (ASN_CONTEXT | ASN_CONSTRUCTOR | 1)){
	ERROR("bad parse of snmpAuthMsg (DES decode probably failed!)");
	return NULL;
    }
    authMsgLen = *length + data - authMsg;
    authMsgInternalLen = *length;
    data = asn_parse_header(data, &authMsgInternalLen, &type);
    if (data == NULL){
	ERROR("bad parse of snmpAuthMsg");
	return NULL;
    }
    if (type == (ASN_UNIVERSAL | ASN_PRIMITIVE | ASN_OCTET_STR)){
	/* noAuth */	    
	pi->version = SMP_1;
	
    } else if (type == (ASN_CONTEXT | ASN_CONSTRUCTOR | 2)){
	pi->version = SMP_1;
	ismd5 = 1;

	digestStart = data;
	authDigestLen = sizeof(authDigest);
	data = asn_parse_string(data, length, &type, authDigest,
				&authDigestLen);
	if (data == NULL){
	    ERROR("Digest");
	    return NULL;
	}
	digestEnd = data;

	data = asn_parse_unsigned_int(data, length, &type, &authDstTimeStamp,
			     sizeof(authDstTimeStamp));
	if (data == NULL){
	    ERROR("DstTimeStamp");
	    return NULL;
	}

	data = asn_parse_unsigned_int(data, length, &type, &authSrcTimeStamp,
				      sizeof(authSrcTimeStamp));
	if (data == NULL){
	    ERROR("SrcTimeStamp");
	    return NULL;
	}
    } else if (type == (ASN_CONTEXT | ASN_CONSTRUCTOR | 1)){
	pi->version = SNMP_SECURITY_1;
	ismd5 = 1;
	data = asn_parse_unsigned_int(data, length, &type, &authSrcTimeStamp,
				      sizeof(authSrcTimeStamp));
	if (data == NULL){
	    ERROR("TimeStamp");
	    return NULL;
	}
	data = asn_parse_int(data, length, &type, &authNonce,
			     sizeof(authNonce));
	if (data == NULL){
	    ERROR("Nonce");
	    return NULL;
	}
	digestStart = data;
	authDigestLen = sizeof(authDigest);
	data = asn_parse_string(data, length, &type, authDigest,
				&authDigestLen);
	if (data == NULL){
	    ERROR("Digest");
	    return NULL;
	}
	digestEnd = data;
    } else {
	ERROR("Bad format for authData");
	return NULL;
    }
    data = asn_parse_header(data, length, &type);
    if (data == NULL){
	ERROR("bad parse of snmpMgmtCom");
	return NULL;
    }
    data = asn_parse_objid(data, length, &type, dstParty2, &dstParty2Length);
    if (data == NULL){
	ERROR("bad parse of dstParty");
	return NULL;
    }
    data = asn_parse_objid(data, length, &type, srcParty, srcPartyLength);
    if (data == NULL){
	ERROR("bad parse of srcParty");
	return NULL;
    }
    if (*dstPartyLength != dstParty2Length
	|| bcmp((char *)dstParty, (char *)dstParty2, dstParty2Length)){
	ERROR("Mismatch of destination parties\n");
	return NULL;
    }
    
    srcp = party_getEntry(srcParty, *srcPartyLength);
    if (!srcp)
	return NULL;
    /* Only perform the following authentication checks if this is the
     * first time called for this packet.
     */
    if (srcp->partyAuthProtocol == SMPMD5AUTHPROT && pi->version != SMP_1)
	return NULL;
    if (srcp->partyAuthProtocol == MD5AUTHPROT
	&& pi->version != SNMP_SECURITY_1)
	return NULL;
    if ((pass & FIRST_PASS) && (srcp->partyAuthProtocol == MD5AUTHPROT ||
				srcp->partyAuthProtocol == SMPMD5AUTHPROT)){
	/* P. 19, 6.2:1 */
	if (!ismd5)
	    return NULL;
	
	/* P. 19, 6.2:3 */
	gettimeofday(&now, (struct timezone *)0);
	srcp->partyAuthClock = now.tv_sec - srcp->tv.tv_sec;
	dstp->partyAuthClock = now.tv_sec - dstp->tv.tv_sec;
	if (srcp->partyAuthClock
	    > authSrcTimeStamp + srcp->partyAuthLifetime){
	    ERROR("Late message");
	    return NULL;
	}
	
	if (pi->version == SNMP_SECURITY_1){
	    /* P. 19, 6.2:4 */
	    if (authSrcTimeStamp < srcp->partyLastTimeStamp){
		ERROR("Out of order message:1");
		return NULL;
	    }
	    
	    /* P. 19, 6.2:5 */
	    if (authSrcTimeStamp == srcp->partyLastTimeStamp
		&& authNonce <= srcp->partyAuthNonce){
		ERROR("Out of order message:2");
		return NULL;
	    }
	}	
	/* P. 19, 6.2:7 */
	biglen = 5000;
	if (digestEnd != asn_build_string(digestStart, &biglen,
					  (u_char)(ASN_UNIVERSAL
						   | ASN_PRIMITIVE
						   | ASN_OCTET_STR),
					  srcp->partySecretsAuthPrivate,
					  srcp->partySecretsAuthPrivateLen)){
	    ERROR("couldn't stuff digest");
	    return NULL;
	}
	md5Digest(authMsg, authMsgLen, digest);
	
	/* P. 20, 6.2:8 */
	if (authDigestLen != 16 || bcmp(authDigest, digest, 16)){
	    ERROR("unauthentic");
	    return NULL;
	}
	/* As per P. 20, 6.2.9, the message is authentic */
	
	biglen = 5000;
	if (digestEnd != asn_build_string(digestStart, &biglen,
					  (u_char)(ASN_UNIVERSAL
						   | ASN_PRIMITIVE
						   | ASN_OCTET_STR),
					  authDigest, 16)){
	    ERROR("couldn't stuff digest back");
	    return NULL;
	}
	
	/* Now that we know the message is authentic, update
	 * the lastTimeStamp and Nonce.
	 */
	/* P. 20 6.2:9 */
	if (srcp->partyAuthClock < authSrcTimeStamp){
	    srcp->partyAuthClock = authSrcTimeStamp;
	    gettimeofday(&srcp->tv, (struct timezone *)0);
	    srcp->tv.tv_sec -= srcp->partyAuthClock;
	}
	if (pi->version == SMP_1){
	    if (dstp->partyAuthClock < authDstTimeStamp){
		dstp->partyAuthClock = authDstTimeStamp;
		gettimeofday(&dstp->tv, (struct timezone *)0);
		dstp->tv.tv_sec -= dstp->partyAuthClock;
	    }
	} else {
	    srcp->partyLastTimeStamp = authSrcTimeStamp;
	    srcp->partyAuthNonce = authNonce;
	}
    } else if ((pass & FIRST_PASS) && dstp->partyPrivProtocol == DESPRIVPROT){
	/* noAuth and desPriv */
	ERROR("noAuth and desPriv");
	return NULL;
    }
    return data;
}


u_char *
snmp_auth_build(data, length, sid, slen, version, messagelen)
    u_char          *data;
    int             *length;
    u_char          *sid;
    int             *slen;
    long            *version;
    int             messagelen;
{
    data = asn_build_header(data, length, (u_char)(ASN_SEQUENCE | ASN_CONSTRUCTOR), messagelen + *slen + 5);
    if (data == NULL){
        ERROR("buildheader");
        return NULL;
    }
    data = asn_build_int(data, length,
            (u_char)(ASN_UNIVERSAL | ASN_PRIMITIVE | ASN_INTEGER),
            (long *)version, sizeof(*version));
    if (data == NULL){
        ERROR("buildint");
        return NULL;
    }
    data = asn_build_string(data, length,
            (u_char)(ASN_UNIVERSAL | ASN_PRIMITIVE | ASN_OCTET_STR),
            sid, *slen);
    if (data == NULL){
        ERROR("buildstring");
        return NULL;
    }
    return (u_char *)data;
}


u_char *
snmp_secauth_build(data, length, pi, messagelen, srcParty, srcPartyLen,
		   dstParty, dstPartyLen, packet_len, pass)
    u_char	    *data;
    int		    *length;
    struct packet_info *pi;
    int		    messagelen;
    oid		    *srcParty;
    int		    srcPartyLen;
    oid		    *dstParty;
    int		    dstPartyLen;
    int		    *packet_len;    /* OUT - length of complete packet */
    int		    pass;  /* FIRST_PASS, LAST_PASS, none, or both */
{
    int snmpMgmtComLength, authInformationLength, snmpAuthMsgLength;
    int snmpPrivMsgLength;
    int srcPartyEncodedLen, dstPartyEncodedLen;
    struct partyEntry *srcp, *dstp;
    struct timeval now;
    u_char *cp, *cp1, *block, *endOfPacket;
    int dummyLength, count;
    u_char key[8], iv[8];	/* initialization vector */
    u_char *digestStart = NULL, *digestEnd, *authMsgStart;
    int privDataLength, authDataLength;
    u_char authDigest[16];

    srcp = party_getEntry(srcParty, srcPartyLen);
    if (!srcp)
	return NULL;
    dstp = party_getEntry(dstParty, dstPartyLen);
    if (!dstp)
	return NULL;
    predict_objid_lengths(srcParty, srcPartyLen, &srcPartyEncodedLen);
    predict_objid_lengths(dstParty, dstPartyLen, &dstPartyEncodedLen);
    if (srcp->partyAuthProtocol == MD5AUTHPROT
	|| srcp->partyAuthProtocol == SMPMD5AUTHPROT){
	if (pass & FIRST_PASS){
	    /* get timestamp and nonce now because they are needed for the
	     * length predictions.
	     */
	    gettimeofday(&now, (struct timezone *)0);
	    srcp->partyAuthClock = now.tv_sec - srcp->tv.tv_sec;
	    if (pi->version == SNMP_SECURITY_1){
		if (srcp->partyAuthClock == srcp->partyLastTimeStamp){
		    srcp->partyAuthNonce++;
		} else {
		    srcp->partyAuthNonce = 0;
		    srcp->partyLastTimeStamp = srcp->partyAuthClock;
		}
	    }
	}
	/* What if we don't actually send the message?  Are we now
	 * out of sync due to the above line?  Answer: No, this
	 * is just like dropping a packet, except that it is dropped
	 * due to some error detected in the software protocol layers
	 * between here and the network.
	 */
	predict_md5Auth_lengths(pi, messagelen, dstp->partyPrivProtocol,
				srcPartyEncodedLen, dstPartyEncodedLen,
				srcp->partyAuthClock, dstp->partyAuthClock,
				srcp->partyAuthNonce,
				&snmpMgmtComLength, &authInformationLength,
				&snmpAuthMsgLength, &authDataLength,
				&privDataLength, &snmpPrivMsgLength,
				packet_len);
    } else {
	/* Don't send noAuth/desPriv. User interface should check for
	 * this so that it can give a reasonable error message
	 */
	if (dstp->partyPrivProtocol == DESPRIVPROT)
	    return NULL;
	predict_noAuth_lengths(messagelen, srcPartyEncodedLen,
			       dstPartyEncodedLen, &snmpMgmtComLength,
			       &snmpAuthMsgLength, &privDataLength,
			       &snmpPrivMsgLength, packet_len);
    }
    data = asn_build_header(data, length,
			    (u_char)(ASN_CONTEXT | ASN_CONSTRUCTOR | 1),
			    snmpPrivMsgLength);
    if (data == NULL){
	ERROR("build_header2");
	return NULL;
    }
    data = asn_build_objid(data, length, (u_char)(ASN_UNIVERSAL | ASN_PRIMITIVE | ASN_OBJECT_ID), dstParty, dstPartyLen);
    if (data == NULL){
	ERROR("build_objid");
	return NULL;
    }
    data = asn_build_header(data, length,
			    (u_char)(ASN_CONTEXT | 1), privDataLength);
    if (data == NULL){
	ERROR("build_header2");
	return NULL;
    }
    authMsgStart = data;
    data = asn_build_header(data, length,
			    (u_char)(ASN_CONTEXT | ASN_CONSTRUCTOR | 1),
			    snmpAuthMsgLength);
    if (data == NULL){
	ERROR("build_header2");
	return NULL;
    }
    if (srcp->partyAuthProtocol == MD5AUTHPROT
	|| srcp->partyAuthProtocol == SMPMD5AUTHPROT){
	if (pi->version == SMP_1){
	    data = asn_build_header(data, length,
				    (u_char)(ASN_CONTEXT |ASN_CONSTRUCTOR | 2),
				    authInformationLength);
	    if (data == NULL){
		ERROR("build_header2");
		return NULL;
	    }

  	    digestStart = data;
	    data = asn_build_string(data, length,
				    (u_char)(ASN_UNIVERSAL | ASN_PRIMITIVE
					     | ASN_OCTET_STR), 
				    srcp->partySecretsAuthPrivate,
				    srcp->partySecretsAuthPrivateLen);
	    if (data == NULL){
		ERROR("build_string");
		return NULL;
	    }
	    digestEnd = data;

	    data = asn_build_unsigned_int(data, length,
					  (u_char)(ASN_UNIVERSAL
						   | ASN_PRIMITIVE
						   | ASN_INTEGER),
					  &dstp->partyAuthClock,
					  sizeof(dstp->partyAuthClock));
	    if (data == NULL){
		ERROR("build_unsigned_int");
		return NULL;
	    }

	    data = asn_build_unsigned_int(data, length,
					  (u_char)(ASN_UNIVERSAL
						   | ASN_PRIMITIVE
						   | ASN_INTEGER),
					  &srcp->partyAuthClock,
					  sizeof(srcp->partyAuthClock));
	    if (data == NULL){
		ERROR("build_unsigned_int");
		return NULL;
	    }
	} else {
	    data = asn_build_header(data, length,
				    (u_char)(ASN_CONTEXT |ASN_CONSTRUCTOR | 1),
				    authInformationLength);
	    if (data == NULL){
		ERROR("build_header2");
		return NULL;
	    }
	    data = asn_build_unsigned_int(data, length,
					  (u_char)(ASN_UNIVERSAL
						   | ASN_PRIMITIVE
						   | ASN_INTEGER),
					  &srcp->partyAuthClock,
					  sizeof(srcp->partyAuthClock));
	    if (data == NULL){
		ERROR("build_unsigned_int");
		return NULL;
	    }
	    data = asn_build_int(data, length,
				 (u_char)(ASN_UNIVERSAL | ASN_PRIMITIVE
					  | ASN_INTEGER),
				 &srcp->partyAuthNonce,
				 sizeof(srcp->partyAuthNonce));
	    if (data == NULL){
		ERROR("build_int");
		return NULL;
	    }
	    digestStart = data;
	    data = asn_build_string(data, length,
				    (u_char)(ASN_UNIVERSAL | ASN_PRIMITIVE
					     | ASN_OCTET_STR), 
				    srcp->partySecretsAuthPrivate,
				    srcp->partySecretsAuthPrivateLen);
	    if (data == NULL){
		ERROR("build_string");
		return NULL;
	    }
	    digestEnd = data;
	}
    } else {
	data = asn_build_string(data, length, (u_char)(ASN_UNIVERSAL
						       | ASN_PRIMITIVE
						       | ASN_OCTET_STR),
				(u_char *)"", 0);
	if (data == NULL){
	    ERROR("build_string");
	    return NULL;
	}
    }
    data = asn_build_header(data, length,
			    (u_char)(ASN_CONTEXT | ASN_CONSTRUCTOR | 1),
			    snmpMgmtComLength);
    if (data == NULL){
	ERROR("build_header2");
	return NULL;
    }
    data = asn_build_objid(data, length,
			   (u_char)(ASN_UNIVERSAL
				    | ASN_PRIMITIVE | ASN_OBJECT_ID),
			   dstParty, dstPartyLen);
    if (data == NULL){
	ERROR("build_objid");
	return NULL;
    }
    data = asn_build_objid(data, length,
			   (u_char)(ASN_UNIVERSAL
				    | ASN_PRIMITIVE | ASN_OBJECT_ID),
			   srcParty, srcPartyLen);
    if (data == NULL){
	ERROR("build_objid");
	return NULL;
    }
    endOfPacket = data;
    /* if not last pass, skip md5 and des computation */
    if (!(pass & LAST_PASS))
	return (u_char *)endOfPacket;

    /* if it isn't MD5, we'll never do DES, so we're done */
    if (srcp->partyAuthProtocol != MD5AUTHPROT
	&& srcp->partyAuthProtocol != SMPMD5AUTHPROT)
	return (u_char *)endOfPacket;
/*    xdump(srcp->partySecretsAuthPrivate, 16, "authPrivate: "); */
    md5Digest(authMsgStart, authDataLength, authDigest);
    dummyLength = 5000;
    data = asn_build_string(digestStart, &dummyLength,
			    (u_char)(ASN_UNIVERSAL | ASN_PRIMITIVE
				     | ASN_OCTET_STR), 
			    authDigest, sizeof(authDigest));
    if (data != digestEnd){
	ERROR("stuffing digest");
	return NULL;
    }
    if (dstp->partyPrivProtocol != DESPRIVPROT)
	return (u_char *)endOfPacket;

    if (dstp->partySecretsPrivPrivateLen != 16)
	return NULL;
    return NULL;  /* without des, this will never work */
/*    return (u_char *)endOfPacket; */
}

md5Digest(start, length, digest)
    u_char *start;
    int length;
    u_char *digest;
{
    u_char *buf, *cp;
    MDstruct MD;
    int i, j;

#if 0
    int count, sum;

    sum = 0;
    for(count = 0; count < length; count++)
	sum += start[count];
    printf("sum %d (%d)\n", sum, length);
#endif

    cp = buf = (u_char *)malloc(length + 8);
    bcopy((char *)start, buf, length);
    MDbegin(&MD);
    while(length >= 64){
	MDupdate(&MD, cp, 64 * 8);
	cp += 64;
	length -= 64;
    }
    MDupdate(&MD, cp, length * 8);
/*    MDprint(&MD); */
   for (i=0;i<4;i++)
     for (j=0;j<32;j=j+8)
	 *digest++ = (MD.buffer[i]>>j) & 0xFF;
    free((char *)buf);
}

/*
 * Predicts the encoding length of an object identifier (without tag and length).
 */
predict_objid_lengths(objid, len, encoded_length)
    oid *objid;
    int len;
    int *encoded_length;
{
    int count, length = 0;
    oid subid;
    
    for(count = 0; count < len; count++){
	subid = *objid++;
	if (subid < 0x80)
	    length += 1;
	else if (subid < 0x4000)
	    length += 2;
	else if (subid < 0x200000)
	    length += 3;
	else if (subid < 0x10000000)
	    length += 4;
	else
	    length += 5;
    }
    *encoded_length = length - 1;   /* account for encoding of first two subids */
}

/*
 * Predicts the encoding lengths in octets for an unauthenticated message
 * with the corresponding parameters.
 */
predict_noAuth_lengths(messagelen, srcPartyLen, dstPartyLen,snmpMgmtComLength,
		snmpAuthMsgLength, privDataLength, snmpPrivMsgLength,
		       packet_len)
    int messagelen;	/* IN - length of PDU */
    int srcPartyLen, dstPartyLen; /* IN - number of subid's in these oid's */
    int	*snmpMgmtComLength;	/* OUT */
    int *snmpAuthMsgLength;	/* OUT */
    int *privDataLength;	/* OUT */
    int *snmpPrivMsgLength;	/* OUT */
    int *packet_len;		/* OUT */
{
    int length;
    int lol;	/* Length Of Length encoding */

    *snmpMgmtComLength = length = messagelen + srcPartyLen + dstPartyLen + 4;

    if (length < 0x80)
	lol = 1;
    else if (length <= 0xFF)
	lol = 2;
    else
	lol = 3;
    *snmpAuthMsgLength = length = length + lol + 3;

    if (length < 0x80)
	lol = 1;
    else if (length <= 0xFF)
	lol = 2;
    else
	lol = 3;
    *privDataLength = length = length + lol + 1;

    if (length < 0x80)
	lol = 1;
    else if (length <= 0xFF)
	lol = 2;
    else
	lol = 3;
    *snmpPrivMsgLength = length = length + lol + 1 + dstPartyLen + 2;

    if (length < 0x80)
	lol = 1;
    else if (length <= 0xFF)
	lol = 2;
    else
	lol = 3;
    *packet_len = length + lol + 1;
}

/*
 * Predicts the encoding lengths in octets for an MD5 authenticated message
 * with the corresponding parameters.
 */
predict_md5Auth_lengths(pi, messagelen, privProtocol,
			srcPartyLen, dstPartyLen,
			authSrcTimeStamp, authDstTimeStamp, authNonce,
			snmpMgmtComLength, authInformationLength,
			snmpAuthMsgLength, authDataLength, privDataLength,
			snmpPrivMsgLength, packet_len)
    struct packet_info *pi;	/* IN - Contains version info about packet */
    int messagelen;	/* IN - length of PDU */
    int privProtocol;	/* IN - NOPRIV or DESPRIVPROT */
    int srcPartyLen, dstPartyLen; /* IN - number of subid's in these oid's */
    u_long authSrcTimeStamp; /* IN - value of authSrcTimeStamp.  This must not
				change between call to predict_lengths and
				subsequent encoding. */
    u_long authDstTimeStamp; /* IN - value of authDstTimeStamp.  This must not
				change between call to predict_lengths and
				subsequent encoding. */
    u_long authNonce;  	/* IN - value of authNonce.  As above, this must not
			   change before encoding occurs */
    int	*snmpMgmtComLength;	/* OUT */
    int *authInformationLength; /* OUT */
    int *snmpAuthMsgLength;	/* OUT */
    int *authDataLength;	/* OUT */
    int *privDataLength;	/* OUT */
    int *snmpPrivMsgLength;	/* OUT */
    int *packet_len;		/* OUT */
{
    int length, pad;
    int lol;	/* Length Of Length encoding */
    int total_authMsgLength;	/* inclusive of SEQUENCE tag and length */
    int authSrcTimeStampLen, authDstTimeStampLen, authNonceLen;

    *snmpMgmtComLength = length = messagelen + srcPartyLen + dstPartyLen + 4;

    if (length < 0x80)
	lol = 1;
    else if (length <= 0xFF)
	lol = 2;
    else
	lol = 3;
    total_authMsgLength = length + lol + 1;

    if (authSrcTimeStamp < 0x80)
	authSrcTimeStampLen = 1;
    else if (authSrcTimeStamp < 0x8000)
	authSrcTimeStampLen = 2;
    else if (authSrcTimeStamp < 0x800000)
	authSrcTimeStampLen = 3;
    else if (authSrcTimeStamp < 0x80000000)
	authSrcTimeStampLen = 4;
    else
	authSrcTimeStampLen = 5;

    if (pi->version == SMP_1){
	if (authDstTimeStamp < 0x80)
	    authDstTimeStampLen = 1;
	else if (authDstTimeStamp < 0x8000)
	    authDstTimeStampLen = 2;
	else if (authDstTimeStamp < 0x800000)
	    authDstTimeStampLen = 3;
	else if (authDstTimeStamp < 0x80000000)
	    authDstTimeStampLen = 4;
	else
	    authDstTimeStampLen = 5;
	*authInformationLength = length = 18 + authSrcTimeStampLen +
	    authDstTimeStampLen + 4;
	/* authDigest is fixed at 16 bytes + 2 for tag and length */
    } else {
	if (authNonce < 0x80)
	    authNonceLen = 1;
	else if (authNonce < 0x8000)
	    authNonceLen = 2;
	else if (authNonce < 0x800000)
	    authNonceLen = 3;
	else if (authNonce < 0x80000000)
	    authNonceLen = 4;
	else
	    authNonceLen = 5;
	*authInformationLength = length = 18 + authSrcTimeStampLen +
	    authNonceLen + 4;
	/* authDigest is fixed at 16 bytes + 2 for tag and length */
    }

    if (length < 0x80)
	lol = 1;
    else if (length <= 0xFF)
	lol = 2;
    else
	lol = 3;
    *snmpAuthMsgLength = length = total_authMsgLength + length + lol + 1;

    if (length < 0x80)
	lol = 1;
    else if (length <= 0xFF)
	lol = 2;
    else
	lol = 3;
    *authDataLength = length += lol + 1;
    /* "length" is now length of encoding.  Now add pad for DES */
    
    if (privProtocol == DESPRIVPROT && ((length % 8) != 0))
	pad = 8 - (length % 8);
    else
	pad = 0;
    *privDataLength = length += pad;

    if (length < 0x80)
	lol = 1;
    else if (length <= 0xFF)
	lol = 2;
    else
	lol = 3;
    *snmpPrivMsgLength = length += lol + 1 + dstPartyLen + 2;

    if (length < 0x80)
	lol = 1;
    else if (length <= 0xFF)
	lol = 2;
    else
	lol = 3;
    *packet_len = length + lol + 1;
}

int
has_access(msg_type, srcParty, srcPartyLength, dstParty, dstPartyLength)
    u_char msg_type;
    oid *srcParty, *dstParty;
    int srcPartyLength, dstPartyLength;
{
    struct aclEntry *ap;

    ap = acl_getEntry(dstParty, dstPartyLength, srcParty, srcPartyLength);
    if (!ap)
	return 0;
    if (ap->aclPriveleges & (1 << (msg_type & 0x1F)))
	return 1;
    return 0;
}

