/*
 * alarmman - Configure alarm tables.
 *
 */
/***********************************************************************
	Copyright 1992 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
************************************************************************/
#include <sys/types.h>
#include <netinet/in.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>


#include "snmp.h"
#include "asn1.h"
#include "snmp_impl.h"
#include "snmp_api.h"
#include "snmp_client.h"
#include "party.h"
#include "alarmman.h"

extern int  errno;
int	snmp_dump_packet = 0;

oid oid_notifyTable[] = {2, 6, 6, 200, 4, 2, 8, 5, 1};
int oid_notifyTableLength = sizeof(oid_notifyTable) / sizeof(oid);

int nt_readable[] = {1, 1, 1, 1, 1};
int nt_writable[] = {1, 1, 0, 0, 1};

char *nt_names[] = {"Src Party",
			"Int Requested",
			"Ret Requested",
			"Lifetime",
			"Status",
			"test"};

char *nt_lnames[] = {"Source Party",
			"Interval Requested",
			"Retries Requested",
			"Lifetime",
			"Status",
			"test"};

int nt_types[] = {OBJID, 
		      INTEGER,
		      INTEGER,
		      INTEGER,
		      INTEGER};

oid oid_eventTable[] = {2, 6, 6, 200, 4, 2, 8, 2, 1};
int oid_eventTableLength = sizeof(oid_eventTable) / sizeof(oid);

int et_readable[] = {0, 1, 1, 1, 1, 1};
int et_writable[] = {0, 1, 1, 0, 0, 1};

char *et_names[] = {"Index", 
			"Event ID",
			"Event Description",
			"Event Events",
			"Last Time",
			"Status"};

char *et_lnames[] = {"Index", 
			"Event ID",
			"Event Description",
			"Event Events",
			"Last Time",
			"Status"};

int et_types[] = {INTEGER,
		      OBJID,
		      STRING,
		      INTEGER,
		      INTEGER,
		      INTEGER};

oid oid_alarmTable[] = {2, 6, 6, 200, 4, 2, 7, 2, 1};
int oid_alarmTableLength = sizeof(oid_alarmTable) / sizeof(oid);

int at_readable[] = {0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1};
int at_writable[] = {0, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1};

char *at_names[] = {"Index", 
			"Src Pty",
			"Variable",
			"Intrvl",
			"S_Type",
			"Value",
			"S_Alarm",
			"R_Thrsh",
			"F_Thrsh",
			"R_Event",
			"F_Event",
			"U_Event",
			"Status"};

char *at_lnames[] = {"Index", 
			 "Source Party",
			 "Variable",
			 "Interval",
			 "Sample Type",
			 "Value",
			 "Startup Alarm",
			 "Rising Threshold",
			 "Falling Threshold",
			 "Rising Event",
			 "Falling Event",
			 "Unavailable Event",
			 "Status"};

int at_types[] = {INTEGER,
		      OBJID,
		      OBJID,
		      INTEGER,
		      INTEGER,
		      INTEGER,
		      INTEGER,
		      INTEGER,
		      INTEGER,
		      INTEGER,
		      INTEGER,
		      INTEGER,
		      INTEGER};

		      

usage() {
    printf("Usage: alarmman gateway srcParty dstParty [-d]\n");
}

get_table_indexes(ss, name, name_length, index_sizes, table, table_length)
    struct snmp_session *ss;
    oid *name;
    int name_length;
    int **index_sizes;
    oid ***table;
    int *table_length;
{
    struct snmp_pdu *pdu, *response;
    struct variable_list *vars;
    oid our_name[100];
    int our_name_length;
    int status;
    oid last_index[100];
    int count = 0;
    int quit = 0;

    int i, j;

#define MAX_TABLE_INDEXES 100

    /* We will only retrieve 100 entries AT MOST. */

    *table = (oid **) malloc(sizeof(oid *) * MAX_TABLE_INDEXES);
    *index_sizes = (int *) malloc(sizeof(oid) * MAX_TABLE_INDEXES);
    if (!(*table)) {
	printf("Out of mem\n");
	exit(1);
    }
    bzero((char *) our_name, sizeof(our_name));
    bzero((char *) last_index, sizeof(last_index));

    bcopy((char *) name, (char *) our_name, name_length * sizeof(oid));
    our_name_length = name_length;


    /* Create newest request */
    while (!quit) {

	/* Copy in the most recent index, to get the next.. */

	bcopy((char *) last_index, 
	      (char *)&(our_name[name_length]),
	      sizeof(oid) * (our_name_length - name_length));

	pdu = snmp_pdu_create(GETNEXT_REQ_MSG);
	snmp_add_null_var(pdu, our_name, our_name_length);

	status = snmp_synch_response(ss, pdu, &response);
	if (status == STAT_SUCCESS){
	    if (response->errstat == SNMP_ERR_NOERROR){
		vars = response->variables;
		if (vars->name_length < name_length) {
		    quit = 1;
		} else {
		    if (bcmp((char *) vars->name, (char *) name, sizeof(oid) * name_length)) {
			quit = 1;
		    } else {
			our_name_length = vars->name_length;
			(*index_sizes)[count] = our_name_length;
			(*table)[count] = (oid *) malloc(sizeof(oid) * (our_name_length - name_length));
			/* Save this to return it... */
			bcopy((char *)&(vars->name[name_length]),
			      (char *)(*table)[count], 
			      sizeof(oid) * (our_name_length - name_length));
			/* Save this so we can get the next. */
			bcopy((char *)&(vars->name[name_length]),
			      (char *)last_index, 
			      sizeof(oid) * (our_name_length - name_length));
			count++;
			if(count == MAX_TABLE_INDEXES) {
			    quit = 1;
			}
		    }
		}
	    } else {
		quit = 1;
		if (response->errstat == SNMP_ERR_NOSUCHNAME){
		    printf("End of MIB.\n");
		} else {
		    printf("Error in packet.\nReason: %s\n", snmp_errstring(response->errstat));
		    if (response->errstat == SNMP_ERR_NOSUCHNAME){
			printf("The request for this object identifier failed: ");
			for(count = 1, vars = response->variables; vars && count != response->errindex;
			    vars = vars->next_variable, count++)
			    /*EMPTY*/;
			if (vars)
			    print_objid(vars->name, vars->name_length);
			printf("\n");
		    }
		}
	    }
	    
	} else if (status == STAT_TIMEOUT){
	    quit = 1;
	    printf("No Response from gateway\n");
	} else {    /* status == STAT_ERROR */
	    quit = 1;
	    printf("An error occurred, Quitting\n");
	}
	
	if (response)
	    snmp_free_pdu(response);
    }
    *table_length = count;
}
    
copyvar(svar, dvar)
    struct variable_list *svar;
    struct variable_list *dvar;
{
    bcopy((char *)svar, (char *)dvar, sizeof(struct variable_list));
    dvar->name = (oid *) malloc (sizeof(oid) * svar->name_length);
    bcopy((char *)svar->name, (char *)dvar->name, sizeof(oid) * svar->name_length);
    dvar->val.string = (u_char *) malloc(svar->val_len);
    bcopy((char *)svar->val.string, (char *)dvar->val.string, svar->val_len);
}

snmp_add_var(pdu, name, name_len, type, val_len, val)
    struct snmp_pdu *pdu;
    oid *name;
    int name_len;
    int type;
    int val_len;
    void *val;
{
    struct variable_list *vars;

    if (pdu->variables == NULL){
        pdu->variables = vars = (struct variable_list *)malloc(sizeof(struct variable_list));
    } else {
        for(vars = pdu->variables; vars->next_variable; vars = vars->next_variable)
            /*EXIT*/;
        vars->next_variable = (struct variable_list *)malloc(sizeof(struct variable_list));
        vars = vars->next_variable;
    }

    vars->next_variable = NULL;
    vars->name = (oid *)malloc(name_len * sizeof(oid));
    bcopy((char *)name, (char *)vars->name, name_len * sizeof(oid));
    vars->name_length = name_len;
    vars->type = type;
    vars->val.string = (u_char *)malloc(val_len);
    if (!vars->val.string) {
	printf("More memory, please\n");
	exit(0);
    }
    bcopy((char *) val, (char *) vars->val.string, val_len);
    vars->val_len = val_len;
}


int find_row(ss, table)
    struct snmp_session *ss;
    control_table_p table;
{

#define RAND_MULT 20
#define MAX_TRIES 20

    oid name[128];
    int name_length;
    int found = 0;
    struct snmp_pdu *pdu, *response;
    int rand_range = 1;
    int count = MAX_TRIES;
    int entry;

    int status;
    int valid = 1;

    bcopy((char *) table->create_row, (char *) name, sizeof(oid) * (1 + table->name_length + table->index_length));
    name_length = table->name_length + table->index_length + 1;
    /* Set up to look at status which is conveniently always the last column */

    /* Create request message */

    while (!found && count) {
	count--;
	if (rand_range < 1500) {
	    rand_range *= RAND_MULT;	
	}
	entry = random() % rand_range;
	if (table->index_pos == INDEX_END) {
	    name[name_length - 1] = entry;
	} else {
	    name[table->name_length + 1] = entry;
	}
	pdu = snmp_pdu_create(SET_REQ_MSG);
	if (!pdu){
	    printf("Error: could not create PDU!\n");
	    exit(0);
	}
	snmp_add_var(pdu, name, name_length, INTEGER, sizeof(int), (void *) &valid);

	status = snmp_synch_response(ss, pdu, &response);

	if (status == STAT_SUCCESS) {
	    if (response->errstat == SNMP_ERR_NOERROR) {
		found = 1;
	    } else if (response->errstat == SNMP_ERR_NOSUCHNAME) {
		snmp_free_pdu(response);
		/* Error: maybe a naming conflict-- try again. */
	    } else {
		snmp_free_pdu(response);
		/* Error: maybe a naming conflict-- try again. */
	    }
	} else {
	    if (response) {
		snmp_free_pdu(response);
	    }
	    printf("Error getting response.\n");
	    if (status == STAT_TIMEOUT) {
		printf("No response from probe.\n");
	    }
	}
    }
    if (!count) {
	printf("Error: exceeded count limit finding suitable entry.\n");
	exit(0);
    }
    if (response) {
	if (response->variables) {
	    if (response->variables->type == INTEGER && *response->variables->val.integer == 1) {
		return(entry);
	    }
	}
    }
    printf("Error..... No response\n");
    return(0);
}

setup_index(table)
    control_table_p table;
{
    u_char buf[256];
    int len;
    int done = 0, i;
    struct partyEntry *pp;
    oid test[555];
    int test_length;

    if (table->type == EVENT_TABLE_TYPE) {
	table->index_length = 1;
	bcopy((char *) table->name, (char *) table->create_row, sizeof(oid) * table->name_length);
	table->create_row[table->name_length] = table->table_columns;
    } else {
	bcopy((char *) table->name, (char *) table->create_row, sizeof(oid) * table->name_length);
	table->create_row[table->name_length] = table->table_columns;

	if (table->type == ALARM_TABLE_TYPE) {
	    /* Set up the length of this sucker. */
	    while (!done) {
		printf("Please enter a destination party for row creation.  (party.conf aliases acceptable)\n");
		gets(buf);


		if (!read_party_database("/etc/party.conf")){
		    fprintf(stderr,
			    "Couldn't read party database from /etc/party.conf\n");
		    exit(0);
		}
		if (!read_acl_database("/etc/acl.conf")){
		    fprintf(stderr,
			    "Couldn't read access control database from /etc/acl.conf\n");
		    exit(0);
		}
		party_scanInit();
		for(pp = party_scanNext(); pp; pp = party_scanNext()){
		    if (!strcasecmp(pp->partyName, buf)){
			table->index_length = pp->partyIdentityLen + 2;
			bcopy((char *)pp->partyIdentity, (char *)test, table->index_length * sizeof(oid));
			done++;
			break;
		    }
		}
		if (!pp){
		    test_length = MAX_NAME_LEN;
		    if (!read_objid(buf, test, &test_length)){
			printf("Invalid party: %s\n", buf);
		    } else {
			table->index_length = test_length + 2;
			done++;
		    }
		}
	    }
	    table->create_row[table->name_length + 1] = table->index_length - 2;
	    bcopy(test, &(table->create_row[table->name_length + 2]), (table->index_length - 2) * sizeof(oid));
	} else if (table->type == NOTIFY_TABLE_TYPE) {
	    /* Set up the length of this sucker. */
	    while (!done) {
		printf("Please enter a destination party for row creation.  (party.conf aliases acceptable)\n");
		gets(buf);


		if (!read_party_database("/etc/party.conf")){
		    fprintf(stderr,
			    "Couldn't read party database from /etc/party.conf\n");
		    exit(0);
		}
		if (!read_acl_database("/etc/acl.conf")){
		    fprintf(stderr,
			    "Couldn't read access control database from /etc/acl.conf\n");
		    exit(0);
		}
		party_scanInit();
		for(pp = party_scanNext(); pp; pp = party_scanNext()){
		    if (!strcasecmp(pp->partyName, buf)){
			table->index_length = pp->partyIdentityLen + 2;
			bcopy((char *)pp->partyIdentity, (char *)test, table->index_length * sizeof(oid));
			done++;
			break;
		    }
		}
		if (!pp){
		    test_length = MAX_NAME_LEN;
		    if (!read_objid(buf, test, &test_length)){
			printf("Invalid party: %s\n", buf);
		    } else {
			table->index_length = test_length + 2;
			done++;
		    }
		}
	    }
	    table->create_row[table->name_length + 2] = table->index_length - 2;
	    bcopy(test, &(table->create_row[table->name_length + 3]), (table->index_length - 2) * sizeof(oid));
	} else {
	    while (!done) {
		printf("Please enter object id to base index for row creation.\n");
		gets(buf);
		len = 128;
		read_objid(buf, &(table->create_row[table->name_length + 1]), &len);
		for (i = table->name_length + 1; i < table->name_length + 1 + table->index_length; i++) {
		    printf(".%i", table->create_row[i]);
		} 
		printf("\n");
		printf("Is this correct?\n");
		gets(buf);
		if (buf[0] == 'y' || buf[0] == 'Y') {
		    done++;
		}
	    }
	}
    }
}
		
	

int alarm_int_defaults[] = {0, 0, 0, 10, 1, 0, 1, 100, 0, 0, 0, 0, 0};
char *alarm_str_defaults[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
int alarm_str_defaults_len[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
oid *alarm_oid_defaults[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
int alarm_oid_defaults_len[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

oid dummyoid[] = {1, 3};
int dummyoid_length = 2;


struct variable_list **setup_alarm_defaults(table, index)
    control_table_p table;
    int index;
{
    struct variable_list **mdef_list;
    int i;
    int len;

    mdef_list = (struct variable_list **) malloc ((table->table_columns) * sizeof(struct variable_list *));
    if (!mdef_list) {
	printf("Out of memory\n");
	exit(0);
    }

    for (i = 0; i < table->table_columns; i++){
	mdef_list[i] = NULL;
	if (table->index_writable[i]) {
	    mdef_list[i] = (struct variable_list *) malloc (sizeof(struct variable_list));
	    if (!mdef_list[i]) {
		printf("Error: out of memory\n");
		exit(0);
	    }

	    mdef_list[i]->next_variable = NULL;

	    mdef_list[i]->name = (oid *) malloc (100 * sizeof(oid));
	    
	    len = table->name_length + table->index_length + 1;

	    /* Set up name....  WITHOUT entry associated... */
	    bcopy((char *) table->create_row, (char *) mdef_list[i]->name, sizeof(oid) * len);
	    mdef_list[i]->name[len - 1] = index;
	    mdef_list[i]->name_length = len;
	    mdef_list[i]->name[table->name_length] = i + 1;

	    mdef_list[i]->type = table->index_types[i];

	    if (table->index_types[i] == INTEGER) {
		mdef_list[i]->val.integer = &(alarm_int_defaults[i]);
		mdef_list[i]->val_len = sizeof(int);
	    } else if (table->index_types[i] == STRING) {
		mdef_list[i]->val.string = (u_char *)alarm_str_defaults[i];
		mdef_list[i]->val_len = alarm_str_defaults_len[i];
	    } else {
		mdef_list[i]->val.objid = dummyoid;
		mdef_list[i]->val_len = dummyoid_length * sizeof(oid);
	    }
	}
    }
    return(mdef_list);
}

int event_int_defaults[] = {0, 0, 0, 0, 0, 0};
char *event_str_defaults[] = {0, 0, "An Event", 0, 0, 0};
int event_str_defaults_len[] = {0, 0, 8, 0, 0, 0};
oid *event_oid_defaults[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
int event_oid_defaults_len[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

struct variable_list **setup_event_defaults(table, index)
    control_table_p table;
    int index;
{
    struct variable_list **mdef_list;
    int i;
    int len;

    mdef_list = (struct variable_list **) malloc ((table->table_columns) * sizeof(struct variable_list *));
    if (!mdef_list) {
	printf("Out of memory\n");
	exit(0);
    }

    for (i = 0; i < table->table_columns; i++){
	mdef_list[i] = NULL;
	if (table->index_writable[i]) {
	    mdef_list[i] = (struct variable_list *) malloc (sizeof(struct variable_list));
	    if (!mdef_list[i]) {
		printf("Error: out of memory\n");
		exit(0);
	    }

	    mdef_list[i]->next_variable = NULL;

	    mdef_list[i]->name = (oid *) malloc (100 * sizeof(oid));
	    
	    len = table->name_length + table->index_length + 1;

	    /* Set up name....  WITHOUT entry associated... */
	    bcopy((char *) table->create_row, (char *) mdef_list[i]->name, sizeof(oid) * len);
	    mdef_list[i]->name[len - 1] = index;
	    mdef_list[i]->name_length = len;
	    mdef_list[i]->name[table->name_length] = i + 1;

	    mdef_list[i]->type = table->index_types[i];

	    if (table->index_types[i] == INTEGER) {
		mdef_list[i]->val.integer = &(event_int_defaults[i]);
		mdef_list[i]->val_len = sizeof(int);
	    } else if (table->index_types[i] == STRING) {
		mdef_list[i]->val.string = (u_char *)event_str_defaults[i];
		mdef_list[i]->val_len = event_str_defaults_len[i];
	    } else {
		mdef_list[i]->val.objid = dummyoid;
		mdef_list[i]->val_len = dummyoid_length * sizeof(oid);
	    }
	}
    }
    return(mdef_list);
}

int notify_int_defaults[] = {0, 10, 5, 600, 0};
char *notify_str_defaults[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
int notify_str_defaults_len[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
oid *notify_oid_defaults[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
int notify_oid_defaults_len[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

struct variable_list **setup_notify_defaults(table, index)
    control_table_p table;
    int index;
{
    struct variable_list **mdef_list;
    int i;
    int len;

    mdef_list = (struct variable_list **) malloc ((table->table_columns) * sizeof(struct variable_list *));
    if (!mdef_list) {
	printf("Out of memory\n");
	exit(0);
    }

    for (i = 0; i < table->table_columns; i++){
	mdef_list[i] = NULL;
	if (table->index_writable[i]) {
	    mdef_list[i] = (struct variable_list *) malloc (sizeof(struct variable_list));
	    if (!mdef_list[i]) {
		printf("Error: out of memory\n");
		exit(0);
	    }

	    mdef_list[i]->next_variable = NULL;

	    mdef_list[i]->name = (oid *) malloc (100 * sizeof(oid));
	    
	    len = table->name_length + table->index_length + 1;

	    /* Set up name....  WITHOUT entry associated... */
	    bcopy((char *) table->create_row, (char *) mdef_list[i]->name, sizeof(oid) * len);

	    if (table->index_pos == INDEX_END) {

		mdef_list[i]->name[len - 1] = index;
		mdef_list[i]->name_length = len;
		mdef_list[i]->name[table->name_length] = i + 1;
	    } else {
		mdef_list[i]->name[table->name_length + 1] = index;
		mdef_list[i]->name_length = len;
		mdef_list[i]->name[table->name_length] = i + 1;
	    }
	    mdef_list[i]->type = table->index_types[i];

	    if (table->index_types[i] == INTEGER) {
		mdef_list[i]->val.integer = &(notify_int_defaults[i]);
		mdef_list[i]->val_len = sizeof(int);
	    } else if (table->index_types[i] == STRING) {
		mdef_list[i]->val.string = (u_char *)notify_str_defaults[i];
		mdef_list[i]->val_len = notify_str_defaults_len[i];
	    } else {
		mdef_list[i]->val.objid = dummyoid;
		mdef_list[i]->val_len = dummyoid_length * sizeof(oid);
	    }
	}
    }
    return(mdef_list);
}

void get_defaults(ss, table, index, def_list)
    struct snmp_session *ss;
    control_table_p table;
    int index;
    struct variable_list **def_list;
{
    oid name[128];
    int name_length;
    struct snmp_pdu *pdu, *response;
    struct variable_list *vars;
    int i, status;

    name_length = table->name_length + 1 + table->index_length;
    bcopy((char *)table->create_row, (char *) name, sizeof(oid) * name_length);

    if (table->index_pos == INDEX_END) {
	name[name_length - 1] = index;
    } else {
	name[table->name_length + 1] = index;
    }

    for (i = 0; i < table->table_columns; i++) {
	def_list[i] = NULL;
	if (table->index_readable[i]){

	    name[table->name_length] = i + 1;

	    pdu = snmp_pdu_create(GET_REQ_MSG);
	    if (!pdu){
		printf("Error: could not create pdu!! \n");
		exit(0);
	    }
	    snmp_add_null_var(pdu, name, name_length);

	    status = snmp_synch_response(ss, pdu, &response);
	    
	    if (status == STAT_SUCCESS) {
		if (response->errstat == SNMP_ERR_NOERROR) {
		    vars = response->variables;
		    if (vars) {
			if (vars->type != SMP_ENDOFMIBVIEW
			    && vars->type != SMP_NOSUCHOBJECT  /* for robustness */
			    && vars->type != SMP_NOSUCHINSTANCE){
			    /* Success.... we have a default value... */
			    def_list[i] = (struct variable_list *) malloc(sizeof(struct variable_list));
			    copyvar(response->variables, def_list[i]);
			}
		    } else {
			printf("Error: no response variables for default value request\n");
		    }
		} else {
		    if (response->errstat == SNMP_ERR_NOSUCHNAME){
			/* This is okay... there may be no default value.... */
		    } else {
			/* This is probably a real error... */
			printf("Error in packet.\nReason: <%s>", snmp_errstring(response->errstat));
		    }
		}
	    } else {
		if (status == STAT_TIMEOUT) {
		    printf("Timed out\n");
		}
	    }
	    if (response) {
		snmp_free_pdu(response);
	    }
	}
    }
}


struct variable_list *add_variable(v, s)
    struct variable_list *v;
    struct variable_list *s;
{
    if (!s) {
	s = v;
        return(s);
    }
    v->next_variable = s;
    s = v;
    return(s);
}

int
ascii_to_binary(cp, bufp)
    u_char  *cp;
    u_char *bufp;
{
    int subidentifier;
    u_char *bp = bufp;

    for(; *cp != '\0'; cp++){
        if (isspace(*cp))
            continue;
        if (!isdigit(*cp)){
            fprintf(stderr, "Input error\n");
            return -1;
        }
        subidentifier = atoi(cp);
        if (subidentifier > 255){
            fprintf(stderr, "subidentifier %d is too large ( > 255)\n", subidentifier);
            return -1;
        }
        *bp++ = (u_char)subidentifier;
        while(isdigit(*cp))
            cp++;
        cp--;
    }
    return bp - bufp;
}


int
hex_to_binary(cp, bufp)
    u_char  *cp;
    u_char *bufp;
{
    int subidentifier;
    u_char *bp = bufp;

    for(; *cp != '\0'; cp++){
        if (isspace(*cp))
            continue;
        if (!isxdigit(*cp)){
            fprintf(stderr, "Input error\n");
            return -1;
        }
        sscanf(cp, "%x", &subidentifier);
        if (subidentifier > 255){
            fprintf(stderr, "subidentifier %d is too large ( > 255)\n", subidentifier);
            return -1;
        }
        *bp++ = (u_char)subidentifier;
        while(isxdigit(*cp))
            cp++;
        cp--;
    }
    return bp - bufp;
}

read_value(vp, buf)
    struct variable_list *vp;
    char *buf;
{	
    u_char value[256], ch;
    struct partyEntry *pp;
    
    if (vp->type == ASN_OCTET_STR) {
	ch = 's';
    } else {
	ch = 's';
    }
    switch(vp->type){
	case INTEGER:
	    vp->val.integer = (long *)malloc(sizeof(long));
	    *(vp->val.integer) = atoi(buf);
	    vp->val_len = sizeof(long);
	    break;
	case STRING:
	    if (ch == 'd'){
		vp->val_len = ascii_to_binary(buf, value);
	    } else if (ch == 's'){
		strcpy(value, buf);
		vp->val_len = strlen(buf);
	    } else if (ch == 'x'){
		vp->val_len = hex_to_binary(buf, value);
	    }
	    vp->val.string = (u_char *)malloc(vp->val_len);
	    bcopy((char *)value, (char *)vp->val.string, vp->val_len);
	    break;
	case NULLOBJ:
	    vp->val_len = 0;
	    vp->val.string = NULL;
	    break;
	case OBJID:

	    if (!read_party_database("/etc/party.conf")){
		fprintf(stderr,
			"Couldn't read party database from /etc/party.conf\n");
		exit(0);
	    }
	    if (!read_acl_database("/etc/acl.conf")){
		fprintf(stderr,
			"Couldn't read access control database from /etc/acl.conf\n");
		exit(0);
	    }
	    party_scanInit();
	    for(pp = party_scanNext(); pp; pp = party_scanNext()){
		if (!strcasecmp(pp->partyName, buf)){
		    vp->val_len = pp->partyIdentityLen * sizeof(oid);
		    vp->val.objid = (oid *) malloc (vp->val_len);
		    bcopy(pp->partyIdentity, vp->val.objid, vp->val_len);
		    break;
		}
	    }
	    if (!pp){
		vp->val_len = MAX_NAME_LEN;;
		read_objid(buf, value, &vp->val_len);
		vp->val_len *= sizeof(oid);
		vp->val.objid = (oid *)malloc(vp->val_len);
		bcopy((char *)value, (char *)vp->val.objid, vp->val_len);
	    }
	    break;
	case TIMETICKS:
	    vp->val.integer = (long *)malloc(sizeof(long));
	    *(vp->val.integer) = atoi(buf);
	    vp->val_len = sizeof(long);
	    break;
	case IPADDRESS:
	    vp->val.integer = (long *)malloc(sizeof(long));
	    *(vp->val.integer) = inet_addr(buf);
	    vp->val_len = sizeof(long);
	    break;
	default:
	    fprintf(stderr, "Internal error\n");
	    break;
    }
}

struct variable_list *choose_variables(table, def_list, mdef_list, var_list)
    control_table_p table;
    struct variable_list **def_list;
    struct variable_list **mdef_list;
    struct variable_list *var_list;
{
    int i;

    char ch[255];
    char buf[200];
    struct variable_list *v;

    for (i = 0; i < table->table_columns - 1; i++) {
	if (table->index_writable[i]) {
	    printf("\n Choose value for <%s>\n", table->index_lnames[i]);
	    if (!def_list[i]) {
		printf("Suggested value: ");
		print_value(mdef_list[i]->name, mdef_list[i]->name_length, mdef_list[i]);
		printf("Enter value. [Enter for default] :");
	        gets(ch);
		if (ch[0] == 0){
		    var_list = add_variable(mdef_list[i], var_list);
		} else {
		    v = (struct variable_list *) malloc (sizeof(struct variable_list));
		    if (!v) {
			printf("Error: out of mem.\n");
			exit(0);
		    }
		    bcopy ((char *)mdef_list[i], (char *)v, sizeof(struct variable_list));
		    v->val_len = 0;
		    v->val.integer = NULL;
		    read_value(v, ch);
		    var_list = add_variable(v, var_list);
		}
	    } else {
		printf("Default value: ");
		print_value(def_list[i]->name, def_list[i]->name_length, def_list[i]);
		printf("Enter value. [Enter for default] :");
		gets(ch);
		if (ch[0] == 0) {
		    var_list = add_variable(def_list[i], var_list);
		} else {
		    v = (struct variable_list *) malloc (sizeof(struct variable_list));
		    if (!v) {
			printf("Error: out of mem.\n");
			exit(0);
		    }
		    bcopy ((char *)mdef_list[i], (char *)v, sizeof(struct variable_list));
		    v->val_len = 0;
		    v->val.integer = NULL;
		    read_value(v, ch);
		    var_list = add_variable(v, var_list);
		}
	    }
	}
    }
    return(var_list);
}

send_var_list(ss, table, var_list, entry)
    struct snmp_session *ss;
    control_table_p table;
    struct variable_list *var_list;
    int entry;
{
    struct snmp_pdu *pdu, *response;
    struct variable_list *vars;
    int status;
    int *valid;

    oid name[100];
    int name_length;

    valid = (int *) malloc (sizeof(int));
    *valid = 4;

    
    name_length = table->name_length + 1 + table->index_length;
    bcopy((char *) table->create_row, (char *) name, sizeof(oid) * name_length);

    name[table->name_length] = table->table_columns;
    
    if (table->index_pos == INDEX_END) {
	name[name_length - 1] = entry;
    } else {
	name[table->name_length + 1] = entry;
    }

    pdu = snmp_pdu_create(SET_REQ_MSG);
    pdu->variables = var_list;

    /* Okay, let's be cheesy and add the validation variable to the end here.... */

    snmp_add_var(pdu, name, name_length, INTEGER, sizeof(int), (void *) valid);

    status = snmp_synch_response(ss, pdu, &response);

    if (status == STAT_SUCCESS) {
	if (response->errstat == SNMP_ERR_NOERROR) {
	    /* It worked.... do nothing. */
	} else {
	    printf("Error in packet.\nReason: <%s>", snmp_errstring(response->errstat));
	    exit(0);
	}
    } else {
	printf("Error getting response.\n");
	if (status == STAT_TIMEOUT) {
	    printf("Timeout\n");
	}
	exit(0);
    }
    if (response) {
	snmp_free_pdu(response);
    }
}


int create_row(ss, table)
    struct snmp_session *ss;
    control_table_p table;
{
    struct variable_list **def_list, **mdef_list;
    struct variable_list *var_list = NULL;
    struct variable_list *tmp;

    int entry;
    int i;

    setup_index(table);

    entry = find_row(ss, table);
    
    if (!entry) {
	printf("Error: could not find suitable entry in probe\n");
	exit(0);
    }

    def_list = (struct variable_list **) malloc(table->table_columns * sizeof(struct variable_list *));
    if (!def_list) {
	printf("Out of memory\n");
	exit(0);
    }

    get_defaults(ss, table, entry, def_list);

    switch (table->type) {
	case ALARM_TABLE_TYPE:
	
	mdef_list = setup_alarm_defaults(table, entry);
	break;

	case EVENT_TABLE_TYPE:

	mdef_list = setup_event_defaults(table, entry);
	break;

	case NOTIFY_TABLE_TYPE:

	mdef_list = setup_notify_defaults(table, entry);
	break;
    }

    var_list = choose_variables(table, def_list, mdef_list, var_list);

    send_var_list(ss, table, var_list, entry);
}


read_table(ss, table)
    struct snmp_session *ss;
    control_table_p table;
{
    struct snmp_pdu *pdu, *response;
    struct variable_list *vars;
    int status, count;

    oid **valid_indexes;
    int *valid_lengths;
    int table_length;

    oid name[128];
    int name_length;

    int i, j;

    bcopy((char *)table->name, (char *)name, sizeof(oid) * table->name_length);
    name_length = table->name_length;

    /* Add a column index to the oid.  The last column 
     * index should be status, which is eminently stable.
     */
    name[name_length] = table->table_columns;
    name_length++;

    get_table_indexes(ss, name, name_length, &valid_lengths, &valid_indexes, &table_length);
    table->table_rows = table_length;

    /* Allocate room for all the variables we might have coming back... */
    table->index_values = (struct variable_list **) malloc(sizeof(struct variable_list *) * table->table_rows);
    if (!table->index_values){
	printf("Out of memory\n");
	exit(0);
    }

    for (i = 0; i < table_length; i++) {
	name_length = valid_lengths[i];
	table->index_values[i] = (struct variable_list *) malloc(sizeof(struct variable_list) * table->table_columns);
	if (!table->index_values[i]){
	    printf("Error: out of memory\n");
	    exit(0);
	}

	/* Set row index */
	bcopy((char *) valid_indexes[i], 
	      (char *) (&(name[table->name_length + 1])), 
	      sizeof(oid) * (name_length - table->name_length - 1));
	
	pdu = snmp_pdu_create(GET_REQ_MSG);

	for (j = 1; j <= table->table_columns; j++){
	    if (table->index_readable[j - 1]) {
		/* Set up for the j'th column.... */
		name[table->name_length] = j;
		snmp_add_null_var(pdu, name, name_length);
	    }
	}

	status = snmp_synch_response(ss, pdu, &response);
	if (status == STAT_SUCCESS){
	    if (response->errstat == SNMP_ERR_NOERROR){
		for (vars = response->variables; vars; vars = vars->next_variable){


		    if (vars->type != SMP_ENDOFMIBVIEW
			&& vars->type != SMP_NOSUCHOBJECT  /* for robustness */
			&& vars->type != SMP_NOSUCHINSTANCE){

			/* Save the returned variable for further reference... */
			copyvar(vars, &(table->index_values[i][vars->name[table->name_length] - 1]));
			
		    } else {
			bzero(&(table->index_values[i][vars->name[table->name_length] - 1]), sizeof(struct variable_list));
		    }
		}
	    } else {
		if (response->errstat == SNMP_ERR_NOSUCHNAME){
		    printf("End of MIB.\n");
		} else {
		    printf("Error in packet.\nReason: %s\n", snmp_errstring(response->errstat));
		    if (response->errstat == SNMP_ERR_NOSUCHNAME){
			printf("The request for this object identifier failed: ");
			for(count = 1, vars = response->variables; vars && count != response->errindex;
			    vars = vars->next_variable, count++)
			    /*EMPTY*/;
			if (vars)
			    print_objid(vars->name, vars->name_length);
			printf("\n");
		    }
		}
	    }
	    
	} else if (status == STAT_TIMEOUT){
	    printf("No Response from gateway\n");
	} else {    /* status == STAT_ERROR */
	    printf("An error occurred, Quitting\n");
	}
	
	if (response)
	    snmp_free_pdu(response);
    }


}

draw_line(table)
    control_table_p table;
{
    int i, j, k;

    printf("--------");
    for (j = 0; j < table->table_columns; j++) {
	if (table->index_readable[j] && table->index_types[j] != OBJID) {
	    for (i = 0; table->index_names[j][i]; i++){
		printf("-");
	    }
	    if (j < table->table_columns - 1) {
		k = 8 - i % 8;
		for (i = 0; i < k; i++){
		    printf("-");
		}
	    }
	}
    }
    printf("\n");
}

display_table(table)
    control_table_p table;
{
    /* Funnn!!!!! */
    struct variable_list *vars;
    char str[200];
    char str2[200];
    int i, j, k, start, len;

    printf("\n\n Format:\n\n");

    draw_line(table);

    printf("Index\t");
    for (j = 0; j < table->table_columns; j++) {
	if (table->index_readable[j] && table->index_types[j] != OBJID) {
	    printf("%s\t", table->index_names[j]);
	}
    }
    printf("\n");
    if (table->type == ALARM_TABLE_TYPE || table->type == NOTIFY_TABLE_TYPE) {
	printf("Destination party\n");
    }
    for (j = 0; j < table->table_columns; j++) {
	if (table->index_readable[j] && table->index_types[j] == OBJID) {
	    printf("%s\n", table->index_names[j]);
	}

    }

    draw_line(table);


    for (i = 0; i < table->table_rows; i++) {

	/* Display the index. */
	if (table->index_pos == INDEX_END) {
	    printf("%i\t", 
		   table->index_values[i][table->table_columns - 1].name[table->index_values[i][table->table_columns - 1].name_length - 1]);
	} else {
	    printf("%i\t", 
		   table->index_values[i][table->table_columns - 1].name[table->name_length + 1]);
	}
	for (j = 0; j < table->table_columns; j++) {
	    vars = &(table->index_values[i][j]);
	    if (table->index_readable[j] && table->index_types[j] != OBJID) {
		if (vars->name != NULL) {
		    switch (table->index_types[j]) {
			case STRING:
			
			if (vars->type != STRING) {
			    printf("Received wrong type! expected string!\n");
			    print_variable(vars->name, vars->name_length, vars);
			} else {
			    sprintf(str, "%%%is\t", strlen(table->index_names[j]));
			    bcopy((char *)vars->val.string, (char *) str2, vars->val_len);
			    str2[vars->val_len] = 0;
			    printf(str, str2);
			}
			break;
			
			case INTEGER:
			
			if ((vars->type != INTEGER) &&
			    (vars->type != TIMETICKS) &&
			    (vars->type != COUNTER) &&
			    (vars->type != GAUGE)) {
			    printf("Received wrong type! expected integer!\n");
			    print_variable(vars->name, vars->name_length, vars);
			} else {
			    sprintf(str, "%%%id\t", strlen(table->index_names[j]));
			    printf(str, *(vars->val.integer));
			}
			break;
			
			default:
			
			printf("Error\n");
			printf("Received wrong type! expected integer or string!\n");
			print_variable(vars->name, vars->name_length, vars);
		    }
		} else {
		    sprintf(str, "%%%is\t", strlen(table->index_names[j]));
		    printf(str, "<N/A>");
		}
	    }
	}
	printf("\n");
	if (table->type == ALARM_TABLE_TYPE || table->type == NOTIFY_TABLE_TYPE) {
	    vars = &table->index_values[i][table->table_columns - 1];
	    if (table->index_pos == INDEX_END) {
		len = vars->name[table->name_length + 1];
		start = table->name_length + 2;
	    } else {
		len = vars->name[table->name_length + 2];
		start = table->name_length + 3;
	    }
	    printf("OID: ");
	    print_objid((oid *)&(vars->name[start]), len);
	}
	for (j = 0; j < table->table_columns; j++) {
	    vars = &(table->index_values[i][j]);
	    if (table->index_readable[j] && table->index_types[j] == OBJID) {
		if (vars->name != NULL) {
		    if (vars->type != OBJID) {
			printf("Received wrong type! expected objid!\n");
			print_variable(vars->name, vars->name_length, vars);
		    } else {
			print_value(vars->name, vars->name_length, vars);
		    }
		} else {
		    printf("Not applicable\n");
		}
	    }
	}
	draw_line(table);
    }
    printf("\n\n");
}
			
delete_row(ss, table)
    struct snmp_session *ss;
    control_table_p table;
{
    struct snmp_pdu *pdu, *response;
    struct variable_list *vars;
    int status, valid = 2;

    oid *name;
    int name_length;

    int i, j, killme;
    char buf[255];
    
    
    printf("\n\n");

    
    printf("Please enter the number of the row you wish to delete.\n");
    printf("Choose from one of:\n");
    
    if (table->index_pos == INDEX_END) {
	for (i = 0; i < table->table_rows; i++) {
	    printf("%i\t", table->index_values[i][table->table_columns - 1].name[table->index_values[i][table->table_columns - 1].name_length - 1]);
	}
    } else {
	for (i = 0; i < table->table_rows; i++) {
	    printf("%i\t", table->index_values[i][table->table_columns - 1].name[table->name_length + 1]);
	}
    }
    printf("\n\n");
    printf("Your choice : ");

    gets(buf);
    killme = atoi(buf);

    /* find matching index. */
    if (table->index_pos == INDEX_END) {
	for (i = 0; 
	     i < table->table_rows && 
	     killme != 
	     table->index_values[i][table->table_columns - 1].name[table->index_values[i][table->table_columns - 1].name_length - 1];
	     i++);
    } else {
	for (i = 0; 
	     i < table->table_rows && 
	     killme != 
	     table->index_values[i][table->table_columns - 1].name[table->name_length + 1]; 
	     i++);
    }
    if (i == table->table_rows) {
	printf("Aborted!\n");
    } else {

	name = table->index_values[i][table->table_columns - 1].name;
	name_length = table->index_values[i][table->table_columns - 1].name_length;

	pdu = snmp_pdu_create(SET_REQ_MSG);
	
	snmp_add_var(pdu, name, name_length, INTEGER, sizeof(int), (void *) &valid);

	status = snmp_synch_response(ss, pdu, &response);

	if (status == STAT_SUCCESS) {
	    if (response->errstat == SNMP_ERR_NOERROR) {
		vars = response->variables;
		if (vars->type != SMP_ENDOFMIBVIEW
		    && vars->type != SMP_NOSUCHOBJECT  /* for robustness */
		    && vars->type != SMP_NOSUCHINSTANCE){
		    printf("Deleted\n");
		} else {
		    printf("Error\n");
		}
	    } else if (response->errstat == SNMP_ERR_NOSUCHNAME) {
		snmp_free_pdu(response);
		/* Error: maybe a naming conflict-- try again. */
	    } else {
		snmp_free_pdu(response);
		/* Error: maybe a naming conflict-- try again. */
	    }
	} else {
	    if (response) {
		snmp_free_pdu(response);
	    }
	    printf("Error getting response.\n");
	    if (status == STAT_TIMEOUT) {
		printf("No response from probe.\n");
	    }
	}
    }
}
	
free_table_indexes(table)
    control_table_p table;
{
    int i, j;

    for (i = 0; i < table->table_rows; i++) {
	for (j = 0; j < table->table_columns; j++){
	    free(table->index_values[i][j].name);
	    free(table->index_values[i][j].val.string);
	}
	free(table->index_values[i]);
    }
    free(table->index_values);
    table->index_values = NULL;
}


get_table(ss, table)
    struct snmp_session *ss;
    control_table_p table;
{
    int done = 0;
    char buf[255];

    while (!done) {
	read_table(ss, table);
	display_table(table);

	printf("You may:\n");
	printf(" A. Add an entry to the table\n");
	printf(" D. Delete an entry from the table\n");
	printf(" S. Scan the table again.\n");
	printf(" Q. Quit\n");
	printf("\nEnter Choice : ");

	gets(buf);
	switch (buf[0]){
	    case 'a':
	    case 'A':

	    create_row(ss, table);
	    break;

	    case 'd':
	    case 'D':
	    
	    delete_row(ss, table);
	    break;

	    case 's':
	    case 'S':
	    
	    break;

	    case 'q':
	    case 'Q':

	    done = 1;
	    break;

	    default:

	    printf("Please enter one of <a, d, s, q>\n");
	    break;
	}
	free_table_indexes(table);
    }
}


do_alarm_table(ss) 
    struct snmp_session *ss;
{
    control_table_t alarm_table, *at;

    at = &alarm_table;

    bcopy((char *)oid_alarmTable, (char *) at->name, sizeof(oid) * oid_alarmTableLength);
    at->name_length = oid_alarmTableLength;

    at->table_columns = 13;
    at->table_rows = 0;
    at->index_values = NULL;
    at->index_names = at_names;
    at->index_lnames = at_lnames;
    at->index_types = at_types;
    at->index_length = 14;
    at->index_readable = at_readable;
    at->index_writable = at_writable;
    at->type = ALARM_TABLE_TYPE;
    at->index_pos = INDEX_END;

    get_table(ss, at);
}


do_event_table(ss) 
    struct snmp_session *ss;
{
    control_table_t event_table, *et;

    et = &event_table;

    bcopy((char *)oid_eventTable, (char *) et->name, sizeof(oid) * oid_eventTableLength);
    et->name_length = oid_eventTableLength;

    et->table_columns = 6;
    et->table_rows = 0;
    et->index_values = NULL;
    et->index_names = et_names;
    et->index_lnames = et_lnames;
    et->index_types = et_types;
    et->index_length = 1;
    et->index_readable = et_readable;
    et->index_writable = et_writable;
    et->type = EVENT_TABLE_TYPE;
    et->index_pos = INDEX_END;

    get_table(ss, et);
}

do_notify_table(ss) 
    struct snmp_session *ss;
{
    control_table_t notify_table, *nt;

    nt = &notify_table;

    bcopy((char *)oid_notifyTable, (char *) nt->name, sizeof(oid) * oid_notifyTableLength);
    nt->name_length = oid_notifyTableLength;

    nt->table_columns = 5;
    nt->table_rows = 0;
    nt->index_values = NULL;
    nt->index_names = nt_names;
    nt->index_lnames = nt_lnames;
    nt->index_types = nt_types;
    nt->index_length = 14;
    nt->index_readable = nt_readable;
    nt->index_writable = nt_writable;
    nt->type = NOTIFY_TABLE_TYPE;
    nt->index_pos = INDEX_BEGIN;

    get_table(ss, nt);
}

do_tables(ss)
    struct snmp_session *ss;
{
    char buf[255];
    int quit = 0;

    while (!quit) {
	printf("Work with which of the following?\n");
	printf("  A. Alarm Tables\n");
	printf("  E. Event Tables\n");
	printf("  N. Notify Tables\n");
	printf("  Q. Quit\n");
	printf("\nYour Choice : ");
	gets(buf);
	switch(buf[0]) {
	    case 'a':
	    case 'A':
	    
	    do_alarm_table(ss);
	    break;
	    
	    case 'e':
	    case 'E':
	    
	    do_event_table(ss);
	    break;

	    case 'n':
	    case 'N':
	    
	    do_notify_table(ss);
	    break;
	    
	    case 'q':
	    case 'Q':
	    
	    quit++;
	    break;
	    
	    default:

	    printf("Please type one of <a, e, q>\n");
	    break;
	}
    }
}


main(argc, argv)
    int	    argc;
    char    *argv[];
{
    struct snmp_session session, *ss;
    int	arg;
    char *gateway = NULL;
    char *community = NULL;

    int port_flag = 0;
    int dest_port = 0;
    oid src[MAX_NAME_LEN], dst[MAX_NAME_LEN];
    int srclen = 0, dstlen = 0;
    struct partyEntry *pp;
    
    init_mib();

    for(arg = 1; arg < argc; arg++){
	if (argv[arg][0] == '-'){
	    switch(argv[arg][1]){
	      case 'd':
		snmp_dump_packet++;
		break;
	      default:
		printf("invalid option: -%c\n", argv[arg][1]);
		break;
	    }
	    continue;
	}
	if (gateway == NULL){
	    gateway = argv[arg];
	} else if (srclen == 0){

	    if (!read_party_database("/etc/party.conf")){
		fprintf(stderr,
			"Couldn't read party database from /etc/party.conf\n");
		exit(0);
	    }
	    if (!read_acl_database("/etc/acl.conf")){
		fprintf(stderr,
			"Couldn't read access control database from /etc/acl.conf\n");
		exit(0);
	    }
	    party_scanInit();
	    for(pp = party_scanNext(); pp; pp = party_scanNext()){
		if (!strcasecmp(pp->partyName, argv[arg])){
		    srclen = pp->partyIdentityLen;
		    bcopy(pp->partyIdentity, src, srclen * sizeof(oid));
		    break;
		}
	    }
	    if (!pp){
		srclen = MAX_NAME_LEN;
		if (!read_objid(argv[arg], src, &srclen)){
		    printf("Invalid source party: %s\n", argv[arg]);
		    srclen = 0;
		    usage();
		    exit(1);
		}
	    }
	} else if (dstlen == 0){
	    dstlen = MAX_NAME_LEN;
	    party_scanInit();
	    for(pp = party_scanNext(); pp; pp = party_scanNext()){
		if (!strcasecmp(pp->partyName, argv[arg])){
		    dstlen = pp->partyIdentityLen;
		    bcopy(pp->partyIdentity, dst, dstlen * sizeof(oid));
		    break;
		}
	    }
	    if (!pp){
		if (!read_objid(argv[arg], dst, &dstlen)){
		    printf("Invalid destination party: %s\n", argv[arg]);
		    dstlen = 0;
		    usage();
		    exit(1);
		}
	    }
	} else {
	    usage();
	    exit(1);
	}
    }
    
    if (!(gateway && srclen && dstlen)){
	usage();
	exit(1);
    }
    
    bzero((char *)&session, sizeof(struct snmp_session));
    session.peername = gateway;

    session.version = 2;

    session.srcParty = src;
    session.srcPartyLen = srclen;
    session.dstParty = dst;
    session.dstPartyLen = dstlen;
    

    session.retries = SNMP_DEFAULT_RETRIES;
    session.timeout = SNMP_DEFAULT_TIMEOUT;
    session.timeout = 2000000L;
    session.authenticator = NULL;
    snmp_synch_setup(&session);
    ss = snmp_open(&session);
    if (ss == NULL){
	printf("Couldn't open snmp\n");
	exit(-1);
    }

    do_tables(ss);
    snmp_close(ss);
}

