#include <sys/types.h>
#include <sys/time.h>

#include "asn1.h"
#include "snmp.h"
#include "snmp_impl.h"
#include "snmp_vars.h"

#include "party.h"
#include "acl.h"
#include "view.h"

oid rfc1351Domain[] = {1, 3, 6, 1, 2, 1, 20, 1, 2, 1};
oid smpUdpDomain[] = {2, 6, 6, 200, 4, 1, 1, 1};
/* no others defined yet */

oid noAuth[] = {1, 3, 6, 1, 2, 1, 20, 1, 1, 1};
oid mD5AuthProt[] = {1, 3, 6, 1, 2, 1, 20, 1, 1, 5};
oid smpMD5AuthProt[] = {2, 6, 6, 200, 4, 3, 1};

oid noPriv[] = {1, 3, 6, 1, 2, 1, 20, 1, 1, 3};
oid dESPrivProt[] = {1, 3, 6, 1, 2, 1, 20, 1, 1, 4};

oid noProxy[] = {1, 3, 6, 1, 2, 1, 20, 1, 3, 1};

#define OIDCMP(l1, l2, o1, o2) (((l1) == (l2)) \
				&& !bcmp((char *)(o1), (char *)(o2), \
					 (l1)*sizeof(oid)))

#define	PARTYIDENTITY_MASK		1
#define	PARTYTDOMAIN_MASK		2
#define	PARTYTADDRESS_MASK		4
#define	PARTYPROXYFOR_MASK		8
#define	PARTYAUTHPROTOCOL_MASK		16
#define	PARTYAUTHCLOCK_MASK		32
#define	PARTYAUTHPUBLIC_MASK		64
#define	PARTYAUTHLIFETIME_MASK		128
#define	PARTYPRIVPROTOCOL_MASK		256
#define	PARTYPRIVPUBLIC_MASK		512
#define PARTYMAXMESSAGESIZE_MASK	1024
#define PARTYSTATUS_MASK		2048
#define	PARTYSECRETSIDENTITY_MASK	4096
#define	PARTYSECRETSAUTHPRIVATE_MASK	8192
#define	PARTYSECRETSPRIVPRIVATE_MASK	16384
#define	PARTYSECRETSSTATUS_MASK		32768

#define PARTYCOMPLETE_MASK		65535	/* all collumns */

struct partyEntry *
party_rowCreate(partyID, partyIDLen)
    oid *partyID;
    int partyIDLen;
{
    struct partyEntry *pp;

    if (partyIDLen > 32)
	return NULL;
    pp = party_createEntry(partyID, partyIDLen);
    pp->partyBitMask = 0;
    pp->reserved->partySecretsStatus = PARTYINVALID;

    pp->partyBitMask = pp->reserved->partyBitMask =
	PARTYIDENTITY_MASK | PARTYSECRETSIDENTITY_MASK;
    /* Watch out for this becoming permanent by accident:
     * If during FREE stage below we discover row didn't exist before,
     * free row.
     */
    return pp;
}

party_rowDelete(partyID, partyIDLen)
    oid *partyID;
    int partyIDLen;
{
    party_destroyEntry(partyID, partyIDLen);
}

/*
 * If statP is non-NULL, the referenced object is at that location.
 * If statP is NULL and pp is non-NULL, the instance exists, but not this
 * variable.
 * If statP is NULL and pp is NULL, then neither this instance nor the
 * variable exists.
 */
int
write_party(action, var_val, var_val_type, var_val_len, statP, name, length)
   int      action;
   u_char   *var_val;
   u_char   var_val_type;
   int      var_val_len;
   u_char   *statP;
   oid      *name;
   int      length;
{
    struct partyEntry *pp, *rp;
    int var, indexlen, len;
    oid *index;
    long val;
    oid buf[32];
    int bigsize = 1000, size;
    struct aclEntry *ap;
    struct viewEntry *vp;
    u_long get_myaddr(), myaddr;
    
    if (length < 13)
	return SMP_ERR_NOCREATION;
    var = name[10];
    indexlen = name[11];
    index = name + 12;
    if (length != 12 + indexlen)
	return SMP_ERR_NOCREATION;

    /* gross hack to make it appear as if all these vars are in the
       same table.  Adding 12 is putting the secret vars contiguously at the
       end of the public vars.
     */
    if (name[6] == 21)
	var += 12;

    pp = party_getEntry(index, indexlen);
    if (pp)
	rp = pp->reserved;
    if (action == RESERVE1 && !pp){
	if ((pp = party_rowCreate(index, indexlen)) == NULL)
	    return SMP_ERR_RESOURCEUNAVAILABLE;
	rp = pp->reserved;
	/* create default vals here in reserve area */
	rp->partyTDomain = DOMAINSMPUDP;
	bzero((char *)rp->partyTAddress, 6);
	rp->partyTAddressLen = 6;
	bcopy((char *)noProxy, (char *)rp->partyProxyFor, sizeof(noProxy));
	rp->partyProxyForLen = sizeof(noProxy)/sizeof(oid);
	rp->partyAuthProtocol = NOAUTH;
	rp->partyAuthClock = 0;
	rp->partyAuthPublicLen = 0;
	rp->partyAuthLifetime = 300;
	rp->partyPrivProtocol = NOPRIV;
	rp->partyPrivPublicLen = 0;
	rp->partyMaxMessageSize = 484;
	rp->partySecretsAuthPrivateLen = 0;
	rp->partySecretsPrivPrivateLen = 0;
	rp->partySecretsStatus = PARTYVALID;
	rp->partyBitMask = PARTYCOMPLETE_MASK;
    } else if (action == COMMIT){
	if (pp->partySecretsStatus == PARTYNONEXISTENT){
	    /* commit the default vals */
	    /* This happens at most once per entry because the status is set to
	       valid after the first pass.  After that, this commit code
	       does not get executed.  It is also important to note that this
	       gets executed before any of the commits below (and never after
	       them), so they overlay their data on top of these defaults.
	       This commit code should allow for the object specific code
	       to have overlayed data after the code above has executed.
	      */
	    pp->partyTDomain = rp->partyTDomain;
	    bcopy(rp->partyTAddress, pp->partyTAddress, rp->partyTAddressLen);
	    pp->partyTAddressLen = rp->partyTAddressLen;
	    bcopy((char *)rp->partyProxyFor, (char *)pp->partyProxyFor,
		  rp->partyProxyForLen * sizeof(oid));
	    pp->partyProxyForLen = rp->partyProxyForLen;
	    pp->partyAuthProtocol = rp->partyAuthProtocol;
	    pp->partyAuthClock = rp->partyAuthClock;
	    gettimeofday(&pp->tv, (struct timezone *)0);
	    pp->tv.tv_sec -= pp->partyAuthClock;
	    pp->partyAuthPublicLen = rp->partyAuthPublicLen;
	    pp->partyAuthLifetime = rp->partyAuthLifetime;
	    pp->partyPrivProtocol = rp->partyPrivProtocol;
	    pp->partyPrivPublicLen = rp->partyPrivPublicLen;
	    pp->partyMaxMessageSize = rp->partyMaxMessageSize;
	    pp->partySecretsAuthPrivateLen = rp->partySecretsAuthPrivateLen;
	    pp->partySecretsPrivPrivateLen = rp->partySecretsPrivPrivateLen;
	    pp->partySecretsStatus = rp->partySecretsStatus;
	    pp->partyBitMask = rp->partyBitMask;
	    
	}
    } else if (action == FREE){
	if (pp && pp->partySecretsStatus == PARTYNONEXISTENT){
	    party_rowDelete(index, indexlen);
	    pp = rp = NULL;
	}
	if (pp)	/* satisfy postcondition for bitMask */
	    rp->partyBitMask = pp->partyBitMask;
    }

/* XXX !!! check return values from the asn_parse_* routines */
    switch(var){
      case PARTYTDOMAIN:
	if (action == RESERVE1){
	    if (var_val_type != ASN_OBJECT_ID)
		return SMP_ERR_WRONGTYPE;
	    size = sizeof(buf)/sizeof(oid);
	    asn_parse_objid(var_val, &bigsize, &var_val_type, buf, &size);
	    if (OIDCMP(size, sizeof(rfc1351Domain)/sizeof(oid), buf,
		       rfc1351Domain)){
		rp->partyTDomain = DOMAIN1351;
		rp->partyBitMask |= PARTYTDOMAIN_MASK;
	    } else if (OIDCMP(size, sizeof(smpUdpDomain)/sizeof(oid), buf,
		       smpUdpDomain)){
		rp->partyTDomain = DOMAINSMPUDP;
		rp->partyBitMask |= PARTYTDOMAIN_MASK;
	    } else {
		return SMP_ERR_WRONGVALUE;
	    }
	} else if (action == COMMIT){
	    pp->partyTDomain = rp->partyTDomain;
	}
	break;
      case PARTYTADDRESS:
	if (action == RESERVE1){
	    if (var_val_type != ASN_OCTET_STR)
		return SMP_ERR_WRONGTYPE;
	    size = sizeof(rp->partyTAddress);
	    asn_parse_string(var_val, &bigsize, &var_val_type,
			     rp->partyTAddress, &size);
	    rp->partyTAddressLen = size;
	    /* if other TDomains were possible, it would be necessary to
	       check the size in the reserve2 phase to see if it was
	       consistent with the TDomain.
	       Also: what if TAddr is changed to a local party: consider
	       implications for MaxMessageSize.
	     */
	    if (size != 6)
		return SMP_ERR_WRONGLENGTH;
	    rp->partyBitMask |= PARTYTADDRESS_MASK;
	} else if (action == COMMIT){
	    pp->partyTAddressLen = rp->partyTAddressLen;
	    bcopy(rp->partyTAddress, pp->partyTAddress, pp->partyTAddressLen);
	}
	break;
      case PARTYPROXYFOR:
	if (action == RESERVE1){
	    if (var_val_type != ASN_OBJECT_ID)
		return SMP_ERR_WRONGTYPE;
	    size = sizeof(rp->partyProxyFor)/sizeof(oid);
	    asn_parse_objid(var_val, &bigsize, &var_val_type,
			    rp->partyProxyFor, &size);
	    rp->partyProxyForLen = size;
	    /* We only allow noProxy now */
	    if (OIDCMP(size, sizeof(noProxy)/sizeof(oid),
		       rp->partyProxyFor, noProxy))
		return SMP_ERR_WRONGVALUE;
	    rp->partyBitMask |= PARTYPROXYFOR_MASK;
	} else if (action == COMMIT){
	    pp->partyProxyForLen = rp->partyProxyForLen;
	    bcopy((char *)rp->partyProxyFor, (char *)pp->partyProxyFor,
		  pp->partyProxyForLen * sizeof(oid));
	}
	break;
      case PARTYAUTHPROTOCOL:
	if (action == RESERVE1){
	    if (var_val_type != ASN_OBJECT_ID)
		return SMP_ERR_WRONGTYPE;
	    size = sizeof(buf)/sizeof(oid);
	    asn_parse_objid(var_val, &bigsize, &var_val_type, buf, &size);
	    if (OIDCMP(size, sizeof(noAuth)/sizeof(oid), buf, noAuth)){
		rp->partyAuthProtocol = NOAUTH;
	    } else if (OIDCMP(size, sizeof(mD5AuthProt)/sizeof(oid), buf,
			      mD5AuthProt)){
		rp->partyAuthProtocol = MD5AUTHPROT;
	    } else if (OIDCMP(size, sizeof(smpMD5AuthProt)/sizeof(oid), buf,
			      smpMD5AuthProt)){
		rp->partyAuthProtocol = SMPMD5AUTHPROT;
	    } else {
		/* no other currently defined */
		return SMP_ERR_WRONGVALUE ;
	    }
	    rp->partyBitMask |= PARTYAUTHPROTOCOL_MASK;
	} else if (action == COMMIT){
	    pp->partyAuthProtocol = rp->partyAuthProtocol;
	}
	break;
      case PARTYAUTHCLOCK:
	if (action == RESERVE1){
	    if (var_val_type != ASN_INTEGER)
		return SMP_ERR_WRONGTYPE;
	    asn_parse_int(var_val, &bigsize, &var_val_type, &val, sizeof(val));
	    rp->partyAuthClock = val;
	    rp->partyBitMask |= PARTYAUTHCLOCK_MASK;
	} else if (action == COMMIT){
	    pp->partyAuthClock = rp->partyAuthClock;
	    gettimeofday(&pp->tv, (struct timezone *)0);
	    pp->tv.tv_sec -= pp->partyAuthClock;
	}
	break;
      case PARTYAUTHPUBLIC:
	if (action == RESERVE1){
	    if (var_val_type != ASN_OCTET_STR)
		return SMP_ERR_WRONGTYPE;
	    size = sizeof(rp->partyAuthPublic);
	    asn_parse_string(var_val, &bigsize, &var_val_type,
			     rp->partyAuthPublic, &size);
	    rp->partyAuthPublicLen = size;
	    if (size > 32)
		return SMP_ERR_WRONGLENGTH;
	    rp->partyBitMask |= PARTYAUTHPUBLIC_MASK;
	} else if (action == COMMIT){
	    pp->partyAuthPublicLen = rp->partyAuthPublicLen;
	    bcopy((char *)rp->partyAuthPublic,
		  (char *)pp->partyAuthPublic, pp->partyAuthPublicLen);
	}
	break;
      case PARTYAUTHLIFETIME:
	if (action == RESERVE1){
	    if (var_val_type != ASN_INTEGER)
		return SMP_ERR_WRONGTYPE;
	    asn_parse_int(var_val, &bigsize, &var_val_type, &val, sizeof(val));
	    /* what range should I check for ???
	    if (val < 1 || val > 3600)
		return SMP_ERR_WRONGVALUE;
	    */
	    rp->partyAuthLifetime = val;
	    rp->partyBitMask |= PARTYAUTHLIFETIME_MASK;
	} else if (action == COMMIT){
	    pp->partyAuthLifetime = rp->partyAuthLifetime;
	}
	break;
      case PARTYPRIVPROTOCOL:
	if (action == RESERVE1){
	    if (var_val_type != ASN_OBJECT_ID)
		return SMP_ERR_WRONGTYPE;
	    size = sizeof(buf)/sizeof(oid);
	    asn_parse_objid(var_val, &bigsize, &var_val_type, buf, &size);
	    if (OIDCMP(size, sizeof(noPriv)/sizeof(oid), buf, noPriv)){
		rp->partyPrivProtocol = NOPRIV;
	    } else if (OIDCMP(size, sizeof(dESPrivProt)/sizeof(oid), buf,
			      dESPrivProt)){
		rp->partyPrivProtocol = DESPRIVPROT;
	    } else {
		/* no other currently defined */
		return SMP_ERR_WRONGVALUE;
	    }
	    rp->partyBitMask |= PARTYPRIVPROTOCOL_MASK;
	} else if (action == COMMIT){
	    pp->partyPrivProtocol = rp->partyPrivProtocol;
	}
	break;
      case PARTYPRIVPUBLIC:
	if (action == RESERVE1){
	    if (var_val_type != ASN_OCTET_STR)
		return SMP_ERR_WRONGTYPE;
	    size = sizeof(rp->partyPrivPublic);
	    asn_parse_string(var_val, &bigsize, &var_val_type,
			     rp->partyPrivPublic, &size);
	    rp->partyPrivPublicLen = size;
	    if (size > 32)
		return SMP_ERR_WRONGLENGTH;
	    rp->partyBitMask |= PARTYPRIVPUBLIC_MASK;
	} else if (action == COMMIT){
	    bcopy((char *)rp->partyPrivPublic, (char *)pp->partyPrivPublic,
		  rp->partyPrivPublicLen);
	    pp->partyPrivPublicLen = rp->partyPrivPublicLen;
	}
	break;
      case PARTYMAXMESSAGESIZE:
	if (action == RESERVE1){
	    if (var_val_type != ASN_INTEGER)
		return SMP_ERR_WRONGTYPE;
	    asn_parse_int(var_val, &bigsize, &var_val_type, &val, sizeof(val));
	    if (val < 484 || val > 65507)
		return SMP_ERR_WRONGVALUE;
	    rp->partyMaxMessageSize = val;
	    rp->partyBitMask |= PARTYMAXMESSAGESIZE_MASK;
	} else if (action == RESERVE2){
	    myaddr = get_myaddr();
	    if ((rp->partyTDomain == DOMAIN1351
		 || rp->partyTDomain == DOMAINSMPUDP)
		&& !bcmp((char *)&myaddr, rp->partyTAddress, 4)){
		/* party is local */
		/* 1500 should be constant in snmp_impl.h */
		if (rp->partyMaxMessageSize > 1500)
		    return SMP_ERR_INCONSISTENTVALUE;
	    }
	} else if (action == COMMIT){
	    pp->partyMaxMessageSize = rp->partyMaxMessageSize;
	}
	break;
      case PARTYSECRETSAUTHPRIVATE:
	if (action == RESERVE1){
	    if (var_val_type != ASN_OCTET_STR)
		return SMP_ERR_WRONGTYPE;
	    size = sizeof(rp->partySecretsAuthPrivate);
	    asn_parse_string(var_val, &bigsize, &var_val_type,
			     rp->partySecretsAuthPrivate, &size);
	    rp->partySecretsAuthPrivateLen = size;
	    if (size > 16)
		return SMP_ERR_WRONGLENGTH;
	    rp->partyBitMask |= PARTYSECRETSAUTHPRIVATE_MASK;
	} else if (action == COMMIT){
	    if (!(pp->partyBitMask & PARTYSECRETSAUTHPRIVATE_MASK))
		pp->partySecretsAuthPrivateLen = 0;
	    for(len = 0; (len < pp->partySecretsAuthPrivateLen)
		&& (len < rp->partySecretsAuthPrivateLen); len++){
		pp->partySecretsAuthPrivate[len] ^=
		    rp->partySecretsAuthPrivate[len];
	    }
	    while(len < rp->partySecretsAuthPrivateLen)
		pp->partySecretsAuthPrivate[len] =
		    rp->partySecretsAuthPrivate[len];
	    pp->partySecretsAuthPrivateLen = rp->partySecretsAuthPrivateLen;
	}
	break;
      case PARTYSECRETSPRIVPRIVATE:
	if (action == RESERVE1){
	    if (var_val_type != ASN_OCTET_STR)
		return SMP_ERR_WRONGTYPE;
	    size = sizeof(rp->partySecretsPrivPrivate);
	    asn_parse_string(var_val, &bigsize, &var_val_type,
			     rp->partySecretsPrivPrivate, &size);
	    rp->partySecretsPrivPrivateLen = size;
	    if (size > 16)
		return SMP_ERR_WRONGLENGTH;
	    rp->partyBitMask |= PARTYSECRETSPRIVPRIVATE_MASK;
	} else if (action == COMMIT){
	    if (!(pp->partyBitMask & PARTYSECRETSPRIVPRIVATE_MASK))
		pp->partySecretsPrivPrivateLen = 0;
	    for(len = 0; (len < pp->partySecretsPrivPrivateLen)
		&& (len < rp->partySecretsPrivPrivateLen); len++){
		pp->partySecretsPrivPrivate[len] ^=
		    rp->partySecretsPrivPrivate[len];
	    }
	    while(len < rp->partySecretsPrivPrivateLen)
		pp->partySecretsPrivPrivate[len] =
		    rp->partySecretsPrivPrivate[len];
	    pp->partySecretsPrivPrivateLen = rp->partySecretsPrivPrivateLen;
	}
	break;
      case PARTYSECRETSSTATUS: /* read-write access */
	if (action == RESERVE1){
	    if (var_val_type != ASN_INTEGER)
		return SMP_ERR_WRONGTYPE;
	    asn_parse_int(var_val, &bigsize, &var_val_type, &val, sizeof(val));
	    if (val < 1 || val > 2)
		return SMP_ERR_WRONGVALUE;
	    rp->partySecretsStatus = val;
	    rp->partyBitMask |= PARTYSECRETSSTATUS_MASK | PARTYSTATUS_MASK;
	} else if (action == RESERVE2){
	    if ((rp->partySecretsStatus == PARTYVALID)
		&& (rp->partyBitMask != PARTYCOMPLETE_MASK))
		return SMP_ERR_INCONSISTENTVALUE;
	    /* tried to set incomplete row valid */
	} else if (action == COMMIT){
	    pp->partySecretsStatus = rp->partySecretsStatus;
	} else if (action == ACTION && pp->partySecretsStatus == PARTYINVALID){

	    /* delete all related acl entries */
	    acl_scanInit();
	    ap = acl_scanNext();
	    do {
		for(; ap; ap = acl_scanNext()){
		    if ((ap->aclTargetLen == pp->partyIdentityLen
			 && !bcmp(ap->aclTarget, pp->partyIdentity,
				  ap->aclTargetLen * sizeof(oid)))
			|| (ap->aclSubjectLen == pp->partyIdentityLen
			    && !bcmp(ap->aclSubject, pp->partyIdentity,
				     ap->aclSubjectLen * sizeof(oid)))){
			acl_destroyEntry(ap->aclTarget, ap->aclTargetLen,
					 ap->aclSubject, ap->aclSubjectLen);
			acl_scanInit();
			ap = acl_scanNext();
			break;
			/* ap is still set, so we'll start over again */
		    }
		}
	    } while (ap);
		
	    /* delete all related view entries */
	    view_scanInit();
	    vp = view_scanNext();
	    do {
		for(; vp; vp = view_scanNext()){
		    if (vp->viewPartyLen == pp->partyIdentityLen
			&& !bcmp(vp->viewParty, pp->partyIdentity,
				 vp->viewPartyLen * sizeof(oid))){
			view_destroyEntry(vp->viewParty, vp->viewPartyLen,
					  vp->viewSubtree, vp->viewSubtreeLen);
			view_scanInit();
			vp = view_scanNext();
			break;
			/* vp is still set, so we'll start over again */
		    }
		}
	    } while (vp);
		
	    /* then delete the party itself */
	    party_rowDelete(pp->partyIdentity, pp->partyIdentityLen);
	}
	break;
      case PARTYIDENTITY:
      case PARTYSECRETSIDENTITY:
      default:
	    return SMP_ERR_NOCREATION;
    }
    if (action == COMMIT)	/* make any new collumns appear */
	pp->partyBitMask = rp->partyBitMask;

    return SNMP_ERR_NOERROR;
}

u_char *
var_party(vp, name, length, exact, var_len, write_method)
    register struct variable *vp;   /* IN - pointer to variable entry that points here */
    register oid *name;      /* IN/OUT - input name requested, output name found */
    register int *length;    /* IN/OUT - length of input and output oid's */
    int          exact;      /* IN - TRUE if an exact match was requested. */
    int          *var_len;   /* OUT - length of variable or 0 if function returned. */
    int          (**write_method)(); /* OUT - pointer to function to set variable, otherwise 0 */
{
    oid newname[MAX_NAME_LEN], lowname[MAX_NAME_LEN];
    int newnamelen, lownamelen;
    struct partyEntry *pp, *lowpp = NULL;
    u_long mask;
    struct timeval now;
/*
 * This routine handles requests for variables of the form:
 * .iso.org.dod.internet.mgmt.mib.snmpParty.partyPublic.partyTable.partyEntry.X.oidlen.oid
 * or .1.3.6.1.2.1.20.2.1.1.X.oidlen.oid, where the oid suffix is variable length
 * Therefore, the length of the index is name[9] and the index starts
 * at name[12].
 * It also handles three objects in the partySecrets table:
 * .iso.org.dod.internet.mgmt.mib.snmpSecrets.partyPrivate.partySecretsTable.partySecretsEntry.X.oidlen.oid.
 * It is only able to handle both because they are the same length and
 * general structure.
 */

    mask = 1 << (vp->magic - 1);
    bcopy((char *)vp->name, (char *)newname, (int)vp->namelen * sizeof(oid));
    if (exact){
        if (*length < 13 ||
	    bcmp((char *)name, (char *)vp->name, 9 * sizeof(oid)))
	    return NULL;
    	*write_method = write_party;
        pp = party_getEntry(name + 12, (int)name[11]);
	if (pp == NULL)
	    return NULL;
	if (!(pp->partyBitMask & mask))
	    return NULL;
    } else {
      /* find "next" control entry */
      party_scanInit();
      for(pp = party_scanNext(); pp; pp = party_scanNext()){
	if (!(pp->partyBitMask & mask))
	    continue;
	newname[11] = (oid)pp->partyIdentityLen;
	bcopy((char *)pp->partyIdentity, (char *)(newname + 11),
	      pp->partyIdentityLen * sizeof(oid));
	newnamelen = 11 + pp->partyIdentityLen;
	if ((compare(newname, newnamelen, name, *length) > 0) &&
	    (!lowpp || compare(newname, newnamelen,
			       lowname, lownamelen) < 0)){
	    /*
	     * if new one is greater than input and closer to input than
	     * previous lowest, save this one as the "next" one.
	     */
	    bcopy((char *)newname, (char *)lowname, newnamelen * sizeof(oid));
	    lownamelen = newnamelen;
	    lowpp = pp;
	}
      }
      if (lowpp == NULL)
	  return NULL;
      pp = lowpp;
      bcopy((char *)lowname, (char *)name, lownamelen * sizeof(oid));
      *length = lownamelen;
    }

    *var_len = sizeof(long);
    long_return = 0;

    switch (vp->magic){
      case PARTYIDENTITY:
	*var_len = pp->partyIdentityLen * sizeof(oid);
	return (u_char *)pp->partyIdentity;
      case PARTYTDOMAIN:
	if (pp->partyTDomain == DOMAINSMPUDP){
	    *var_len = sizeof(smpUdpDomain);
	    return (u_char *)smpUdpDomain;
	} else if (pp->partyTDomain == DOMAIN1351){
	    *var_len = sizeof(rfc1351Domain);
	    return (u_char *)rfc1351Domain;
	}
      case PARTYTADDRESS:
	*var_len = pp->partyTAddressLen;
	return (u_char *)pp->partyTAddress;
      case PARTYPROXYFOR:
	*var_len = pp->partyProxyForLen * sizeof(oid);
	return (u_char *)pp->partyProxyFor;
      case PARTYAUTHPROTOCOL:
	if (pp->partyAuthProtocol == MD5AUTHPROT){
	    *var_len = sizeof(mD5AuthProt);
	    return (u_char *)mD5AuthProt;
	} else if (pp->partyAuthProtocol == SMPMD5AUTHPROT){
	    *var_len = sizeof(smpMD5AuthProt);
	    return (u_char *)smpMD5AuthProt;
	} else { /* noAuth */
	    *var_len = sizeof(noAuth);
	    return (u_char *)noAuth;
	}
      case PARTYAUTHCLOCK:
	gettimeofday(&now, (struct timezone *)0);
	long_return = now.tv_sec - pp->tv.tv_sec;
	return (u_char *)&long_return;
      case PARTYAUTHPUBLIC:
	*var_len = pp->partyAuthPublicLen;
	return (u_char *)pp->partyAuthPublic;
      case PARTYAUTHLIFETIME:
	return (u_char *)&pp->partyAuthLifetime;
      case PARTYPRIVPROTOCOL:
	if (pp->partyPrivProtocol == DESPRIVPROT){
	    *var_len = sizeof(dESPrivProt);
	    return (u_char *)dESPrivProt;
	} else { /* noPriv */
	    *var_len = sizeof(noPriv);
	    return (u_char *)noPriv;
	}
	/*NOTREACHED*/
      case PARTYPRIVPUBLIC:
	*var_len = pp->partyPrivPublicLen;
	return (u_char *)pp->partyPrivPublic;
      case PARTYMAXMESSAGESIZE:
	return (u_char *)&pp->partyMaxMessageSize;
      case PARTYSECRETSIDENTITY:
	*var_len = pp->partyIdentityLen * sizeof(oid);
	return (u_char *)pp->partyIdentity;
      case PARTYSECRETSAUTHPRIVATE:
	*var_len = 0;
	return (u_char *)pp->partyIdentity;
      case PARTYSECRETSPRIVPRIVATE:
	*var_len = 0;
	return (u_char *)pp->partyIdentity;
      case PARTYSTATUS:
      case PARTYSECRETSSTATUS:
	return (u_char *)&pp->partySecretsStatus;
      default:
            ERROR("");
    }
    return NULL;
}
