/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * Symbol table handling
 */

/*
 * Exports:
 * 	symtab_init();
 *	symtab_free();	Remove all entries
 * 	symtab_print();
 *
 * 	struct symbol *symbol_lookup(s: char *) 
 * 	symbol_enter(s: char *; tag: int; token: int;
 *			client_proc, server_proc: (*int)()) 
 * 	symbol_print(se: struct symbol *) 
 *
 *	char *symbol_lookup_token(token: int)
 *
 *	symbol_remove(s: char *)
 */

#include <general.h>

/*  */

typedef struct ste {
    struct symbol *entry;
    struct ste	*next;
} symtab_t;
#define NULLSTE (symtab_t *)NULL
    
symtab_t *symtab;

#define allocsymbol() 	(struct symbol *)malloc(sizeof(struct symbol))
#define freesymbol(se)	free((char *)se)

#define allocste() 	(symtab_t *)malloc(sizeof(symtab_t))
#define freeste(ste)	free((char *)ste)



symtab_init()
{
    tprintf("symtab_init()\n");
    symtab = NULLSTE;
}

/*  */

struct symbol *symbol_lookup(s)
    char  *s;
{
    symtab_t *st = symtab;

    tprintf("symbol_lookup (%s)\n", s);

    lowercase(s);
    
    while (st != NULLSTE) {
	if (st->entry == NULLSE) {
	    eprintf(EF_IN4, INTERNAL, "Symbol table problem", "symbol_lookup",
	       "\n\tNull entry");
	    return NULLSE;
	}
	if (strcmp(s, st->entry->se_name) == 0)
	    return st->entry;
	st = st->next;
    }
    return NULLSE;
} /* symbol_lookup */

/*  */

char *symbol_lookup_token(token)
    int token;
{
    symtab_t *st = symtab;
    char  *s;

    tprintf("symbol_lookup_token(%d)\n", token);
    
    while (st != NULLSTE) {
	if (st->entry == NULLSE) {
	    eprintf(EF_IN4, INTERNAL, "Symbol table problem", "symbol_lookup_token",
	       "\n\tNull entry");
	    return NULL;
	}
	if (symbol_istoken(st->entry) && (st->entry->se_token == token))
	    return st->entry->se_name;
	st = st->next;
    }
    return NULL;
} /* symbol_lookup_token */

/*  */

/* VARARGS3 */
symbol_enter(s, tag, token, client_proc, server_proc)
    char *s;
    int tag, token;
    int (*client_proc)(), (*server_proc)();
{
    symtab_t **a_st = &symtab;
    struct symbol *se;

    tprintf("symbol_enter (%s, tag:%d, token:%d, 0x%x, 0x%X)\n",
		s, tag, token, client_proc, server_proc);

    lowercase(s);
    
    /* Check for duplicates */
    if (symbol_lookup(s) != NULLSE) {
	eprintf(EF_IN4, INTERNAL, "Duplicate symbol name", s, "symbol table");
	symtab_print();
	return NOTOK;
    }

    while (*a_st != NULLSTE)
	a_st = &(*a_st)->next;

    *a_st = allocste();

    if (*a_st == NULLSTE) {
	eprintf(EF_IN4, INTERNAL, "Resource shortage", "symbol_enter",
	       "\n\tAllocste failed");
	return NOTOK;
    }

    (*a_st)->next = NULLSTE;

    se = allocsymbol();
    if (se == NULLSE) {
	eprintf(EF_IN4, INTERNAL, "Resource shortage", "symbol_enter",
	       "\n\tAllocsymbol failed");
	return NOTOK;
    }
    (*a_st)->entry = se;

    se->se_name = s;
    se->se_tag = tag;
    if (tag == SET_TOKEN)
	se->se_token = token;
    else {
	se->se_client_proc = client_proc;
	se->se_server_proc = server_proc;
    }

    /* Check input parameters */
    if (se->se_tag == SET_TOKEN && se->se_token == 0) {
	eprintf(EF_IN4, INTERNAL, "Bad parameter", "symbol_enter",
	       "\n\tA token entry must have a non-zero token number");
	symbol_print(se);
	return NOTOK;
    }
    return OK;
} /* symbol_enter */
	
/*  */

symtab_free()
{
    symtab_t *st = symtab, *st1;


    tprintf("symtab_free()\n");

    while (st != NULLSTE) {
	symbol_free(st->entry);
	st1 = st;
	st = st->next;
	freeste(st1);
    }
    symtab = NULLSTE;
} /* symtab_free */
	
/*  */

symbol_free(se)
    struct symbol *se;
{
    
    if (se == NULLSE) {
	printf("symbol_free: bad entry\n");
	return;
    }
    freesymbol(se);
} /* symbol_free */

/*  */

symtab_print()
{
    symtab_t *st = symtab;


    tprintf("symtab_print()\n");

    printf("The symbol table:\n");
    while (st != NULLSTE) {
	symbol_print(st->entry);
	st = st->next;
    }
} /* symtab_print */
	
/*  */

symbol_print(se)
    struct symbol *se;
{
    char *tstr;
    
    if (se == NULLSE) {
	printf("symbol_print: bad entry\n");
	return;
    }
    switch(se->se_tag) {
    case SET_TOKEN:
	tstr = "TOKEN";
	break;
    case SET_DIST:
	tstr = "DIST";
	break;
    case SET_BASIC:
	tstr = "BASIC";
	break;
    case SET_COMP:
	tstr = "COMP";
	break;
    }
    printf("Symbol: %s, tag %s", se->se_name, tstr);
    if (se->se_tag == SET_TOKEN)
	printf(", token = %d", se->se_token);
    else
	printf(", client_proc 0x%x, server_proc 0x%x",
	       se->se_client_proc, se->se_server_proc);
    printf("\n");
} /* symbol_print */

/*  */

static lowercase(str)
    char *str;
{
    while (*str != '\0') {
	if (isupper(*str))
	    *str = tolower(*str);
	str++;
    }
}

