/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * The Benchmark Control Protocol - signal part
 */

#include <general.h>
#include <ipc.h>

#define NO_KILLPG	1


/*
 * Exports:
 *	int sigunblock(mask)
 *	    int mask;
 *	void setup_bench_sighdl()
 *	void setup_demon_sighdl()
 *	void setup_child_sighdl()
 *	int sig_send_msg(msg)
 *	    msg_t *msg;
 *
 *	int interupted	- global variable set here and check in the benchmark
 *			  and demon procedures
 */

int interupted;

/*  Implementation dependent #include's */

#include <signal.h>
#include <ipc_sig.h>

/* Imported from ipc_system.c */

extern int 		system_client_nproc;
extern commstate_t 	*system_client_procs;
extern int 		system_server_nproc;
extern commstate_t 	*system_server_procs;

/*  */

/*
 * Signal to message translators:
 *
 * This will work even when the process is sleeping on a select() in recv_msg.
 * The select will return when the signal is caught (with errno == EINTR)
 * so that the message will be processed.
 */

static int hdl_alarm(sig)
    int sig;
{
    msg_t msgs, *msg = &msgs;
    
#ifdef SYSV
    (void)signal(sig, SIG_IGN);
#endif SYSV

    tprintf("hdl_alarm()\n");

    
    msg->msg_from.ba_addr = BA_SYSTEM;
    msg->msg_from.ba_index = 0;
    msg->msg_to = get_myaddr();
    msg->msg_code = MSG_ALARM;
    msg->msg_datalen = 0;
    bcp_send_msg(msg);
#ifdef SYSV
    (void)signal(sig, hdl_alarm);
#endif SYSV
    return OK;
} /* hdl_alarm */

static int hdl_fatal(sig)
    int sig;
{
    msg_t msgs, *msg = &msgs;
    
#ifdef SYSV
    (void)signal(sig, SIG_IGN);
#endif SYSV

    tprintf("hdl_fatal()\n");

    msg->msg_from.ba_addr = BA_SYSTEM;
    msg->msg_from.ba_index = 0;
    msg->msg_to = get_myaddr();
    msg->msg_code = MSG_FATAL;
    msg->msg_datalen = 0;
    bcp_send_msg(msg);
#ifdef SYSV
    (void)signal(sig, hdl_fatal);
#endif SYSV
    return OK;
} /* hdl_fatal */

/*  */

/*
 * Signal handler for benchmark and demon procedures
 * Set the interupted flag which will cause the procedure terminate and
 * the process will pick up the message from its local message queue.
 */
static int hdl_child_signals(sig)
    int sig;
{
    msg_t msgs, *msg = &msgs;
    
#ifdef SYSV
    (void)signal(sig, SIG_IGN);
#endif SYSV
    tprintf("hdl_child_signals(%d)\n", sig);

    msg->msg_to = get_myaddr();
    msg->msg_from.ba_addr = BA_SYSTEM;
    msg->msg_from.ba_index = 0;
    msg->msg_datalen = 0;
    
    switch (sig) {
    case SIG_TERMINATE:
	dprintf("hdl_child_signals: received a SIG_TERMINATE\n");
	interupted = 1;
	msg->msg_code = MSG_TERMINATE;
	break;

    case SIG_START:
	dprintf("hdl_child_signals: received a SIG_START\n");
	msg->msg_code = MSG_START;
	break;

    case SIG_ABORT:
	dprintf("hdl_child_signals: received a SIG_ABORT\n");
	interupted = 1;
	msg->msg_code = MSG_ABORT;
	break;

    default:
	dprintf("hdl_child_signals: received an unknown signal: %d\n", sig);
	interupted = 1;
	msg->msg_code = MSG_FATAL;
	break;
	
    }
    (void)bcp_send_msg(msg);
    
#ifdef SYSV
    (void)signal(sig, hdl_child_signals);
#endif SYSV
    return OK;
} /* hdl_child_signals */

/*  */

int sigunblock(mask)
    int mask;
{
    int oldmask = 0;
#ifdef BSD
    oldmask = sigblock(0);
    mask = oldmask & ~mask;
    sigsetmask(mask);
#endif BSD
    return oldmask;
}

/*  */

void setup_bench_sighdl()
{
    tprintf("setup_bench_sighdl()\n");

    (void)signal(SIGALRM, hdl_alarm);

#ifdef BSD
    (void)sigunblock(sigmask(SIGALRM));

    (void)sigblock(sigmask(SIGPIPE));
#endif BSD
    (void)signal(SIGHUP, hdl_fatal);
    (void)signal(SIGINT, hdl_fatal);
    (void)signal(SIGQUIT, hdl_fatal);
    (void)signal(SIGTERM, hdl_fatal);
} /* setup_bench_sighdl */

/*  */

void setup_demon_sighdl()
{
    tprintf("setup_demon_sighdl()\n");
#ifdef BSD
    (void)sigblock(sigmask(SIGPIPE));
#endif BSD
    (void)signal(SIGHUP, hdl_fatal);
    (void)signal(SIGINT, hdl_fatal);
    (void)signal(SIGQUIT, hdl_fatal);
    (void)signal(SIGTERM, hdl_fatal);
} /* setup_demon_sighdl */

/*  */

void setup_child_sighdl()
{
    tprintf("setup_child_sighdl()\n");
#ifdef BSD
    (void)sigblock(sigmask(SIGPIPE));
#endif BSD
    (void)signal(SIG_TERMINATE, hdl_child_signals);
    (void)signal(SIG_START, hdl_child_signals);
    (void)signal(SIG_ABORT, hdl_child_signals);
    (void)signal(SIGHUP, hdl_child_signals);
    (void)signal(SIGINT, hdl_child_signals);
    (void)signal(SIGQUIT, hdl_child_signals);
    (void)signal(SIGTERM, hdl_child_signals);
} /* setup_child_sighdl */

/*  */

int sig_send_msg(msg)
    msg_t *msg;
{
    int sig, pgrp;
    int i;
    bcpaddr_t dest;
    
    tprintf("sig_send_msg(0x%x)\n", msg);

    if (msg->msg_datalen > 0) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "Can't send data using signals",
		"sig_send_msg");
	dprintf("\tDatalen %d, data %s\n", msg->msg_datalen, msg->msg_data);
	return NOTOK;
    }
    
    switch (msg->msg_code) {
    case MSG_TERMINATE:
	sig = SIG_TERMINATE;
	break;
    case MSG_START:
	sig = SIG_START;
	break;
    case MSG_ABORT:
	sig = SIG_ABORT;
	break;
    case MSG_FATAL:
	sig = SIG_FATAL;
	break;
    default:
	eprintf(EF_IN4, INTERNAL, PROTOCOL,
		"Can't send this code using signals",
		"sig_send_msg");
	dprintf("\tCode = %s\n", code2str(msg->msg_code));
	return NOTOK;
    }

    if (msg->msg_to.ba_addr & BA_CLIENTS) {
	(void)group_send_msg(sig, system_client_nproc, system_client_procs);
    } else if (msg->msg_to.ba_addr & BA_CLIENT) {
	dprintf("Sending signal %d to process %d (index %d)\n",
		sig, system_client_procs[msg->msg_to.ba_index].c_pid,
		msg->msg_to.ba_index);

	if (msg->msg_to.ba_index >= system_client_nproc) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "Too large ba_index",
		    "sig_send_msg");
	    return NOTOK;
	}
	
	if (kill(system_client_procs[msg->msg_to.ba_index].c_pid, sig) == NOTOK) {
	    eprintf(EF_SYSCALL, PROTOCOL, "kill", "sig_send_msg", getsyserr());
	    return NOTOK;
	}
    }
    if (msg->msg_to.ba_addr & BA_SERVERS) {
	(void)group_send_msg(sig, system_server_nproc, system_server_procs);
    } else if (msg->msg_to.ba_addr & BA_SERVER) {
	dprintf("Sending signal %d to process %d (index %d)\n",
		sig, system_server_procs[msg->msg_to.ba_index].c_pid,
		msg->msg_to.ba_index);

	if (msg->msg_to.ba_index >= system_server_nproc) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "Too large ba_index",
		    "sig_send_msg");
	    return NOTOK;
	}
	
	if (kill(system_server_procs[msg->msg_to.ba_index].c_pid, sig) == NOTOK) {
	    eprintf(EF_SYSCALL, PROTOCOL, "kill", "sig_send_msg", getsyserr());
	    return NOTOK;
	}
    }
    return OK;
} /* sig_send_msg */

/*  */

static int group_send_msg(sig, system_nproc, system_procs)
    int sig;
    int system_nproc;
    commstate_t *system_procs;
{
    int pgrp, i;

    tprintf("group_send_msg(%d, %d, 0x%x)\n", sig, system_nproc, system_procs);
    
    if (system_nproc == 0) {
	dprintf("group_send_msg: no processes to signal\n");
	return OK;
    }
    
    pgrp = 0;
#ifdef REVERSE
    for (i = system_nproc - 1; i >= 0; i--) {
#else
    for (i = 0; i < system_nproc; i++) {
#endif REVERSE
	if (system_procs[i].c_pid == 0)
	    continue;
	
	if (pgrp != 0 && pgrp == system_procs[i].c_pgrp) {
	    dprintf("group_send_msg: skipping process in group %d\n", pgrp);
	    continue;
	}

#ifndef NO_KILLPG
	pgrp = system_procs[i].c_pgrp;
#endif NO_KILLPG
	
	if (pgrp == 0) {
	    dprintf("Sending signal %d to process %d\n",
		    sig, system_procs[i].c_pid);
	    
	    if (kill(system_procs[i].c_pid, sig) == NOTOK) {
		eprintf(EF_SYSCALL, PROTOCOL, "kill", "group_send_msg",
			getsyserr());
		return NOTOK;
	    }
	} else {
#ifdef BSD_PGRP
	    dprintf("Sending signal %d to process group %d\n",
		    sig, pgrp);
	    
	    if (killpg(pgrp, sig) == NOTOK) {
		eprintf(EF_SYSCALL, PROTOCOL, "killpg", "group_send_msg",
			getsyserr());
		return NOTOK;
	    }
#else
	    eprintf(EF_IN3, INTERNAL, "Process group handling",
		    "group_send_msg");
	    return NOTOK;
#endif !BSD_PGRP		
	}
    }
} /* group_send_msg */    
