/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * The Benchmark Control Protocol - presentation level
 */

/*
 * Exports:
 *	char *spec2str(spec: struct spec *, len: int*)
 *	struct spec *str2spec(str: char *, len: int *)
 *
 *	char *statistics2str(stat: statistics_t *, len: int*)
 *	statistics_t *str2statistics(str: char *, len: int *);
 *
 * Protocol dependent routines imported:
 * 	char *address2str(addr: address_t *, len: int *)
 *	address_t *str2address(str: char *, len int *)
 *
 * NOTE: All of the above return dynamically allocated data - use:
 *	free(char *)
 *	spec_free(struct spec *)
 *	statistics_free(statistics_t *)
 *	address_free(address_t *)
 */

/*
 *
 *	Revision: String manipulation buffer buf changed to dynamic
 *		buffer allocation (malloc). 1089/PerG
 *
 */

#include <general.h>
#include <ipc.h>
#include <ipc_pres.h>


/*#define SENDEMONSPEC Not in use! */

/* Forwards */    
char *strtos();

#define BIGBUF		1024
#define SMALLBUF	50

#define strtoi (int)strtol

double strtod();

static char *timing2str();
static char *msize2str();
static struct timing *str2timing();
static struct msize *str2msize();

/*  */

void statistics_free(stat)
    statistics_t *stat;
{
    tprintf("statistics_free(0x%x)\n", stat);
    free((char *)stat);
} /* statistics_free */
#ifdef SENDDEMONSPEC

/*  */

char *spec2str(sp, len)
    struct spec *sp;
    int * len;
{
    static char abuf[SMALLBUF];
    char *res, *str, *this;
    int alen;

    tprintf("Entering spec2str(0x%x, 0x%x)\n", sp, len);

    this = abuf;
    
    sprintf(abuf, "%d %f %s ",
	    spec_number(sp),
	    spec_timelimit(sp),
	    (bench_isbasic(spec_bench(sp)) ? "basic" : "comp"));
    tprintf("spec2str: abuf is:%s", abuf);
    alen = strlen(this);
    this = str + alen;		/* advance to end */
    if (alen >= SMALLBUF) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "SMALLBUF exceeded",
		"spec2str");
	return NULL;
    }

    if (bench_isbasic(spec_bench(sp))) {
	char *bstr;
	int blen, total;
	
	if ((bstr = basic2str(spec_bench(sp), &blen)) == NULL)
	    return NULL;
	blen = blen - 1;	/* ignore NULL character included in len */
	total = alen + blen + 2;
	res = (char *)malloc(total);
	if (res == NULL) {
	    eprintf(EF_IN3, RESOURCE, "No memory for malloc()",
		    "spec2str");
	    return NULL;
	}
	str = res;
	bcopy(abuf, str, alen);
	str = str + alen;
	bcopy(bstr, str, blen);
	str = str + blen;
	free(bstr);

    } else {
	char *cstr;
	int clen, total;
	
	if ((cstr = comp2str(spec_bench(sp), &clen)) == NULL)
	    return NULL;
	dprintf("comp2str returned OK\n");
	clen = clen - 1;
	total = alen + clen + 2;
	res = (char *)malloc(total);
	if (res == NULL) {
	    eprintf(EF_IN3, RESOURCE, "No memory for malloc()",
		    "spec2str");
	    return NULL;
	}
	str = res;
	bcopy(abuf, str, alen);
	str = str + alen;
	bcopy(cstr, str, clen);
	str = str + clen;
	free(cstr);
    }

    bcopy("\n", str, 1);
    tprintf("spec2str: abuf bstr/cstr+newline is:%s", res);

    *len = strlen(res);
    return res;
} /* spec2str */

#endif

#ifdef SENDDEMONSPEC

/*  */

struct spec *str2spec(str, a_len)
    char *str;
    int *a_len;
{
    int len = *a_len;
    struct spec *sp = NULLSP;
    int num;
    char *type, *this, *next;
    struct bench *be;
    double timelimit;

    tprintf("str2spec(0x%x, 0x%x, len %d)/n", str, a_len, len);

    this = str;
    num = strtoi(this, &next, 10);

    if (next == this) {
	eprintf("str2spec: No benchmark repetition count in %s\n", str);
	return NULLSP;
    }
    this = next;

    timelimit = strtod(this, &next);

    if (next == this) {
	eprintf("str2spec: No benchmark timelimit in %s\n", str);
	return NULLSP;
    }
    this = next;
    
    type = strtos(this, &next);
    if (type == NULL) {
	eprintf("str2spec: No benchmark type in %s\n", str);
	return NULL;
    }
    if (next == this) {
	eprintf("str2spec: No benchmark type in %s\n", str);
	return NULL;
    }
    this = next;

    if (strcmp(type, "comp") == 0) {
	be = str2comp(this, &next);
    } else if (strcmp(type, "basic") == 0) {
	be = str2basic(this, &next);
    } else {
	eprintf("str2spec: unknown benchmark type: %s\n", type);
	free(type);
	return NULLSP;
    }
    free(type);
    if (be == NULLBE)
	return NULLSP;

    sp = spec_create(num, be, timelimit);

    if (Debug && next != NULL) {
	while (isspace(*next))
	    next++;
	if (*next != '\0')
	    dprintf("str2spec: remaining string %s\n", next);
    }

    *a_len = next - str;
    return sp;
    
} /* str2spec */

#endif

#ifdef SENDDEMONSPEC

/*  */

struct bench *str2comp(str, a_len)
    char *str;
    int *a_len;
{
    int len = *a_len, used = 0;
    struct bench *basic;
    char *name, *this, *next;
    int num, replicas, i;
    struct symbol *se;
    struct part *pt;

    tprintf("str2comp(0x%x,0x%x len %d) %s\n",(int)str, a_len, len, str);

    this = str;
    name = strtos(this, &next);
    if (name == NULL) {
	eprintf("str2comp: found no name in %s\n", str);
	return NULL;
    }
    if ((se = symbol_lookup(name)) == NULLSE) {
	eprintf("str2comp: name isn't in symbol table: %s\n", name);
	free(name);
	return NULL;
    }
    free(name);
    if (!symbol_iscomp(se)) {
	eprintf("str2comp: name isn't a composite benchmark: %s\n", name);
	return NULL;
    }
    
    if (next == this) {
	eprintf("str2comp: No composite name in %s\n", str);
	return NULL;
    }
    this = next;

    num = strtoi(this, &next, 10);
    if (next == this) {
	eprintf("str2comp: No number of parts in %s\n", str);
	return NULL;
    }
    this = next;
    pt = NULLPT;
    for (i = 0; i < num; i++) {
	int length;
	
	replicas = strtoi(this, &next, 10);
	if (next == this) {
	    eprintf("str2comp: No number of replicas in %s\n", str);
	    return NULL;
	}
	this = next;
	length = len - (this - str);
	basic = str2basic(this, &length);
	if (basic == NULLBE) {
	    eprintf("str2comp: No basic spec in part in %s\n", str);
	    return NULL;
	}
	if (next == this) {
	    eprintf("str2comp: No basic spec in part in %s\n", str);
	    return NULL;
	}
	next = this + length;
	this = next;
	pt = part_append(pt, part_create(replicas, basic));
    }

    *a_len = next - str;    
    return comp_create(se, pt, hostspec_create(NULL, NULL));
} /* str2comp */

#endif

#ifdef SENDDEMONSPEC

/*  */

char *comp2str(be, a_len)
    struct bench *be;
    int *a_len;
{
    static char abuf[SMALLBUF];
    static char *bstr[SMALLBUF];
    static int len_bstr[SMALLBUF];
    char *str = abuf, *res, *cstr;
    char *next;
    int parts, i;
    struct part *pt;
    int alength, length, total;

    tprintf("comp2str(0x%x, 0x%x)\n", str, a_len);
    
    if (!bench_iscomp(be)) {
	eprintf(EF_IN4, INTERNAL, PARAMETER, "comp2str",
	       "A Basic benchmark supplied");
	return NULL;
    }
    
    sprintf(str, "%s ", bench_name(be));
    tprintf("comp2str bench-name is:%s", str);
    alength = strlen(str);
    str = abuf + alength;
    if (alength >= SMALLBUF) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "SMALLBUF exceeded",
		"comp2str");
	return NULL;
    }
    
    if ((parts = comp_number(be)) == NP_UNSPEC) {
	parts = 0;
	pt = comp_parts(be);
	while (pt != NULLPT) {
	    parts++;
	    pt = part_next(pt);
	}
	comp_number(be) = parts;
    }
    if (parts >= SMALLBUF) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "Number of parts exceeded",
		"comp2str");
	return NULL;
    }
    sprintf(str, "%d ", parts);
    tprintf("comp2str: parts is added and  is:%s", abuf);
    alength += strlen(str);
    str = abuf + alength;
    if (alength >= SMALLBUF) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "SMALLBUF exceeded",
		"comp2str");
	return NULL;
    }
    
    pt = comp_parts(be);
    total = 0;
    for (i = 0; i < parts; i++) {
	
	if ((bstr[i] = basic2str(part_bench(pt), &length)) == NULL) {
	    return NULL;
	}
	len_bstr[i] = strlen(bstr[i]);
	total += len_bstr[i]+5;		/* 5 for part number, see below */
    }

    res = (char *)malloc(total+1);
    if (res == NULL) {
	eprintf(EF_IN3, RESOURCE, "No memory for malloc()",
		"comp2str");
	return NULL;
    }

    cstr = res;
    for (i = 0; i < parts; i++) {
	int len_dstr;
	char *dstr[10];
	
	sprintf(dstr, "\n%d ", part_number(pt));
	len_dstr = strlen(dstr);
	bcopy(dstr, cstr, len_dstr);
	tprintf("comp2str part-number is added:%s", res);
	cstr += len_dstr;
	bcopy(bstr[i], cstr, len_bstr[i]);
	free(bstr[i]);
	cstr += len_bstr[i] - 1;/* ignore NULL character included in len */
	pt = part_next(pt);
    }

    tprintf("comp2str leaving comp2str, res is:%s", res);
    *a_len = strlen(res);
    tprintf("comp2str leaving, alen is:%d", a_len);
    return res;
} /* comp2str */

#endif

/*  */

struct bench *str2basic(str, a_len)
    char *str;
    int *a_len;
{
    int len = *a_len;
    struct bench *be;
    char *name, *this, *next;
    int type, iterations;
    struct msize *msgsize1, *msgsize2;
    struct symbol *se;
    int acctype;
    char *filename;
    struct access *src, *dst;
    struct timing *ti;
    int endoftiming;
    double val;
    int background, priority;
    
    tprintf("str2basic(0x%x, 0x%x, len %d) %s\n",(int)str, a_len, len, str);

    this = str;
    name = strtos(this, &next);
    if (name == NULL) {
	eprintf("str2basic: found no name in %s\n", str);
	return NULL;
    }
    if ((se = symbol_lookup(name)) == NULLSE) {
	eprintf("str2basic: name isn't in symbol table: %s\n", name);
	free(name);
	return NULL;
    }
    free(name);
    if (!symbol_isbasic(se)) {
	eprintf("str2basic: name isn't a basic benchmark: %s\n", name);
	return NULL;
    }
    
    if (next == this) {
	eprintf("str2basic: No data specification<1> in %s\n", str);
	return NULL;
    }
    this = next;

    background = strtoi(this, &next, 10);
    if (next == this) {
	eprintf("str2basic: No background flag in %s\n", str);
	return NULL;
    }
    this = next;

    priority = strtoi(this, &next, 10);
    if (next == this) {
	eprintf("str2basic: No priority in %s\n", str);
	return NULL;
    }
    this = next;

    type = strtoi(this, &next, 10);
    if (next == this) {
	eprintf("str2basic: No data type in %s\n", str);
	return NULL;
    }
    this = next;

    msgsize1 = str2msize(this, &next);
    if (next == this) {
	eprintf("str2basic: No send message size in %s\n", str);
	return NULL;
    }
    this = next;

    msgsize2 = str2msize(this, &next);
    if (next == this) {
	eprintf("str2basic: No message size in %s\n", str);
	return NULL;
    }
    this = next;

    iterations = strtoi(this, &next, 10);
    if (next == this) {
	eprintf("str2basic: No iteration count in %s\n", str);
	return NULL;
    }
    this = next;

    acctype = strtoi(this, &next, 10);
    if (next == this) {
	eprintf("str2basic: No source type in %s\n", str);
	return NULL;
    }
    this = next;

    if (acctype == ACT_FILE) {
	filename = strtos(this, &next);
	
	if (next == this) {
	    eprintf("str2basic: No source filename in %s\n", str);
	    return NULL;
	}

	this = next;
    } else
	filename = NULL;
    
    src = access_create(acctype, filename);
    if (filename != NULL)
	free(filename);

    acctype = strtoi(this, &next, 10);
    if (next == this) {
	eprintf("str2basic: No destination type in %s\n", str);
	return NULL;
    }
    this = next;

    if (acctype == ACT_FILE) {
	filename = strtos(this, &next);
	
	if (next == this) {
	    eprintf("str2basic: No destination filename in %s\n", str);
	    return NULL;
	}
	this = next;
    } else
	filename = NULL;

    dst = access_create(acctype, filename);
    if (filename != NULL)
	free(filename);


    ti = NULLTI;

    while (1) {
	struct timing *ti2;
	
	endoftiming = strtoi(this, &next, 10);
	if (next == this) {
	    eprintf("str2basic: No endoftiming in %s\n", str);
	    return NULL;
	}
	this = next;
	
	dprintf("endoftiming %d\n", endoftiming);
	
	if (endoftiming == 0)
	    break;

	ti2 = str2timing(this, &next);
	if (this == next) {
	    eprintf("str2basic: found no timing in %s\n", str);
	    return NULL;
	}
	this = next;
	ti = timing_append(ti, ti2);
    }
    
    be = basic_create(se,
		      datasp_create(type,
				    sizespec_create(msgsize1, msgsize2,
						    iterations),
				    src, dst),
		      ti,
		      hostspec_create(NULL, NULL));

    if (background)
	basic_setbackground(be);
    basic_setpriority(be, priority);
    
    dprintf("str2basic: ...\n");
    if (Debug)
	bench_print(be);

    *a_len = next - str;
    
    return be;
} /* str2basic */

/*  */

char *basic2str(be, a_len)
    struct bench *be;
    int *a_len;
{
    int length, maxsize;
    static char abuf[SMALLBUF], bbuf[SMALLBUF];
    char *str, *res, *msizesend_res, *msizerec_res, *timing_res;
    int abuf_len, bbuf_len, msend_len, mrec_len, timing_len;
    struct timing *ti;
    struct param *pm;
    
    tprintf("basic2str(0x%x, 0x%x)\n", be, a_len);

    if (!bench_isbasic(be)) {
	eprintf(EF_IN4, INTERNAL, PARAMETER, "basic2str",
	       "A Composite benchmark supplied");
	return NULL;
    }

    if (basic_data(be) == NULLDT) {
	eprintf(EF_IN4, INTERNAL, PARAMETER, "basic2str",
	       "No data specification");
	return NULL;
    }
    if (datasp_size(basic_data(be)) == NULLSS) {
	eprintf(EF_IN4, INTERNAL, PARAMETER, "basic2str",
	       "No size specification");
	return NULL;
    } 

    if (access_type(datasp_src(basic_data(be))) != ACT_FILE)
	access_filename(datasp_src(basic_data(be))) = "";
    if (access_type(datasp_dst(basic_data(be))) != ACT_FILE)
	access_filename(datasp_dst(basic_data(be))) = "";
    
    sprintf(abuf, "%s %d %d %d",
	    bench_name(be),
	    basic_background(be),
	    basic_priority(be),
	    datasp_type(basic_data(be)));
    abuf_len = strlen(abuf);
    if (abuf_len >= SMALLBUF) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "SMALLBUF exceeded",
		"basic2str");
	return NULL;
    }
    msizesend_res =  msize2str(sizespec_sendmsg(datasp_size(basic_data(be))));
    msend_len = strlen(msizesend_res);
    msizerec_res =  msize2str(sizespec_recvmsg(datasp_size(basic_data(be))));
    mrec_len = strlen(msizerec_res);
    sprintf(bbuf, " %d %d %s %d %s",    
	    sizespec_iterations(datasp_size(basic_data(be))),
	    access_type(datasp_src(basic_data(be))),
	    access_filename(datasp_src(basic_data(be))),
	    access_type(datasp_dst(basic_data(be))),
	    access_filename(datasp_dst(basic_data(be))));
    bbuf_len = strlen(bbuf);
    if (bbuf_len >= SMALLBUF) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "SMALLBUF exceeded", "basic2str");
	return NULL;
    }

    ti = basic_timing(be);
    timing_len = 0;
    while (ti != NULLTI) {
	timing_res = timing2str(ti);
	timing_len += strlen(timing_res)+3;
	free(timing_res);
	ti = timing_next(ti);
    }

    /*Calculate max string for malloc */
    maxsize = abuf_len+msend_len+mrec_len+bbuf_len+timing_len+100;
    res = (char *)malloc(maxsize+1);
    if (res == NULL) {
	eprintf(EF_IN3, RESOURCE, "No memory for malloc()",
		"basic2str");
	return NULL;
    }
    tprintf("spec2str: maxsize is: %d\n", maxsize);
    
    str = res;
    bcopy(abuf, str, abuf_len);
    str +=abuf_len;
    tprintf("spec2str: res after abuf is: is: %s\n", res);
    
    bcopy(msizesend_res, str, msend_len);
    tprintf("spec2str: res after msize has copied is: %s\n", res);
    str += msend_len;
    bcopy(msizerec_res, str,  mrec_len);
    str += mrec_len;
    bcopy(bbuf, str, bbuf_len);
    str += bbuf_len;

    free(msizesend_res);
    free(msizerec_res);

    ti = basic_timing(be);
    while (ti != NULLTI) {
	timing_res = timing2str(ti);
	sprintf(str, " %d ", 1);
	str += 3;
	timing_len = strlen(timing_res);
	bcopy(timing_res, str, timing_len);
	str += timing_len;
	free(timing_res);
	ti = timing_next(ti);
    }

    tprintf("spec2str: res after timing has copied is: %s\n", res);
    sprintf(str, " %d", 0);

    tprintf("basic2str: result string is: %s\n", res);

    *a_len = strlen(res)+2;
    return res;
} /* basic2str */

/*  */

static char *timing2str(ti)
    struct timing *ti;
{
    static char buf[SMALLBUF], dbuf[SMALLBUF];
    char *bstr, *res, *str = buf;
    struct param *pm;
    int i, maxsize, length;

    sprintf(str,"%s ", timing_name(ti));
    if (strlen(str) >= SMALLBUF) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "SMALLBUF exceeded",
		    "basic2str");
	    return NULL;
	}

    pm = timing_param(ti);
#ifdef notdef
    while (pm != NULLPM) {
	str += strlen(str);
	sprintf(str, " %f", param_value(pm));
	if (strlen(buf) >= SMALLBUF) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "SMALLBUF exceeded",
		    "basic2str");
	    return NULL;
	}
	pm = param_next(pm);

    }
#endif notdef    
    str += strlen(str);

    sprintf(str, " %d", timing_nvalue(ti));
    if (strlen(buf) >= SMALLBUF) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "SMALLBUF exceeded",
		    "basic2str");
	    return NULL;
	}
    str += strlen(str);

    maxsize = 0; /* Calculate max size of string for malloc */
    
    for (i = 0; i < timing_nvalue(ti); i++) {
	sprintf(dbuf, " %f", timing_values(ti)[i]);
	maxsize += strlen(dbuf);
    }
    length = strlen(buf);
    bstr = (char *)malloc(maxsize+length+1);
    res = bstr;
    if (bstr == NULL) {
	eprintf(EF_IN3, RESOURCE, "No memory for malloc()",
		"msize2str");
	return NULL;
    }

    bcopy(buf, bstr, length+1);
    bstr += strlen(bstr);
    
    for (i = 0; i < timing_nvalue(ti); i++) {
	sprintf(bstr, " %f", timing_values(ti)[i]);
	bstr += strlen(bstr);
    }
    tprintf("timing2str:result string is:%s\n", res);
    return res;
} /* timing2str */

static struct timing *str2timing(str, left)
    char *str;
    char **left;
{
    struct timing *ti;
    char *distname;
    struct symbol *distse;
    int nvalue, i;
    double *values;
    struct param *pm;
    char *this = str;
    char *next;
    
    distname = strtos(this, &next);
    if (distname == NULL) {
	eprintf("str2timing: found no distname in %s\n", str);
	*left = str;
	return NULL;
    }
    if ((distse = symbol_lookup(distname)) == NULLSE) {
	eprintf("str2basic: name isn't in symbol table: %s\n", distname);
	free(distname);
	*left = str;
	return NULL;
    }
    free(distname);
    if (!symbol_isdist(distse)) {
	eprintf("str2basic: name isn't a distribution: %s\n", distname);
	*left = str;
	return NULL;
    }
    
    this = next;
    
    pm = NULLPM;
#ifdef notdef
    val = strtod(this, &next);
    while (this != next) {
	pm = param_append(pm, param_create(val));
	this = next;
	val = strtod(this, &next);
    }
#endif notdef

    nvalue = strtoi(this, &next, 10);
    if (next == this) {
	eprintf("str2timing: No nvalue in %s\n", str);
	*left = str;
	return NULL;
    }
    this = next;

    values = (double *)malloc(nvalue * sizeof(double));
    if (values == NULL) {
	eprintf(EF_IN3, RESOURCE, "No memory for malloc()",
		"str2timing");
	*left = str;
	return NULL;
    }
    for (i = 0; i < nvalue; i++) {
	values[i] = strtod(this, &next);
	if (next == this) {
	    eprintf("str2timing: Missing value in %s\n", str);
	    *left = str;
	    return NULL;
	}
	this = next;
    }
    
    ti = timing_create(dist_create(distse, pm), nvalue, values);

    *left = next;

    return ti;
} /* str2timing */

/*  */

static char *msize2str(ms)
    struct msize *ms;
{
    static char buf[SMALLBUF], dbuf[SMALLBUF];
    char *bstr, *res, *str = buf;
    struct param *pm;
    int i, maxsize, length;

    if (msize_dist(ms) == NULLDI) {
	sprintf(str," 0 %d ", msize_value(ms, 0));
	res = (char *)malloc(strlen(str)+1);
	if (res == NULL) {
	    eprintf(EF_IN3, RESOURCE, "No memory for malloc()",
		    "msize2str");
	    return NULL;
	}
	bcopy(str, res, strlen(str)+1);
	
    } else {
	sprintf(str," 1 %s ", msize_name(ms));
	length = strlen(buf);
	if (length >= SMALLBUF) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "SMALLBUF exceeded",
		    "basic2str");
	    return NULL;
	}
	pm = msize_param(ms);
#ifdef notdef
	while (pm != NULLPM) {
    	    str += strlen(str);
	    sprintf(str, " %f", param_value(pm));
	    pm = param_next(pm);
	}
#endif notdef
	str += strlen(str);

	sprintf(str, " %d", msize_nvalue(ms));
	str += strlen(str);
	if (strlen(buf) >= SMALLBUF) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "SMALLBUF exceeded",
		    "basic2str");
	    return NULL;
	}

	maxsize = 0; /* Calculate max size of string for malloc */
	for (i = 0; i < msize_nvalue(ms); i++) {
	    sprintf(dbuf, " %d", msize_values(ms)[i]);
	    maxsize + = strlen(dbuf);
	}
	length = strlen(buf);
	bstr = (char *)malloc(maxsize+length+1);
	res = bstr;
	if (bstr == NULL) {
	    eprintf(EF_IN3, RESOURCE, "No memory for malloc()",
		    "msize2str");
	    return NULL;
	}
	
	bcopy(buf, bstr, length+1);
	tprintf("msize2str:msize_name in bstr is:%s\n", bstr);
	bstr += strlen(bstr);

	for (i = 0; i < msize_nvalue(ms); i++) {
	    sprintf(bstr, " %d", msize_values(ms)[i]);
	    bstr += strlen(bstr);
	}
    
    }
    tprintf("msize2str: return result is:%s\n", res);
    return res;
} /* msize2str */

static struct msize *str2msize(str, left)
    char *str;
    char **left;
{
    struct msize *ms;
    int type;
    char *distname;
    struct symbol *distse;
    int nvalue, i, constant;
    int *values;
    struct param *pm;
    char *this = str;
    char *next;

    type = strtoi(this, &next, 10);
    if (next == this) {
	eprintf("str2msize: found no type in %s\n", str);
	*left = str;
	return NULL;
    }
    this = next;
    if (type == 0) {
	/* constant */
	constant = strtoi(this, &next, 10);
	if (next == this) {
	    eprintf("str2msize: found no constant in %s\n", str);
	    *left = str;
	    return NULL;
	}
	this = next;
	ms = msize_create(MS_CONSTANT, constant, NULL, 0, NULL);
    } else {
	/* distribution case */
	
	distname = strtos(this, &next);
	if (distname == NULL) {
	    eprintf("str2msize: found no distname in %s\n", str);
	    *left = str;
	    return NULL;
	}
	if ((distse = symbol_lookup(distname)) == NULLSE) {
	    eprintf("str2basic: name isn't in symbol table: %s\n", distname);
	    free(distname);
	    *left = str;
	    return NULL;
	}
	free(distname);
	if (!symbol_isdist(distse)) {
	    eprintf("str2basic: name isn't a distribution: %s\n", distname);
	    *left = str;
	    return NULL;
	}
	
	this = next;
	
	pm = NULLPM;
#ifdef notdef
	val = strtod(this, &next);
	while (this != next) {
	    pm = param_append(pm, param_create(val));
	    this = next;
	    val = strtod(this, &next);
	}
#endif notdef
	
	nvalue = strtoi(this, &next, 10);
	if (next == this) {
	    eprintf("str2msize: No nvalue in %s\n", str);
	    *left = str;
	    return NULL;
	}
	this = next;
	
	values = (int *)malloc(nvalue * sizeof(int));
	if (values == NULL) {
	    eprintf(EF_IN3, RESOURCE, "No memory for malloc()",
		    "str2msize");
	    *left = str;
	    return NULL;
	}
	for (i = 0; i < nvalue; i++) {
	    values[i] = strtoi(this, &next, 10);
	    if (next == this) {
		eprintf("str2msize: Missing value in %s\n", str);
		*left = str;
		return NULL;
	    }
	    this = next;
	}
	
	ms = msize_create(MS_DIST, 0, dist_create(distse, pm), nvalue, values);
    }
    if (ms == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "str2msize");
	*left = str;
	return NULL;
    }

    if (Debug)
	msize_print(ms);
    
    *left = next;

    return ms;
} /* str2msize */

/*  */

char *statistics2str(stat, len)
    statistics_t *stat;
    int *len;
{
    static char str[BIGBUF];
    char *next, *res, *this;
    int length = 0;
    
    tprintf("statistics2str(0x%x, 0x%x)\n", stat, len);

    this = str;
    
    /* work */
#define write_float(val) \
    sprintf(this, "%f ", val); \
    this += strlen(this); 

    write_float(stat->st_exitstatus);
    write_float(stat->st_starttime);
    write_float(stat->st_stoptime);
    write_float(stat->st_elapsed);
    write_float(stat->st_cputime);
    write_float(stat->st_stime);
    write_float(stat->st_utime);
    write_float(stat->st_delay);
    write_float(stat->st_datasent);
    write_float(stat->st_datarcvd);
    write_float(stat->st_msgsent);
    write_float(stat->st_msgrcvd);
    write_float(stat->st_packsent);
    write_float(stat->st_packrcvd);
    write_float(stat->st_signals);
    write_float(stat->st_blockout);
    write_float(stat->st_blockin);
    write_float(stat->st_pageio);
    write_float(stat->st_pagerecl);
    write_float(stat->st_swaps);
    write_float(stat->st_ivcsw);
    write_float(stat->st_vcsw);
    write_float(stat->st_maxrss);

    length = this - str;
    if (length >= BIGBUF) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "BIGBUF exceeded",
		"statistics2str");
	return NULL;
    }

    res = (char *)malloc(length+1);
    if (res == NULL) {
	eprintf(EF_IN3, RESOURCE, "No memory for malloc()",
		"statistics2str");
	return NULL;
    }
	
    bcopy(str, res, length+1);

    *len = length+1;
    return res;
} /* statistics2str */

/*  */

statistics_t *str2statistics(str, a_len)
    char *str;
    int *a_len;
{
    int len = *a_len;
    statistics_t *stat;
    int num;
    char *this, *next;

    tprintf("str2statistics(0x%x, 0x%x, len %d)\n", str, a_len, len);

    stat = (statistics_t *)malloc(sizeof(statistics_t));
    if (stat == NULL) {
	eprintf(EF_IN3, RESOURCE, "No memory for malloc()",
		"str2statistics");
	return NULL;
    }

    this = str;

#define read_float(dest) \
    dest = strtod(this, &next); \
    if (next == this) { \
	eprintf(EF_IN3, INTERNAL, PROTOCOL, "Bad number while reading dest", \
		"str2statistics"); \
	free((char *)stat); \
	return NULL; \
    } \
    this = next;
	
    /* work */

    read_float(stat->st_exitstatus);
    read_float(stat->st_starttime);
    read_float(stat->st_stoptime);
    read_float(stat->st_elapsed);
    read_float(stat->st_cputime);
    read_float(stat->st_stime);
    read_float(stat->st_utime);
    read_float(stat->st_delay);
    read_float(stat->st_datasent);
    read_float(stat->st_datarcvd);
    read_float(stat->st_msgsent);
    read_float(stat->st_msgrcvd);
    read_float(stat->st_packsent);
    read_float(stat->st_packrcvd);
    read_float(stat->st_signals);
    read_float(stat->st_blockout);
    read_float(stat->st_blockin);
    read_float(stat->st_pageio);
    read_float(stat->st_pagerecl);
    read_float(stat->st_swaps);
    read_float(stat->st_ivcsw);
    read_float(stat->st_vcsw);
    read_float(stat->st_maxrss);

    if (Debug && next != NULL) {
	while (isspace(*next))
	    next++;
	if (*next != '\0')
	    dprintf("str2statistics: remaining string %s\n", next);
    }

    if (Debug) {
	int max = sizeof(statistics_t)/sizeof(double);
	int i;
	double *dp = (double *)stat;

	for (i = 0; i < max; i++)
	    printf("%.3f ", dp[i]);
	printf("\n");
    }
    
    *a_len = next - str;
    return stat;
} /* str2statistics */

/*  */

double str2time(str, a_len)
    char *str;
    int *a_len;
{
    int len = *a_len, used = 0;
    char *this, *next;
    int sec, usec;
    double res;

    tprintf("str2time(0x%x,0x%x len %d) %s\n",(int)str, a_len, len, str);

    this = str;

    sec = strtoi(this, &next, 10);
    if (next == this) {
	eprintf("str2time: No seconds in %s\n", str);
	return NULL;
    }
    this = next;

    usec = strtoi(this, &next, 10);
    if (next == this) {
	eprintf("str2time: No micro seconds in %s\n", str);
	return NULL;
    }
    this = next;

    used = this - str;
    res = sec + ((double)usec)/1000000;

    dprintf("str2time: read %f\n", res);
    *a_len = used;
    return res;
} /* str2time */
    
/*  */

/*
 * This routine scans space terminated strings.
 * Parameter semantics are the same as for strtol(3).
 * NOTE: the result is a malloc'ed copy of the original string.
 */
char *strtos(str, ptr)
    char *str, **ptr;
{
    register char *last, *dest;
    register int len;
    
    while (isspace(*str))
	str++;
    if (*str == '\0') {
	last = str;
	dest = NULL;
    } else {
	last = str;
	len = 0;
	while (!isspace(*last)) 
	    last++, len++;
	dest = (char *)malloc(len+1);
	if (dest == NULL) {
	    fprintf(stderr, "Malloc failed in strtos\n");
	    exit(1);
	}
	strncpy(dest, str, len);
	dest[len] = '\0';
    }
    if (ptr != NULL)
    	*ptr = last;
    return dest;
} /* strtos */
