%{
/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */

/* Copyright (c) 1989, Swedish Institute of Computer Science */

#include <stdio.h>
#include <symbol.h>
#include <spec.h>

extern int Debug;
extern struct spec *spec;
%}

%start specifications

%union {
	struct spec 	*uspec;
	struct bench	*ubench;
	struct part	*upart;
	struct datasp	*udatasp;
	struct access	*uaccess;
	struct sizespec	*usizespec;
	struct timing	*utiming;
	struct msize	*umsize;
	struct dist	*udist;
	struct param 	*uparam;
	struct symbol 	*usymbol;
	struct hostspec *uhostspec;
	char 		*ustring;
	int		uint;
	double		ufloat;
}

%token <uint> INTEGER DELAY ARRIVAL BYTES ASCII INT FILENAME MEMORY VIRTUAL
%token <uint> TIMELIMIT FROM TO
%token <ufloat> FLOAT 
%token <ustring> STRING 

%type <uspec> specifications specification
%type <ubench> benchmark_spec basic_benchmark composite_benchmark
%type <upart> parts part
%type <udatasp> data_spec
%type <uaccess> source destination access
%type <usizespec> size_spec 
%type <uhostspec> host_spec
%type <utiming> timing_list timing
%type <umsize> message_size 
%type <udist> distribution
%type <uparam> parameters parameter	
%type <usymbol> basic_name composite_name distribution_name
%type <ustring> file_specification hostname
%type <uint> repetitions num_replicas 
%type <uint> num_messages data_volume data_type
%type <ufloat> 


%% /* beginning of the rules */


specifications :
	/* empty */			{ $$ = NULLSP; spec = $$; }
	| specifications specification	{ $$ = spec_append($1, $2);
					  spec = $$; }
;

specification :
		repetitions benchmark_spec ';'
					{ $$ = spec_create($1, $2, TL_UNLIM); }
	|	repetitions TIMELIMIT FLOAT benchmark_spec ';'
					{ $$ = spec_create($1, $4, $3); }
	|	repetitions TIMELIMIT INTEGER benchmark_spec ';'
				{ $$ = spec_create($1, $4, (double)$3); }
	| 	error ';'
		{ 	yyerrok;
			fprintf(stderr, "Bad input!\n");
		}
		specification
					{ $$ = $4; }
;

benchmark_spec :
	basic_benchmark		{ $$ = $1;}
	| composite_benchmark	{ $$ = $1;}
;

host_spec :
        /* empty */	{ $$ = hostspec_create(NULL, NULL); }
	| FROM hostname { $$ = hostspec_create($2, NULL); string_free($2); }
	| TO hostname	{ $$ = hostspec_create(NULL, $2); string_free($2); }
	| FROM hostname TO hostname	{ $$ = hostspec_create($2, $4);
				       string_free($2); string_free($4);}
	| TO hostname FROM hostname	{  $$ = hostspec_create($4, $2);
				       string_free($2); string_free($4);}
;

basic_benchmark :
	host_spec basic_name data_spec timing_list 
		{ $$ = basic_create($2, $3, $4, $1); }
	| host_spec basic_name data_spec 
		{ $$ = basic_create($2, $3, NULLTI, $1); }
;

composite_benchmark :
	host_spec composite_name ':' parts
					{ $$ = comp_create($2, $4, $1); }
;

parts :
	part
	| parts '|' part		{ $$ = part_append($1, $3); }
;

part :
	num_replicas basic_benchmark	{ $$ = part_create($1, $2); }
	| basic_benchmark		{ $$ = part_create(1, $1); }
;			

data_spec :
	data_type size_spec source destination
					{ $$ = datasp_create($1, $2, $3, $4); }
;

source :
	/* empty */			{ $$ = access_create(ACT_MEM, NULL); }
	| '<' access			{ $$ = $2; }
;

destination :
	/* empty */			{ $$ = access_create(ACT_MEM, NULL); }
	| '>' access			{ $$ = $2; }
;

access :
	FILENAME file_specification	{ $$ = access_create(ACT_FILE, $2); }
	| MEMORY			{ $$ = access_create(ACT_MEM, NULL); }
	| VIRTUAL MEMORY		{ $$ = access_create(ACT_VM, NULL); }
;

size_spec :
	message_size '*' num_messages	{ $$ = sizespec_create($1, $1, $3);}
	| message_size ',' message_size '*' num_messages
					{ $$ = sizespec_create($1, $3, $5);}
/* 	| message_size data_volume	{ $$ = sizespec_create($1, $1, $2/$1);} */
	| message_size 			{ $$ = sizespec_create($1, $1, SS_DEFAULT);}
;

message_size  :
    INTEGER			{ $$ = msize_create(MS_CONSTANT, $1, NULLDI); }
    | distribution		{ $$ = msize_create(MS_DIST, 0, $1, 0, NULL);
			      	  msize_generate($$); }
;

timing_list :
	timing			{ $$ = $1; }
	| timing_list ',' timing	{ $$ = timing_append($1, $3); }
;

    	
timing :
	distribution			{ $$ = timing_create($1, 0, NULL);
				          timing_generate($$); }
;

distribution :
	distribution_name '(' parameters ')'
    				{ $$ = dist_create($1, $3); }

distribution_name : 	STRING
	{ $$ = symbol_lookup($1);
	  if ($$ == NULLSE) {
		fprintf(stderr, "Unknown name: %s\n", $1);
		string_free($1);
		YYERROR;
	  }
	  if ($$->se_tag != SET_DIST) {
		fprintf(stderr, "Not the name of a distribution: %s\n", $1);
		string_free($1);
		YYERROR;
	  }
	  string_free($1);
	}
;

parameters :
	parameter
	| parameters ',' parameter	{ $$ = param_append($1, $3); }
;

data_type :
	/* empty */			{ $$ = BYTES; }
	| BYTES				{ $$ = BYTES; }
	| ASCII				{ $$ = ASCII; }
	| INT				{ $$ = INT; }
;

file_specification : STRING
;

hostname : STRING
;    

repetitions : INTEGER
;
num_replicas : INTEGER
;
num_messages  : INTEGER
;
data_volume  : INTEGER
;
parameter :
	FLOAT			{ $$ = param_create($1); }
	| INTEGER			{ $$ = param_create((double)$1); }
;

basic_name : STRING
	{ $$ = symbol_lookup($1);
	  if ($$ == NULLSE) {
		fprintf(stderr, "Unknown name: %s\n", $1);
		string_free($1);
		YYERROR;
	  }
	  if ($$->se_tag != SET_BASIC) {
		fprintf(stderr, "Not the name of a basic benchmark: %s\n", $1);
		string_free($1);
		YYERROR;
	  }
	  string_free($1);
	}
;

composite_name : STRING
	{ $$ = symbol_lookup($1);
	  if ($$ == NULLSE) {
		fprintf(stderr, "Unknown name: %s\n", $1);
		string_free($1);
		YYERROR;
	  }
	  if ($$->se_tag != SET_COMP) {
		fprintf(stderr, "Not the name of a composite benchmark: %s\n",
			$1);
		string_free($1);
		YYERROR;
	  }
	  string_free($1);
	}
;

%%
    
extern int yychar;	/* lookahead token */

/*
 * Yacc parsing errors
 */
yyerror(s)
    char *s;
{
    fprintf(stderr, "Input syntax error: %s with lookahead token %d\n",
	    s, yychar);
}
    
