/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * Header file for benchmarks on the TSAP protocol
 */

#ifndef __TSAP__
#define __TSAP__

#include <isode/tsap.h>

#include "../protoaddrs/tsapaddr.h"

/* A channel is an assocation descriptor */
typedef int 		channel_t;
#define server_t	address_t	/* a struct TSAPaddr */


/*
 * Error handling
 */
typedef struct {
    int er_ret;
    struct TSAPdisconnect er_td;
} error_t;

#define failed(errind)	((errind)->er_ret == NOTOK)

#define report_error(errind,str)	tsap_report_error(errind,str)


#define InitClient()		tsap_init_client()

#define ConnRequest(addr,a_ch,errind) \
    				tsap_conn_request(addr, a_ch, errind)
#define DiscRequest(ch, errind) \
    				tsap_disc_request(ch, errind)
#define AwaitConnInd(serverp, a_ch, errind) \
			    	tsap_await_conn_ind(serverp, a_ch, errind)

#define AwaitDiscInd(ch, errind) \
    				tsap_await_disc_ind(ch, errind)



#define CreateServer(aa_server, aa_addr, errind) \
    				tsap_create_server(aa_server, aa_addr, errind)
#define DestroyServer(a_server, errind) \
    				tsap_destroy_server(a_server, errind)

#define DataRequest(channel,buffer,amount, errind) \
				tsap_data_request(channel,buffer,amount,errind)

#define AwaitDataIndication(channel, buffer, amount, errind)	\
			tsap_await_data_ind(channel,buffer,amount,errind)

#define BulkGetDataReq		DataRequest
#define BulkPutDataReq  	DataRequest
#define BulkGetAwaitDataInd	AwaitDataIndication
#define BulkPutAwaitDataInd	AwaitDataIndication

#define RRSendRequest 		BulkPutDataReq
#define RRAwaitResponseInd	BulkGetAwaitDataInd
#define RRAwaitRequestInd	BulkPutAwaitDataInd
#define RRSendResponse		BulkGetDataReq

/*
 * Synchronize with the sender to avoid buffering anomalies (the
 * sender starting before this receiver and the receiver just retrieves
 * data from the buffers!)
 */

#define BulkGetStart(ch, errind) \
{	\
    char buffer;	/* small buffer */	\
    DataRequest(ch, &buffer, 1, errind);	\
}

#define AwaitBulkGetStart(ch, errind)	\
{	\
    char buffer;	/* small buffer */	\
    AwaitDataIndication(ch, &buffer, 1, errind);	\
}

#define BulkGetStop(ch, errind) 	\
	    			{(errind)->er_ret = OK; }

#define AwaitBulkGetStop(ch, errind)	\
	    			{(errind)->er_ret = OK; }



#define BulkPutStart(ch, errind)	\
	    			{(errind)->er_ret = OK; }

#define AwaitBulkPutStart(ch, errind)\
	    			{(errind)->er_ret = OK; }

#define BulkPutStop(ch, errind)	\
{	\
    char buffer;	/* small buffer */	\
    AwaitDataIndication(ch, &buffer, 1, errind);	\
}

#define AwaitBulkPutStop(ch, errind)	\
{	\
    char buffer;	/* small buffer */	\
    DataRequest(ch, &buffer, 1, errind);	\
}


#define QueryCall(addrp, srcbuf, srclen, dstbuf, dstlen, errind) \
{ \
  channel_t ch; \
 \
  ConnRequest(addrp, &ch, errind); \
  if (!failed(errind)) { \
    RPCCall(ch, srcbuf, srclen, dstbuf, dstlen, errind); \
    if (!failed(errind)) { \
      DiscRequest(ch, errind); \
    } \
  } \
}

#define QueryAwaitCallInd(serverp, dstbuf, dstlen, chp, errind) \
{ \
    AwaitConnInd(serverp, chp, errind); \
	if (!failed(errind)) { \
	RPCAwaitCallInd(*(chp), dstbuf, dstlen, errind); \
    } \
}

#define QueryReturn(ch, srcbuf, srclen, errind) \
{ \
    RPCReturn(ch, srcbuf, srclen, errind); \
    if (!failed(errind)) { \
	AwaitDiscInd(ch, errind); \
    } \
}

#define RPCCall(ch, srcbuf, srclen, dstbuf, dstlen, errind) \
{ \
      DataRequest(ch, srcbuf, srclen, errind); \
      if (!failed(errind)) { \
	     AwaitDataIndication(ch, dstbuf, dstlen, errind); \
	     if (failed(errind)) { \
		report_error(errind, "AwaitDataIndication in RPCCall"); \
		return NOTOK; \
	     } \
      } \
}

#define RPCAwaitCallInd(ch, dstbuf, dstlen, errind) \
{ \
    AwaitDataIndication(ch, dstbuf, dstlen, errind); \
}

#define RPCReturn(ch, srcbuf, srclen, errind) \
{ \
    DataRequest(ch, srcbuf, srclen, errind); \
}

#endif __TSAP__
