/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * Header file for benchmarks on the ISO ROS protocol
 */

#ifndef __ISOROS__
#define __ISOROS__

#include <isode/rosap.h>
#include <isode/tsap.h>

/* Address structure definition */
#include "../protoaddrs/psapaddr.h"

/* A channel is an assocation descriptor */
typedef int 		channel_t;
#define server_t	address_t	/* a struct PSAPaddr */

#define INF_WAIT NOTOK

#define ROS_NOOP	0		/* New parameter to RoResultRequest */

/*
 * Error handling
 */
typedef struct {
    int ret;
    int tag;
    union {
	struct AcSAPindication eru_aci;
	struct RoSAPindication eru_roi;
	struct TSAPdisconnect eru_td;
    } er_u;
#define aci	er_u.eru_aci
#define roi	er_u.eru_roi
#define td	er_u.eru_td    
} error_t;

#define ERT_NONE	0	/* The error need no further attention */
#define ERT_ACI		1
#define ERT_ROI		2
#define ERT_TD		3

#define Aci(ERR) (&(ERR)->aci)
#define Aca_reason(ERR) ((ERR)->aci.aci_abort.aca_reason)
#define Roi(ERR) (&(ERR)->roi)
#define Rop_reason(ERR) ((ERR)->roi.roi_preject.rop_reason)
#define Td(ERR) (&(ERR)->td)
#define Td_reason(ERR) ((ERR)->td.td_reason)


#define failed(errind)	((errind)->ret == NOTOK)

#define report_error(errind,str)	isoros_report_error(errind,str)


#define InitClient	isoros_init_client

#define ConnRequest(addr,a_ch,errind) \
    			isoros_conn_request(addr, a_ch, errind)
#define DiscRequest(ch, errind) \
    			isoros_disc_request(ch, errind)
#define AwaitConnInd(serverp, a_ch, errind) \
			isoros_await_conn_ind(serverp, a_ch, errind)

#define AwaitDiscInd(ch, errind) \
    			isoros_await_disc_ind(ch, errind)



#define CreateServer(aa_server, aa_addr, errind) \
    				isoros_create_server(aa_server, aa_addr, errind)
#define DestroyServer(a_server, errind) \
    				isoros_destroy_server(a_server, errind)

#define DataRequest(channel,buffer,amount, errind) \
			isoros_data_request(channel,buffer,amount,errind)

#define AwaitDataIndication(channel, buffer, amount, errind)	\
			isoros_await_data_ind(channel,buffer,amount,errind)

#define BulkGetDataReq		DataRequest
#define BulkPutDataReq  	DataRequest
#define BulkGetAwaitDataInd	AwaitDataIndication
#define BulkPutAwaitDataInd	AwaitDataIndication

#define RRSendRequest 		BulkPutDataReq
#define RRAwaitResponseInd	BulkGetAwaitDataInd
#define RRAwaitRequestInd	BulkPutAwaitDataInd
#define RRSendResponse		BulkGetDataReq

#define ISOROS_Data		50
#define ISOROS_BulkGetStart	51
#define ISOROS_BulkGetStop	52
#define ISOROS_BulkPutStart	53
#define ISOROS_BulkPutStop	54
#define ISOROS_RPC		55

extern void isoros_await_operation_call();

#define BulkGetStart(ch, errind) \
  isoros_bulk_general(ch, "isoros_BulkGetStart", \
		      ISOROS_BulkGetStart, errind)
#define BulkGetStop(ch, errind) \
  isoros_bulk_general(ch, "isoros_BulkGetStop", \
		      ISOROS_BulkGetStop, errind)
#define AwaitBulkGetStart(ch, errind) \
  isoros_await_general(ch, "isoros_AwaitBulkGetStart", \
		       ISOROS_BulkGetStart, errind)
#define AwaitBulkGetStop(ch, errind) \
  isoros_await_general(ch, "isoros_AwaitBulkGetStop", \
		       ISOROS_BulkGetStop, errind)
#define BulkPutStart(ch, errind) \
  isoros_bulk_general(ch, "isoros_BulkPutStart", \
		      ISOROS_BulkPutStart, errind)
#define BulkPutStop(ch, errind) \
  isoros_bulk_general(ch, "isoros_BulkPutStop", \
		      ISOROS_BulkPutStop, errind)
#define AwaitBulkPutStart(ch, errind) \
  isoros_await_general(ch, "isoros_AwaitBulkPutStart", \
		       ISOROS_BulkPutStart, errind)
#define AwaitBulkPutStop(ch, errind) \
  isoros_await_general(ch, "isoros_AwaitBulkPutStop", \
		       ISOROS_BulkPutStop, errind)

#define isoros_await_general(ch, name, opcode, errind) \
{ \
  struct RoSAPinvoke *res; \
 \
  pprintf("%s(%d, %#x)\n", name, ch, errind); \
  isoros_await_operation_call(opcode, name, ch, errind, &res); \
  if ((errind)->ret == NOTOK) return; \
  isoros_do_operation_reply(ch, res->rox_id, NULLPE, name, errind); \
  ROXFREE(res); \
  pprintf("<%s\n", name); \
} /* isoros_await_general */

#define isoros_bulk_general(ch, name, id, errind) \
{ \
  pprintf("%s(%#x, %d, %#x)\n", name, ch, id, errind); \
  (errind)->ret = RoInvokeRequest(ch, id, ROS_SYNC, NULLPE, id, NULLIP, \
				  ROS_NOPRIO, Roi(errind)); \
  if ((errind)->ret == NOTOK) { \
    eprintf(EF_IN3, COMMUNICATION, "RoInvokeRequest", name); \
    (errind)->tag = ERT_ROI; \
  } else \
    pprintf("<%s\n", name); \
} /* isoros_bulk_general */


#define QueryCall(addrp, srcbuf, srclen, dstbuf, dstlen, errind) \
{ \
  channel_t ch; \
 \
  ConnRequest(addrp, &ch, errind); \
  if (!failed(errind)) { \
    RPCCall(ch, srcbuf, srclen, dstbuf, dstlen, errind); \
    if (!failed(errind)) { \
      DiscRequest(ch, errind); \
    } \
  } \
}

#define QueryAwaitCallInd(serverp, dstbuf, dstlen, chp, errind) \
{ \
    AwaitConnInd(serverp, chp, errind); \
	if (!failed(errind)) { \
	RPCAwaitCallInd(*(chp), dstbuf, dstlen, errind); \
    } \
}

#define QueryReturn(ch, srcbuf, srclen, errind) \
{ \
    RPCReturn(ch, srcbuf, srclen, errind); \
    if (!failed(errind)) { \
	AwaitDiscInd(ch, errind); \
    } \
}

#define RPCCall(ch, srcbuf, srclen, dstbuf, dstlen, errind) \
  isoros_rpc_call(ch, srcbuf, srclen, dstbuf, dstlen, errind)

extern int RPCid;		/* Not very neat (should be in channel_t) */
extern void isoros_await_operation_call();

#define RPCAwaitCallInd(ch, dstbuf, dstlen, errind) \
{ struct RoSAPinvoke *res; \
  int length; \
  char *str; \
  pprintf("isoros_RPCAwaitCallInd(%#x, %#x:%d, %#x)\n", \
	  ch, dstbuf, dstlen, errind); \
  isoros_await_operation_call(ISOROS_RPC, "isoros_RPCAwaitCallInd", \
			      ch, errind, &res); \
  RPCid = res->rox_id; \
  str = prim2str(res->rox_args, &length); \
  bcopy(str, dstbuf, dstlen); \
  if (str) free(str); \
  ROXFREE(res); \
  pprintf("<isoros_RPCAwaitCallInd\n"); \
}

#define RPCReturn(ch, srcbuf, srclen, errind) \
{ \
  PE pe; \
  pprintf("isoros_RPCReturn(%d, %#x:%d, %#x)\n", ch, srcbuf, srclen,errind); \
  isoros_do_operation_reply(ch, RPCid, pe = oct2prim(srcbuf, srclen), \
			    "isoros_RPCReturn", errind); \
  pe_free(pe); \
}

#define isoros_do_operation_reply(ch, id, pe, name, errind) \
  (errind)->ret = RoResultRequest(ch, id, ROS_NOOP, pe, ROS_NOPRIO, Roi(errind)); \
  if ((errind)->ret == NOTOK) { \
    (errind)->tag = ERT_ROI; \
    eprintf(EF_IN3, COMMUNICATION, "RoResultRequest failed", name); \
  }

#endif __ISOROS__
