#define HSI_DEBUG
#define HSI_TIME

#ifndef KERNEL
#define KERNEL
#endif
#ifndef INET
#define INET
#endif

#include <sun4c/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <sys/mbuf.h>
#include <sys/errno.h>
#include <sys/time.h>
#include <net/if.h>
#include <net/netisr.h>
#include <sun/openprom.h>
#include <sun4c/clock.h>
#include "syncmode.h"

#define NULL 0
#define BUF_SIZE 2048
#define PCLK_FREQ 9830400

#ifdef HSI_DEBUG
#define DEBUG(FLAG, ACTION) { \
	if (hsi_debug & (FLAG)) { \
		ACTION; \
	} \
}
#define D_DUMP_ISCC		(1 << 0)
#define D_IOCTL			(1 << 1)
#define D_INIT			(1 << 2)
#define D_INTERRUPT		(1 << 3)
#define D_DUMP_ISCC_ON_INTR	(1 << 4)
#define D_EXTERNAL_STATUS	(1 << 5)
#define D_RX_STATUS_FIFO	(1 << 6)
#define D_RX_ERROR		(1 << 7)
#define D_DUMP_ISCC_ON_RX_ERROR	(1 << 8)
#define D_RX_BUFFER		(1 << 9)
#define D_RX_TRACE		(1 << 10)
#define D_TX_ERROR		(1 << 11)
#define D_TX_BUFFER_STATUS	(1 << 12)
#define D_TX_BUFFER		(1 << 13)
#define D_TX_TRACE		(1 << 14)
#define D_TX_SPECIAL		(1 << 15)
#define D_EXT_STAT		(1 << 16)
#define D_TRACE			(1 << 17)
#else
#define DEBUG(FLAG, ACTION)
#endif

#ifdef HSI_TIME
#define TIME_SIZE 512
#define TIME(EVENT) { \
	htp->ht_time = time; \
	htp->ht_counter = COUNTER->counter10; \
	htp->ht_event = EVENT; \
	if (++htp == &hsi_time[TIME_SIZE]) \
		htp = hsi_time; \
}
#else
#define TIME(EVENT)
#endif

static char hsi_version[] = "working";

static int nhsi;

int hsi_identify(), hsi_attach();

struct dev_ops hsi_ops = {
	1,
	hsi_identify,
	hsi_attach
};

struct hsi_iscc_regs {
	u_char hsi_dma_cmd_reg;
	u_char hsi_scc_cmd_reg_chan_b;
	u_char hsi_iscc_pad_0;
	u_char hsi_scc_cmd_reg_chan_a;
	u_char hsi_iscc_pad_1[0xc];
};

#define hsi_iscc_bcr hsi_dma_cmd_reg

struct hsi_latch_regs {
	u_char hsi_latch_ext_rx_intr_status;
	u_char hsi_latch_pad_0[0xf];
	u_char hsi_latch_rx_clk[4];
	u_char hsi_latch_rx_data[4];
	u_char hsi_latch_pad_1[0x8];
	u_char hsi_latch_tx_data[4];
	u_char hsi_latch_tx_clk[4];
	u_char hsi_latch_pad_2[0x8];
	u_char hsi_latch_internal_clk[4];
	u_char hsi_latch_enable_ext_rx_intr[4];
	u_char hsi_latch_pad_3[0x8];
	u_char hsi_latch_reset_ext_rx_intr[4];
	u_char hsi_latch_pad_4[0xc];
	u_char hsi_latch_status;
	u_char hsi_latch_pad_5[0xf];
	u_char hsi_latch_reset_ab;
	u_char hsi_latch_pad_6[0xf];
	u_char hsi_latch_reset_cd;
};

struct hsi_regs {
	/*		 ISCC 1/2			*/
	/*		 |  channel A/B			*/
	/*		 |  |  transmit/receive		*/
	/*		 |  |  |  buffer 0/1		*/
	/*		 |  |  |  | 			*/
	u_char (*hr_ram)[2][2][2][2][BUF_SIZE];
	struct hsi_iscc_regs *hr_iscc_regs[2];
	struct hsi_latch_regs *hr_latch_regs;
} *hsi_regs;

#define BITS(B7, B6, B5, B4, B3, B2, B1, B0) (B7 * 0x80 | B6 * 0x40 | B5 * 0x20 | B4 * 0x10 | B3 * 0x08 | B2 * 0x04 | B1 * 0x02 | B0)
#define D0 0x01
#define D1 0x02
#define D2 0x04
#define D3 0x08
#define D4 0x10
#define D5 0x20
#define D6 0x40
#define D7 0x80

struct scc_init_table_entry {
	int site_reg;
	int site_val;
};

struct scc_init_table {
	struct scc_init_table_entry sit_scc_wr9_1a;
	struct scc_init_table_entry sit_scc_wr4_1;
	struct scc_init_table_entry sit_scc_wr1_1;
	struct scc_init_table_entry sit_scc_wr3_1;
	struct scc_init_table_entry sit_scc_wr5_1;
	struct scc_init_table_entry sit_scc_wr7_1;
	struct scc_init_table_entry sit_scc_wr9_1b;
	struct scc_init_table_entry sit_scc_wr10_1;
	struct scc_init_table_entry sit_scc_wr11_1;
	struct scc_init_table_entry sit_scc_wr12_1;
	struct scc_init_table_entry sit_scc_wr13_1;
	struct scc_init_table_entry sit_scc_wr14_1a;
	struct scc_init_table_entry sit_scc_wr14_1b;
	struct scc_init_table_entry sit_scc_wr14_2;
	struct scc_init_table_entry sit_scc_wr3_2;
	struct scc_init_table_entry sit_scc_wr5_2;
	struct scc_init_table_entry sit_scc_wr0_2;
	struct scc_init_table_entry sit_scc_wr1_2;
	struct scc_init_table_entry sit_scc_wr15_3;
	struct scc_init_table_entry sit_scc_wr0_3a;
	struct scc_init_table_entry sit_scc_wr0_3b;
	struct scc_init_table_entry sit_scc_wr1_3;
	struct scc_init_table_entry sit_scc_wr9_3;
} scc_default_init_table = {
	        /* 7 6 5 4  3 2 1 0 */
	{9,  BITS( 0,0,0,0, 0,0,0,0 )},		/* sit_scc_wr9_1a  Hardware Reset */
	{4,  BITS( 0,0,1,0, 0,0,0,0 )},		/* sit_scc_wr4_1   Tx/Rx con, Async or Sync Mode */
	{1,  BITS( 0,1,1,0, 0,0,0,0 )},		/* sit_scc_wr1_1   Select W/REQ */
	{3,  BITS( 1,1,0,0, 1,0,0,0 )},		/* sit_scc_wr3_1   Select Rx Control */
	{5,  BITS( 0,1,1,0, 0,0,1,1 )},		/* sit_scc_wr5_1   Select Tx Control */
	{7,  BITS( 0,1,1,1, 1,1,1,0 )},		/* sit_scc_wr7_1   Program sync character */
	{9,  BITS( 0,0,0,0, 0,0,1,0 )},		/* sit_scc_wr9_1b  Select Interrupt Control */
	{10, BITS( 1,0,1,0, 0,0,0,0 )},		/* sit_scc_wr10_1  Misc Transmitter/Receiver Control Bits */
	{11, BITS( 0,0,0,1, 0,1,1,0 )},		/* sit_scc_wr11_1  Clock Control */
	{12, 126}, 				/* sit_scc_wr12_1  Time constant lower byte */
	{13, 0}, 				/* sit_scc_wr13_1  Time constant upper byte */
	{14, BITS( 0,0,0,0, 0,0,1,0 )},		/* sit_scc_wr14_1a Miscellaneous Control */
	{14, BITS( 0,0,0,0, 0,0,1,0 )},		/* sit_scc_wr14_1b Commands */
	{14, BITS( 0,0,0,0, 0,1,1,1 )},		/* sit_scc_wr14_2  Baud Rate Enable */
	{3,  BITS( 1,1,0,0, 1,0,0,1 )},		/* sit_scc_wr3_2   Rx Enable */ 
	{5,  BITS( 0,1,1,0, 1,0,1,1 )},		/* sit_scc_wr5_2   Tx Enable */
	{0,  BITS( 1,0,0,0, 0,0,0,0 )},		/* sit_scc_wr0_2   Reset Tx CRG */
	{1,  BITS( 1,1,1,0, 0,0,0,0 )},		/* sit_scc_wr1_2   DMA Enable */
	{15, BITS( 1,1,0,0, 0,1,0,0 )},		/* sit_scc_wr15_3  Enable External/Status */
	{0,  BITS( 0,0,0,1, 0,0,0,0 )},		/* sit_scc_wr0_3a  Reset External Status */
	{0,  BITS( 0,0,0,1, 0,0,0,0 )},		/* sit_scc_wr0_3b  Reset External Status twice */
	{1,  BITS( 1,1,1,0, 0,0,0,1 )},		/* sit_scc_wr1_3   Enable Rx, Tx and Ext/Status */
	{9,  BITS( 0,0,0,0, 1,0,1,0 )}		/* sit_scc_wr9_3   Enable Master Interrupt Enable */
	        /* 7 6 5 4  3 2 1 0 */
};

#define SCC_INIT_TABLE_SIZE (sizeof(struct scc_init_table) / sizeof(struct scc_init_table_entry))

struct dma_chan {
	int dc_status;
	u_char *dc_cmd_reg;
	int dc_addr_reg;
	int dc_cnt_reg;
	u_char dc_enable;
};

#define dc_dma_reg dc_cmd_reg

struct hsi_buf_ctl_blk {
	int hsi_next_cpu_buf;
	int hsi_next_iscc_buf;
	struct {
		int hsi_buf_status;
		int hsi_buf_len;
		u_char *hsi_buf_addr;
	} hsi_buf[2];
};

struct hsi_softc {
	struct ifnet hs_if;
	struct dma_chan hs_tx_dma_chan;
	struct dma_chan hs_rx_dma_chan;
	u_char *hs_scc_cmd_reg;
	struct hsi_buf_ctl_blk hs_tx_buf_ctl_blk;
	u_int hs_rx_buf_dma_base_addr;
	u_char *hs_rx_buf_addr;
	u_char *hs_rx_buf_end_addr;
	int hs_rx_buf_relative_sweep_count;
	int hs_rx_scc_byte_count;
	int hs_rx_dma_byte_count;
	struct syncmode hs_syncmode;
	struct hsi_regs *hs_hsi_regs;
} *hsi_softc;

#define hs_tx_dma_status hs_tx_dma_chan.dc_status
#define hs_next_cpu_tx_buf hs_tx_buf_ctl_blk.hsi_next_cpu_buf
#define hs_next_iscc_tx_buf hs_tx_buf_ctl_blk.hsi_next_iscc_buf
#define hs_tx_buf_addr(I) hs_tx_buf_ctl_blk.hsi_buf[I].hsi_buf_addr
#define hs_tx_buf_status(I) hs_tx_buf_ctl_blk.hsi_buf[I].hsi_buf_status
#define hs_tx_buf_len(I) hs_tx_buf_ctl_blk.hsi_buf[I].hsi_buf_len 

#define read_iscc_reg(wr0, reg, val) { \
	wr0 = reg; \
	val = wr0; \
}

#define write_iscc_reg(wr0, reg, val) { \
	wr0 = reg; \
	wr0 = val; \
}

#define read_dma_reg_pair(wr0, reg, val) { \
	wr0 = reg; \
	val = wr0; \
	wr0 = (reg) + 1; \
	val |= wr0 << 8; \
}

#define write_dma_reg_pair(wr0, reg, val) { \
	wr0 = reg; \
	wr0 = (val) & 0xff; \
	wr0 = (reg) + 1; \
	wr0 = (val) >> 8 & 0xff; \
}

#define start_dma(chan, addr, cnt) { \
	u_char *dma_cmd_reg; \
	u_char dma_enable; \
	dma_cmd_reg = (chan)->dc_cmd_reg; \
	read_iscc_reg(*dma_cmd_reg, 4, dma_enable); \
	dma_enable &= ~(chan)->dc_enable; \
	write_iscc_reg(*dma_cmd_reg, 4, dma_enable); \
	write_dma_reg_pair(*dma_cmd_reg, (chan)->dc_cnt_reg, cnt); \
	write_dma_reg_pair(*dma_cmd_reg, (chan)->dc_addr_reg, addr); \
	dma_enable |= (chan)->dc_enable; \
	write_iscc_reg(*dma_cmd_reg, 4, dma_enable); \
	if (*(chan)->dc_cmd_reg = 0x4, *(chan)->dc_cmd_reg != dma_enable) { \
		printf("DMA hung\n"); \
	} \
}

int hsi_init(), hsi_output(), hsi_ioctl(), hsi_reset(), hsi_poll();

extern int hz;
extern struct ifqueue ipintrq;

#ifdef HSI_DEBUG
static int hsi_debug = D_RX_ERROR | D_TX_ERROR;
#endif

#ifdef HSI_TIME
extern struct timeval time;
static struct {
	struct timeval ht_time;
	u_int ht_counter;
	char *ht_event;
} hsi_time[TIME_SIZE], *htp = hsi_time;
#endif

#ifdef HSI_DEBUG
static int hsi_interrupts = 0;
static int hsi_tx_byte_count[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static int hsi_rx_byte_count[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static int hsi_rx_dma_byte_count[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static int hsi_rx_cpu_byte_count[8] = {0, 0, 0, 0, 0, 0, 0, 0};
#endif

hsi_identify(name)
	char *name;
{
	if (strcmp(name, "HSI") == 0) {
		nhsi++;
		return (1);
	} else {
		return (0);
	}
}

hsi_attach(devp)
	struct dev_info *devp;
{
	static int cur_hsi;
	struct hsi_regs *hrp;
	struct hsi_softc *hsp;
	int i;

	if (hsi_regs == NULL) {
		hsi_regs = (struct hsi_regs *)kmem_zalloc((u_int)(nhsi * sizeof(struct hsi_regs)));
	}
	if (hsi_softc == NULL) {
		hsi_softc = (struct hsi_softc *)kmem_zalloc((u_int)(4 * nhsi * sizeof(struct hsi_softc)));
		for (hsp = &hsi_softc[0]; hsp != &hsi_softc[4 * nhsi]; ++hsp) {
			hsp->hs_if.if_name = "hsi";
			hsp->hs_if.if_unit = hsp - &hsi_softc[0];
			hsp->hs_if.if_mtu = 1504;
			hsp->hs_if.if_init = hsi_init;
	        	hsp->hs_if.if_output = hsi_output;
			hsp->hs_if.if_ioctl = hsi_ioctl;
			hsp->hs_if.if_reset = hsi_reset;
			if_attach(&hsp->hs_if);
/*
			hsp->hs_syncmode.sm_baudrate = 1544000;
*/
			hsp->hs_syncmode.sm_baudrate = 500000;
		}
	}

	devp->devi_unit = cur_hsi;
	devp->devi_data = (caddr_t)&hsi_regs[cur_hsi];
	hrp = &hsi_regs[cur_hsi];

	hrp->hr_ram =
	    (u_char (*)[2][2][2][2][BUF_SIZE])map_regs(devp->devi_reg[0].reg_addr, 0x20000, devp->devi_reg[0].reg_bustype);

	hrp->hr_iscc_regs[0] = 
	    (struct hsi_iscc_regs *)map_regs(devp->devi_reg[0].reg_addr + 0x20000, 0x20, devp->devi_reg[0].reg_bustype);

	hrp->hr_iscc_regs[1] = hsi_regs[cur_hsi].hr_iscc_regs[0] + 1;

	hrp->hr_latch_regs =
	    (struct hsi_latch_regs *)map_regs(devp->devi_reg[0].reg_addr + 0x40000, 0x80, devp->devi_reg[0].reg_bustype);

	addintr(devp->devi_intr->int_pri, hsi_poll, devp->devi_name, cur_hsi);

	hsp = &hsi_softc[4 * cur_hsi];

	for (i = 0; i < 2; ++i) {
		hsp->hs_scc_cmd_reg = &hsi_regs[cur_hsi].hr_iscc_regs[i]->hsi_scc_cmd_reg_chan_a;
		hsp->hs_tx_dma_chan.dc_cmd_reg  = &hsi_regs[cur_hsi].hr_iscc_regs[i]->hsi_dma_cmd_reg;
		hsp->hs_tx_dma_chan.dc_addr_reg = 0x14;
		hsp->hs_tx_dma_chan.dc_cnt_reg  = 0x0a;
		hsp->hs_tx_dma_chan.dc_enable   = D6;
		hsp->hs_rx_dma_chan.dc_cmd_reg  = &hsi_regs[cur_hsi].hr_iscc_regs[i]->hsi_dma_cmd_reg;
		hsp->hs_rx_dma_chan.dc_addr_reg = 0x10;
		hsp->hs_rx_dma_chan.dc_cnt_reg  = 0x08;
		hsp->hs_rx_dma_chan.dc_enable   = D7;
		hsp->hs_tx_buf_addr(0) = &((*hsi_regs[cur_hsi].hr_ram)[i][0][0][0][0]);
		hsp->hs_tx_buf_addr(1) = &((*hsi_regs[cur_hsi].hr_ram)[i][0][0][1][0]);
		hsp->hs_rx_buf_dma_base_addr = &((*hsi_regs[cur_hsi].hr_ram)[i][0][1][0][0]) - (u_char *)hrp->hr_ram;
		hsp->hs_hsi_regs = hrp;
		++hsp;
		hsp->hs_scc_cmd_reg = &hsi_regs[cur_hsi].hr_iscc_regs[i]->hsi_scc_cmd_reg_chan_b;
		hsp->hs_tx_dma_chan.dc_cmd_reg  = &hsi_regs[cur_hsi].hr_iscc_regs[i]->hsi_dma_cmd_reg;
		hsp->hs_tx_dma_chan.dc_addr_reg = 0x1c;
		hsp->hs_tx_dma_chan.dc_cnt_reg  = 0x0e;
		hsp->hs_tx_dma_chan.dc_enable   = D4;
		hsp->hs_rx_dma_chan.dc_cmd_reg  = &hsi_regs[cur_hsi].hr_iscc_regs[i]->hsi_dma_cmd_reg;
		hsp->hs_rx_dma_chan.dc_addr_reg = 0x18;
		hsp->hs_rx_dma_chan.dc_cnt_reg  = 0x0c;
		hsp->hs_rx_dma_chan.dc_enable   = D5;
		hsp->hs_tx_buf_addr(0) = &((*hsi_regs[cur_hsi].hr_ram)[i][1][0][0][0]);
		hsp->hs_tx_buf_addr(1) = &((*hsi_regs[cur_hsi].hr_ram)[i][1][0][1][0]);
		hsp->hs_rx_buf_dma_base_addr = &((*hsi_regs[cur_hsi].hr_ram)[i][1][1][0][0]) - (u_char *)hrp->hr_ram;
		hsp->hs_hsi_regs = hrp;
		++hsp;
	}

	hrp->hr_latch_regs->hsi_latch_reset_ab = 1;
	hrp->hr_latch_regs->hsi_latch_reset_cd = 1;
	hrp->hr_latch_regs->hsi_latch_enable_ext_rx_intr[0] = 0;
	hrp->hr_latch_regs->hsi_latch_enable_ext_rx_intr[1] = 0;
	hrp->hr_latch_regs->hsi_latch_enable_ext_rx_intr[2] = 0;
	hrp->hr_latch_regs->hsi_latch_enable_ext_rx_intr[3] = 0;
	hrp->hr_latch_regs->hsi_latch_reset_ext_rx_intr[0] = 1;
	hrp->hr_latch_regs->hsi_latch_reset_ext_rx_intr[1] = 1;
	hrp->hr_latch_regs->hsi_latch_reset_ext_rx_intr[2] = 1;
	hrp->hr_latch_regs->hsi_latch_reset_ext_rx_intr[3] = 1;
	hrp->hr_iscc_regs[0]->hsi_iscc_bcr = 0;
	hrp->hr_iscc_regs[1]->hsi_iscc_bcr = 0;
	hrp->hr_iscc_regs[0]->hsi_dma_cmd_reg = 0x60;
	hrp->hr_iscc_regs[1]->hsi_dma_cmd_reg = 0x60;
	hrp->hr_iscc_regs[0]->hsi_dma_cmd_reg = 1;
	hrp->hr_iscc_regs[0]->hsi_dma_cmd_reg = 0x20;
	hrp->hr_iscc_regs[1]->hsi_dma_cmd_reg = 1;
	hrp->hr_iscc_regs[1]->hsi_dma_cmd_reg = 0x20;
	hrp->hr_iscc_regs[0]->hsi_dma_cmd_reg = 5;
	hrp->hr_iscc_regs[0]->hsi_dma_cmd_reg = 0xbf;
	hrp->hr_iscc_regs[1]->hsi_dma_cmd_reg = 5;
	hrp->hr_iscc_regs[1]->hsi_dma_cmd_reg = 0xbf;
	write_iscc_reg(hrp->hr_iscc_regs[0]->hsi_scc_cmd_reg_chan_a, 9, D7 | D6);
	write_iscc_reg(hrp->hr_iscc_regs[0]->hsi_scc_cmd_reg_chan_a, 2, 0);

	report_dev(devp);
	++cur_hsi;
	return(0);
}

hsi_ioctl(ifp, cmd, ifrp)
	struct ifnet *ifp;
	int cmd;
	struct ifreq *ifrp;
{
		DEBUG(D_IOCTL, printf("hsi_ioctl: Request = %x,   Ifreq = %x %x %x %x\n", cmd, *(int *)ifrp, *((int *)ifrp + 1), *((int *)ifrp + 2), *((int *)ifrp + 3)))
	switch (cmd) {
	    case SIOCSIFFLAGS:
		hsi_init(ifp->if_unit);
		return(0);
	    case SIOCSETSYNC:
		hsi_softc[ifp->if_unit].hs_syncmode = *(struct syncmode *)ifrp->ifr_data;
		return(0);
	    case SIOCGETSYNC:
		*(struct syncmode *)ifrp->ifr_data = hsi_softc[ifp->if_unit].hs_syncmode;
		return(0);
	    default:
		return (EINVAL);
	}
}

hsi_reset(unit)
	int unit;
{
}

#define splhsi spl5

hsi_init(unit)
	int unit;
{
	struct hsi_softc *hsp;
	struct hsi_regs *hrp;
	struct scc_init_table scc_init_table;
	struct scc_init_table_entry *sitp;
	int time_constant;
	int clock_mode = 1;
	struct mbuf *m;
	int s;
	static u_char channel_reset[] = {D7, D6};

	int hsi_external_status_change_intr(), hsi_special_receive_condition_intr();

	s = splhsi();
	hsp = &hsi_softc[unit];
	hrp = &hsi_regs[unit >> 2];

	hsp->hs_if.if_flags &= ~(IFF_UP | IFF_RUNNING);

	while (hsp->hs_if.if_snd.ifq_len) {
		IF_DEQUEUE(&hsp->hs_if.if_snd, m);
		m_freem(m);
	}

	write_iscc_reg(*hsp->hs_scc_cmd_reg, 9, channel_reset[unit & 0x01]);
	hrp->hr_latch_regs->hsi_latch_enable_ext_rx_intr[unit & 0x03] = 1;
	hrp->hr_latch_regs->hsi_latch_tx_clk[unit & 0x03] = 1;

	hsp->hs_next_cpu_tx_buf = 0;
	hsp->hs_next_iscc_tx_buf = 0;
	hsp->hs_tx_buf_status(0) = 0;
	hsp->hs_tx_buf_status(1) = 0;
	hsp->hs_tx_dma_status = 0;

	bcopy(&scc_default_init_table, &scc_init_table, sizeof(struct scc_init_table));

	scc_init_table.sit_scc_wr9_1a.site_val = channel_reset[unit & 0x01];

	switch (hsp->hs_syncmode.sm_txclock) {
	    case TXC_IS_TXC:
		break;
	    case TXC_IS_RXC:
		break;
	    case TXC_IS_BAUD:
		break;
	    case TXC_IS_PLL:
		break;
	}
	switch (hsp->hs_syncmode.sm_rxclock) {
	    case RXC_IS_RXC:
		break;
	    case RXC_IS_TXC:
		break;
	    case RXC_IS_BAUD:
		break;
	    case RXC_IS_PLL:
		break;
	}

	time_constant = PCLK_FREQ / (2 * clock_mode * hsp->hs_syncmode.sm_baudrate) - 2;
	scc_init_table.sit_scc_wr12_1.site_val = time_constant & 0xff;
	scc_init_table.sit_scc_wr13_1.site_val = time_constant >> 8;

	if (hsp->hs_syncmode.sm_loopback) {
		scc_init_table.sit_scc_wr14_1a.site_val |= D4;
	}

	start_dma(&hsp->hs_rx_dma_chan, hsp->hs_rx_buf_dma_base_addr, 2 * BUF_SIZE);
	hsp->hs_rx_buf_addr = (u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr;
	hsp->hs_rx_buf_end_addr = (u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr + 2 * BUF_SIZE;
	hsp->hs_rx_buf_relative_sweep_count = 0;
	hsp->hs_rx_scc_byte_count = 0;
	hsp->hs_rx_dma_byte_count = 0;

	for (sitp = (struct scc_init_table_entry *)&scc_init_table;
	    sitp != (struct scc_init_table_entry *)&scc_init_table + SCC_INIT_TABLE_SIZE; ++sitp) {
		if (sitp->site_reg == 0) {
			*hsp->hs_scc_cmd_reg = sitp->site_val;
		} else {
			write_iscc_reg(*hsp->hs_scc_cmd_reg, sitp->site_reg, sitp->site_val);
		}
		DEBUG(D_INIT, printf("hsi_init: Write Register %d = %x\n", sitp->site_reg, sitp->site_val);)
	}

	hsp->hs_if.if_flags |= IFF_UP | IFF_RUNNING;
	splx(s);
/*
	hsi_clock();
*/
}

hsi_clock()
{
	struct hsi_softc *hsp;
	int time_constant;

	hsp = &hsi_softc[3];
	write_iscc_reg(*hsp->hs_scc_cmd_reg, 9, BITS( 0,1,0,0, 0,0,0,0 ));
	time_constant = PCLK_FREQ / 2000 - 2;
	write_iscc_reg(*hsp->hs_scc_cmd_reg, 12, time_constant & 0xff);
	write_iscc_reg(*hsp->hs_scc_cmd_reg, 13, time_constant >> 8);
	write_iscc_reg(*hsp->hs_scc_cmd_reg, 15,  BITS( 0,0,0,0, 0,0,1,0 ));
	write_iscc_reg(*hsp->hs_scc_cmd_reg, 1,  BITS( 0,0,0,0, 0,0,0,1 ));
	write_iscc_reg(*hsp->hs_scc_cmd_reg, 9,  BITS( 0,0,0,0, 1,0,1,0 ));
}

hsi_output(ifp, m0)
	struct ifnet *ifp;
	struct mbuf *m0;
{
	struct mbuf *m;
	u_char *bufp;
	int s;
	int hsi_tx_watchdog();

#define hsp ((struct hsi_softc *)ifp)

	s = splhsi();
	DEBUG(D_TX_BUFFER_STATUS, printf("hsi_output[0]: Unit = %d,   DMA Status = %x,   Buffer Status = (%x, %x),   Next CPU Buffer = %x,   Next ISCC buffer = %x\n",
	 ifp->if_unit, hsp->hs_tx_dma_status, hsp->hs_tx_buf_status(0), hsp->hs_tx_buf_status(1), hsp->hs_next_cpu_tx_buf, hsp->hs_next_iscc_tx_buf);)
	if (hsp->hs_tx_buf_status(hsp->hs_next_cpu_tx_buf) == 0) {
		bufp = hsp->hs_tx_buf_addr(hsp->hs_next_cpu_tx_buf);
		hsp->hs_tx_buf_len(hsp->hs_next_cpu_tx_buf) = 0;
		for (m = m0; m; m = m->m_next) {
			bcopy(mtod(m, u_char *), bufp, m->m_len);
 			bufp += m->m_len;
			hsp->hs_tx_buf_len(hsp->hs_next_cpu_tx_buf) += m->m_len;
		}
		m_freem(m0);
		hsp->hs_tx_buf_status(hsp->hs_next_cpu_tx_buf) = 1;
		hsp->hs_next_cpu_tx_buf ^= 1;
		if (hsp->hs_tx_dma_status == 0) {
			hsi_start_transmitter(hsp);
		}
		DEBUG(D_TX_BUFFER_STATUS, printf("hsi_output[1]: Unit = %d,   DMA Status = %x,   Buffer Status = (%x, %x),   Next CPU Buffer = %x,   Next ISCC buffer = %x\n",
			ifp->if_unit, hsp->hs_tx_dma_status, hsp->hs_tx_buf_status(0), hsp->hs_tx_buf_status(1), hsp->hs_next_cpu_tx_buf,
			 hsp->hs_next_iscc_tx_buf);)
		splx(s);
		return(0);
	} else {
		DEBUG(D_TX_BUFFER_STATUS, printf("hsi_output[2]: Unit = %d,   DMA Status = %x,   Buffer Status = (%x, %x),   Next CPU Buffer = %x,   Next ISCC buffer = %x\n",
			ifp->if_unit, hsp->hs_tx_dma_status, hsp->hs_tx_buf_status(0), hsp->hs_tx_buf_status(1), hsp->hs_next_cpu_tx_buf,
			hsp->hs_next_iscc_tx_buf);)
		if (!IF_QFULL(&ifp->if_snd)) {
			IF_ENQUEUE(&ifp->if_snd, m0);
			splx(s);
			return(0);
		} else {
			IF_DROP(&ifp->if_snd);
			splx(s);
			m_freem(m0);
			return(ENOBUFS);
		}
	}

#undef hsp
}

hsi_poll()
{
	struct hsi_regs *hrp;
	u_char interrupt_pending;
	u_char byte_counter_msb;
	u_char ext_rx_intr_status;
	int unit, i;
	int return_value = 0;
	int hsi_external_status_change_intr(), hsi_special_receive_condition_intr();
	int j;

	TIME("hsi_poll: enter");

	DEBUG(D_TRACE | D_INTERRUPT, printf("hsi_poll:\n");)

	DELAY(128);

	for (hrp = &hsi_regs[0], unit = 0; hrp != &hsi_regs[nhsi]; ++hrp) {
		ext_rx_intr_status = hrp->hr_latch_regs->hsi_latch_ext_rx_intr_status;

		for (i = 0; i < 2; ++i, unit += 2) {

			if ((ext_rx_intr_status & (1 << unit)) == 0) {
				for (j = 0; j < 16; ++j) {
					read_iscc_reg(hrp->hr_iscc_regs[i]->hsi_scc_cmd_reg_chan_a, 7, byte_counter_msb);
					if (byte_counter_msb & D6) {
						DEBUG(D_INTERRUPT, printf("hsi%d: hsi_poll: SCC RR7 = %x\n", unit, byte_counter_msb);)
						hsi_special_receive_condition_intr(unit);
						return_value = 1;
						break;
					} else {
						DELAY(128);
					}
				}
			}

			if ((ext_rx_intr_status & (1 << (unit + 1))) == 0) {
				for (j = 0; j < 16; ++j) {
					read_iscc_reg(hrp->hr_iscc_regs[i]->hsi_scc_cmd_reg_chan_b, 7, byte_counter_msb);
					if (byte_counter_msb & D6) {
						DEBUG(D_INTERRUPT, printf("hsi%d: hsi_poll: SCC RR7 = %x\n", unit + 1, byte_counter_msb);)
						hsi_special_receive_condition_intr(unit + 1);
						return_value = 1;
						break;
					} else {
						DELAY(128);
					}
				}
			}

			read_iscc_reg(hrp->hr_iscc_regs[i]->hsi_scc_cmd_reg_chan_a, 3, interrupt_pending);
			DEBUG(D_INTERRUPT, printf("hsi_poll: Interrupt Pending = %x\n", interrupt_pending);)
			if (interrupt_pending & D3) {
				hsi_external_status_change_intr(unit);
				return_value = 1;
			}
			if (interrupt_pending & D0) {
				hsi_external_status_change_intr(unit + 1);
				return_value = 1;
			}
		}
		if (hrp->hr_latch_regs->hsi_latch_status & 0x30) {
			return_value = 1;
		}
		if ((ext_rx_intr_status & 0xf) != 0) {
			DEBUG(D_INTERRUPT, printf("hsi: hsi_poll: External Receive Interrupt Status = %x\n", ext_rx_intr_status);)
			for (i = 0; i < 4; ++i) {
				if ((ext_rx_intr_status & (1 << i)) == 0) {
					hrp->hr_latch_regs->hsi_latch_reset_ext_rx_intr[i] = 1;
				}
			}
			return_value = 1;
		}
		DEBUG(D_DUMP_ISCC_ON_INTR, hsi_dump_iscc(0, 0);)
	}

#ifdef HSI_DEBUG
	if (return_value) {
		++hsi_interrupts;
	}
#endif
	TIME("hsi_poll: exit");

	return (return_value);
}

hsi_special_receive_condition_intr(unit)
	int unit;
{
	struct hsi_softc *hsp;
	u_char *dma_cmd_reg;
	u_char dma_status_reg;
	u_char special_receive_condition_status;
	u_char byte_counter_lsb;
	u_char byte_counter_msb;
	u_char dma_enable_reg;
	u_char dma_enable_reg2;
	u_short byte_counter;
	int rx_dma_cnt, rx_dma_cnt2;
	u_char *bufp;
	struct mbuf *m0, *m, *m_next;
	int mlen, len, len0;
	int s;
	int j;
/*
	u_char rx_data[16], *rx_datap;
*/
	static u_char rx_dma_abort_bit[2] = {1 << 3, 1 << 1};

	s = spl8();
	TIME("TIME: enter");
	TIME("TIME: exit");
	TIME("TIME: enter");
	TIME("TIME: exit");
	splx(s);

	TIME("hsi_special_receive_condition_intr: enter");
	DEBUG(D_TRACE, printf("hsi_special_receive_condition_intr: Unit = %d\n", unit);)

	hsp = &hsi_softc[unit];

	if ((hsp->hs_if.if_flags & (IFF_UP | IFF_RUNNING)) != (IFF_UP | IFF_RUNNING)) {
		return;
	}

	TIME("start_dma: enter");

	dma_cmd_reg = hsp->hs_rx_dma_chan.dc_cmd_reg;
	read_iscc_reg(*dma_cmd_reg, 0, dma_status_reg);
	if (dma_status_reg & rx_dma_abort_bit[unit & 0x1]) {
		printf("hsi%d: rx reset on DMA abort\n", unit);
		hsi_reset_receiver_only(hsp);
		return;
	}
	read_dma_reg_pair(*dma_cmd_reg, hsp->hs_rx_dma_chan.dc_cnt_reg, rx_dma_cnt);

	if (hsp->hs_rx_buf_relative_sweep_count > 0 && (u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr + (2 * BUF_SIZE - rx_dma_cnt)
	    > hsp->hs_rx_buf_addr) {
		/* DMA overrun */
		printf("hsi%d: rx reset on DMA overrun\n", unit);
		hsi_reset_receiver_only(hsp);
		return;
	}

	if (hsp->hs_rx_buf_relative_sweep_count == 0 && rx_dma_cnt < BUF_SIZE) {

		/* DMA buffer almost full so begin new sweep */
/*
		rx_datap = rx_data;
*/
		s = spl8();

		{ 
			int i;

			for (i = 0; i < 256; ++i) {
				if ((*hsp->hs_scc_cmd_reg & D0) == 0) {
					break;
				}
				DELAY(16);
			}
		}

		/* disable DMA */
/*
		read_iscc_reg(*dma_cmd_reg, 4, dma_enable_reg);
		dma_enable_reg &= ~hsp->hs_rx_dma_chan.dc_enable;
		write_iscc_reg(*dma_cmd_reg, 4, dma_enable_reg);
*/
		write_iscc_reg(*hsp->hs_scc_cmd_reg, 1, scc_default_init_table.sit_scc_wr1_3.site_val & 0x7f);

		/* to prevent receive overruns while the DMA is not enabled the cpu will transfer characters from the receiver */
/*
		while (*hsp->hs_scc_cmd_reg & D0) {
			read_iscc_reg(*hsp->hs_scc_cmd_reg, 8, *rx_datap++);
		}
*/

		read_dma_reg_pair(*dma_cmd_reg, hsp->hs_rx_dma_chan.dc_cnt_reg, rx_dma_cnt);
		write_dma_reg_pair(*dma_cmd_reg, hsp->hs_rx_dma_chan.dc_cnt_reg, 2 * BUF_SIZE);
		write_dma_reg_pair(*dma_cmd_reg, hsp->hs_rx_dma_chan.dc_addr_reg, hsp->hs_rx_buf_dma_base_addr);

		DEBUG(D_RX_ERROR, {
			if (rx_dma_cnt == 0) {
				printf("hsi%d: special_receive_condition_intr:  DMA TERMINAL COUNT ERROR\n", unit);
			}
		})

		/* to prevent receive overruns while the DMA is not enabled the cpu will transfer characters from the receiver */
/*
		while (*hsp->hs_scc_cmd_reg & D0) {
			read_iscc_reg(*hsp->hs_scc_cmd_reg, 8, *rx_datap++);
		}
*/

		/* re-enable DMA */
/*
		dma_enable_reg |= hsp->hs_rx_dma_chan.dc_enable; 
		write_iscc_reg(*dma_cmd_reg, 4, dma_enable_reg);
		read_iscc_reg(*dma_cmd_reg, 4, dma_enable_reg2);
		if (dma_enable_reg2 != dma_enable_reg) {
			printf("hsi%d: special_receive_condition_intr: DMA ENABLE FAILED\n", unit);
		}
*/
		write_iscc_reg(*hsp->hs_scc_cmd_reg, 1, scc_default_init_table.sit_scc_wr1_3.site_val);

		splx(s);
		hsp->hs_rx_buf_end_addr = (u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr + (2 * BUF_SIZE - rx_dma_cnt);
		hsp->hs_rx_dma_byte_count += 2 * BUF_SIZE - rx_dma_cnt;
#ifdef HSI_DEBUG
			hsi_rx_dma_byte_count[unit] += 2 * BUF_SIZE - rx_dma_cnt;
#endif
/*
		if (rx_datap - rx_data) {
			bcopy(rx_data, hsp->hs_rx_buf_end_addr, rx_datap - rx_data);
			hsp->hs_rx_buf_end_addr += rx_datap - rx_data;
#ifdef HSI_DEBUG
			hsi_rx_cpu_byte_count[unit] += rx_datap - rx_data;
#endif
		}
*/
		hsp->hs_rx_buf_relative_sweep_count++;
	} else if (hsp->hs_rx_buf_relative_sweep_count == 1 && rx_dma_cnt < BUF_SIZE) {
		printf("hsi%d: special_receive_condition_intr: DMA RELOAD FAILURE\n", unit);
	}
	TIME("start_dma: exit");

	while (1) {
		read_iscc_reg(*hsp->hs_scc_cmd_reg, 7, byte_counter_msb);
		DEBUG(D_RX_STATUS_FIFO, printf("hsi_special_receive_condition_intr: Unit = %d,   SCC: RR7 = %x\n", unit, byte_counter_msb);)
		if (byte_counter_msb & D7) {
			++hsp->hs_if.if_ierrors;
			DEBUG(D_RX_ERROR, 
				printf("hsi%d: special_receive_condition_intr:  RX ERROR SCC: RR7 = %x, RR6 = %x, RR1 = %x; bufp = %x, buf_endp = %x\n",
					unit, byte_counter_msb, byte_counter_lsb, special_receive_condition_status, bufp, hsp->hs_rx_buf_end_addr);)
			printf("hsi%d: rx reset on FIFO overflow\n", unit);
			hsi_reset_receiver_only(hsp);
			return;
		}
		if ((byte_counter_msb & D6) == 0) {

			/* status FIFO is empty so no more frames in buffer and we are done */
			read_iscc_reg(*hsp->hs_scc_cmd_reg, 1, special_receive_condition_status);
			break;
		}

		read_iscc_reg(*hsp->hs_scc_cmd_reg, 6, byte_counter_lsb);
		read_iscc_reg(*hsp->hs_scc_cmd_reg, 1, special_receive_condition_status);
		byte_counter = (byte_counter_msb & 0x3f) << 8 | byte_counter_lsb;

		DEBUG(D_RX_TRACE, printf("hsi%d: special_receive_condition_intr: packet no = %d, len = %d, buf addr = %x\n", 
			unit, hsp->hs_if.if_ipackets, byte_counter, hsp->hs_rx_buf_addr);)

#ifdef HSI_DEBUG
		hsi_rx_byte_count[unit] += byte_counter;
#endif
		hsp->hs_rx_scc_byte_count += byte_counter;
		for (j = 0; j < 16; ++j) {
			read_dma_reg_pair(*dma_cmd_reg, hsp->hs_rx_dma_chan.dc_cnt_reg, rx_dma_cnt2);
			if (hsp->hs_rx_scc_byte_count <=  hsp->hs_rx_dma_byte_count + (2 * BUF_SIZE - rx_dma_cnt2)) {
				break;
			} else {
				DELAY(256);
			}
		}
		if (hsp->hs_rx_scc_byte_count >  hsp->hs_rx_dma_byte_count + (2 * BUF_SIZE - rx_dma_cnt2)) {
			DEBUG(D_RX_ERROR, printf("hsi%d: special_receive_condition_intr: SCC byte count = %d > DMA byte count = %d\n",
				unit, hsp->hs_rx_scc_byte_count, hsp->hs_rx_dma_byte_count + (2 * BUF_SIZE - rx_dma_cnt2));)
			hsi_reset_receiver_only(hsp);
			return;
		}
		bufp = hsp->hs_rx_buf_addr;
		/* adjust pointer to next frame */
		hsp->hs_rx_buf_addr += byte_counter;
		if (hsp->hs_rx_buf_addr >= hsp->hs_rx_buf_end_addr) {
			/* frame wraps around so readjust pointer to next frame accordingly */
			hsp->hs_rx_buf_addr = (u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr
			    + (hsp->hs_rx_buf_addr - hsp->hs_rx_buf_end_addr);
			hsp->hs_rx_buf_relative_sweep_count--;
		}

		DEBUG(D_RX_STATUS_FIFO, printf("hsi%d: special_receive_condition_intr:  SCC: RR7 = %x, RR6 = %x, RR1 = %x; bufp = %x, buf_endp = %x\n",
			 unit, byte_counter_msb, byte_counter_lsb, special_receive_condition_status, bufp, hsp->hs_rx_buf_end_addr);)

		if (special_receive_condition_status & (D6 | D5)) {

			DEBUG(D_RX_ERROR, {
				printf("hsi%d: special_receive_condition_intr:  RX ERROR SCC: RR7 = %x, RR6 = %x, RR1 = %x\n",
					unit, byte_counter_msb, byte_counter_lsb, special_receive_condition_status);
				printf("        bufp = %x, len = %x, buf_endp = %x, dma_cnt = %x\n",
					bufp, byte_counter, hsp->hs_rx_buf_end_addr, rx_dma_cnt);
				printf("        buf       = %x %x %x %x ...\n",
				 	*bufp        << 24 | *(bufp + 1)  << 16 | *(bufp + 2)  << 8 | *(bufp + 3), 
					*(bufp + 4)  << 24 | *(bufp + 5)  << 16 | *(bufp + 6)  << 8 | *(bufp + 7),
					*(bufp + 8)  << 24 | *(bufp + 9)  << 16 | *(bufp + 10) << 8 | *(bufp + 11),
					*(bufp + 12) << 24 | *(bufp + 13) << 16 | *(bufp + 14) << 8 | *(bufp + 15));
				if (bufp + 16 > hsp->hs_rx_buf_end_addr) {
					printf("buf_start = %x %x %x %x ...\n",
						*(int *)((u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr),
						*(int *)((u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr + 4),
						*(int *)((u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr + 8),
						*(int *)((u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr + 12));
				}
				if (bufp + byte_counter > hsp->hs_rx_buf_end_addr) {
					printf("        RX WRAPAROUND OCCURRED IN THIS PACKET!\n");
				}
				if (hsp->hs_rx_buf_addr >= (u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr + 16) {
					bufp = hsp->hs_rx_buf_addr - 16;
				} else {
					bufp = (u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr;
				}
				printf("   next bufp - 16 = %x\n", bufp);
				printf("   next buf - 16  = %x %x %x %x ...\n",
				 	*bufp        << 24 | *(bufp + 1)  << 16 | *(bufp + 2)  << 8 | *(bufp + 3), 
					*(bufp + 4)  << 24 | *(bufp + 5)  << 16 | *(bufp + 6)  << 8 | *(bufp + 7),
					*(bufp + 8)  << 24 | *(bufp + 9)  << 16 | *(bufp + 10) << 8 | *(bufp + 11),
					*(bufp + 12) << 24 | *(bufp + 13) << 16 | *(bufp + 14) << 8 | *(bufp + 15));
			})
			DEBUG(D_DUMP_ISCC_ON_RX_ERROR, hsi_dump_iscc(0, (unit & 0x2) >> 1);)
			if (special_receive_condition_status & D6) {
				printf("hsi%d: rx CRC\n", unit);
			}
			if (special_receive_condition_status & D5) {
				printf("hsi%d: rx overrun\n", unit);
			}

			/* CRC or overrun error */
			++hsp->hs_if.if_ierrors;

			/* reset the receiver */
			hsi_reset_receiver_only(hsp);
			return;
		}

		if (byte_counter <= 2) {

			/* frames with length less than or equal to 2 cannot contain data since CRC is 2 bytes */
			++hsp->hs_if.if_ierrors;
			DEBUG(D_RX_ERROR, printf("hsi_special_receive_condition_intr: Unit = %d,   Bad Packet Length = %d\n", unit, byte_counter);)
			continue;
		}

		if ((special_receive_condition_status & (D3 | D2 | D1)) != (D2 | D1)) {

			/* frames with length not an integral multiple of 8 bits must have an error */
			++hsp->hs_if.if_ierrors;
			DEBUG(D_RX_ERROR, printf("hsi_special_receive_condition_intr: Unit = %d,   Bad Residue Code = %x\n", unit,
				special_receive_condition_status & (D3 | D2 | D1) >> 1);)
			continue;
		}
		DEBUG(D_RX_BUFFER, printf("hsi_special_receive_condition_intr: Unit = %d,   Packet Length = %d,   Contents = %x %x %x %x ...\n",
			unit, byte_counter,
			*bufp << 24 | *(bufp + 1) << 16 | *(bufp + 2) << 8 | *(bufp + 3), 
			*(bufp + 4) << 24 | *(bufp + 5) << 16 | *(bufp + 6) << 8 | *(bufp + 7),
			*(bufp + 8) << 24 | *(bufp + 9) << 16 | *(bufp + 10) << 8 | *(bufp + 11),
			*(bufp + 12) << 24 | *(bufp + 13) << 16 | *(bufp + 14) << 8 | *(bufp + 15));)
			
		/* copy packet into mbufs */
		for (m0 = 0, byte_counter -= 2; byte_counter; byte_counter -= len, bufp += len) {
			TIME("MGET: enter");
			MGET(m_next, M_DONTWAIT, MT_DATA);
			TIME("MGET: exit");
			if (m_next == 0) {
				if (m0 != 0) {
					m_freem(m0);
				}
				break;
			}
			if (m0 == 0) {
				m0 = m_next;
			} else {
				m->m_next = m_next;
			}
			m = m_next;
			if (byte_counter > 3 * MLEN) {

				/* many bytes left in this frame so use a cluster */
				TIME("MCLGET: enter");
				MCLGET(m);
				TIME("MCLGET: exit");
				if (m->m_len != MCLBYTES) {
					m_freem(m0);
					continue;
				}
				mlen = MCLBYTES;

			} else {
				mlen = MLEN;
			}
			len = min(byte_counter, mlen);
			if (bufp + len <= hsp->hs_rx_buf_end_addr) {

				/* frame is contiguous */
				TIME("bcopy: enter");
				bcopy(bufp, mtod(m, caddr_t), len);
				TIME("bcopy: exit");
			} else {

				/* frame wraps around */
				/* copy first half of frame */
				len0 = hsp->hs_rx_buf_end_addr - bufp;
				TIME("bcopy: enter");
				bcopy(bufp, mtod(m, caddr_t), len0);
				TIME("bcopy: exit");

				/* copy second half of frame */
				bufp = (u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr;
				TIME("bcopy: enter");
				bcopy(bufp, mtod(m, caddr_t) + len0, len - len0);
				TIME("bcopy: exit");
				/* adjust pointer to end of buffer */
				hsp->hs_rx_buf_end_addr = (u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr + 2 * BUF_SIZE;
			}
			m->m_len = len;
		}
		TIME("if_upper->if_input: enter");
		(*hsp->hs_if.if_upper->if_input)(hsp->hs_if.if_upper, m0);
		TIME("if_upper->if_input: exit");
		++hsp->hs_if.if_ipackets;
	}

	TIME("hsi_special_receive_condition_intr: exit");
}

hsi_reset_receiver_only(hsp)
	struct hsi_softc *hsp;
{
	u_char rx_data;

	/* reset the receiver */

	/* disable receiver and status FIFO */
	write_iscc_reg(*hsp->hs_scc_cmd_reg, 3, BITS( 1,1,0,0, 1,0,0,0 ));
	write_iscc_reg(*hsp->hs_scc_cmd_reg, 15, BITS( 1,1,0,0, 0,0,0,0 ));

	/* drain receive FIFO */
	while (*hsp->hs_scc_cmd_reg & D0) {
		read_iscc_reg(*hsp->hs_scc_cmd_reg, 8, rx_data);
		printf("%x ", rx_data);
	}

	/* restart DMA */
	start_dma(&hsp->hs_rx_dma_chan, hsp->hs_rx_buf_dma_base_addr, 2 * BUF_SIZE);

	/* enable status FIFO and receiver */
	write_iscc_reg(*hsp->hs_scc_cmd_reg, 15, BITS( 1,1,0,0, 0,1,0,0 ));
	write_iscc_reg(*hsp->hs_scc_cmd_reg, 3, BITS( 1,1,0,0, 1,0,0,1 ));

	hsp->hs_rx_buf_addr = (u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr;
	hsp->hs_rx_buf_end_addr = (u_char *)hsp->hs_hsi_regs->hr_ram + hsp->hs_rx_buf_dma_base_addr + 2 * BUF_SIZE;
	hsp->hs_rx_buf_relative_sweep_count = 0;
	hsp->hs_rx_scc_byte_count = 0;
	hsp->hs_rx_dma_byte_count = 0;
}

hsi_external_status_change_intr(unit)
	int unit;
{
	struct hsi_softc *hsp;
	u_char external_status;
	struct mbuf *m0, *m;
	u_char *bufp;
	int tx_dma_cnt;
	u_char dma_status_reg;

	int s;

	static u_char tx_dma_abort_bit[2] = {1 << 2, 1 << 0};

	DEBUG(D_EXT_STAT, printf("hsi%d: external_status_change_intr: RR0 = %x\n", unit, external_status);)
	s = spl8();
	TIME("TIME: enter");
	TIME("TIME: exit");
	TIME("TIME: enter");
	TIME("TIME: exit");
	splx(s);

	TIME("hsi_external_status_change_intr: enter");

	hsp = &hsi_softc[unit];

	external_status = *hsp->hs_scc_cmd_reg;

	DEBUG(D_EXT_STAT, printf("hsi%d: external_status_change_intr: RR0 = %x\n", unit, external_status);)

	*hsp->hs_scc_cmd_reg = D4;
	*hsp->hs_scc_cmd_reg = (D5 | D4 | D3);

	if ((hsp->hs_if.if_flags & (IFF_UP | IFF_RUNNING)) != (IFF_UP | IFF_RUNNING)) {
		return;
	}

	if (external_status & D6) {
		DEBUG(D_EXTERNAL_STATUS, printf("hsi_external_status_change_intr[0]: Unit = %d,   DMA Status = %x,   Buffer Status = (%x, %x),   Next CPU Buffer = %x,   Next ISCC buffer = %x\n",
		unit, hsp->hs_tx_dma_status, hsp->hs_tx_buf_status(0), hsp->hs_tx_buf_status(1), hsp->hs_next_cpu_tx_buf, hsp->hs_next_iscc_tx_buf);)

		read_iscc_reg(*hsp->hs_tx_dma_chan.dc_cmd_reg, 0, dma_status_reg);
		if (dma_status_reg & tx_dma_abort_bit[unit & 0x1]) {
			printf("hsi%d: tx DMA abort\n", unit);
		}

		read_dma_reg_pair(*hsp->hs_tx_dma_chan.dc_cmd_reg, hsp->hs_tx_dma_chan.dc_cnt_reg, tx_dma_cnt);
		if (tx_dma_cnt != 0) {
			++hsp->hs_if.if_oerrors;
			DEBUG(D_EXTERNAL_STATUS | D_TX_ERROR, {
				printf("hsi%d: hsi_external_status_change_intr: PREMATURE TX UNDERRUN, INCOMPLETE PACKET SENT %d bytes remaining in buffer\n", unit, tx_dma_cnt);
				printf("        packet no = %d,  len = %d\n", hsp->hs_if.if_opackets, hsp->hs_tx_buf_len(hsp->hs_next_iscc_tx_buf ^ 1));
			})
		}
		write_iscc_reg(*hsp->hs_tx_dma_chan.dc_cmd_reg, 0, BITS( 0,1,0,0, 0,0,0,0));
		untimeout(hsi_tx_watchdog, unit);
		hsp->hs_tx_dma_status = 0;
		hsp->hs_tx_buf_status(hsp->hs_next_iscc_tx_buf ^ 1) = 0;
/*
		write_iscc_reg(*hsp->hs_scc_cmd_reg, 14, BITS( 0,0,0,0, 0,0,1,1 ));
*/
		if (hsp->hs_tx_buf_status(hsp->hs_next_iscc_tx_buf)) {
			hsi_start_transmitter(hsp);
			if (hsp->hs_if.if_snd.ifq_len) {
				IF_DEQUEUE(&hsp->hs_if.if_snd, m0);
				bufp = hsp->hs_tx_buf_addr(hsp->hs_next_cpu_tx_buf);
				hsp->hs_tx_buf_len(hsp->hs_next_cpu_tx_buf) = 0;
				for (m = m0; m; m = m->m_next) {
					bcopy(mtod(m, u_char *), bufp, m->m_len);
		 			bufp += m->m_len;
					hsp->hs_tx_buf_len(hsp->hs_next_cpu_tx_buf) += m->m_len;
				}
				m_freem(m0);
				hsp->hs_tx_buf_status(hsp->hs_next_cpu_tx_buf) = 1;
				hsp->hs_next_cpu_tx_buf ^= 1;
			}
		}
		DEBUG(D_EXTERNAL_STATUS, printf("hsi_external_status_change_intr[1]: Unit = %d,   DMA Status = %x,   Buffer Status = (%x, %x),   Next CPU Buffer = %x,   Next ISCC buffer = %x\n",
		unit, hsp->hs_tx_dma_status, hsp->hs_tx_buf_status(0), hsp->hs_tx_buf_status(1), hsp->hs_next_cpu_tx_buf, hsp->hs_next_iscc_tx_buf);)
	} else if (external_status & D7) {
		++hsp->hs_if.if_ierrors;
		DEBUG(D_EXTERNAL_STATUS, printf("hsi_external_status_change_intr[2]: Unit = %d,   External Status = %x\n", unit, external_status);)
		hsi_reset_receiver_only(hsp);
	}

	TIME("hsi_external_status_change_intr: exit");
}

hsi_start_transmitter(hsp)
	struct hsi_softc *hsp;
{
	int tx_dma_cnt;
	u_char current_external_status;

	if (hsp->hs_tx_buf_len(hsp->hs_next_iscc_tx_buf) > 1504) {
		printf("tx len  = %d > 1504\n", hsp->hs_tx_buf_len(hsp->hs_next_iscc_tx_buf));
		return(-1);
	}
	/* wait until CRC is sent before resetting transmitter CRC generator */
	while (*hsp->hs_scc_cmd_reg & D2 == 0) {
		DELAY(16);
	}
	DELAY(64);
	*hsp->hs_scc_cmd_reg = D7;
	DELAY(64);
	DEBUG(D_TX_TRACE, printf("hsi%d: start_transmitter: packet no = %d, len = %d\n", 
		hsp - hsi_softc, hsp->hs_if.if_opackets, hsp->hs_tx_buf_len(hsp->hs_next_iscc_tx_buf) + 2);)
	DEBUG(D_TX_BUFFER, {
		u_char *bufp;

		bufp = hsp->hs_tx_buf_addr(hsp->hs_next_iscc_tx_buf);
		printf("hsi%d: tx started len = %d, buf_addr = %x, \nbuf = %x %x %x %x ...\n",
		hsp - hsi_softc, hsp->hs_tx_buf_len(hsp->hs_next_iscc_tx_buf), bufp, 
	 	*bufp        << 24 | *(bufp + 1)  << 16 | *(bufp + 2)  << 8 | *(bufp + 3), 
		*(bufp + 4)  << 24 | *(bufp + 5)  << 16 | *(bufp + 6)  << 8 | *(bufp + 7),
		*(bufp + 8)  << 24 | *(bufp + 9)  << 16 | *(bufp + 10) << 8 | *(bufp + 11),
		*(bufp + 12) << 24 | *(bufp + 13) << 16 | *(bufp + 14) << 8 | *(bufp + 15));
	})
	start_dma(&hsp->hs_tx_dma_chan, hsp->hs_tx_buf_addr(hsp->hs_next_iscc_tx_buf) - (u_char *)hsp->hs_hsi_regs->hr_ram,
	    hsp->hs_tx_buf_len(hsp->hs_next_iscc_tx_buf));
/*
	write_iscc_reg(*hsp->hs_scc_cmd_reg, 14, BITS( 0,0,0,0, 0,1,1,1 ));
*/
	/* wait until first character is written before resetting Tx Underrun/EOM latch */
	while (1) {
		read_dma_reg_pair(*hsp->hs_tx_dma_chan.dc_cmd_reg, hsp->hs_tx_dma_chan.dc_cnt_reg, tx_dma_cnt);
		if (tx_dma_cnt < hsp->hs_tx_buf_len(hsp->hs_next_iscc_tx_buf) - 3) {
			break;
		}
		DELAY(16);
	}
	*hsp->hs_scc_cmd_reg = (D7 | D6);
	while ((current_external_status = *hsp->hs_scc_cmd_reg) & D6) {
		DEBUG(D_TX_SPECIAL, printf("hsi_external_status_change_intr: Tx Underrun Bit Set?   Unit = %d,   Current External Status = %x\n",
			hsp - hsi_softc, current_external_status);)
		*hsp->hs_scc_cmd_reg = (D7 | D6);
		DELAY(16);
	}
	timeout(hsi_tx_watchdog, hsp - hsi_softc, 500);
	hsp->hs_tx_dma_status = 1;
#ifdef HSI_DEBUG
	hsi_tx_byte_count[hsp - hsi_softc] += hsp->hs_tx_buf_len(hsp->hs_next_iscc_tx_buf) + 2;
#endif
	hsp->hs_next_iscc_tx_buf ^= 1;
	++hsp->hs_if.if_opackets;
}

hsi_tx_watchdog(unit)
	int unit;
{
	struct hsi_softc *hsp;
	u_char external_status;

	hsp = &hsi_softc[unit];
	if (hsp->hs_tx_dma_status) {
		external_status = *hsp->hs_scc_cmd_reg;
		printf("hsi%d: Transmitter Hung,   External Status = %x,   DMA Status = %d\n", unit, external_status, hsp->hs_tx_dma_status);
		if (external_status & D6) {
			hsi_external_status_change_intr(unit);
		}
	}
}

#ifdef HSI_DEBUG
static hsi_dump_iscc(board, chip)
	int board, chip;
{
	static int scc_reg[] = {1, 2, 3, 0};
	static int dma_reg[] = {3, 0};
	int i;
	int val;
	int s;

	val = hsi_regs[board].hr_iscc_regs[chip]->hsi_scc_cmd_reg_chan_a;
	printf("Board %d,   Chip %d,   Channel A,   SCC Read Register 0 = %x\n", board, chip, val);
	for (i = 0; scc_reg[i]; ++i) {
		read_iscc_reg(hsi_regs[board].hr_iscc_regs[chip]->hsi_scc_cmd_reg_chan_a, scc_reg[i], val);
		printf("Board %d,   Chip %d,   Channel A,   SCC Read Register %d = %x\n", board, chip, scc_reg[i], val);
	}
	val = hsi_regs[board].hr_iscc_regs[chip]->hsi_scc_cmd_reg_chan_b;
	printf("Board %d,   Chip %d,   Channel B,   SCC Read Register 0 = %x\n", board, chip, val);
	for (i = 0; scc_reg[i]; ++i) {
		read_iscc_reg(hsi_regs[board].hr_iscc_regs[chip]->hsi_scc_cmd_reg_chan_b, scc_reg[i], val);
		printf("Board %d,   Chip %d,   Channel B,   SCC Read Register %d = %x\n", board, chip, scc_reg[i], val);
	}
	val = hsi_regs[board].hr_iscc_regs[chip]->hsi_dma_cmd_reg;
	printf("Board %d,   Chip %d,   DMA Read Register 0 = %x\n", board, chip, val);
	for (i = 0; dma_reg[i]; ++i) {
		read_iscc_reg(hsi_regs[board].hr_iscc_regs[chip]->hsi_dma_cmd_reg, dma_reg[i], val);
		printf("Board %d,  Chip %d,   DMA Read Register %d = %x\n", board, chip, dma_reg[i], val);
	}
}
#endif
